<?php
require_once 'admin_auth.php';
require_once '../includes/config.php';
require_once '../utils/ads_helper.php';

$success_message = '';
$error_message = '';

// Reklam istatistiklerini al
try {
    // Toplam reklam sayısı
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM ads");
    $total_ads = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Aktif reklam sayısı
    $stmt = $pdo->query("SELECT COUNT(*) as active FROM ads WHERE is_active = 1");
    $active_ads = $stmt->fetch(PDO::FETCH_ASSOC)['active'];
    
    // Detaylı reklam listesi
    $stmt = $pdo->prepare("
        SELECT 
            a.id,
            a.name,
            a.position,
            a.page_type,
            a.ad_type,
            a.is_active,
            COALESCE(SUM(s.views), 0) as total_views,
            COALESCE(SUM(s.clicks), 0) as total_clicks,
            CASE 
                WHEN COALESCE(SUM(s.views), 0) > 0 
                THEN ROUND((COALESCE(SUM(s.clicks), 0) / COALESCE(SUM(s.views), 0)) * 100, 2)
                ELSE 0 
            END as ctr
        FROM ads a
        LEFT JOIN ad_stats s ON a.id = s.ad_id
        GROUP BY a.id, a.name, a.position, a.page_type, a.ad_type, a.is_active
        ORDER BY a.created_at DESC
    ");
    $stmt->execute();
    $ad_stats = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $total_ads = 0;
    $active_ads = 0;
    $ad_stats = [];
}

// Sayfa tipleri ve pozisyonlar
$page_types = [
    'home' => 'Ana Sayfa',
    'category' => 'Kategori',
    'site_detail' => 'Site Detay',
    'blog' => 'Blog',
    'newly_added' => 'Yeni Eklenen',
    'popular' => 'Popüler',
    'add_site' => 'Site Ekle',
    'register' => 'Kayıt',
    'login' => 'Giriş',
    'search' => 'Arama'
];

$positions = [
    'header' => 'Header (Üst)',
    'footer' => 'Footer (Alt)',
    'sidebar' => 'Sidebar (Yan)',
    'content_top' => 'İçerik Üstü',
    'content_bottom' => 'İçerik Altı'
];

$ad_types = [
    'adsense' => 'Google AdSense',
    'adsterra' => 'Adsterra',
    'media_net' => 'Media.net',
    'custom' => 'Özel Kod'
];

// AJAX işlemleri
if (isset($_POST['ajax_action'])) {
    header('Content-Type: application/json');
    
    try {
        switch ($_POST['ajax_action']) {
            case 'get_ads':
                $page_type = $_POST['page_type'] ?? 'home';
                $position = $_POST['position'] ?? 'header';
                
                $stmt = $pdo->prepare("
                    SELECT * FROM ads 
                    WHERE page_type = ? AND position = ? 
                    ORDER BY display_order ASC, id ASC
                ");
                $stmt->execute([$page_type, $position]);
                $ads = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                echo json_encode(['success' => true, 'ads' => $ads]);
                break;
                
            case 'get_all_ads':
                $stmt = $pdo->query("
                    SELECT * FROM ads 
                    ORDER BY page_type ASC, position ASC, display_order ASC, id ASC
                ");
                $ads = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                echo json_encode(['success' => true, 'ads' => $ads]);
                break;
                
            case 'get_ad':
                $id = $_POST['id'] ?? null;
                if ($id) {
                    $stmt = $pdo->prepare("SELECT * FROM ads WHERE id = ?");
                    $stmt->execute([$id]);
                    $ad = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($ad) {
                        echo json_encode(['success' => true, 'ad' => $ad]);
                    } else {
                        echo json_encode(['success' => false, 'message' => 'Reklam bulunamadı']);
                    }
                } else {
                    echo json_encode(['success' => false, 'message' => 'Geçersiz reklam ID']);
                }
                break;
                
            case 'save_ad':
                $id = $_POST['id'] ?? null;
                $name = $_POST['name'] ?? '';
                $page_type = $_POST['page_type'] ?? '';
                $position = $_POST['position'] ?? '';
                $ad_code = $_POST['ad_code'] ?? '';
                $ad_type = $_POST['ad_type'] ?? 'custom';
                $is_active = isset($_POST['is_active']) ? 1 : 0;
                $display_order = $_POST['display_order'] ?? 0;
                
                // Reklam kodunu test et
                if (empty(trim($name))) {
                    echo json_encode(['success' => false, 'message' => 'Reklam adı boş olamaz']);
                    break;
                }
                
                if (empty(trim($ad_code))) {
                    echo json_encode(['success' => false, 'message' => 'Reklam kodu boş olamaz']);
                    break;
                }
                
                if ($id) {
                    // Güncelle
                    $stmt = $pdo->prepare("
                        UPDATE ads SET 
                        name = ?, page_type = ?, position = ?, ad_code = ?, 
                        ad_type = ?, is_active = ?, display_order = ?, updated_at = NOW()
                        WHERE id = ?
                    ");
                    $stmt->execute([$name, $page_type, $position, $ad_code, $ad_type, $is_active, $display_order, $id]);
                } else {
                    // Yeni ekle
                    $stmt = $pdo->prepare("
                        INSERT INTO ads (name, page_type, position, ad_code, ad_type, is_active, display_order) 
                        VALUES (?, ?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([$name, $page_type, $position, $ad_code, $ad_type, $is_active, $display_order]);
                }
                
                echo json_encode(['success' => true, 'message' => 'Reklam başarıyla kaydedildi']);
                break;
                
            case 'delete_ad':
                $id = $_POST['id'] ?? null;
                if ($id) {
                    $stmt = $pdo->prepare("DELETE FROM ads WHERE id = ?");
                    $stmt->execute([$id]);
                    echo json_encode(['success' => true, 'message' => 'Reklam silindi']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Geçersiz reklam ID']);
                }
                break;
                
            case 'toggle_ad_status':
                $id = $_POST['id'] ?? null;
                if ($id) {
                    $stmt = $pdo->prepare("UPDATE ads SET is_active = NOT is_active WHERE id = ?");
                    $stmt->execute([$id]);
                    echo json_encode(['success' => true, 'message' => 'Reklam durumu güncellendi']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Geçersiz reklam ID']);
                }
                break;
                
            default:
                echo json_encode(['success' => false, 'message' => 'Geçersiz işlem']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Hata: ' . $e->getMessage()]);
    }
    exit;
}

// Form gönderildiğinde
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !isset($_POST['ajax_action'])) {
    try {
        $pdo->beginTransaction();
        
        // Toplu güncelleme
        if (isset($_POST['bulk_update'])) {
            foreach ($_POST['ads'] as $id => $data) {
                $stmt = $pdo->prepare("
                    UPDATE ads SET 
                    name = ?, ad_code = ?, ad_type = ?, is_active = ?, display_order = ?
                    WHERE id = ?
                ");
                $stmt->execute([
                    $data['name'],
                    $data['ad_code'],
                    $data['ad_type'],
                    isset($data['is_active']) ? 1 : 0,
                    $data['display_order'],
                    $id
                ]);
            }
        }
        
        $pdo->commit();
        $success_message = "Reklamlar başarıyla güncellendi!";
        
    } catch (Exception $e) {
        $pdo->rollback();
        $error_message = 'Hata: ' . $e->getMessage();
    }
}
?>

<?php include 'menu.php'; ?>

<div class="flex-1 bg-background min-h-screen">
    <div class="p-8">
        <?php
        $page_title = "Reklam Ayarları";
        $page_subtitle = "Reklam kodlarınızı yönetin ve tüm sayfalarda görüntüleyin";
        include 'includes/admin_header.php';
        ?>
        
        <!-- İstatistikler -->
        <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
            <div class="bg-gradient-to-r from-blue-500 to-blue-600 text-white p-6 rounded-xl shadow-lg">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-blue-100 text-sm">Toplam Reklam</p>
                        <p class="text-3xl font-bold"><?php echo $total_ads; ?></p>
                    </div>
                    <div class="w-12 h-12 bg-blue-400 rounded-lg flex items-center justify-center">
                        <span class="material-symbols-outlined text-2xl">campaign</span>
                    </div>
                </div>
            </div>
            
            <div class="bg-gradient-to-r from-green-500 to-green-600 text-white p-6 rounded-xl shadow-lg">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-green-100 text-sm">Aktif Reklam</p>
                        <p class="text-3xl font-bold"><?php echo $active_ads; ?></p>
                    </div>
                    <div class="w-12 h-12 bg-green-400 rounded-lg flex items-center justify-center">
                        <span class="material-symbols-outlined text-2xl">visibility</span>
                    </div>
                </div>
            </div>
            
            <div class="bg-gradient-to-r from-purple-500 to-purple-600 text-white p-6 rounded-xl shadow-lg">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-purple-100 text-sm">Sayfa Türü</p>
                        <p class="text-3xl font-bold"><?php echo count($page_types); ?></p>
                    </div>
                    <div class="w-12 h-12 bg-purple-400 rounded-lg flex items-center justify-center">
                        <span class="material-symbols-outlined text-2xl">pages</span>
                    </div>
                </div>
            </div>
            
            <div class="bg-gradient-to-r from-orange-500 to-orange-600 text-white p-6 rounded-xl shadow-lg">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-orange-100 text-sm">Pozisyon</p>
                        <p class="text-3xl font-bold"><?php echo count($positions); ?></p>
                    </div>
                    <div class="w-12 h-12 bg-orange-400 rounded-lg flex items-center justify-center">
                        <span class="material-symbols-outlined text-2xl">place</span>
                    </div>
                </div>
            </div>
        </div>

        <?php if ($success_message): ?>
            <div class="bg-green-50 border border-green-200 text-green-800 px-6 py-4 rounded-xl mb-6">
                <div class="flex items-center gap-2">
                    <span class="material-symbols-outlined">check_circle</span>
                    <?php echo htmlspecialchars($success_message); ?>
                </div>
            </div>
        <?php endif; ?>

        <?php if ($error_message): ?>
            <div class="bg-red-50 border border-red-200 text-red-700 px-6 py-4 rounded-xl mb-6">
                <div class="flex items-center gap-2">
                    <span class="material-symbols-outlined">error</span>
                    <?php echo htmlspecialchars($error_message); ?>
                </div>
            </div>
        <?php endif; ?>

        <!-- Filtreler -->
        <div class="bg-white dark:bg-gray-800 rounded-xl shadow-lg border border-gray-200 dark:border-gray-700 p-6 mb-6">
            <div class="flex flex-wrap gap-4 items-end">
                <div class="flex-1 min-w-48">
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Sayfa Türü</label>
                    <select id="pageTypeFilter" class="w-full px-4 py-3 rounded-xl border border-gray-200 dark:border-gray-600 focus:ring-2 focus:ring-primary focus:border-primary transition-all">
                        <?php foreach ($page_types as $key => $label): ?>
                            <option value="<?php echo $key; ?>"><?php echo $label; ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="flex-1 min-w-48">
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Pozisyon</label>
                    <select id="positionFilter" class="w-full px-4 py-3 rounded-xl border border-gray-200 dark:border-gray-600 focus:ring-2 focus:ring-primary focus:border-primary transition-all">
                        <?php foreach ($positions as $key => $label): ?>
                            <option value="<?php echo $key; ?>"><?php echo $label; ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <button onclick="loadAds(true)" class="bg-primary hover:bg-blue-600 text-white px-6 py-3 rounded-xl font-semibold transition-all duration-200 flex items-center gap-2">
                    <span class="material-symbols-outlined">search</span>
                    Filtrele
                </button>
                
                <button onclick="showAddAdModal()" class="bg-green-600 hover:bg-green-700 text-white px-6 py-3 rounded-xl font-semibold transition-all duration-200 flex items-center gap-2">
                    <span class="material-symbols-outlined">add</span>
                    Yeni Reklam
                </button>
            </div>
        </div>


        <!-- Reklam Listesi -->
        <div id="adsList" class="space-y-4">
            <!-- Reklamlar buraya yüklenecek -->
        </div>

        <!-- Yeni Reklam Modal -->
        <div id="addAdModal" class="fixed inset-0 bg-black bg-opacity-50 z-50 hidden">
            <div class="flex items-center justify-center min-h-screen p-4">
                <div class="bg-white dark:bg-gray-800 rounded-xl shadow-2xl w-full max-w-2xl max-h-[90vh] overflow-y-auto">
                    <div class="p-6">
                        <div class="flex items-center justify-between mb-6">
                            <h3 class="text-xl font-semibold text-gray-900 dark:text-white">Yeni Reklam Ekle</h3>
                            <button onclick="hideAddAdModal()" class="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300">
                                <span class="material-symbols-outlined text-2xl">close</span>
                            </button>
                        </div>
                        
                        <form id="addAdForm">
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Reklam Adı</label>
                                    <input type="text" id="adName" name="name" required class="w-full px-4 py-3 rounded-xl border border-gray-200 dark:border-gray-600 focus:ring-2 focus:ring-primary focus:border-primary transition-all">
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Reklam Tipi</label>
                                    <select id="adType" name="ad_type" class="w-full px-4 py-3 rounded-xl border border-gray-200 dark:border-gray-600 focus:ring-2 focus:ring-primary focus:border-primary transition-all">
                                        <?php foreach ($ad_types as $key => $label): ?>
                                            <option value="<?php echo $key; ?>"><?php echo $label; ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Sayfa Türü</label>
                                    <select id="adPageType" name="page_type" required class="w-full px-4 py-3 rounded-xl border border-gray-200 dark:border-gray-600 focus:ring-2 focus:ring-primary focus:border-primary transition-all">
                                        <?php foreach ($page_types as $key => $label): ?>
                                            <option value="<?php echo $key; ?>"><?php echo $label; ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Pozisyon</label>
                                    <select id="adPosition" name="position" required class="w-full px-4 py-3 rounded-xl border border-gray-200 dark:border-gray-600 focus:ring-2 focus:ring-primary focus:border-primary transition-all">
                                        <?php foreach ($positions as $key => $label): ?>
                                            <option value="<?php echo $key; ?>"><?php echo $label; ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Görüntüleme Sırası</label>
                                    <input type="number" id="adDisplayOrder" name="display_order" value="0" class="w-full px-4 py-3 rounded-xl border border-gray-200 dark:border-gray-600 focus:ring-2 focus:ring-primary focus:border-primary transition-all">
                                </div>
                                
                                <div class="flex items-center">
                                    <input type="checkbox" id="adIsActive" name="is_active" checked class="w-5 h-5 text-primary border-gray-300 rounded focus:ring-primary">
                                    <label for="adIsActive" class="ml-2 text-sm text-gray-700 dark:text-gray-300">Aktif</label>
                                </div>
                            </div>
                            
                            <div class="mb-4">
                                <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Reklam Kodu</label>
                                <textarea id="adCode" name="ad_code" rows="8" required class="w-full px-4 py-3 rounded-xl border border-gray-200 dark:border-gray-600 focus:ring-2 focus:ring-primary focus:border-primary transition-all font-mono text-sm" placeholder="Reklam kodunuzu buraya yapıştırın..."></textarea>
                                <p class="text-sm text-gray-500 dark:text-gray-400 mt-2">
                                    AdSense, Adsterra, Media.net veya özel reklam kodlarınızı yapıştırabilirsiniz.
                                </p>
                            </div>
                            
                            <div class="flex justify-end gap-3">
                                <button type="button" onclick="hideAddAdModal()" class="px-6 py-3 text-gray-600 dark:text-gray-400 hover:text-gray-800 dark:hover:text-gray-200 transition-colors">
                                    İptal
                                </button>
                                <button type="submit" class="bg-primary hover:bg-blue-600 text-white px-6 py-3 rounded-xl font-semibold transition-all duration-200">
                                    Kaydet
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
let currentPageType = 'home';
let currentPosition = 'header';

// Sayfa yüklendiğinde
document.addEventListener('DOMContentLoaded', function() {
    loadAds();
    
    // Form submit
    document.getElementById('addAdForm').addEventListener('submit', function(e) {
        e.preventDefault();
        saveAd();
    });
    
    // Filtre değişiklikleri - otomatik yükleme yok
    document.getElementById('pageTypeFilter').addEventListener('change', function() {
        currentPageType = this.value;
    });
    
    document.getElementById('positionFilter').addEventListener('change', function() {
        currentPosition = this.value;
    });
});

function loadAds(useFilter = false) {
    let body;
    
    if (useFilter) {
        const pageType = document.getElementById('pageTypeFilter').value;
        const position = document.getElementById('positionFilter').value;
        body = `ajax_action=get_ads&page_type=${pageType}&position=${position}`;
    } else {
        body = `ajax_action=get_all_ads`;
    }
    
    fetch('', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: body
    })
    .then(response => response.text())
    .then(text => {
        try {
            const data = JSON.parse(text);
            if (data.success) {
                displayAds(data.ads);
            } else {
                showAlert('Hata: ' + data.message, 'error');
            }
        } catch (e) {
            showAlert('Sunucu hatası: ' + text.substring(0, 100), 'error');
        }
    })
    .catch(error => {
        showAlert('Bir hata oluştu: ' + error.message, 'error');
    });
}

function displayAds(ads) {
    const container = document.getElementById('adsList');
    
    if (ads.length === 0) {
        container.innerHTML = `
            <div class="bg-gray-50 dark:bg-gray-800 rounded-xl p-8 text-center">
                <span class="material-symbols-outlined text-4xl text-gray-400 mb-4">campaign</span>
                <h3 class="text-lg font-semibold text-gray-600 dark:text-gray-400 mb-2">Henüz reklam yok</h3>
                <p class="text-gray-500 dark:text-gray-500">Bu sayfa ve pozisyon için henüz reklam eklenmemiş.</p>
            </div>
        `;
        return;
    }
    
    let html = '';
    ads.forEach(ad => {
        const statusClass = ad.is_active ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200' : 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200';
        const statusText = ad.is_active ? 'Aktif' : 'Pasif';
        
        html += `
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow-lg border border-gray-200 dark:border-gray-700 p-6">
                <div class="flex items-start justify-between mb-4">
                    <div class="flex-1">
                        <div class="flex items-center gap-3 mb-2">
                            <h3 class="text-lg font-semibold text-gray-900 dark:text-white">${ad.name}</h3>
                            <span class="px-3 py-1 rounded-full text-xs font-medium ${statusClass}">${statusText}</span>
                        </div>
                        <div class="flex items-center gap-4 text-sm text-gray-600 dark:text-gray-400">
                            <span><strong>Tip:</strong> ${getAdTypeLabel(ad.ad_type)}</span>
                            <span><strong>Sıra:</strong> ${ad.display_order}</span>
                            <span><strong>Oluşturulma:</strong> ${new Date(ad.created_at).toLocaleDateString('tr-TR')}</span>
                        </div>
                    </div>
                    <div class="flex items-center gap-2">
                        <button onclick="toggleAdStatus(${ad.id})" class="p-2 rounded-lg ${ad.is_active ? 'bg-red-100 text-red-600 hover:bg-red-200' : 'bg-green-100 text-green-600 hover:bg-green-200'} transition-colors">
                            <span class="material-symbols-outlined text-sm">${ad.is_active ? 'visibility_off' : 'visibility'}</span>
                        </button>
                        <button onclick="editAd(${ad.id})" class="p-2 rounded-lg bg-blue-100 text-blue-600 hover:bg-blue-200 transition-colors">
                            <span class="material-symbols-outlined text-sm">edit</span>
                        </button>
                        <button onclick="deleteAd(${ad.id})" class="p-2 rounded-lg bg-red-100 text-red-600 hover:bg-red-200 transition-colors">
                            <span class="material-symbols-outlined text-sm">delete</span>
                        </button>
                    </div>
                </div>
                
                <div class="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                    <div class="flex items-center justify-between mb-2">
                        <span class="text-sm font-medium text-gray-700 dark:text-gray-300">Reklam Kodu:</span>
                        <button onclick="copyAdCode(${ad.id})" class="text-xs text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300">
                            Kopyala
                        </button>
                    </div>
                    <pre class="text-xs text-gray-600 dark:text-gray-400 overflow-x-auto whitespace-pre-wrap" id="adCode-${ad.id}">${ad.ad_code}</pre>
                </div>
            </div>
        `;
    });
    
    container.innerHTML = html;
}

function getAdTypeLabel(type) {
    const labels = {
        'adsense': 'Google AdSense',
        'adsterra': 'Adsterra',
        'media_net': 'Media.net',
        'custom': 'Özel Kod'
    };
    return labels[type] || type;
}

function showAddAdModal() {
    document.getElementById('addAdModal').classList.remove('hidden');
    document.getElementById('addAdForm').reset();
    document.getElementById('adPageType').value = currentPageType;
    document.getElementById('adPosition').value = currentPosition;
}

function hideAddAdModal() {
    document.getElementById('addAdModal').classList.add('hidden');
}

function saveAd() {
    const formData = new FormData(document.getElementById('addAdForm'));
    formData.append('ajax_action', 'save_ad');
    
    fetch('', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert(data.message, 'success');
            hideAddAdModal();
            loadAds();
        } else {
            showAlert('Hata: ' + data.message, 'error');
        }
    })
    .catch(error => {
        showAlert('Bir hata oluştu: ' + error.message, 'error');
    });
}

function editAd(id) {
    // Reklam bilgilerini al
    fetch('', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: `ajax_action=get_ad&id=${id}`
    })
    .then(response => response.text())
    .then(text => {
        try {
            const data = JSON.parse(text);
            if (data.success) {
                showEditModal(data.ad);
            } else {
                showAlert('Hata: ' + data.message, 'error');
            }
        } catch (e) {
            showAlert('Sunucu hatası: ' + text.substring(0, 100), 'error');
        }
    })
    .catch(error => {
        showAlert('Bir hata oluştu: ' + error.message, 'error');
    });
}

function deleteAd(id) {
    if (confirm('Bu reklamı silmek istediğinizden emin misiniz?')) {
        const formData = new FormData();
        formData.append('ajax_action', 'delete_ad');
        formData.append('id', id);
        
        fetch('', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert(data.message, 'success');
                loadAds();
            } else {
                showAlert('Hata: ' + data.message, 'error');
            }
        });
    }
}

function toggleAdStatus(id) {
    const formData = new FormData();
    formData.append('ajax_action', 'toggle_ad_status');
    formData.append('id', id);
    
    fetch('', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert(data.message, 'success');
            loadAds();
        } else {
            showAlert('Hata: ' + data.message, 'error');
        }
    });
}

function copyAdCode(id) {
    const codeElement = document.getElementById(`adCode-${id}`);
    const text = codeElement.textContent;
    
    navigator.clipboard.writeText(text).then(() => {
        showAlert('Reklam kodu kopyalandı!', 'success');
    }).catch(() => {
        showAlert('Kopyalama başarısız!', 'error');
    });
}

function showAlert(message, type) {
    const alertClass = type === 'success' ? 'bg-green-50 border-green-200 text-green-800' : 'bg-red-50 border-red-200 text-red-800';
    const icon = type === 'success' ? 'check_circle' : 'error';
    
    const alert = document.createElement('div');
    alert.className = `fixed top-4 right-4 ${alertClass} border px-6 py-4 rounded-xl shadow-lg z-50 flex items-center gap-2`;
    alert.innerHTML = `
        <span class="material-symbols-outlined">${icon}</span>
        ${message}
    `;
    
    document.body.appendChild(alert);
    
    setTimeout(() => {
        alert.remove();
    }, 3000);
}

function showEditModal(ad) {
    // Edit modal'ı oluştur
    const modal = document.createElement('div');
    modal.id = 'editAdModal';
    modal.className = 'fixed inset-0 bg-black bg-opacity-50 z-50';
    modal.innerHTML = `
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow-2xl w-full max-w-2xl max-h-[90vh] overflow-y-auto">
                <div class="p-6">
                    <div class="flex items-center justify-between mb-6">
                        <h3 class="text-xl font-semibold text-gray-800 dark:text-gray-200">Reklam Düzenle</h3>
                        <button onclick="hideEditModal()" class="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300">
                            <span class="material-symbols-outlined">close</span>
                        </button>
                    </div>
                    
                    <form id="editAdForm">
                        <input type="hidden" name="id" value="${ad.id}">
                        
                        <div class="space-y-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Reklam Adı</label>
                                <input type="text" name="name" value="${ad.name}" class="w-full px-4 py-3 rounded-xl border border-gray-200 dark:border-gray-600 focus:ring-2 focus:ring-primary focus:border-primary transition-all" required>
                            </div>
                            
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Sayfa Türü</label>
                                    <select name="page_type" class="w-full px-4 py-3 rounded-xl border border-gray-200 dark:border-gray-600 focus:ring-2 focus:ring-primary focus:border-primary transition-all">
                                        <option value="home" ${ad.page_type === 'home' ? 'selected' : ''}>Ana Sayfa</option>
                                        <option value="category" ${ad.page_type === 'category' ? 'selected' : ''}>Kategori</option>
                                        <option value="site_detail" ${ad.page_type === 'site_detail' ? 'selected' : ''}>Site Detay</option>
                                        <option value="blog" ${ad.page_type === 'blog' ? 'selected' : ''}>Blog</option>
                                        <option value="newly_added" ${ad.page_type === 'newly_added' ? 'selected' : ''}>Yeni Eklenen</option>
                                        <option value="popular" ${ad.page_type === 'popular' ? 'selected' : ''}>Popüler</option>
                                        <option value="add_site" ${ad.page_type === 'add_site' ? 'selected' : ''}>Site Ekle</option>
                                        <option value="register" ${ad.page_type === 'register' ? 'selected' : ''}>Kayıt</option>
                                        <option value="login" ${ad.page_type === 'login' ? 'selected' : ''}>Giriş</option>
                                        <option value="search" ${ad.page_type === 'search' ? 'selected' : ''}>Arama</option>
                                    </select>
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Pozisyon</label>
                                    <select name="position" class="w-full px-4 py-3 rounded-xl border border-gray-200 dark:border-gray-600 focus:ring-2 focus:ring-primary focus:border-primary transition-all">
                                        <option value="header" ${ad.position === 'header' ? 'selected' : ''}>Header (Üst)</option>
                                        <option value="footer" ${ad.position === 'footer' ? 'selected' : ''}>Footer (Alt)</option>
                                        <option value="sidebar" ${ad.position === 'sidebar' ? 'selected' : ''}>Sidebar (Yan)</option>
                                        <option value="content_top" ${ad.position === 'content_top' ? 'selected' : ''}>İçerik Üstü</option>
                                        <option value="content_bottom" ${ad.position === 'content_bottom' ? 'selected' : ''}>İçerik Altı</option>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Reklam Tipi</label>
                                    <select name="ad_type" class="w-full px-4 py-3 rounded-xl border border-gray-200 dark:border-gray-600 focus:ring-2 focus:ring-primary focus:border-primary transition-all">
                                        <option value="adsense" ${ad.ad_type === 'adsense' ? 'selected' : ''}>Google AdSense</option>
                                        <option value="adsterra" ${ad.ad_type === 'adsterra' ? 'selected' : ''}>Adsterra</option>
                                        <option value="media_net" ${ad.ad_type === 'media_net' ? 'selected' : ''}>Media.net</option>
                                        <option value="custom" ${ad.ad_type === 'custom' ? 'selected' : ''}>Özel Kod</option>
                                    </select>
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Görüntüleme Sırası</label>
                                    <input type="number" name="display_order" value="${ad.display_order}" class="w-full px-4 py-3 rounded-xl border border-gray-200 dark:border-gray-600 focus:ring-2 focus:ring-primary focus:border-primary transition-all">
                                </div>
                            </div>
                            
                            <div>
                                <label class="flex items-center gap-2">
                                    <input type="checkbox" name="is_active" ${ad.is_active == 1 ? 'checked' : ''} class="rounded border-gray-300 text-primary focus:ring-primary">
                                    <span class="text-sm font-medium text-gray-700 dark:text-gray-300">Aktif</span>
                                </label>
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Reklam Kodu</label>
                                <textarea name="ad_code" rows="8" class="w-full px-4 py-3 rounded-xl border border-gray-200 dark:border-gray-600 focus:ring-2 focus:ring-primary focus:border-primary transition-all" placeholder="AdSense, Adsterra, Media.net veya özel reklam kodlarınızı yapıştırabilirsiniz.">${ad.ad_code}</textarea>
                            </div>
                        </div>
                        
                        <div class="flex items-center justify-end gap-4 mt-6">
                            <button type="button" onclick="hideEditModal()" class="px-6 py-3 text-gray-600 dark:text-gray-400 hover:text-gray-800 dark:hover:text-gray-200 transition-colors">
                                İptal
                            </button>
                            <button type="submit" class="bg-primary hover:bg-blue-600 text-white px-6 py-3 rounded-xl font-semibold transition-all duration-200">
                                Güncelle
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Form submit event
    document.getElementById('editAdForm').addEventListener('submit', function(e) {
        e.preventDefault();
        updateAd();
    });
}

function hideEditModal() {
    const modal = document.getElementById('editAdModal');
    if (modal) {
        modal.remove();
    }
}

function updateAd() {
    const formData = new FormData(document.getElementById('editAdForm'));
    formData.append('ajax_action', 'save_ad');
    
    fetch('', {
        method: 'POST',
        body: formData
    })
    .then(response => response.text())
    .then(text => {
        try {
            const data = JSON.parse(text);
            if (data.success) {
                showAlert(data.message, 'success');
                hideEditModal();
                loadAds();
            } else {
                showAlert('Hata: ' + data.message, 'error');
            }
        } catch (e) {
            showAlert('Sunucu hatası: ' + text.substring(0, 100), 'error');
        }
    })
    .catch(error => {
        showAlert('Bir hata oluştu: ' + error.message, 'error');
    });
}
</script>
