<?php
session_start();
require_once __DIR__ . '/../includes/config.php';
require_once 'admin_auth.php';

if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header('Location: users.php?error=invalid_user');
    exit;
}

$user_id = intval($_GET['id']);

// Kullanıcıyı getir
$stmt = $pdo->prepare('SELECT * FROM user WHERE id = ?');
$stmt->execute([$user_id]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$user) {
    header('Location: users.php?error=user_not_found');
    exit;
}

// Admin kendini banlayamaz
if ($user['role'] === 'admin') {
    header('Location: users.php?error=cannot_ban_admin');
    exit;
}

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $reason = trim($_POST['reason'] ?? '');
    $ban_ip = isset($_POST['ban_ip']) ? 1 : 0;
    
    if (empty($reason)) {
        $error = 'Ban nedeni gereklidir.';
    } else {
        try {
            $pdo->beginTransaction();
            
            // Kullanıcıyı banla
            $stmt = $pdo->prepare('UPDATE user SET is_banned = 1 WHERE id = ?');
            $stmt->execute([$user_id]);
            
            // Ban geçmişine ekle
            $stmt = $pdo->prepare('INSERT INTO ban_history (user_id, ip_address, action, reason, admin_id) VALUES (?, ?, ?, ?, ?)');
            $stmt->execute([$user_id, $user['ip'], 'ban', $reason, $_SESSION['user_id']]);
            
            // IP'yi de banlamak isteniyorsa
            if ($ban_ip && !empty($user['ip'])) {
                $stmt = $pdo->prepare('INSERT IGNORE INTO banned_ips (ip_address, reason, banned_by) VALUES (?, ?, ?)');
                $stmt->execute([$user['ip'], 'Kullanıcı banı ile birlikte: ' . $reason, $_SESSION['user_id']]);
            }
            
            $pdo->commit();
            header('Location: users.php?success=user_banned');
            exit;
            
        } catch (Exception $e) {
            $pdo->rollback();
            $error = 'Ban işlemi sırasında hata: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="utf-8"/>
    <meta content="width=device-width, initial-scale=1.0" name="viewport"/>
    <title>Kullanıcı Banla - Admin Panel</title>
    <link href="https://fonts.googleapis.com" rel="preconnect"/>
    <link crossorigin="" href="https://fonts.gstatic.com" rel="preconnect"/>
    <link href="https://fonts.googleapis.com/css2?family=Manrope:wght@400;500;600;700;800&display=swap" rel="stylesheet"/>
    <script src="https://cdn.tailwindcss.com?plugins=forms,container-queries"></script>
    <link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined:opsz,wght,FILL,GRAD@20..48,100..700,0..1,-50..200" rel="stylesheet"/>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: "#1877f2",
                        "primary-light": "#e7f3ff",
                        background: "#f0f2f5",
                        surface: "#ffffff",
                        "text-primary": "#1c1e21",
                        "text-secondary": "#606770",
                        "line-light": "#e0e0e0",
                    },
                    fontFamily: {
                        display: ["Manrope", "sans-serif"],
                    },
                    borderRadius: {
                        DEFAULT: "0.75rem",
                        lg: "1rem",
                        xl: "1.5rem",
                        full: "9999px"
                    },
                    boxShadow: {
                        'soft': '0 4px 12px 0 rgb(0 0 0 / 0.07)',
                        'subtle': '0 1px 2px 0 rgb(0 0 0 / 0.05)',
                    }
                },
            },
        };
    </script>
    <style>
        .material-symbols-outlined {
            font-variation-settings: "FILL" 0, "wght" 400, "GRAD" 0, "opsz" 24;
        }
    </style>
</head>
<body class="bg-background font-display text-text-primary">
    <div class="flex min-h-screen">
        <?php include 'menu.php'; ?>
        <main class="flex-1 p-10 min-w-0">
            <header class="flex justify-between items-center mb-10">
                <div>
                    <h1 class="text-4xl font-extrabold text-text-primary">Kullanıcı Banla</h1>
                    <p class="text-text-secondary mt-1">Kullanıcının hesabını banla ve erişimini kısıtla.</p>
                </div>
                <a href="users.php" class="flex items-center gap-2 bg-gray-100 text-gray-700 font-bold py-3 px-5 rounded-lg hover:bg-gray-200 transition-colors">
                    <span class="material-symbols-outlined">arrow_back</span>
                    <span>Geri Dön</span>
                </a>
            </header>

            <div class="max-w-2xl">
                <!-- Kullanıcı Bilgileri -->
                <div class="bg-surface p-6 rounded-xl border border-line-light shadow-soft mb-6">
                    <h2 class="text-xl font-bold text-text-primary mb-4 flex items-center gap-2">
                        <span class="material-symbols-outlined text-red-600">person_off</span>
                        Banlanacak Kullanıcı
                    </h2>
                    <div class="flex items-center gap-4">
                        <img alt="User Avatar" class="w-16 h-16 rounded-full object-cover border-2 border-gray-100" 
                             src="<?php echo htmlspecialchars($user['profile_photo'] ? '../' . $user['profile_photo'] : '../uploads/no-logo.png'); ?>"/>
                        <div>
                            <h3 class="font-bold text-lg text-text-primary"><?php echo htmlspecialchars($user['username']); ?></h3>
                            <p class="text-text-secondary"><?php echo htmlspecialchars($user['email']); ?></p>
                            <p class="text-sm text-text-secondary">IP: <?php echo htmlspecialchars($user['ip'] ?? 'Bilinmiyor'); ?></p>
                            <p class="text-sm text-text-secondary">Kayıt: <?php echo date('d.m.Y H:i', strtotime($user['kayit_tarihi'])); ?></p>
                        </div>
                    </div>
                </div>

                <!-- Ban Formu -->
                <div class="bg-surface p-6 rounded-xl border border-line-light shadow-soft">
                    <h2 class="text-xl font-bold text-text-primary mb-6 flex items-center gap-2">
                        <span class="material-symbols-outlined text-red-600">block</span>
                        Ban Detayları
                    </h2>

                    <?php if (!empty($error)): ?>
                    <div class="bg-red-50 border-l-4 border-red-400 p-4 mb-6">
                        <div class="flex items-center">
                            <span class="material-symbols-outlined text-red-400 mr-2">error</span>
                            <p class="text-red-700"><?php echo htmlspecialchars($error); ?></p>
                        </div>
                    </div>
                    <?php endif; ?>

                    <form method="POST" class="space-y-6">
                        <div>
                            <label for="reason" class="block text-sm font-medium text-text-primary mb-2">
                                Ban Nedeni *
                            </label>
                            <textarea id="reason" name="reason" rows="4" required
                                class="w-full border border-line-light rounded-lg px-4 py-3 focus:ring-2 focus:ring-primary focus:border-transparent placeholder:text-text-secondary"
                                placeholder="Bu kullanıcının neden banlandığını açıklayın..."><?php echo htmlspecialchars($_POST['reason'] ?? ''); ?></textarea>
                        </div>

                        <?php if (!empty($user['ip'])): ?>
                        <div class="flex items-center gap-3">
                            <input type="checkbox" id="ban_ip" name="ban_ip" value="1" 
                                class="rounded border-gray-300 text-red-600 focus:ring-red-500">
                            <label for="ban_ip" class="text-sm text-text-primary">
                                IP adresini de banla (<?php echo htmlspecialchars($user['ip']); ?>)
                            </label>
                        </div>
                        <?php endif; ?>

                        <div class="bg-red-50 border border-red-200 rounded-lg p-4">
                            <div class="flex items-start gap-3">
                                <span class="material-symbols-outlined text-red-600 mt-0.5">warning</span>
                                <div>
                                    <h4 class="font-medium text-red-800 mb-1">Dikkat!</h4>
                                    <ul class="text-sm text-red-700 space-y-1">
                                        <li>• Kullanıcı siteye giriş yapabilir ancak yeni site ekleyemez</li>
                                        <li>• Ban işlemi geri alınabilir</li>
                                        <li>• IP ban seçilirse aynı IP'deki tüm kullanıcılar etkilenir</li>
                                    </ul>
                                </div>
                            </div>
                        </div>

                        <div class="flex gap-4">
                            <button type="submit" 
                                class="flex items-center gap-2 bg-red-600 text-white font-bold py-3 px-6 rounded-lg hover:bg-red-700 transition-colors">
                                <span class="material-symbols-outlined">block</span>
                                <span>Kullanıcıyı Banla</span>
                            </button>
                            <a href="users.php" 
                                class="flex items-center gap-2 bg-gray-100 text-gray-700 font-bold py-3 px-6 rounded-lg hover:bg-gray-200 transition-colors">
                                <span class="material-symbols-outlined">cancel</span>
                                <span>İptal</span>
                            </a>
                        </div>
                    </form>
                </div>
            </div>
        </main>
    </div>
</body>
</html>

