<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
require_once 'admin_auth.php';
require_once __DIR__ . '/../includes/config.php';

$page_title = 'Yeni Blog Yazısı Ekle';
$error = '';
$success = '';

// Form gönderildi mi?
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $title = trim($_POST['title'] ?? '');
    $description = $_POST['description'] ?? '';
    $excerpt = trim($_POST['excerpt'] ?? '');
    $keywords = trim($_POST['keywords'] ?? '');
    // Resim yükleme işlemi
    $featured_image = '';
    if (isset($_FILES['featured_image']) && $_FILES['featured_image']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = '../uploads/blog/';
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }
        
        $fileExtension = strtolower(pathinfo($_FILES['featured_image']['name'], PATHINFO_EXTENSION));
        $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
        
        if (in_array($fileExtension, $allowedExtensions)) {
            $fileName = uniqid() . '_' . time() . '.' . $fileExtension;
            $filePath = $uploadDir . $fileName;
            
            if (move_uploaded_file($_FILES['featured_image']['tmp_name'], $filePath)) {
                $featured_image = '/uploads/blog/' . $fileName;
            } else {
                $error = 'Resim yüklenemedi.';
            }
        } else {
            $error = 'Geçersiz dosya türü. Sadece JPG, PNG, GIF ve WebP dosyalarına izin verilir.';
        }
    }
    $status = $_POST['status'] ?? 'draft';
    $meta_title = trim($_POST['meta_title'] ?? '');
    $meta_description = trim($_POST['meta_description'] ?? '');
    $categories = $_POST['categories'] ?? [];
    $tags = $_POST['tags'] ?? [];
    
    // Slug oluştur
    $slug = createSlug($title);
    
    // Validasyon
    if (empty($title)) {
        $error = 'Blog title is required.';
    } elseif (empty($description)) {
        $error = 'Blog description is required.';
    } else {
        try {
            $pdo->beginTransaction();
            
            // Blog yazısını ekle
            $sql = "INSERT INTO blog_posts (title, slug, description, excerpt, keywords, featured_image, status, author_id, meta_title, meta_description, published_at) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $published_at = ($status === 'published') ? date('Y-m-d H:i:s') : null;
            
            // Admin user ID kontrolü
            if (!isset($_SESSION['admin_user_id'])) {
                $error = 'Admin user ID not found in session. Please login again.';
            } else {
                $stmt = $pdo->prepare($sql);
                $stmt->execute([
                    $title, $slug, $description, $excerpt, $keywords, $featured_image, 
                    $status, $_SESSION['admin_user_id'], $meta_title, $meta_description, $published_at
                ]);
                
                $post_id = $pdo->lastInsertId();
                
                // Kategorileri ekle
                if (!empty($categories)) {
                    foreach ($categories as $category_id) {
                        $stmt = $pdo->prepare("INSERT INTO blog_post_categories (post_id, category_id) VALUES (?, ?)");
                        $stmt->execute([$post_id, $category_id]);
                    }
                }
                
                // Etiketleri ekle (yoksa oluştur)
                if (!empty($tags)) {
                    foreach ($tags as $tag_name) {
                        $tag_name = trim($tag_name);
                        if (!empty($tag_name)) {
                            $tag_slug = createSlug($tag_name);
                            
                            // Etiket var mı kontrol et
                            $stmt = $pdo->prepare("SELECT id FROM blog_tags WHERE slug = ?");
                            $stmt->execute([$tag_slug]);
                            $tag = $stmt->fetch();
                            
                            if (!$tag) {
                                // Yeni etiket oluştur
                                $stmt = $pdo->prepare("INSERT INTO blog_tags (name, slug) VALUES (?, ?)");
                                $stmt->execute([$tag_name, $tag_slug]);
                                $tag_id = $pdo->lastInsertId();
                            } else {
                                $tag_id = $tag['id'];
                            }
                            
                            // Blog yazısına etiket ekle
                            $stmt = $pdo->prepare("INSERT IGNORE INTO blog_post_tags (post_id, tag_id) VALUES (?, ?)");
                            $stmt->execute([$post_id, $tag_id]);
                        }
                    }
                }
                
                $pdo->commit();
                $_SESSION['success'] = 'Blog yazısı başarıyla oluşturuldu!';
                header('Location: blog.php');
                exit;
            }
            
        } catch (Exception $e) {
            $pdo->rollback();
            $error = 'Blog yazısı oluşturulurken hata oluştu: ' . $e->getMessage();
        }
    }
}

// Slug oluşturma fonksiyonu
function createSlug($text) {
    $text = strtolower($text);
    $text = preg_replace('/[^a-z0-9\s-]/', '', $text);
    $text = preg_replace('/[\s-]+/', '-', $text);
    $text = trim($text, '-');
    return $text;
}

// Kategorileri çek
$categoriesStmt = $pdo->query("SELECT * FROM blog_categories ORDER BY name");
$allCategories = $categoriesStmt->fetchAll(PDO::FETCH_ASSOC);

// Include meta head function
include '../includes/meta_head.php';
?>
<?php include 'menu.php'; ?>

<!-- Quill Editor CSS -->
<link href="https://cdn.quilljs.com/1.3.6/quill.snow.css" rel="stylesheet">
<script src="https://cdn.quilljs.com/1.3.6/quill.min.js"></script>
<style>
    .ql-editor {
        min-height: 350px !important;
        font-family: 'Inter', system-ui, -apple-system, sans-serif !important;
        font-size: 14px !important;
        line-height: 1.6 !important;
        color: #1c1e21 !important;
        background: white !important;
    }
    .ql-toolbar {
        border-top: 1px solid #e5e7eb !important;
        border-left: 1px solid #e5e7eb !important;
        border-right: 1px solid #e5e7eb !important;
        border-bottom: none !important;
        background: #f9fafb !important;
        border-radius: 8px 8px 0 0 !important;
    }
    .ql-container {
        border-bottom: 1px solid #e5e7eb !important;
        border-left: 1px solid #e5e7eb !important;
        border-right: 1px solid #e5e7eb !important;
        border-top: none !important;
        font-family: inherit !important;
        border-radius: 0 0 8px 8px !important;
        background: white !important;
    }
    .ql-editor.ql-blank::before {
        color: #9ca3af !important;
        font-style: normal !important;
    }
    #editor {
        background: white !important;
        border-radius: 8px !important;
        overflow: hidden !important;
    }
</style>

<div class="flex-1 bg-background min-h-screen">
    <div class="p-8">
        <!-- Header -->
        <div class="flex items-center gap-4 mb-8">
            <a href="blog.php" class="inline-flex items-center justify-center w-10 h-10 bg-surface hover:bg-surface-hover rounded-xl border border-border-light transition-colors">
                <span class="material-symbols-outlined text-text-secondary">arrow_back</span>
            </a>
            <div>
                <h1 class="text-3xl font-bold text-text-primary mb-2">Yeni Blog Yazısı Ekle</h1>
                <p class="text-text-secondary">Zengin içerikli yeni bir blog yazısı oluşturun</p>
            </div>
        </div>

        <?php if ($error): ?>
            <div class="bg-red-50 border border-red-200 text-red-700 px-6 py-4 rounded-xl mb-6">
                <div class="flex items-center gap-2">
                    <span class="material-symbols-outlined">error</span>
                    <?php echo htmlspecialchars($error); ?>
                </div>
            </div>
        <?php endif; ?>

        <!-- Form -->
        <form method="POST" enctype="multipart/form-data" class="space-y-6">
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <!-- Main Content -->
                <div class="lg:col-span-2 space-y-6">
                    <!-- Title and Excerpt -->
                    <div class="bg-surface rounded-2xl shadow-soft border border-border-light p-6">
                        <h2 class="text-xl font-semibold text-text-primary mb-4">Blog Detayları</h2>
                        
                        <div class="space-y-4">
                            <div>
                                <label for="title" class="block text-sm font-medium text-text-primary mb-2">Blog Başlığı *</label>
                                <input type="text" id="title" name="title" required 
                                       value="<?php echo htmlspecialchars($_POST['title'] ?? ''); ?>"
                                       class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors"
                                       placeholder="Blog başlığını girin">
                            </div>
                            
                            <div>
                                <label for="excerpt" class="block text-sm font-medium text-text-primary mb-2">Özet</label>
                                <textarea id="excerpt" name="excerpt" rows="3"
                                          class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors"
                                          placeholder="Blog yazısının kısa açıklaması"><?php echo htmlspecialchars($_POST['excerpt'] ?? ''); ?></textarea>
                            </div>
                        </div>
                    </div>

                    <!-- Content -->
                    <div class="bg-surface rounded-2xl shadow-soft border border-border-light p-6">
                        <h2 class="text-xl font-semibold text-text-primary mb-4">İçerik</h2>
                        
                        <div class="mb-4">
                            <label for="description" class="block text-sm font-medium text-text-primary mb-2">Blog İçeriği *</label>
                            <div id="editor" style="height: 350px;"></div>
                            <textarea id="description" name="description" style="display: none;"><?php echo htmlspecialchars($_POST['description'] ?? ''); ?></textarea>
                        </div>
                    </div>
                </div>

                <!-- Sidebar -->
                <div class="space-y-6">
                    <!-- Status and Image -->
                    <div class="bg-surface rounded-2xl shadow-soft border border-border-light p-6">
                        <h2 class="text-xl font-semibold text-text-primary mb-4">Ayarlar</h2>
                        
                        <div class="space-y-4">
                            <div>
                                <label for="status" class="block text-sm font-medium text-text-primary mb-2">Durum</label>
                                <select id="status" name="status" 
                                        class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors">
                                    <option value="draft" <?php echo ($_POST['status'] ?? 'draft') === 'draft' ? 'selected' : ''; ?>>Taslak</option>
                                    <option value="published" <?php echo ($_POST['status'] ?? '') === 'published' ? 'selected' : ''; ?>>Yayınlandı</option>
                                    <option value="archived" <?php echo ($_POST['status'] ?? '') === 'archived' ? 'selected' : ''; ?>>Arşivlendi</option>
                                </select>
                            </div>
                            
                            <div>
                                <label for="featured_image" class="block text-sm font-medium text-text-primary mb-2">Öne Çıkan Resim</label>
                                <div class="space-y-3">
                                    <input type="file" id="featured_image" name="featured_image" accept="image/*" 
                                           class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors"
                                           onchange="previewImage(this)">
                                    
                                    <!-- Image Preview -->
                                    <div id="imagePreview" class="hidden">
                                        <img id="previewImg" src="" alt="Preview" class="w-full h-48 object-cover rounded-xl border border-border-light">
                                        <button type="button" onclick="removeImage()" class="mt-2 px-3 py-1 bg-red-500 text-white rounded-lg hover:bg-red-600 transition-colors text-sm">
                                            Resmi Kaldır
                                        </button>
                                    </div>
                                    
                                    <!-- Current Image Display (for editing) -->
                                    <?php if (!empty($_POST['featured_image'])): ?>
                                    <div id="currentImage">
                                        <img src="<?php echo htmlspecialchars($_POST['featured_image']); ?>" alt="Current" class="w-full h-48 object-cover rounded-xl border border-border-light">
                                        <p class="text-sm text-text-secondary mt-1">Mevcut resim</p>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- SEO -->
                    <div class="bg-surface rounded-2xl shadow-soft border border-border-light p-6">
                        <h2 class="text-xl font-semibold text-text-primary mb-4">SEO</h2>
                        
                        <div class="space-y-4">
                            <div>
                                <label for="meta_title" class="block text-sm font-medium text-text-primary mb-2">Meta Başlık</label>
                                <input type="text" id="meta_title" name="meta_title" 
                                       value="<?php echo htmlspecialchars($_POST['meta_title'] ?? ''); ?>"
                                       class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors"
                                       placeholder="Arama motorları için SEO başlığı">
                            </div>
                            
                            <div>
                                <label for="keywords" class="block text-sm font-medium text-text-primary mb-2">Anahtar Kelimeler</label>
                                <input type="text" id="keywords" name="keywords" 
                                       value="<?php echo htmlspecialchars($_POST['keywords'] ?? ''); ?>"
                                       class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors"
                                       placeholder="anahtar1, anahtar2, anahtar3">
                            </div>
                            
                            <div>
                                <label for="meta_description" class="block text-sm font-medium text-text-primary mb-2">Meta Açıklama</label>
                                <textarea id="meta_description" name="meta_description" rows="3"
                                          class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors"
                                          placeholder="Arama motorları için SEO açıklaması"><?php echo htmlspecialchars($_POST['meta_description'] ?? ''); ?></textarea>
                            </div>
                        </div>
                    </div>

                    <!-- Categories & Tags -->
                    <div class="bg-surface rounded-2xl shadow-soft border border-border-light p-6">
                        <h2 class="text-xl font-semibold text-text-primary mb-4">Kategoriler ve Etiketler</h2>
                        
                        <div class="space-y-4">
                            <div>
                                <label class="block text-sm font-medium text-text-primary mb-2">Kategoriler</label>
                                <div class="space-y-2 max-h-40 overflow-y-auto border border-border-light rounded-xl p-3">
                                    <?php foreach ($allCategories as $category): ?>
                                        <label class="flex items-center">
                                            <input type="checkbox" name="categories[]" value="<?php echo $category['id']; ?>"
                                                   <?php echo in_array($category['id'], $_POST['categories'] ?? []) ? 'checked' : ''; ?>
                                                   class="rounded border-border-light text-primary focus:ring-primary">
                                            <span class="ml-2 text-sm text-text-primary"><?php echo htmlspecialchars($category['name']); ?></span>
                                        </label>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                            
                            <div>
                                <label for="tags" class="block text-sm font-medium text-text-primary mb-2">Etiketler</label>
                                <textarea id="tags" name="tags" rows="4"
                                          class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors"
                                          placeholder="Etiketleri virgülle ayırarak girin"><?php echo htmlspecialchars(implode(', ', $_POST['tags'] ?? [])); ?></textarea>
                                <p class="text-sm text-text-secondary mt-1">Etiketleri virgülle ayırın</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Form Actions -->
            <div class="flex justify-end gap-4 pt-6 border-t border-border-light">
                <a href="blog.php" class="px-6 py-3 text-text-secondary border border-border-light rounded-xl hover:bg-surface transition-colors">
                    İptal
                </a>
                <button type="submit" class="px-6 py-3 bg-primary text-white rounded-xl hover:bg-primary-dark transition-colors font-semibold">
                    Blog Yazısı Oluştur
                </button>
            </div>
        </form>
    </div>
</div>

<script>
// Wait for DOM to be ready
document.addEventListener('DOMContentLoaded', function() {
    // Initialize Quill Editor
    var quill = new Quill('#editor', {
        theme: 'snow',
        placeholder: 'Blog içeriğinizi yazmaya başlayın...',
        modules: {
            toolbar: [
                [{ 'header': [1, 2, 3, 4, 5, 6, false] }],
                ['bold', 'italic', 'underline', 'strike'],
                [{ 'color': [] }, { 'background': [] }],
                [{ 'list': 'ordered'}, { 'list': 'bullet' }],
                [{ 'indent': '-1'}, { 'indent': '+1' }],
                [{ 'align': [] }],
                ['link', 'image', 'video'],
                ['blockquote', 'code-block'],
                ['clean']
            ]
        }
    });

    // Set initial content if editing
    <?php if (!empty($_POST['description'])): ?>
    quill.root.innerHTML = <?php echo json_encode($_POST['description']); ?>;
    <?php endif; ?>

    // Update hidden textarea before form submission
    document.querySelector('form').addEventListener('submit', function(e) {
        // Get the HTML content from Quill
        var content = quill.root.innerHTML;
        document.getElementById('description').value = content;
        
        // Validate that content is not empty
        if (quill.getText().trim().length === 0) {
            e.preventDefault();
            alert('Please enter some content for your blog post.');
            return false;
        }
    });

    // Also update on text change
    quill.on('text-change', function() {
        document.getElementById('description').value = quill.root.innerHTML;
    });

    // Process tags input
    const tagsInput = document.getElementById('tags');
    if (tagsInput) {
        tagsInput.addEventListener('blur', function() {
            var tags = this.value.split(',').map(function(tag) {
                return tag.trim();
            }).filter(function(tag) {
                return tag.length > 0;
            });
            this.value = tags.join(', ');
        });
    }
});

// Image preview functions
function previewImage(input) {
    if (input.files && input.files[0]) {
        const reader = new FileReader();
        reader.onload = function(e) {
            document.getElementById('previewImg').src = e.target.result;
            document.getElementById('imagePreview').classList.remove('hidden');
            // Hide current image if exists
            const currentImage = document.getElementById('currentImage');
            if (currentImage) {
                currentImage.style.display = 'none';
            }
        };
        reader.readAsDataURL(input.files[0]);
    }
}

function removeImage() {
    document.getElementById('featured_image').value = '';
    document.getElementById('imagePreview').classList.add('hidden');
    // Show current image again if exists
    const currentImage = document.getElementById('currentImage');
    if (currentImage) {
        currentImage.style.display = 'block';
    }
}
</script>

</body>
</html>
