<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
require_once __DIR__ . '/../includes/config.php';
if (!isset($pdo)) require_once __DIR__ . '/../includes/config.php';

// Bulk delete operation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_delete'])) {
    $category_ids = $_POST['category_ids'] ?? [];
    
    if (!empty($category_ids) && is_array($category_ids)) {
        try {
            // Güvenlik için sadece sayısal ID'leri al
            $category_ids = array_filter($category_ids, 'is_numeric');
            $category_ids = array_map('intval', $category_ids);
            
            if (!empty($category_ids)) {
                $placeholders = str_repeat('?,', count($category_ids) - 1) . '?';
                
                // Transaction başlat
                $pdo->beginTransaction();
                
                try {
                    // First delete related records from blog_post_categories table
                    $sql_posts = "DELETE FROM blog_post_categories WHERE category_id IN ($placeholders)";
                    $stmt_posts = $pdo->prepare($sql_posts);
                    $stmt_posts->execute($category_ids);
                    
                    // Then delete records from blog_categories table
                    $sql_categories = "DELETE FROM blog_categories WHERE id IN ($placeholders)";
                    $stmt_categories = $pdo->prepare($sql_categories);
                    $stmt_categories->execute($category_ids);
                    
                    // Transaction'ı commit et
                    $pdo->commit();
                    
                    $deleted_count = $stmt_categories->rowCount();
                    $_SESSION['bulk_delete_success'] = $deleted_count;
                    
                } catch (PDOException $e) {
                    // Hata durumunda transaction'ı rollback et
                    $pdo->rollback();
                    throw $e;
                }
            }
        } catch (PDOException $e) {
            $_SESSION['bulk_delete_error'] = 'Veritabanı hatası: ' . $e->getMessage();
        }
    } else {
        $_SESSION['bulk_delete_error'] = 'Silme için kategori seçilmedi.';
    }
    
    // Redirect to avoid resubmission
    header('Location: blog_categories.php');
    exit;
}

// Sayfalama ayarları
$perPage = isset($_GET['perPage']) ? intval($_GET['perPage']) : 5;
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$offset = ($page - 1) * $perPage;

// Arama ve filtreleme
$search = $_GET['search'] ?? '';
$status = $_GET['status'] ?? '';

// Create SQL query
$whereConditions = [];
$params = [];

if (!empty($search)) {
    $whereConditions[] = "(name LIKE ? OR description LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

$whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';

// Toplam kayıt sayısını al
$countSql = "SELECT COUNT(*) FROM blog_categories $whereClause";
$countStmt = $pdo->prepare($countSql);
$countStmt->execute($params);
$totalRecords = $countStmt->fetchColumn();

// Sayfalama bilgileri
$totalPages = ceil($totalRecords / $perPage);

// Kategorileri al
$sql = "SELECT bc.*, 
        (SELECT COUNT(*) FROM blog_post_categories bpc WHERE bpc.category_id = bc.id) as post_count
        FROM blog_categories bc 
        $whereClause 
        ORDER BY bc.id DESC 
        LIMIT $perPage OFFSET $offset";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="en"><head>
<meta charset="utf-8"/>
<meta content="width=device-width, initial-scale=1.0" name="viewport"/>
<title>Blog Categories Management</title>
<link href="https://fonts.googleapis.com" rel="preconnect"/>
<link crossorigin="" href="https://fonts.gstatic.com" rel="preconnect"/>
<link href="https://fonts.googleapis.com/css2?family=Manrope:wght@400;500;600;700;800&amp;display=swap" rel="stylesheet"/>
<script src="https://cdn.tailwindcss.com?plugins=forms,container-queries"></script>
<link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined:opsz,wght,FILL,GRAD@20..48,100..700,0..1,-50..200" rel="stylesheet"/>
<script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: "#1877f2", // A modern, friendly blue
                        "primary-light": "#e7f3ff",
                        background: "#f0f2f5",
                        surface: "#ffffff",
                        "text-primary": "#1c1e21",
                        "text-secondary": "#606770",
                        "line-light": "#e0e0e0",
                    },
                    fontFamily: {
                        display: ["Manrope", "sans-serif"],
                    },
                    borderRadius: {
                        DEFAULT: "0.75rem",
                        lg: "1rem",
                        xl: "1.5rem",
                        full: "9999px"
                    },
                    boxShadow: {
                        'soft': '0 4px 12px 0 rgb(0 0 0 / 0.07)',
                        'subtle': '0 1px 2px 0 rgb(0 0 0 / 0.05)',
                    }
                },
            },
        };
    </script>
</head>
<body class="bg-background font-display text-text-primary">
<div class="flex min-h-screen">
<?php include 'menu.php'; ?>
<main class="flex-1 p-10 min-w-0">
<?php
$page_title = "Blog Categories Management";
$page_subtitle = "Manage your blog categories and organize content.";
include 'includes/admin_header.php';
?>

<!-- Add New Category Button -->
<div class="flex justify-end mb-6">
    <a href="blog_category_add.php" class="inline-flex items-center px-6 py-3 bg-primary text-white rounded-xl hover:bg-primary-dark transition-colors font-semibold shadow-soft">
        <span class="material-symbols-outlined mr-2">add</span>
        Add New Category
    </a>
</div>

<!-- Search and Filter Form -->
<form method="GET" class="flex items-center gap-4 bg-surface rounded-xl shadow-soft p-6 mb-6 border border-line-light">
    <div class="flex items-center gap-2 flex-1 min-w-[220px] max-w-md">
        <div class="relative w-full">
            <span class="material-symbols-outlined absolute left-4 top-1/2 -translate-y-1/2 text-text-secondary">search</span>
            <input type="text" name="search" value="<?php echo htmlspecialchars($search); ?>" 
                   placeholder="Search by name or description..." 
                   class="w-full pl-12 pr-4 py-3 border border-line-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors">
        </div>
    </div>
    
    <button type="submit" class="px-6 py-3 bg-primary text-white rounded-xl hover:bg-primary-dark transition-colors font-semibold">
        Search
    </button>
    
    <a href="blog_categories.php" class="px-4 py-3 text-text-secondary border border-line-light rounded-xl hover:bg-surface-hover transition-colors">
        <span class="material-symbols-outlined">clear</span>
    </a>
</form>

<!-- Success/Error Messages -->
<?php if (isset($_SESSION['bulk_delete_success'])): ?>
    <div class="mb-6 p-4 bg-green-100 border border-green-200 text-green-800 rounded-xl">
        <div class="flex items-center">
            <span class="material-symbols-outlined text-green-600 mr-2">check_circle</span>
            Successfully deleted <?php echo $_SESSION['bulk_delete_success']; ?> category(ies).
        </div>
    </div>
    <?php unset($_SESSION['bulk_delete_success']); ?>
<?php endif; ?>

<?php if (isset($_SESSION['bulk_delete_error'])): ?>
    <div class="mb-6 p-4 bg-red-100 border border-red-200 text-red-800 rounded-xl">
        <div class="flex items-center">
            <span class="material-symbols-outlined text-red-600 mr-2">error</span>
            <?php echo htmlspecialchars($_SESSION['bulk_delete_error']); ?>
        </div>
    </div>
    <?php unset($_SESSION['bulk_delete_error']); ?>
<?php endif; ?>

<!-- Bulk Actions -->
<div class="bg-surface rounded-xl shadow-soft p-4 mb-6 border border-line-light" id="bulkActions" style="display: none;">
    <div class="flex items-center justify-between">
        <span class="text-text-secondary" id="selectedCount">0 categories selected</span>
        <div class="flex gap-2">
            <button onclick="bulkDelete()" class="px-4 py-2 bg-red-500 text-white rounded-lg hover:bg-red-600 transition-colors">
                <span class="material-symbols-outlined mr-1">delete</span>
                Delete Selected
            </button>
        </div>
    </div>
</div>

<!-- Categories Table -->
<div class="overflow-x-auto bg-white rounded-xl shadow-soft border border-line-light">
    <table class="w-full text-left table-fixed min-w-[1200px]">
        <thead class="border-b border-line-light">
            <tr class="bg-gradient-to-r from-primary/5 to-primary/10 border-b-2 border-primary/20">
                <th class="px-6 py-4 text-sm font-semibold text-text-primary w-12">
                    <input type="checkbox" id="selectAll" onchange="toggleSelectAll()" class="rounded border-line-light text-primary focus:ring-primary">
                </th>
                <th class="px-6 py-4 text-sm font-semibold text-text-primary">Category</th>
                <th class="px-6 py-4 text-sm font-semibold text-text-primary">Description</th>
                <th class="px-6 py-4 text-sm font-semibold text-text-primary">Color</th>
                <th class="px-6 py-4 text-sm font-semibold text-text-primary">Posts</th>
                <th class="px-6 py-4 text-sm font-semibold text-text-primary">Slug</th>
                <th class="px-6 py-4 text-sm font-semibold text-text-primary">Actions</th>
            </tr>
        </thead>
        <tbody class="divide-y divide-line-light">
            <?php if (empty($categories)): ?>
                <tr>
                    <td colspan="7" class="px-6 py-12 text-center text-text-secondary">
                        <span class="material-symbols-outlined text-4xl mb-2 block">category</span>
                        No blog categories found
                    </td>
                </tr>
            <?php else: ?>
                <?php foreach ($categories as $category): ?>
                    <tr class="hover:bg-primary/5 transition-colors">
                        <td class="px-6 py-4">
                            <input type="checkbox" name="category_ids[]" value="<?php echo $category['id']; ?>" 
                                   class="category-checkbox rounded border-line-light text-primary focus:ring-primary" 
                                   onchange="updateBulkActions()">
                        </td>
                        <td class="px-6 py-4">
                            <div class="flex items-center gap-3">
                                <div class="w-10 h-10 rounded-lg flex items-center justify-center text-white font-semibold" 
                                     style="background-color: <?php echo htmlspecialchars($category['color']); ?>">
                                    <span class="material-symbols-outlined text-lg">category</span>
                                </div>
                                <div>
                                    <div class="font-semibold text-text-primary"><?php echo htmlspecialchars($category['name']); ?></div>
                                </div>
                            </div>
                        </td>
                        <td class="px-6 py-4">
                            <div class="text-text-secondary text-sm max-w-xs truncate">
                                <?php echo htmlspecialchars($category['description'] ?? 'No description'); ?>
                            </div>
                        </td>
                        <td class="px-6 py-4">
                            <div class="flex items-center gap-2">
                                <div class="w-6 h-6 rounded-full border-2 border-white shadow-sm" 
                                     style="background-color: <?php echo htmlspecialchars($category['color']); ?>"></div>
                                <span class="text-sm text-text-secondary"><?php echo htmlspecialchars($category['color']); ?></span>
                            </div>
                        </td>
                        <td class="px-6 py-4">
                            <span class="px-2 py-1 bg-primary/10 text-primary rounded-full text-sm font-semibold">
                                <?php echo $category['post_count']; ?>
                            </span>
                        </td>
                        <td class="px-6 py-4">
                            <code class="text-sm bg-gray-100 px-2 py-1 rounded"><?php echo htmlspecialchars($category['slug']); ?></code>
                        </td>
                        <td class="px-6 py-4">
                            <div class="flex items-center gap-2">
                                <a href="blog_category_view.php?id=<?php echo $category['id']; ?>" 
                                   class="p-2 text-blue-600 hover:bg-blue-50 rounded-lg transition-colors" 
                                   title="View">
                                    <span class="material-symbols-outlined text-sm">visibility</span>
                                </a>
                                <a href="blog_category_edit.php?id=<?php echo $category['id']; ?>" 
                                   class="p-2 text-green-600 hover:bg-green-50 rounded-lg transition-colors" 
                                   title="Edit">
                                    <span class="material-symbols-outlined text-sm">edit</span>
                                </a>
                                <a href="blog_category_delete.php?id=<?php echo $category['id']; ?>" 
                                   class="p-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors" 
                                   title="Delete"
                                   onclick="return confirm('Are you sure you want to delete this category?')">
                                    <span class="material-symbols-outlined text-sm">delete</span>
                                </a>
                            </div>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<!-- Pagination -->
<?php if ($totalPages > 1): ?>
<div class="flex items-center justify-between mt-6">
    <div class="text-text-secondary text-sm">
        Showing <?php echo $offset + 1; ?> to <?php echo min($offset + $perPage, $totalRecords); ?> of <?php echo $totalRecords; ?> categories
    </div>
    
    <div class="flex items-center gap-2">
        <?php if ($page > 1): ?>
            <a href="?page=<?php echo $page - 1; ?>&perPage=<?php echo $perPage; ?>&search=<?php echo urlencode($search); ?>" 
               class="px-3 py-2 text-text-secondary border border-line-light rounded-lg hover:bg-surface-hover transition-colors">
                <span class="material-symbols-outlined text-sm">chevron_left</span>
            </a>
        <?php endif; ?>
        
        <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
            <a href="?page=<?php echo $i; ?>&perPage=<?php echo $perPage; ?>&search=<?php echo urlencode($search); ?>" 
               class="px-3 py-2 rounded-lg transition-colors <?php echo $i === $page ? 'bg-primary text-white' : 'text-text-secondary border border-line-light hover:bg-surface-hover'; ?>">
                <?php echo $i; ?>
            </a>
        <?php endfor; ?>
        
        <?php if ($page < $totalPages): ?>
            <a href="?page=<?php echo $page + 1; ?>&perPage=<?php echo $perPage; ?>&search=<?php echo urlencode($search); ?>" 
               class="px-3 py-2 text-text-secondary border border-line-light rounded-lg hover:bg-surface-hover transition-colors">
                <span class="material-symbols-outlined text-sm">chevron_right</span>
            </a>
        <?php endif; ?>
    </div>
</div>
<?php endif; ?>

<!-- Bulk Delete Modal -->
<div id="bulkDeleteModal" class="fixed inset-0 bg-black bg-opacity-50 z-50 hidden">
    <div class="flex items-center justify-center min-h-screen p-4">
        <div class="bg-white rounded-2xl shadow-2xl w-full max-w-md">
            <div class="p-6">
                <div class="flex items-center gap-3 mb-4">
                    <div class="w-12 h-12 bg-red-100 rounded-full flex items-center justify-center">
                        <span class="material-symbols-outlined text-red-600 text-xl">warning</span>
                    </div>
                    <div>
                        <h3 class="text-lg font-semibold text-text-primary">Delete Categories</h3>
                        <p class="text-text-secondary text-sm">This action cannot be undone</p>
                    </div>
                </div>
                
                <p class="text-text-secondary mb-6">
                    Are you sure you want to delete the selected categories? This will also remove them from all blog posts.
                </p>
                
                <div class="flex gap-3">
                    <button onclick="closeBulkDeleteModal()" class="flex-1 px-4 py-3 text-text-secondary border border-line-light rounded-xl hover:bg-surface-hover transition-colors">
                        Cancel
                    </button>
                    <button onclick="proceedBulkDelete()" class="flex-1 px-4 py-3 bg-red-500 text-white rounded-xl hover:bg-red-600 transition-colors">
                        Delete Categories
                    </button>
                </div>
            </div>
        </div>
    </div>
</div>

</main>
</div>

<script>
// Select All functionality
function toggleSelectAll() {
    const selectAll = document.getElementById('selectAll');
    const checkboxes = document.querySelectorAll('.category-checkbox');
    
    checkboxes.forEach(checkbox => {
        checkbox.checked = selectAll.checked;
    });
    
    updateBulkActions();
}

// Update bulk actions visibility
function updateBulkActions() {
    const checkboxes = document.querySelectorAll('.category-checkbox:checked');
    const bulkActions = document.getElementById('bulkActions');
    const selectedCount = document.getElementById('selectedCount');
    
    if (checkboxes.length > 0) {
        bulkActions.style.display = 'block';
        selectedCount.textContent = `${checkboxes.length} categories selected`;
    } else {
        bulkActions.style.display = 'none';
    }
}

// Bulk delete functions
function bulkDelete() {
    const checkboxes = document.querySelectorAll('.category-checkbox:checked');
    if (checkboxes.length === 0) {
        alert('Please select categories to delete.');
        return;
    }
    
    document.getElementById('bulkDeleteModal').classList.remove('hidden');
}

function closeBulkDeleteModal() {
    document.getElementById('bulkDeleteModal').classList.add('hidden');
}

function proceedBulkDelete() {
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = 'blog_categories.php';
    
    const checkboxes = document.querySelectorAll('.category-checkbox:checked');
    checkboxes.forEach(checkbox => {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'category_ids[]';
        input.value = checkbox.value;
        form.appendChild(input);
    });
    
    const bulkDeleteInput = document.createElement('input');
    bulkDeleteInput.type = 'hidden';
    bulkDeleteInput.name = 'bulk_delete';
    bulkDeleteInput.value = '1';
    form.appendChild(bulkDeleteInput);
    
    document.body.appendChild(form);
    form.submit();
}
</script>

</body>
</html>
