<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
require_once 'admin_auth.php';
require_once __DIR__ . '/../includes/config.php';

$page_title = 'Blog Yazısını Sil';
$error = '';
$post = null;

// Blog ID'yi al
$post_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if (!$post_id) {
    header('Location: blog.php');
    exit;
}

// Blog yazısını çek
try {
    $stmt = $pdo->prepare("SELECT bp.*, u.username as author_name FROM blog_posts bp 
                          LEFT JOIN user u ON bp.author_id = u.id 
                          WHERE bp.id = ?");
    $stmt->execute([$post_id]);
    $post = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$post) {
        header('Location: blog.php');
        exit;
    }
} catch (Exception $e) {
    $error = 'Blog yazısı yüklenirken hata oluştu: ' . $e->getMessage();
}

// Silme işlemi
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['confirm_delete'])) {
    try {
        $pdo->beginTransaction();
        
        // İlişkili kayıtları sil
        $stmt = $pdo->prepare("DELETE FROM blog_post_categories WHERE post_id = ?");
        $stmt->execute([$post_id]);
        
        $stmt = $pdo->prepare("DELETE FROM blog_post_tags WHERE post_id = ?");
        $stmt->execute([$post_id]);
        
        // Blog yazısını sil
        $stmt = $pdo->prepare("DELETE FROM blog_posts WHERE id = ?");
        $stmt->execute([$post_id]);
        
        $pdo->commit();
        $_SESSION['success'] = 'Blog yazısı başarıyla silindi!';
        header('Location: blog.php');
        exit;
        
    } catch (Exception $e) {
        $pdo->rollback();
        $error = 'Blog yazısı silinirken hata oluştu: ' . $e->getMessage();
    }
}

// Include meta head function
include '../includes/meta_head.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - Admin Panel</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined" rel="stylesheet">
    <?php echo generateMetaHead($page_title, "Blog yazısını sil"); ?>
    <style>
        .shake {
            animation: shake 0.5s;
        }
        @keyframes shake {
            0%, 100% { transform: translateX(0); }
            25% { transform: translateX(-5px); }
            75% { transform: translateX(5px); }
        }
        .pulse-border {
            animation: pulse-border 2s infinite;
        }
        @keyframes pulse-border {
            0%, 100% { border-color: #ef4444; }
            50% { border-color: #dc2626; }
        }
    </style>
</head>
<?php include 'menu.php'; ?>

<div class="flex-1 ml-80 flex items-center justify-center min-h-screen">
    <main class="p-4 w-full">
        <div class="max-w-4xl mx-auto">
            <!-- Header -->
            <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-4">
                <div class="flex items-center justify-between">
                    <div class="flex items-center">
                        <span class="material-symbols-outlined text-red-500 text-3xl mr-3">delete</span>
                        <div>
                            <h1 class="text-2xl font-bold text-gray-900">Delete Blog Post</h1>
                            <p class="text-gray-600">Permanently remove blog post from the system</p>
                        </div>
                    </div>
                    <a href="blog.php" class="inline-flex items-center px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition-colors">
                        <span class="material-symbols-outlined mr-2">arrow_back</span>
                        Back to Blog
                    </a>
                </div>
            </div>

            <?php if ($error): ?>
                <div class="bg-red-50 border border-red-200 rounded-lg p-4 mb-4 shake">
                    <div class="flex items-center">
                        <span class="material-symbols-outlined text-red-500 mr-2">error</span>
                        <p class="text-red-800"><?php echo htmlspecialchars($error); ?></p>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Blog Post Details -->
            <div class="bg-white rounded-lg shadow-sm border border-red-200 pulse-border p-4 mb-4">
                <div class="flex items-center mb-3">
                    <span class="material-symbols-outlined text-orange-500 text-xl mr-2">warning</span>
                    <h2 class="text-lg font-semibold text-gray-900">Blog Post Information</h2>
                </div>
                
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-4">
                    <div class="space-y-3">
                        <div class="flex items-center">
                            <span class="material-symbols-outlined text-gray-400 mr-2 text-sm">title</span>
                            <div>
                                <p class="text-xs text-gray-500">Title</p>
                                <p class="font-semibold text-gray-900 text-sm"><?php echo htmlspecialchars($post['title']); ?></p>
                            </div>
                        </div>
                        
                        <div class="flex items-center">
                            <span class="material-symbols-outlined text-gray-400 mr-2 text-sm">person</span>
                            <div>
                                <p class="text-xs text-gray-500">Author</p>
                                <p class="font-semibold text-gray-900 text-sm"><?php echo htmlspecialchars($post['author_name']); ?></p>
                            </div>
                        </div>
                    </div>
                    
                    <div class="space-y-3">
                        <div class="flex items-center">
                            <span class="material-symbols-outlined text-gray-400 mr-2 text-sm">visibility</span>
                            <div>
                                <p class="text-xs text-gray-500">Views</p>
                                <p class="font-semibold text-gray-900 text-sm"><?php echo number_format($post['views']); ?></p>
                            </div>
                        </div>
                        
                        <div class="flex items-center">
                            <span class="material-symbols-outlined text-gray-400 mr-2 text-sm">toggle_on</span>
                            <div>
                                <p class="text-xs text-gray-500">Status</p>
                                <?php
                                $statusColors = [
                                    'draft' => 'bg-yellow-100 text-yellow-800',
                                    'published' => 'bg-green-100 text-green-800',
                                    'archived' => 'bg-gray-100 text-gray-800'
                                ];
                                $statusColor = $statusColors[$post['status']] ?? 'bg-gray-100 text-gray-800';
                                ?>
                                <span class="px-2 py-1 text-xs font-semibold rounded-full <?php echo $statusColor; ?>">
                                    <?php echo ucfirst($post['status']); ?>
                                </span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Warning Card -->
            <div class="bg-red-50 border border-red-200 rounded-lg p-4 mb-4">
                <div class="flex items-start">
                    <span class="material-symbols-outlined text-red-500 text-xl mr-2 flex-shrink-0">dangerous</span>
                    <div>
                        <h3 class="text-lg font-semibold text-red-900 mb-2">⚠️ Warning</h3>
                        <div class="text-red-800 space-y-1 text-sm">
                            <p class="font-medium">Deleting this blog post will:</p>
                            <ul class="list-disc list-inside ml-3 space-y-1">
                                <li>Permanently remove the blog post</li>
                                <li>Delete all associated categories and tags</li>
                                <li>Remove all SEO metadata</li>
                                <li><strong>Cannot be undone</strong></li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Confirmation Form -->
            <form method="POST" class="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
                <h3 class="text-lg font-semibold text-gray-900 mb-4 flex items-center">
                    <span class="material-symbols-outlined text-red-500 mr-2">check_circle</span>
                    Confirm Deletion
                </h3>
                
                <div class="mb-6">
                    <label class="flex items-start space-x-3 cursor-pointer">
                        <input type="checkbox" id="confirm-checkbox" class="mt-1 rounded border-gray-300 text-red-600 focus:ring-red-500" required>
                        <span class="text-gray-700">
                            I understand that deleting blog post <strong>"<?php echo htmlspecialchars($post['title']); ?>"</strong> 
                            will permanently remove it and all associated data. This action cannot be undone.
                        </span>
                    </label>
                </div>
                
                <div class="flex gap-4">
                    <a href="blog.php" class="flex-1 inline-flex items-center justify-center px-6 py-3 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition-colors">
                        <span class="material-symbols-outlined mr-2">cancel</span>
                        Cancel
                    </a>
                    <button type="submit" name="confirm_delete" value="yes" id="delete-btn" 
                            class="flex-1 inline-flex items-center justify-center px-6 py-3 bg-red-600 text-white rounded-lg hover:bg-red-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                            disabled>
                        <span class="material-symbols-outlined mr-2">delete_forever</span>
                        Delete Blog Post
                    </button>
                </div>
            </form>
        </div>
    </main>
</div>

<script>
// Enable delete button only when checkbox is checked
document.getElementById('confirm-checkbox').addEventListener('change', function() {
    document.getElementById('delete-btn').disabled = !this.checked;
});

// Add extra confirmation on form submit
document.querySelector('form').addEventListener('submit', function(e) {
    if (!confirm(`Are you absolutely sure you want to delete blog post "${<?php echo json_encode($post['title']); ?>}"? This action cannot be undone!`)) {
        e.preventDefault();
    }
});
</script>

</body>
</html>

