<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
require_once 'admin_auth.php';
require_once __DIR__ . '/../includes/config.php';

$page_title = 'Blog Yazısını Düzenle';
$error = '';
$success = '';

// Blog ID'yi al
$post_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if (!$post_id) {
    header('Location: blog.php');
    exit;
}

// Blog yazısını çek
try {
    $stmt = $pdo->prepare("SELECT * FROM blog_posts WHERE id = ?");
    $stmt->execute([$post_id]);
    $post = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$post) {
        header('Location: blog.php');
        exit;
    }
    
    // Blog yazısının kategorilerini çek
    $stmt = $pdo->prepare("SELECT category_id FROM blog_post_categories WHERE post_id = ?");
    $stmt->execute([$post_id]);
    $postCategories = array_column($stmt->fetchAll(PDO::FETCH_ASSOC), 'category_id');
    
    // Blog yazısının etiketlerini çek
    $stmt = $pdo->prepare("SELECT bt.name FROM blog_post_tags bpt 
                          JOIN blog_tags bt ON bpt.tag_id = bt.id 
                          WHERE bpt.post_id = ?");
    $stmt->execute([$post_id]);
    $postTags = array_column($stmt->fetchAll(PDO::FETCH_ASSOC), 'name');
    
} catch (Exception $e) {
    $error = 'Blog yazısı yüklenirken hata oluştu: ' . $e->getMessage();
}

// Form gönderildi mi?
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $title = trim($_POST['title'] ?? '');
    $description = $_POST['description'] ?? '';
    $excerpt = trim($_POST['excerpt'] ?? '');
    $keywords = trim($_POST['keywords'] ?? '');
    $status = $_POST['status'] ?? 'draft';
    $meta_title = trim($_POST['meta_title'] ?? '');
    $meta_description = trim($_POST['meta_description'] ?? '');
    $published_at = $_POST['published_at'] ?? '';
    $categories = $_POST['categories'] ?? [];
    $tags = $_POST['tags'] ?? [];
    
    // Resim yükleme işlemi
    $featured_image = $post['featured_image']; // Mevcut resmi koru
    
    // Mevcut resmi silme kontrolü
    if (isset($_POST['remove_current_image']) && $_POST['remove_current_image'] === '1') {
        if (!empty($post['featured_image']) && file_exists('../' . $post['featured_image'])) {
            unlink('../' . $post['featured_image']);
        }
        $featured_image = '';
    }
    
    if (isset($_FILES['featured_image']) && $_FILES['featured_image']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = '../uploads/blog/';
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }
        
        $fileExtension = strtolower(pathinfo($_FILES['featured_image']['name'], PATHINFO_EXTENSION));
        $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
        
        if (in_array($fileExtension, $allowedExtensions)) {
            $fileName = uniqid() . '_' . time() . '.' . $fileExtension;
            $filePath = $uploadDir . $fileName;
            
            if (move_uploaded_file($_FILES['featured_image']['tmp_name'], $filePath)) {
                // Eski resmi sil (eğer varsa ve yeni bir resim yüklendiyse)
                if (!empty($post['featured_image']) && file_exists('../' . $post['featured_image'])) {
                    unlink('../' . $post['featured_image']);
                }
                $featured_image = '/uploads/blog/' . $fileName;
            } else {
                $error = 'Failed to upload image.';
            }
        } else {
            $error = 'Invalid file type. Only JPG, PNG, GIF, and WebP files are allowed.';
        }
    }
    
    // Slug oluştur (eğer başlık değiştiyse)
    $new_slug = createSlug($title);
    $slug = ($new_slug !== $post['slug']) ? $new_slug : $post['slug'];
    
    // Validasyon
    if (empty($title)) {
        $error = 'Blog başlığı gerekli.';
    } elseif (empty($description)) {
        $error = 'Blog açıklaması gerekli.';
    } else {
        try {
            $pdo->beginTransaction();
            
            // Blog yazısını güncelle
            $sql = "UPDATE blog_posts SET title = ?, slug = ?, description = ?, excerpt = ?, keywords = ?, 
                    featured_image = ?, status = ?, meta_title = ?, meta_description = ?, updated_at = NOW(), 
                    published_at = ? WHERE id = ?";
            
            // Tarih işleme
            if (!empty($published_at)) {
                $published_at = date('Y-m-d H:i:s', strtotime($published_at));
            } else {
                $published_at = ($status === 'published' && $post['status'] !== 'published') ? date('Y-m-d H:i:s') : $post['published_at'];
            }
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                $title, $slug, $description, $excerpt, $keywords, $featured_image, 
                $status, $meta_title, $meta_description, $published_at, $post_id
            ]);
            
            // Mevcut kategorileri sil
            $stmt = $pdo->prepare("DELETE FROM blog_post_categories WHERE post_id = ?");
            $stmt->execute([$post_id]);
            
            // Yeni kategorileri ekle
            if (!empty($categories)) {
                foreach ($categories as $category_id) {
                    $stmt = $pdo->prepare("INSERT INTO blog_post_categories (post_id, category_id) VALUES (?, ?)");
                    $stmt->execute([$post_id, $category_id]);
                }
            }
            
            // Mevcut etiketleri sil
            $stmt = $pdo->prepare("DELETE FROM blog_post_tags WHERE post_id = ?");
            $stmt->execute([$post_id]);
            
            // Yeni etiketleri ekle
            if (!empty($tags)) {
                foreach ($tags as $tag_name) {
                    $tag_name = trim($tag_name);
                    if (!empty($tag_name)) {
                        $tag_slug = createSlug($tag_name);
                        
                        // Etiket var mı kontrol et
                        $stmt = $pdo->prepare("SELECT id FROM blog_tags WHERE slug = ?");
                        $stmt->execute([$tag_slug]);
                        $tag = $stmt->fetch();
                        
                        if (!$tag) {
                            // Yeni etiket oluştur
                            $stmt = $pdo->prepare("INSERT INTO blog_tags (name, slug) VALUES (?, ?)");
                            $stmt->execute([$tag_name, $tag_slug]);
                            $tag_id = $pdo->lastInsertId();
                        } else {
                            $tag_id = $tag['id'];
                        }
                        
                        // Blog yazısına etiket ekle
                        $stmt = $pdo->prepare("INSERT INTO blog_post_tags (post_id, tag_id) VALUES (?, ?)");
                        $stmt->execute([$post_id, $tag_id]);
                    }
                }
            }
            
            $pdo->commit();
            $_SESSION['success'] = 'Blog yazısı başarıyla güncellendi!';
            header('Location: blog.php');
            exit;
            
        } catch (Exception $e) {
            $pdo->rollback();
            $error = 'Blog yazısı güncellenirken hata oluştu: ' . $e->getMessage();
        }
    }
}

// Slug oluşturma fonksiyonu
function createSlug($text) {
    $text = strtolower($text);
    $text = preg_replace('/[^a-z0-9\s-]/', '', $text);
    $text = preg_replace('/[\s-]+/', '-', $text);
    $text = trim($text, '-');
    return $text;
}

// Kategorileri çek
$categoriesStmt = $pdo->query("SELECT * FROM blog_categories ORDER BY name");
$allCategories = $categoriesStmt->fetchAll(PDO::FETCH_ASSOC);

// Include meta head function
include '../includes/meta_head.php';
?>
<?php include 'menu.php'; ?>

<!-- Quill Editor CSS -->
<link href="https://cdn.quilljs.com/1.3.6/quill.snow.css" rel="stylesheet">
<script src="https://cdn.quilljs.com/1.3.6/quill.min.js"></script>
<style>
    .ql-editor {
        min-height: 350px !important;
        font-family: 'Inter', system-ui, -apple-system, sans-serif !important;
        font-size: 14px !important;
        line-height: 1.6 !important;
        color: #1c1e21 !important;
        background: white !important;
    }
    .ql-toolbar {
        border-top: 1px solid #e5e7eb !important;
        border-left: 1px solid #e5e7eb !important;
        border-right: 1px solid #e5e7eb !important;
        border-bottom: none !important;
        background: #f9fafb !important;
        border-radius: 8px 8px 0 0 !important;
    }
    .ql-container {
        border-bottom: 1px solid #e5e7eb !important;
        border-left: 1px solid #e5e7eb !important;
        border-right: 1px solid #e5e7eb !important;
        border-top: none !important;
        font-family: inherit !important;
        border-radius: 0 0 8px 8px !important;
        background: white !important;
    }
    .ql-editor.ql-blank::before {
        color: #9ca3af !important;
        font-style: normal !important;
    }
    #editor {
        background: white !important;
        border-radius: 8px !important;
        overflow: hidden !important;
    }
</style>

<div class="flex-1 bg-background min-h-screen">
    <div class="p-8">
        <!-- Header -->
        <div class="flex items-center gap-4 mb-8">
            <a href="blog.php" class="inline-flex items-center justify-center w-10 h-10 bg-surface hover:bg-surface-hover rounded-xl border border-border-light transition-colors">
                <span class="material-symbols-outlined text-text-secondary">arrow_back</span>
            </a>
            <div>
                <h1 class="text-3xl font-bold text-text-primary mb-2">Edit Blog Post</h1>
                <p class="text-text-secondary">Update your blog post content</p>
            </div>
        </div>

        <?php if ($error): ?>
            <div class="bg-red-50 border border-red-200 text-red-700 px-6 py-4 rounded-xl mb-6">
                <div class="flex items-center gap-2">
                    <span class="material-symbols-outlined">error</span>
                    <?php echo htmlspecialchars($error); ?>
                </div>
            </div>
        <?php endif; ?>

        <!-- Form -->
        <form method="POST" enctype="multipart/form-data" class="space-y-6">
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <!-- Main Content -->
                <div class="lg:col-span-2 space-y-6">
                    <!-- Title and Excerpt -->
                    <div class="bg-surface rounded-2xl shadow-soft border border-border-light p-6">
                        <h2 class="text-xl font-semibold text-text-primary mb-4">Blog Details</h2>
                    
                        <div class="space-y-4">
                            <div>
                                <label for="title" class="block text-sm font-medium text-text-primary mb-2">Blog Title *</label>
                                <input type="text" id="title" name="title" required 
                                       value="<?php echo htmlspecialchars($_POST['title'] ?? $post['title']); ?>"
                                       class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors"
                                       placeholder="Enter blog title">
                            </div>
                            
                            <div>
                                <label for="excerpt" class="block text-sm font-medium text-text-primary mb-2">Excerpt</label>
                                <textarea id="excerpt" name="excerpt" rows="3"
                                          class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors"
                                          placeholder="Brief description of the blog post"><?php echo htmlspecialchars($_POST['excerpt'] ?? $post['excerpt']); ?></textarea>
                            </div>
                        </div>
                    </div>

                    <!-- Content -->
                    <div class="bg-surface rounded-2xl shadow-soft border border-border-light p-6">
                        <h2 class="text-xl font-semibold text-text-primary mb-4">Content</h2>
                        
                        <div class="mb-4">
                            <label for="description" class="block text-sm font-medium text-text-primary mb-2">Blog Content *</label>
                            <div id="editor" style="height: 350px;"></div>
                            <textarea id="description" name="description" style="display: none;"><?php echo htmlspecialchars($_POST['description'] ?? $post['description']); ?></textarea>
                        </div>
                    </div>
                </div>

                <!-- Sidebar -->
                <div class="space-y-6">
                    <!-- Status and Image -->
                    <div class="bg-surface rounded-2xl shadow-soft border border-border-light p-6">
                        <h2 class="text-xl font-semibold text-text-primary mb-4">Settings</h2>
                        
                        <div class="space-y-4">
                            <div>
                                <label for="status" class="block text-sm font-medium text-text-primary mb-2">Status</label>
                                <select id="status" name="status" 
                                        class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors">
                                    <option value="draft" <?php echo ($_POST['status'] ?? $post['status']) === 'draft' ? 'selected' : ''; ?>>Draft</option>
                                    <option value="published" <?php echo ($_POST['status'] ?? $post['status']) === 'published' ? 'selected' : ''; ?>>Published</option>
                                    <option value="archived" <?php echo ($_POST['status'] ?? $post['status']) === 'archived' ? 'selected' : ''; ?>>Archived</option>
                                </select>
                            </div>
                            
                            <div>
                                <label for="published_at" class="block text-sm font-medium text-text-primary mb-2">Published Date</label>
                                <input type="datetime-local" id="published_at" name="published_at" 
                                       value="<?php echo isset($_POST['published_at']) ? $_POST['published_at'] : (isset($post['published_at']) && $post['published_at'] ? date('Y-m-d\TH:i', strtotime($post['published_at'])) : ''); ?>"
                                       class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors">
                                <p class="text-xs text-text-secondary mt-1">Leave empty to use current date when publishing</p>
                            </div>
                            
                            <div>
                                <label for="featured_image" class="block text-sm font-medium text-text-primary mb-2">Featured Image</label>
                                <div class="space-y-4">
                                    <!-- Current Image Preview -->
                                    <?php if (!empty($post['featured_image'])): ?>
                                        <div class="current-image-preview">
                                            <p class="text-sm text-text-secondary mb-2">Current Image:</p>
                                            <div class="relative inline-block">
                                                <img src="<?php echo htmlspecialchars($post['featured_image']); ?>" 
                                                     alt="Current featured image" 
                                                     class="w-32 h-32 object-cover rounded-lg border border-border-light">
                                                <button type="button" onclick="removeCurrentImage()" 
                                                        class="absolute -top-2 -right-2 bg-red-500 text-white rounded-full w-6 h-6 flex items-center justify-center text-xs hover:bg-red-600 transition-colors">
                                                    ×
                                                </button>
                                            </div>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <!-- File Upload -->
                                    <div class="file-upload-area">
                                        <input type="file" id="featured_image" name="featured_image" 
                                               accept="image/*" 
                                               class="hidden" 
                                               onchange="previewImage(this)">
                                        <label for="featured_image" 
                                               class="flex flex-col items-center justify-center w-full h-32 border-2 border-dashed border-border-light rounded-xl cursor-pointer hover:bg-surface-hover transition-colors">
                                            <div class="flex flex-col items-center justify-center pt-5 pb-6">
                                                <span class="material-symbols-outlined text-4xl text-text-muted mb-2">cloud_upload</span>
                                                <p class="mb-2 text-sm text-text-secondary">
                                                    <span class="font-semibold">Click to upload</span> or drag and drop
                                                </p>
                                                <p class="text-xs text-text-muted">PNG, JPG, GIF, WebP (MAX. 10MB)</p>
                                            </div>
                                        </label>
                                    </div>
                                    
                                    <!-- Image Preview -->
                                    <div id="image-preview" class="hidden">
                                        <p class="text-sm text-text-secondary mb-2">New Image Preview:</p>
                                        <div class="relative inline-block">
                                            <img id="preview-img" src="" alt="Preview" 
                                                 class="w-32 h-32 object-cover rounded-lg border border-border-light">
                                            <button type="button" onclick="removePreview()" 
                                                    class="absolute -top-2 -right-2 bg-red-500 text-white rounded-full w-6 h-6 flex items-center justify-center text-xs hover:bg-red-600 transition-colors">
                                                ×
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- SEO -->
                    <div class="bg-surface rounded-2xl shadow-soft border border-border-light p-6">
                        <h2 class="text-xl font-semibold text-text-primary mb-4">SEO</h2>
                        
                        <div class="space-y-4">
                            <div>
                                <label for="meta_title" class="block text-sm font-medium text-text-primary mb-2">Meta Title</label>
                                <input type="text" id="meta_title" name="meta_title" 
                                       value="<?php echo htmlspecialchars($_POST['meta_title'] ?? $post['meta_title']); ?>"
                                       class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors"
                                       placeholder="SEO title for search engines">
                            </div>
                            
                            <div>
                                <label for="keywords" class="block text-sm font-medium text-text-primary mb-2">Keywords</label>
                                <input type="text" id="keywords" name="keywords" 
                                       value="<?php echo htmlspecialchars($_POST['keywords'] ?? $post['keywords']); ?>"
                                       class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors"
                                       placeholder="keyword1, keyword2, keyword3">
                            </div>
                            
                            <div>
                                <label for="meta_description" class="block text-sm font-medium text-text-primary mb-2">Meta Description</label>
                                <textarea id="meta_description" name="meta_description" rows="3"
                                          class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors"
                                          placeholder="SEO description for search engines"><?php echo htmlspecialchars($_POST['meta_description'] ?? $post['meta_description']); ?></textarea>
                            </div>
                        </div>
                    </div>

                    <!-- Categories & Tags -->
                    <div class="bg-surface rounded-2xl shadow-soft border border-border-light p-6">
                        <h2 class="text-xl font-semibold text-text-primary mb-4">Categories & Tags</h2>
                        
                        <div class="space-y-4">
                            <div>
                                <label class="block text-sm font-medium text-text-primary mb-2">Categories</label>
                                <div class="space-y-2 max-h-40 overflow-y-auto border border-border-light rounded-xl p-3">
                                    <?php foreach ($allCategories as $category): ?>
                                        <label class="flex items-center">
                                            <input type="checkbox" name="categories[]" value="<?php echo $category['id']; ?>"
                                                   <?php echo in_array($category['id'], $_POST['categories'] ?? $postCategories) ? 'checked' : ''; ?>
                                                   class="rounded border-border-light text-primary focus:ring-primary">
                                            <span class="ml-2 text-sm text-text-primary"><?php echo htmlspecialchars($category['name']); ?></span>
                                        </label>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                            
                            <div>
                                <label for="tags" class="block text-sm font-medium text-text-primary mb-2">Tags</label>
                                <textarea id="tags" name="tags" rows="4"
                                          class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors"
                                          placeholder="Enter tags separated by commas"><?php echo htmlspecialchars(implode(', ', $_POST['tags'] ?? $postTags)); ?></textarea>
                                <p class="text-sm text-text-secondary mt-1">Separate tags with commas</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Form Actions -->
            <div class="flex justify-end gap-4 pt-6 border-t border-border-light">
                <a href="blog.php" class="px-6 py-3 text-text-secondary border border-border-light rounded-xl hover:bg-surface transition-colors">
                    Cancel
                </a>
                <button type="submit" class="px-6 py-3 bg-primary text-white rounded-xl hover:bg-primary-dark transition-colors font-semibold">
                    Update Blog Post
                </button>
            </div>
        </form>
    </div>
</div>

<script>
// Image upload functions
function previewImage(input) {
    if (input.files && input.files[0]) {
        const reader = new FileReader();
        reader.onload = function(e) {
            document.getElementById('preview-img').src = e.target.result;
            document.getElementById('image-preview').classList.remove('hidden');
        };
        reader.readAsDataURL(input.files[0]);
    }
}

function removePreview() {
    document.getElementById('featured_image').value = '';
    document.getElementById('image-preview').classList.add('hidden');
    document.getElementById('preview-img').src = '';
}

function removeCurrentImage() {
    if (confirm('Are you sure you want to remove the current image?')) {
        // Add a hidden input to indicate image removal
        const hiddenInput = document.createElement('input');
        hiddenInput.type = 'hidden';
        hiddenInput.name = 'remove_current_image';
        hiddenInput.value = '1';
        document.querySelector('form').appendChild(hiddenInput);
        
        // Hide current image preview
        document.querySelector('.current-image-preview').style.display = 'none';
    }
}

// Quill Editor initialization
var quill = new Quill('#editor', {
    theme: 'snow',
    modules: {
        toolbar: [
            [{ 'header': [1, 2, 3, false] }],
            ['bold', 'italic', 'underline', 'strike'],
            [{ 'color': [] }, { 'background': [] }],
            [{ 'list': 'ordered'}, { 'list': 'bullet' }],
            [{ 'indent': '-1'}, { 'indent': '+1' }],
            ['link', 'image', 'video'],
            ['blockquote', 'code-block'],
            ['clean']
        ]
    }
});

// Set initial content
quill.root.innerHTML = <?php echo json_encode($post['description']); ?>;

// Update hidden textarea before form submission
document.querySelector('form').addEventListener('submit', function() {
    document.getElementById('description').value = quill.root.innerHTML;
});

// Process tags input
document.getElementById('tags').addEventListener('blur', function() {
    var tags = this.value.split(',').map(function(tag) {
        return tag.trim();
    }).filter(function(tag) {
        return tag.length > 0;
    });
    this.value = tags.join(', ');
});
</script>

</body>
</html>
