<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
require_once 'admin_auth.php';
require_once __DIR__ . '/../includes/config.php';

$page_title = 'Blog Yazısını Görüntüle';
$post = null;

// Blog ID'yi al
$post_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if (!$post_id) {
    header('Location: blog.php');
    exit;
}

// Blog yazısını çek
try {
    $sql = "SELECT bp.*, u.username as author_name,
                   GROUP_CONCAT(DISTINCT bc.name) as categories,
                   GROUP_CONCAT(DISTINCT bt.name) as tags
            FROM blog_posts bp 
            LEFT JOIN user u ON bp.author_id = u.id
            LEFT JOIN blog_post_categories bpc ON bp.id = bpc.post_id 
            LEFT JOIN blog_categories bc ON bpc.category_id = bc.id
            LEFT JOIN blog_post_tags bpt ON bp.id = bpt.post_id
            LEFT JOIN blog_tags bt ON bpt.tag_id = bt.id
            WHERE bp.id = ?
            GROUP BY bp.id";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$post_id]);
    $post = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$post) {
        header('Location: blog.php');
        exit;
    }
} catch (Exception $e) {
    header('Location: blog.php');
    exit;
}

// Include meta head function
include '../includes/meta_head.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - Admin Panel</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined" rel="stylesheet">
    <?php echo generateMetaHead($page_title, "View blog post"); ?>
</head>
<?php include 'menu.php'; ?>

<div class="flex-1 ml-80">
    <main class="p-6">
        <div class="max-w-4xl mx-auto">
            <!-- Header -->
            <div class="flex items-center justify-between mb-6">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900">View Blog Post</h1>
                    <p class="text-gray-600 mt-1">Preview your blog post content</p>
                </div>
                <div class="flex space-x-2">
                    <a href="blog_edit.php?id=<?php echo $post['id']; ?>" class="inline-flex items-center px-4 py-2 bg-primary text-white rounded-lg hover:bg-primary-dark transition-colors">
                        <span class="material-symbols-outlined mr-2">edit</span>
                        Edit
                    </a>
                    <a href="blog.php" class="inline-flex items-center px-4 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600 transition-colors">
                        <span class="material-symbols-outlined mr-2">arrow_back</span>
                        Back to Blog
                    </a>
                </div>
            </div>

            <!-- Blog Post Content -->
            <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
                <!-- Featured Image -->
                <?php if ($post['featured_image']): ?>
                    <div class="aspect-video bg-gray-200">
                        <img src="<?php echo htmlspecialchars($post['featured_image']); ?>" 
                             alt="<?php echo htmlspecialchars($post['title']); ?>" 
                             class="w-full h-full object-cover">
                    </div>
                <?php endif; ?>

                <!-- Post Header -->
                <div class="p-6 border-b border-gray-200">
                    <div class="flex items-center justify-between mb-4">
                        <div class="flex items-center space-x-4">
                            <?php
                            $statusColors = [
                                'draft' => 'bg-yellow-100 text-yellow-800',
                                'published' => 'bg-green-100 text-green-800',
                                'archived' => 'bg-gray-100 text-gray-800'
                            ];
                            $statusColor = $statusColors[$post['status']] ?? 'bg-gray-100 text-gray-800';
                            ?>
                            <span class="px-3 py-1 text-sm font-semibold rounded-full <?php echo $statusColor; ?>">
                                <?php echo ucfirst($post['status']); ?>
                            </span>
                            <span class="text-sm text-gray-500">
                                <span class="material-symbols-outlined text-sm mr-1">visibility</span>
                                <?php echo number_format($post['views']); ?> views
                            </span>
                        </div>
                        <div class="text-sm text-gray-500">
                            Created: <?php echo date('M j, Y \a\t g:i A', strtotime($post['created_at'])); ?>
                        </div>
                    </div>
                    
                    <h1 class="text-3xl font-bold text-gray-900 mb-4"><?php echo htmlspecialchars($post['title']); ?></h1>
                    
                    <?php if ($post['excerpt']): ?>
                        <p class="text-lg text-gray-600 mb-4"><?php echo htmlspecialchars($post['excerpt']); ?></p>
                    <?php endif; ?>
                    
                    <div class="flex items-center justify-between text-sm text-gray-500">
                        <div class="flex items-center">
                            <span class="material-symbols-outlined mr-1">person</span>
                            By <?php echo htmlspecialchars($post['author_name']); ?>
                        </div>
                        <?php if ($post['published_at']): ?>
                            <div class="flex items-center">
                                <span class="material-symbols-outlined mr-1">schedule</span>
                                Published: <?php echo date('M j, Y \a\t g:i A', strtotime($post['published_at'])); ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Post Content -->
                <div class="p-6">
                    <div class="prose max-w-none">
                        <?php echo $post['description']; ?>
                    </div>
                </div>

                <!-- Post Meta -->
                <div class="p-6 bg-gray-50 border-t border-gray-200">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Categories -->
                        <div>
                            <h3 class="text-sm font-semibold text-gray-900 mb-2">Categories</h3>
                            <?php if ($post['categories']): ?>
                                <div class="flex flex-wrap gap-2">
                                    <?php foreach (explode(',', $post['categories']) as $category): ?>
                                        <span class="px-2 py-1 bg-primary/10 text-primary text-xs rounded-full">
                                            <?php echo htmlspecialchars(trim($category)); ?>
                                        </span>
                                    <?php endforeach; ?>
                                </div>
                            <?php else: ?>
                                <p class="text-sm text-gray-500">No categories assigned</p>
                            <?php endif; ?>
                        </div>

                        <!-- Tags -->
                        <div>
                            <h3 class="text-sm font-semibold text-gray-900 mb-2">Tags</h3>
                            <?php if ($post['tags']): ?>
                                <div class="flex flex-wrap gap-2">
                                    <?php foreach (explode(',', $post['tags']) as $tag): ?>
                                        <span class="px-2 py-1 bg-gray-200 text-gray-700 text-xs rounded-full">
                                            #<?php echo htmlspecialchars(trim($tag)); ?>
                                        </span>
                                    <?php endforeach; ?>
                                </div>
                            <?php else: ?>
                                <p class="text-sm text-gray-500">No tags assigned</p>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Keywords -->
                    <?php if ($post['keywords']): ?>
                        <div class="mt-4">
                            <h3 class="text-sm font-semibold text-gray-900 mb-2">Keywords</h3>
                            <p class="text-sm text-gray-600"><?php echo htmlspecialchars($post['keywords']); ?></p>
                        </div>
                    <?php endif; ?>
                </div>

                <!-- SEO Meta -->
                <?php if ($post['meta_title'] || $post['meta_description']): ?>
                    <div class="p-6 bg-blue-50 border-t border-blue-200">
                        <h3 class="text-sm font-semibold text-blue-900 mb-3">SEO Information</h3>
                        <div class="space-y-2">
                            <?php if ($post['meta_title']): ?>
                                <div>
                                    <span class="text-xs font-medium text-blue-700">Meta Title:</span>
                                    <p class="text-sm text-blue-800"><?php echo htmlspecialchars($post['meta_title']); ?></p>
                                </div>
                            <?php endif; ?>
                            <?php if ($post['meta_description']): ?>
                                <div>
                                    <span class="text-xs font-medium text-blue-700">Meta Description:</span>
                                    <p class="text-sm text-blue-800"><?php echo htmlspecialchars($post['meta_description']); ?></p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Actions -->
            <div class="mt-6 flex justify-center space-x-4">
                <a href="blog_edit.php?id=<?php echo $post['id']; ?>" class="inline-flex items-center px-6 py-3 bg-primary text-white rounded-lg hover:bg-primary-dark transition-colors">
                    <span class="material-symbols-outlined mr-2">edit</span>
                    Edit Post
                </a>
                <a href="blog_delete.php?id=<?php echo $post['id']; ?>" class="inline-flex items-center px-6 py-3 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors"
                   onclick="return confirm('Are you sure you want to delete this blog post?')">
                    <span class="material-symbols-outlined mr-2">delete</span>
                    Delete Post
                </a>
            </div>
        </div>
    </main>
</div>

<style>
.prose {
    color: #374151;
    line-height: 1.75;
}

.prose h1, .prose h2, .prose h3, .prose h4, .prose h5, .prose h6 {
    color: #111827;
    font-weight: 600;
    margin-top: 2rem;
    margin-bottom: 1rem;
}

.prose h1 { font-size: 2.25rem; }
.prose h2 { font-size: 1.875rem; }
.prose h3 { font-size: 1.5rem; }
.prose h4 { font-size: 1.25rem; }

.prose p {
    margin-bottom: 1.25rem;
}

.prose ul, .prose ol {
    margin-bottom: 1.25rem;
    padding-left: 1.5rem;
}

.prose li {
    margin-bottom: 0.5rem;
}

.prose blockquote {
    border-left: 4px solid #e5e7eb;
    padding-left: 1rem;
    margin: 1.5rem 0;
    font-style: italic;
    color: #6b7280;
}

.prose code {
    background-color: #f3f4f6;
    padding: 0.125rem 0.25rem;
    border-radius: 0.25rem;
    font-size: 0.875rem;
}

.prose pre {
    background-color: #1f2937;
    color: #f9fafb;
    padding: 1rem;
    border-radius: 0.5rem;
    overflow-x: auto;
    margin: 1.5rem 0;
}

.prose img {
    max-width: 100%;
    height: auto;
    border-radius: 0.5rem;
    margin: 1.5rem 0;
}

.prose a {
    color: #3b82f6;
    text-decoration: underline;
}

.prose a:hover {
    color: #1d4ed8;
}
</style>

</body>
</html>

