<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
require_once 'admin_auth.php';
require_once __DIR__ . '/../includes/config.php';

// Toplu silme işlemi
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_delete'])) {
    $category_ids = $_POST['category_ids'] ?? [];
    
    if (!empty($category_ids) && is_array($category_ids)) {
        try {
            // Güvenlik için sadece sayısal ID'leri al
            $category_ids = array_filter($category_ids, 'is_numeric');
            $category_ids = array_map('intval', $category_ids);
            
            if (!empty($category_ids)) {
                // Önce kategorilerde site olup olmadığını kontrol et
                $placeholders = str_repeat('?,', count($category_ids) - 1) . '?';
                $checkSql = "SELECT category_id, COUNT(*) as site_count FROM site_submissions WHERE category_id IN ($placeholders) GROUP BY category_id";
                $checkStmt = $pdo->prepare($checkSql);
                $checkStmt->execute($category_ids);
                $categoriesWithSites = $checkStmt->fetchAll(PDO::FETCH_KEY_PAIR);
                
                if (!empty($categoriesWithSites)) {
                    $totalSites = array_sum($categoriesWithSites);
                    $_SESSION['bulk_delete_error'] = "Site içeren kategoriler silinemez. Seçilen kategorilerde $totalSites site bulundu. Lütfen önce siteleri taşıyın veya silin.";
                } else {
                    // Kategorileri sil
                    $sql = "DELETE FROM categories WHERE id IN ($placeholders)";
                    $stmt = $pdo->prepare($sql);
                    
                    if ($stmt->execute($category_ids)) {
                        $deleted_count = $stmt->rowCount();
                        $_SESSION['bulk_delete_success'] = $deleted_count;
                    } else {
                        $_SESSION['bulk_delete_error'] = 'Kategoriler silinirken hata oluştu.';
                    }
                }
            }
        } catch (PDOException $e) {
            $_SESSION['bulk_delete_error'] = 'Veritabanı hatası: ' . $e->getMessage();
        }
    } else {
        $_SESSION['bulk_delete_error'] = 'Silme için kategori seçilmedi.';
    }
    
    // Yeniden gönderimi önlemek için yönlendir
    header('Location: category.php');
    exit;
}

// Message handling
$success = $error = '';
if (isset($_GET['success'])) {
    switch ($_GET['success']) {
        case 'category_deleted':
            $success = 'Category "' . htmlspecialchars($_GET['name'] ?? 'Unknown') . '" has been deleted successfully.';
            break;
        default:
            $success = 'Operation completed successfully.';
    }
}

if (isset($_GET['error'])) {
    switch ($_GET['error']) {
        case 'invalid_category_id':
            $error = 'Invalid category ID provided.';
            break;
        case 'category_not_found':
            $error = 'Category not found.';
            break;
        case 'category_has_sites':
            $count = intval($_GET['count'] ?? 0);
            $error = "Cannot delete category. It contains $count site(s). Please move or delete the sites first.";
            break;
        case 'delete_failed':
            $error = 'Failed to delete category.';
            break;
        case 'database_error':
            $error = 'Database error occurred. Please try again.';
            break;
        default:
            $error = 'An error occurred.';
    }
}

$perPage = isset($_GET['perPage']) ? intval($_GET['perPage']) : 10;
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$offset = ($page - 1) * $perPage;
$whereArr = [];
$params = [];
if (isset($_GET['search']) && $_GET['search'] !== '') {
    $whereArr[] = '(name LIKE :search OR slug LIKE :search)';
    $params[':search'] = '%' . $_GET['search'] . '%';
}
$whereSQL = count($whereArr) ? ('WHERE ' . implode(' AND ', $whereArr)) : '';
$orderSQL = 'ORDER BY id DESC';
if (isset($_GET['sort'])) {
    if ($_GET['sort'] == 'name') $orderSQL = 'ORDER BY name ASC';
    elseif ($_GET['sort'] == 'sitecount') $orderSQL = 'ORDER BY site_count DESC';
}
$totalStmt = $pdo->prepare('SELECT COUNT(*) FROM categories ' . $whereSQL);
$totalStmt->execute($params);
$totalRows = $totalStmt->fetchColumn();
$totalPages = ceil($totalRows / $perPage);
$sql = 'SELECT c.*, (SELECT COUNT(*) FROM site_submissions s WHERE s.category_id = c.id) as site_count FROM categories c ' . $whereSQL . ' ' . $orderSQL . ' LIMIT :limit OFFSET :offset';
$stmt = $pdo->prepare($sql);
foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value);
}
$stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();
$categories = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en"><head>
<meta charset="utf-8"/>
<meta content="width=device-width, initial-scale=1.0" name="viewport"/>
<title>Categories Management</title>
<link href="https://fonts.googleapis.com" rel="preconnect"/>
<link crossorigin="" href="https://fonts.gstatic.com" rel="preconnect"/>
<link href="https://fonts.googleapis.com/css2?family=Manrope:wght@400;500;600;700;800&amp;display=swap" rel="stylesheet"/>
<script src="https://cdn.tailwindcss.com?plugins=forms,container-queries"></script>
<link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined:opsz,wght,FILL,GRAD@20..48,100..700,0..1,-50..200" rel="stylesheet"/>
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
<script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: "#1877f2", // A modern, friendly blue
                        "primary-light": "#e7f3ff",
                        background: "#f0f2f5",
                        surface: "#ffffff",
                        "text-primary": "#1c1e21",
                        "text-secondary": "#606770",
                        "line-light": "#e0e0e0",
                    },
                    fontFamily: {
                        display: ["Manrope", "sans-serif"],
                    },
                    borderRadius: {
                        DEFAULT: "0.75rem",
                        lg: "1rem",
                        xl: "1.5rem",
                        full: "9999px"
                    },
                    boxShadow: {
                        'soft': '0 4px 12px 0 rgb(0 0 0 / 0.07)',
                        'subtle': '0 1px 2px 0 rgb(0 0 0 / 0.05)',
                    }
                },
            },
        };
    </script>
<style>
        .material-symbols-outlined {
            font-variation-settings: "FILL" 0, "wght" 400, "GRAD" 0, "opsz" 24;
        }
        .active .material-symbols-outlined {
            font-variation-settings: 'FILL' 1;
        }
    </style>
</head>
<body class="bg-background font-display text-text-primary">
<div class="flex min-h-screen">
<?php include 'menu.php'; ?>
<main class="flex-1 p-10">
<?php
$page_title = "Categories";
$page_subtitle = "Manage your directory's categories.";
include 'includes/admin_header.php';
?>

<!-- Flash Messages -->
<?php if (isset($_SESSION['bulk_delete_success'])): ?>
<div class="mb-6 p-4 bg-green-100 border border-green-200 rounded-lg flex items-center">
    <span class="material-symbols-outlined text-green-600 mr-3">check_circle</span>
    <div>
        <p class="text-green-800 font-medium">Bulk deletion successful!</p>
        <p class="text-green-700 text-sm"><?php echo $_SESSION['bulk_delete_success']; ?> categories have been deleted.</p>
    </div>
</div>
<?php 
unset($_SESSION['bulk_delete_success']);
endif; 
?>

<?php if (isset($_SESSION['bulk_delete_error'])): ?>
<div class="mb-6 p-4 bg-red-100 border border-red-200 rounded-lg flex items-center">
    <span class="material-symbols-outlined text-red-600 mr-3">error</span>
    <div>
        <p class="text-red-800 font-medium">Bulk deletion failed!</p>
        <p class="text-red-700 text-sm"><?php echo htmlspecialchars($_SESSION['bulk_delete_error']); ?></p>
    </div>
</div>
<?php 
unset($_SESSION['bulk_delete_error']);
endif; 
?>

<!-- Messages -->
<?php if ($success): ?>
<div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded-lg mb-6 flex items-center gap-3">
    <span class="material-symbols-outlined">check_circle</span>
    <span><?php echo $success; ?></span>
</div>
<?php endif; ?>

<?php if ($error): ?>
<div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded-lg mb-6 flex items-center gap-3">
    <span class="material-symbols-outlined">error</span>
    <span><?php echo $error; ?></span>
</div>
<?php endif; ?>

<form method="GET" class="flex items-center gap-4 bg-surface rounded-xl shadow-soft p-6 mb-6 border border-line-light">
    <div class="flex items-center gap-2 flex-1 min-w-[220px] max-w-md">
        <div class="relative w-full">
            <span class="material-symbols-outlined absolute left-4 top-1/2 -translate-y-1/2 text-text-secondary">search</span>
            <input class="w-full pl-12 pr-4 py-3 bg-background border-none rounded-lg focus:ring-2 focus:ring-primary" name="search" value="<?php echo htmlspecialchars($_GET['search'] ?? ''); ?>" placeholder="İsim veya slug ile ara..." type="text"/>
        </div>
        <button type="submit" class="px-4 py-2 rounded-lg bg-primary text-white font-bold">Ara</button>
    </div>
    <select name="sort" class="px-4 py-2 bg-background rounded-lg text-text-secondary font-medium hover:bg-line-light w-52" onchange="this.form.submit()">
        <option value="name" <?php if(!isset($_GET['sort'])||$_GET['sort']=='name') echo 'selected'; ?>>Sırala: İsim</option>
        <option value="sitecount" <?php if(isset($_GET['sort'])&&$_GET['sort']=='sitecount') echo 'selected'; ?>>Sırala: Site Sayısı</option>
    </select>
    <select name="perPage" class="px-4 py-2 bg-background rounded-lg text-text-secondary font-medium hover:bg-line-light" onchange="this.form.submit()">
        <?php $perPageOptions = [5,10,20,50,100,200,500,1000];
        $selectedPerPage = isset($_GET['perPage']) ? intval($_GET['perPage']) : 10;
        foreach($perPageOptions as $opt) {
            echo '<option value="'.$opt.'"'.($selectedPerPage==$opt?' selected':'').'>'.$opt.' / sayfa</option>';
        }
        ?>
    </select>
    <input type="hidden" name="page" value="<?php echo $page; ?>">
    
    <!-- New Category Button moved here -->
    <a href="category_new.php" class="flex items-center gap-2 bg-primary text-white font-bold py-3 px-5 rounded-lg hover:bg-primary/90 transition-colors ml-4">
        <span class="material-symbols-outlined">add</span>
        <span>Yeni Kategori</span>
    </a>
</form>

<!-- Bulk Actions -->
<div class="mb-4 flex items-center justify-between">
    <div class="flex items-center gap-4">
        <button id="bulk-delete-btn" onclick="bulkDelete()" disabled 
                class="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors disabled:bg-gray-300 disabled:cursor-not-allowed flex items-center gap-2">
            <span class="material-symbols-outlined text-sm">delete</span>
            Seçilenleri Sil
        </button>
        <span id="selected-count" class="text-sm text-gray-600">0 kategori seçildi</span>
    </div>
</div>

<div class="overflow-x-auto bg-white rounded-xl shadow-soft border border-line-light">
<table class="w-full text-left">
<thead>
<tr class="bg-gradient-to-r from-primary/5 to-primary/10 border-b-2 border-primary/20">
    <th class="p-6 font-bold text-text-primary text-lg w-16">
        <input type="checkbox" id="select-all" class="w-4 h-4 text-primary bg-gray-100 border-gray-300 rounded focus:ring-primary focus:ring-2" onchange="toggleSelectAll()">
    </th>
    <th class="p-6 font-bold text-text-primary text-lg">
        <div class="flex items-center gap-3">
            <span class="material-symbols-outlined text-primary">category</span>
            Kategori Adı
        </div>
    </th>
    <th class="p-6 font-bold text-text-primary text-lg">
        <div class="flex items-center gap-3">
            <span class="material-symbols-outlined text-primary">link</span>
            Slug
        </div>
    </th>
    <th class="p-6 font-bold text-text-primary text-lg">
        <div class="flex items-center gap-3">
            <span class="material-symbols-outlined text-primary">description</span>
            Açıklama
        </div>
    </th>
    <th class="p-6 font-bold text-text-primary text-lg text-center">
        <div class="flex items-center justify-center gap-3">
            <span class="material-symbols-outlined text-primary">bar_chart</span>
            Site Sayısı
        </div>
    </th>
    <th class="p-6 font-bold text-text-primary text-lg text-center">
        <div class="flex items-center justify-center gap-3">
            <span class="material-symbols-outlined text-primary">settings</span>
            İşlemler
        </div>
    </th>
</tr>
</thead>
<tbody>
<?php foreach ($categories as $index => $cat): ?>
<tr class="border-b border-line-light hover:bg-gradient-to-r hover:from-primary/5 hover:to-transparent transition-all duration-200 group">
    <td class="p-6">
        <input type="checkbox" class="category-checkbox w-4 h-4 text-primary bg-gray-100 border-gray-300 rounded focus:ring-primary focus:ring-2" 
               value="<?php echo $cat['id']; ?>" onchange="updateBulkActions()">
    </td>
    <td class="p-6">
        <div class="flex items-center gap-4">
            <div class="w-12 h-12 rounded-xl flex items-center justify-center shadow-soft" style="background-color: <?php echo htmlspecialchars($cat['color'] ?? '#38BDF8'); ?>20;">
                <?php if (!empty($cat['icon'])): ?>
                    <i class="<?php echo htmlspecialchars($cat['icon']); ?> text-xl" style="color: <?php echo htmlspecialchars($cat['color'] ?? '#38BDF8'); ?>"></i>
                <?php else: ?>
                    <span class="material-symbols-outlined text-xl" style="color: <?php echo htmlspecialchars($cat['color'] ?? '#38BDF8'); ?>">folder</span>
                <?php endif; ?>
            </div>
            <div>
                <div class="font-bold text-text-primary text-lg group-hover:text-primary transition-colors">
                    <?php echo htmlspecialchars($cat['name']); ?>
                </div>
                <div class="text-text-secondary text-sm">ID: #<?php echo $cat['id']; ?></div>
            </div>
        </div>
    </td>
    <td class="p-6">
        <div class="inline-flex items-center gap-2 bg-background px-4 py-2 rounded-lg">
            <span class="material-symbols-outlined text-text-secondary text-sm">tag</span>
            <code class="text-text-secondary font-mono"><?php echo htmlspecialchars($cat['slug'] ?? 'No slug'); ?></code>
        </div>
    </td>
    <td class="p-6">
        <div class="text-text-secondary text-sm max-w-sm">
            <?php 
            $description = $cat['description'] ?? '';
            if (!empty($description)) {
                echo htmlspecialchars(strlen($description) > 80 ? substr($description, 0, 80) . '...' : $description);
            } else {
                echo '<span class="italic text-gray-400">No description</span>';
            }
            ?>
        </div>
    </td>
    <td class="p-6 text-center">
        <div class="inline-flex items-center justify-center w-16 h-16 rounded-xl <?php echo $cat['site_count'] > 0 ? 'bg-green-100 text-green-700' : 'bg-gray-100 text-gray-500'; ?> font-bold text-xl shadow-soft">
            <?php echo $cat['site_count'] ?? 0; ?>
        </div>
    </td>
    <td class="p-6">
        <div class="flex items-center justify-center gap-3">
            <a href="category_view.php?id=<?php echo $cat['id']; ?>" class="w-12 h-12 flex items-center justify-center rounded-xl bg-green-100 text-green-600 hover:bg-green-200 hover:scale-105 transition-all duration-200 shadow-soft" title="View Category">
                <span class="material-symbols-outlined text-xl">visibility</span>
            </a>
            <a href="category_edit.php?id=<?php echo $cat['id']; ?>" class="w-12 h-12 flex items-center justify-center rounded-xl bg-blue-100 text-blue-600 hover:bg-blue-200 hover:scale-105 transition-all duration-200 shadow-soft" title="Edit Category">
                <span class="material-symbols-outlined text-xl">edit</span>
            </a>
            <a href="category_delete.php?id=<?php echo $cat['id']; ?>" onclick="return confirmDelete('<?php echo addslashes($cat['name']); ?>', <?php echo $cat['id']; ?>);" class="w-12 h-12 flex items-center justify-center rounded-xl bg-red-100 text-red-600 hover:bg-red-200 hover:scale-105 transition-all duration-200 shadow-soft" title="Delete Category">
                <span class="material-symbols-outlined text-xl">delete</span>
            </a>
        </div>
    </td>
</tr>
<?php endforeach; ?>
</tbody>
</table>

<?php if (empty($categories)): ?>
<div class="p-12 text-center">
    <div class="w-20 h-20 rounded-full bg-background mx-auto mb-4 flex items-center justify-center">
        <span class="material-symbols-outlined text-4xl text-text-secondary">folder_off</span>
    </div>
    <h3 class="text-xl font-bold text-text-primary mb-2">No Categories Found</h3>
    <p class="text-text-secondary mb-6">Start by creating your first category to organize your sites.</p>
    <a href="category_new.php" class="inline-flex items-center gap-2 bg-primary text-white font-bold py-3 px-6 rounded-lg hover:bg-primary/90 transition-colors">
        <span class="material-symbols-outlined">add</span>
        <span>Create Category</span>
    </a>
</div>
<?php endif; ?>

</div>

<!-- Pagination -->
<?php if ($totalPages > 1): ?>
<div class="flex items-center justify-center gap-2 mt-6">
    <?php if ($page > 1): ?>
        <a href="?page=<?php echo ($page - 1); ?>&<?php echo http_build_query(array_filter(['search' => $_GET['search'] ?? '', 'sort' => $_GET['sort'] ?? '', 'perPage' => $_GET['perPage'] ?? ''])); ?>" class="flex items-center justify-center w-10 h-10 rounded-lg bg-background hover:bg-line-light transition-colors">
            <span class="material-symbols-outlined">chevron_left</span>
        </a>
    <?php endif; ?>
    
    <?php
    $startPage = max(1, $page - 2);
    $endPage = min($totalPages, $page + 2);
    
    if ($startPage > 1): ?>
        <a href="?page=1&<?php echo http_build_query(array_filter(['search' => $_GET['search'] ?? '', 'sort' => $_GET['sort'] ?? '', 'perPage' => $_GET['perPage'] ?? ''])); ?>" class="flex items-center justify-center w-10 h-10 rounded-lg bg-background hover:bg-line-light transition-colors font-medium">1</a>
        <?php if ($startPage > 2): ?>
            <span class="px-2 text-text-secondary">...</span>
        <?php endif; ?>
    <?php endif; ?>
    
    <?php for ($i = $startPage; $i <= $endPage; $i++): ?>
        <a href="?page=<?php echo $i; ?>&<?php echo http_build_query(array_filter(['search' => $_GET['search'] ?? '', 'sort' => $_GET['sort'] ?? '', 'perPage' => $_GET['perPage'] ?? ''])); ?>" class="flex items-center justify-center w-10 h-10 rounded-lg <?php echo $i == $page ? 'bg-primary text-white' : 'bg-background hover:bg-line-light'; ?> transition-colors font-medium">
            <?php echo $i; ?>
        </a>
    <?php endfor; ?>
    
    <?php if ($endPage < $totalPages): ?>
        <?php if ($endPage < $totalPages - 1): ?>
            <span class="px-2 text-text-secondary">...</span>
        <?php endif; ?>
        <a href="?page=<?php echo $totalPages; ?>&<?php echo http_build_query(array_filter(['search' => $_GET['search'] ?? '', 'sort' => $_GET['sort'] ?? '', 'perPage' => $_GET['perPage'] ?? ''])); ?>" class="flex items-center justify-center w-10 h-10 rounded-lg bg-background hover:bg-line-light transition-colors font-medium"><?php echo $totalPages; ?></a>
    <?php endif; ?>
    
    <?php if ($page < $totalPages): ?>
        <a href="?page=<?php echo ($page + 1); ?>&<?php echo http_build_query(array_filter(['search' => $_GET['search'] ?? '', 'sort' => $_GET['sort'] ?? '', 'perPage' => $_GET['perPage'] ?? ''])); ?>" class="flex items-center justify-center w-10 h-10 rounded-lg bg-background hover:bg-line-light transition-colors">
            <span class="material-symbols-outlined">chevron_right</span>
        </a>
    <?php endif; ?>
</div>
<?php endif; ?>

</div>
</main>
</div>

<script>
// Checkbox selection functions
function toggleSelectAll() {
    const selectAllCheckbox = document.getElementById('select-all');
    const categoryCheckboxes = document.querySelectorAll('.category-checkbox');
    
    categoryCheckboxes.forEach(checkbox => {
        checkbox.checked = selectAllCheckbox.checked;
    });
    
    updateBulkActions();
}

function updateBulkActions() {
    const categoryCheckboxes = document.querySelectorAll('.category-checkbox');
    const checkedBoxes = document.querySelectorAll('.category-checkbox:checked');
    const selectAllCheckbox = document.getElementById('select-all');
    const bulkDeleteBtn = document.getElementById('bulk-delete-btn');
    const selectedCount = document.getElementById('selected-count');
    
    // Update select all checkbox state
    if (checkedBoxes.length === 0) {
        selectAllCheckbox.indeterminate = false;
        selectAllCheckbox.checked = false;
    } else if (checkedBoxes.length === categoryCheckboxes.length) {
        selectAllCheckbox.indeterminate = false;
        selectAllCheckbox.checked = true;
    } else {
        selectAllCheckbox.indeterminate = true;
    }
    
    // Update bulk actions
    bulkDeleteBtn.disabled = checkedBoxes.length === 0;
    selectedCount.textContent = `${checkedBoxes.length} categories selected`;
}

// Bulk delete function
function bulkDelete() {
    const checkedBoxes = document.querySelectorAll('.category-checkbox:checked');
    
    if (checkedBoxes.length === 0) {
        alert('Please select categories to delete.');
        return;
    }
    
    const categoryNames = Array.from(checkedBoxes).map(checkbox => {
        const row = checkbox.closest('tr');
        return row.querySelector('td:nth-child(2) .font-bold').textContent;
    });
    
    const modal = document.createElement('div');
    modal.className = 'fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50';
    modal.innerHTML = `
        <div class="bg-white rounded-lg p-6 max-w-md w-full mx-4 transform scale-95 transition-transform duration-200">
            <div class="flex items-center mb-4">
                <span class="material-symbols-outlined text-red-500 text-3xl mr-3">warning</span>
                <h3 class="text-lg font-semibold text-gray-900">Confirm Bulk Deletion</h3>
            </div>
            <p class="text-gray-600 mb-4">
                Are you sure you want to delete <strong>${checkedBoxes.length} categories</strong>?
            </p>
            <div class="max-h-32 overflow-y-auto mb-4 p-3 bg-gray-50 rounded border">
                <ul class="text-sm text-gray-700">
                    ${categoryNames.map(name => `<li>• ${name}</li>`).join('')}
                </ul>
            </div>
            <p class="text-sm text-red-600 mb-6">
                <strong>Warning:</strong> Categories containing sites cannot be deleted. This action cannot be undone.
            </p>
            <div class="flex gap-3">
                <button onclick="closeBulkDeleteModal()" class="flex-1 px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50">
                    Cancel
                </button>
                <button onclick="proceedBulkDelete()" class="flex-1 px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700">
                    Delete Categories
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Animate in
    setTimeout(() => {
        modal.querySelector('.bg-white').style.transform = 'scale(1)';
    }, 10);
    
    // Store modal reference
    window.currentBulkDeleteModal = modal;
}

function closeBulkDeleteModal() {
    if (window.currentBulkDeleteModal) {
        const modal = window.currentBulkDeleteModal;
        modal.querySelector('.bg-white').style.transform = 'scale(0.95)';
        setTimeout(() => {
            document.body.removeChild(modal);
        }, 200);
        window.currentBulkDeleteModal = null;
    }
}

function proceedBulkDelete() {
    closeBulkDeleteModal();
    
    const checkedBoxes = document.querySelectorAll('.category-checkbox:checked');
    
    // Show loading indicator
    const loader = document.createElement('div');
    loader.className = 'fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50';
    loader.innerHTML = `
        <div class="bg-white rounded-lg p-6 text-center">
            <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-3"></div>
            <p class="text-gray-600">Deleting categories...</p>
        </div>
    `;
    document.body.appendChild(loader);
    
    // Create form and submit
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = 'category.php';
    
    const bulkDeleteInput = document.createElement('input');
    bulkDeleteInput.type = 'hidden';
    bulkDeleteInput.name = 'bulk_delete';
    bulkDeleteInput.value = '1';
    form.appendChild(bulkDeleteInput);
    
    // Add selected category IDs
    checkedBoxes.forEach(checkbox => {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'category_ids[]';
        input.value = checkbox.value;
        form.appendChild(input);
    });
    
    document.body.appendChild(form);
    form.submit();
}

// Close modal on escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape' && window.currentBulkDeleteModal) {
        closeBulkDeleteModal();
    }
    if (e.key === 'Escape' && window.currentDeleteModal) {
        closeDeleteModal();
    }
});

// Single category delete confirmation
function confirmDelete(categoryName, categoryId) {
    const modal = document.createElement('div');
    modal.className = 'fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50';
    modal.innerHTML = `
        <div class="bg-white rounded-lg p-6 max-w-md w-full mx-4 transform scale-95 transition-transform duration-200">
            <div class="flex items-center mb-4">
                <span class="material-symbols-outlined text-red-500 text-3xl mr-3">warning</span>
                <h3 class="text-lg font-semibold text-gray-900">Confirm Deletion</h3>
            </div>
            <p class="text-gray-600 mb-6">
                Are you sure you want to delete category <strong>"${categoryName}"</strong>?<br>
                <span class="text-sm text-red-600">This action cannot be undone.</span>
            </p>
            <div class="flex gap-3">
                <button onclick="closeDeleteModal()" class="flex-1 px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50">
                    Cancel
                </button>
                <button onclick="proceedSingleDelete(${categoryId})" class="flex-1 px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700">
                    Delete Category
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Animate in
    setTimeout(() => {
        modal.querySelector('.bg-white').style.transform = 'scale(1)';
    }, 10);
    
    // Store modal reference
    window.currentDeleteModal = modal;
    
    return false; // Prevent default link behavior
}

function closeDeleteModal() {
    if (window.currentDeleteModal) {
        const modal = window.currentDeleteModal;
        modal.querySelector('.bg-white').style.transform = 'scale(0.95)';
        setTimeout(() => {
            document.body.removeChild(modal);
        }, 200);
        window.currentDeleteModal = null;
    }
}

function proceedSingleDelete(categoryId) {
    closeDeleteModal();
    
    // Show loading indicator
    const loader = document.createElement('div');
    loader.className = 'fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50';
    loader.innerHTML = `
        <div class="bg-white rounded-lg p-6 text-center">
            <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-3"></div>
            <p class="text-gray-600">Deleting category...</p>
        </div>
    `;
    document.body.appendChild(loader);
    
    // Proceed with deletion
    window.location.href = `category_delete.php?id=${categoryId}`;
}
</script>
</body>
</html>


