<?php
include 'admin_auth.php';
include '../includes/config.php';

$category_id = isset($_GET['id']) ? intval($_GET['id']) : 0;
$success = $error = '';

if ($category_id <= 0) {
    header('Location: category.php');
    exit;
}

// Kategori bilgilerini çek
$stmt = $pdo->prepare('SELECT * FROM categories WHERE id = ?');
$stmt->execute([$category_id]);
$category = $stmt->fetch();

if (!$category) {
    header('Location: category.php?error=category_not_found');
    exit;
}

// Form işleme
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $color = trim($_POST['color'] ?? '#38BDF8');
    $icon = trim($_POST['icon'] ?? '');
    
    if (empty($name)) {
        $error = 'Kategori adı gerekli.';
    } else {
        try {
            // Check if name already exists (excluding current category)
            $stmt = $pdo->prepare('SELECT id FROM categories WHERE name = ? AND id != ?');
            $stmt->execute([$name, $category_id]);
            if ($stmt->rowCount() > 0) {
                $error = 'Bu isimde bir kategori zaten mevcut.';
            } else {
                // Generate slug
                $slug = strtolower(preg_replace('/[^a-zA-Z0-9]+/', '-', trim($name)));
                $slug = trim($slug, '-');
                
                // Update category
                $stmt = $pdo->prepare('UPDATE categories SET name = ?, description = ?, color = ?, icon = ?, slug = ? WHERE id = ?');
                if ($stmt->execute([$name, $description, $color, $icon, $slug, $category_id])) {
                    $success = 'Kategori başarıyla güncellendi!';
                    // Refresh category data
                    $stmt = $pdo->prepare('SELECT * FROM categories WHERE id = ?');
                    $stmt->execute([$category_id]);
                    $category = $stmt->fetch();
                } else {
                    $error = 'Kategori güncellenemedi.';
                }
            }
        } catch (PDOException $e) {
            $error = 'Veritabanı hatası: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8"/>
    <meta content="width=device-width, initial-scale=1.0" name="viewport"/>
    <title>Kategori Düzenle - <?php echo htmlspecialchars($category['name']); ?></title>
    <link href="https://fonts.googleapis.com" rel="preconnect"/>
    <link crossorigin="" href="https://fonts.gstatic.com" rel="preconnect"/>
    <link href="https://fonts.googleapis.com/css2?family=Manrope:wght@400;500;600;700;800&display=swap" rel="stylesheet"/>
    <script src="https://cdn.tailwindcss.com?plugins=forms,container-queries"></script>
    <link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined:opsz,wght,FILL,GRAD@20..48,100..700,0..1,-50..200" rel="stylesheet"/>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: "#1877f2",
                        "primary-light": "#e7f3ff",
                        background: "#f0f2f5",
                        surface: "#ffffff",
                        "text-primary": "#1c1e21",
                        "text-secondary": "#606770",
                        "line-light": "#e0e0e0",
                    },
                    fontFamily: {
                        display: ["Manrope", "sans-serif"],
                    },
                    borderRadius: {
                        DEFAULT: "0.75rem",
                        lg: "1rem",
                        xl: "1.5rem",
                        full: "9999px"
                    },
                    boxShadow: {
                        'soft': '0 4px 12px 0 rgb(0 0 0 / 0.07)',
                        'subtle': '0 1px 2px 0 rgb(0 0 0 / 0.05)',
                    }
                },
            },
        };
    </script>
</head>
<body class="bg-background font-display text-text-primary">
    <div class="flex h-screen bg-background">
        <?php include 'menu.php'; ?>
        
        <main class="flex-1 overflow-auto">
            <div class="p-8">
                <!-- Header -->
                <div class="flex items-center justify-between mb-8">
                    <div>
                        <h1 class="text-3xl font-bold text-text-primary">Kategori Düzenle</h1>
                        <p class="text-text-secondary mt-1">Kategori bilgilerini değiştirin</p>
                    </div>
                    <div class="flex items-center gap-4">
                        <a href="category_view.php?id=<?php echo $category['id']; ?>" class="flex items-center gap-2 bg-green-600 text-white font-bold py-3 px-5 rounded-lg hover:bg-green-700 transition-colors">
                            <span class="material-symbols-outlined">visibility</span>
                            <span>Kategoriyi Görüntüle</span>
                        </a>
                        <a href="category.php" class="flex items-center gap-2 bg-gray-600 text-white font-bold py-3 px-5 rounded-lg hover:bg-gray-700 transition-colors">
                            <span class="material-symbols-outlined">arrow_back</span>
                            <span>Kategorilere Dön</span>
                        </a>
                    </div>
                </div>

                <!-- Messages -->
                <?php if ($success): ?>
                <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded-lg mb-6">
                    <?php echo htmlspecialchars($success); ?>
                </div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded-lg mb-6">
                    <?php echo htmlspecialchars($error); ?>
                </div>
                <?php endif; ?>

                <!-- Edit Form -->
                <div class="bg-surface rounded-xl shadow-soft p-6">
                    <form method="POST" class="space-y-6">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <!-- Category Name -->
                            <div>
                                <label for="name" class="block text-sm font-medium text-text-secondary mb-2">Kategori Adı *</label>
                                <input type="text" id="name" name="name" value="<?php echo htmlspecialchars($category['name']); ?>" required 
                                       class="w-full px-4 py-3 border border-line-light rounded-lg focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent">
                            </div>

                            <!-- Color -->
                            <div>
                                <label for="color" class="block text-sm font-medium text-text-secondary mb-2">Renk</label>
                                <div class="flex items-center gap-3">
                                    <input type="color" id="color" name="color" value="<?php echo htmlspecialchars($category['color'] ?? '#38BDF8'); ?>" 
                                           class="w-16 h-12 border border-line-light rounded-lg cursor-pointer">
                                    <input type="text" id="colorText" value="<?php echo htmlspecialchars($category['color'] ?? '#38BDF8'); ?>" 
                                           class="flex-1 px-4 py-3 border border-line-light rounded-lg focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent">
                                </div>
                            </div>
                        </div>

                        <!-- Description -->
                        <div>
                            <label for="description" class="block text-sm font-medium text-text-secondary mb-2">Açıklama</label>
                            <textarea id="description" name="description" rows="4" 
                                      class="w-full px-4 py-3 border border-line-light rounded-lg focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent"
                                      placeholder="Bu kategori için isteğe bağlı açıklama"><?php echo htmlspecialchars($category['description'] ?? ''); ?></textarea>
                        </div>

                        <!-- Icon -->
                        <div>
                            <label for="icon" class="block text-sm font-medium text-text-secondary mb-2">İkon Sınıfı</label>
                            <div class="flex items-center gap-3">
                                <div id="iconPreview" class="w-12 h-12 bg-gray-100 rounded-lg flex items-center justify-center">
                                    <i class="<?php echo htmlspecialchars($category['icon'] ?? 'fa-solid fa-folder'); ?> text-xl" style="color: <?php echo htmlspecialchars($category['color'] ?? '#38BDF8'); ?>"></i>
                                </div>
                                <input type="text" id="icon" name="icon" value="<?php echo htmlspecialchars($category['icon'] ?? ''); ?>" 
                                       class="flex-1 px-4 py-3 border border-line-light rounded-lg focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent"
                                       placeholder="e.g., fa-solid fa-folder">
                            </div>
                            <p class="text-xs text-text-secondary mt-1">Font Awesome ikon sınıflarını kullanın (örn., fa-solid fa-folder)</p>
                        </div>

                        <!-- Submit Button -->
                        <div class="flex items-center gap-4 pt-4">
                            <button type="submit" class="bg-primary text-white font-bold py-3 px-6 rounded-lg hover:bg-primary/90 transition-colors">
                                Kategoriyi Güncelle
                            </button>
                            <a href="category.php" class="bg-gray-500 text-white font-bold py-3 px-6 rounded-lg hover:bg-gray-600 transition-colors">
                                İptal
                            </a>
                        </div>
                    </form>
                </div>
            </div>
        </main>
    </div>

    <script>
        // Color picker sync
        const colorPicker = document.getElementById('color');
        const colorText = document.getElementById('colorText');
        const iconPreview = document.getElementById('iconPreview');
        const iconInput = document.getElementById('icon');

        colorPicker.addEventListener('change', function() {
            colorText.value = this.value;
            updateIconPreview();
        });

        colorText.addEventListener('input', function() {
            if (/^#[0-9A-F]{6}$/i.test(this.value)) {
                colorPicker.value = this.value;
                updateIconPreview();
            }
        });

        // Icon preview update
        iconInput.addEventListener('input', updateIconPreview);

        function updateIconPreview() {
            const iconClass = iconInput.value || 'fa-solid fa-folder';
            const color = colorText.value || '#38BDF8';
            iconPreview.innerHTML = `<i class="${iconClass} text-xl" style="color: ${color}"></i>`;
        }

        // Initial preview update
        updateIconPreview();
    </script>
</body>
</html>

