<?php
// Admin header component
// Bu dosya tüm admin sayfalarında kullanılacak ortak header bölümüdür

// Admin bilgilerini al
$admin_id = $_SESSION['admin_user_id'] ?? null;
$admin_username = $_SESSION['admin_username'] ?? 'Admin';

// Admin profil fotoğrafını ve email'ini al (varsayılan olarak no-logo.png)
$admin_avatar = '/uploads/no-logo.png';
$admin_email = '';

// Eğer admin tablosunda profil fotoğrafı ve email varsa onu kullan
try {
    $stmt = $pdo->prepare("SELECT profile_photo, email FROM user WHERE id = ? AND role = 'admin'");
    $stmt->execute([$admin_id]);
    $admin_data = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($admin_data) {
        if (!empty($admin_data['profile_photo'])) {
            $admin_avatar = $admin_data['profile_photo'];
        }
        if (!empty($admin_data['email'])) {
            $admin_email = $admin_data['email'];
        }
    }
} catch (Exception $e) {
    // Hata durumunda varsayılan değerleri kullan
}

// Pending sites sayısını al
$pending_sites = 0;
try {
    $stmt = $pdo->query("SELECT COUNT(*) FROM site_submissions WHERE status = 'pending'");
    $pending_sites = $stmt->fetchColumn();
} catch (Exception $e) {
    // Hata durumunda 0 kullan
}
?>

<!-- Admin Header -->
<header class="flex justify-between items-center mb-10">
    <div>
        <h1 class="text-4xl font-extrabold text-text-primary"><?php echo $page_title ?? 'Admin Panel'; ?></h1>
        <p class="text-text-secondary mt-1"><?php echo $page_subtitle ?? 'Welcome back, ' . htmlspecialchars($admin_username); ?></p>
    </div>
    <div class="flex items-center gap-4">
        <!-- Notifications Button -->
        <button class="relative w-10 h-10 flex items-center justify-center rounded-full bg-surface hover:bg-background border border-line-light transition-colors">
            <span class="material-symbols-outlined text-text-secondary">notifications</span>
            <?php if ($pending_sites > 0): ?>
            <div class="absolute top-1 right-1 w-2.5 h-2.5 bg-red-500 rounded-full border-2 border-surface"></div>
            <?php endif; ?>
        </button>
        
        <!-- Profile Section -->
        <div class="flex items-center gap-3">
            <button onclick="openProfileModal()" class="flex items-center gap-3 hover:bg-surface rounded-lg p-2 transition-colors">
                <img alt="Admin Avatar" class="w-12 h-12 rounded-full object-cover border-2 border-line-light" src="<?php echo htmlspecialchars($admin_avatar); ?>"/>
                <div class="text-left">
                    <p class="font-bold text-text-primary"><?php echo htmlspecialchars($admin_username); ?></p>
                    <p class="text-sm text-text-secondary">Administrator</p>
                </div>
            </button>
        </div>
    </div>
</header>

<!-- Profile Edit Modal -->
<div id="profileModal" class="fixed inset-0 bg-black bg-opacity-50 z-50 hidden">
    <div class="flex items-center justify-center min-h-screen p-4">
        <div class="bg-white dark:bg-gray-800 rounded-2xl shadow-2xl w-full max-w-md transform transition-all">
            <!-- Modal Header -->
            <div class="flex items-center justify-between p-6 border-b border-gray-200 dark:border-gray-700">
                <h3 class="text-xl font-bold text-gray-900 dark:text-white">Edit Profile</h3>
                <button onclick="closeProfileModal()" class="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300 transition-colors">
                    <span class="material-symbols-outlined text-2xl">close</span>
                </button>
            </div>
            
            <!-- Modal Body -->
            <form id="profileForm" class="p-6 space-y-4">
                <!-- Profile Photo -->
                <div class="flex flex-col items-center space-y-4">
                    <div class="relative">
                        <img id="profilePreview" src="<?php echo htmlspecialchars($admin_avatar); ?>" alt="Profile Preview" class="w-24 h-24 rounded-full object-cover border-4 border-gray-200 dark:border-gray-600">
                        <input type="file" id="profilePhoto" name="profile_photo" accept="image/*" class="hidden" onchange="previewProfilePhoto(this)">
                        <button type="button" onclick="document.getElementById('profilePhoto').click()" class="absolute bottom-0 right-0 bg-primary text-white rounded-full p-2 hover:bg-primary-dark transition-colors">
                            <span class="material-symbols-outlined text-sm">camera_alt</span>
                        </button>
                    </div>
                </div>
                
                <!-- Username -->
                <div>
                    <label for="username" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Username</label>
                    <input type="text" id="username" name="username" value="<?php echo htmlspecialchars($admin_username); ?>" class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white transition-colors">
                </div>
                
                <!-- Email -->
                <div>
                    <label for="email" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Email</label>
                    <input type="email" id="email" name="email" value="<?php echo htmlspecialchars($admin_email); ?>" class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white transition-colors">
                </div>
                
                <!-- Current Password -->
                <div>
                    <label for="current_password" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Current Password</label>
                    <div class="relative">
                        <input type="password" id="current_password" name="current_password" class="w-full px-4 py-3 pr-12 border border-gray-300 dark:border-gray-600 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white transition-colors">
                        <button type="button" onclick="togglePassword('current_password')" class="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-gray-600 dark:hover:text-gray-300">
                            <span class="material-symbols-outlined">visibility</span>
                        </button>
                    </div>
                </div>
                
                <!-- New Password -->
                <div>
                    <label for="new_password" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">New Password (optional)</label>
                    <div class="relative">
                        <input type="password" id="new_password" name="new_password" class="w-full px-4 py-3 pr-12 border border-gray-300 dark:border-gray-600 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white transition-colors">
                        <button type="button" onclick="togglePassword('new_password')" class="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-gray-600 dark:hover:text-gray-300">
                            <span class="material-symbols-outlined">visibility</span>
                        </button>
                    </div>
                </div>
                
                <!-- Confirm New Password -->
                <div>
                    <label for="confirm_password" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Confirm New Password</label>
                    <div class="relative">
                        <input type="password" id="confirm_password" name="confirm_password" class="w-full px-4 py-3 pr-12 border border-gray-300 dark:border-gray-600 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white transition-colors">
                        <button type="button" onclick="togglePassword('confirm_password')" class="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-gray-600 dark:hover:text-gray-300">
                            <span class="material-symbols-outlined">visibility</span>
                        </button>
                    </div>
                </div>
            </form>
            
            <!-- Modal Footer -->
            <div class="flex gap-3 p-6 border-t border-gray-200 dark:border-gray-700">
                <button onclick="closeProfileModal()" class="flex-1 px-4 py-3 text-gray-600 dark:text-gray-300 border border-gray-300 dark:border-gray-600 rounded-xl hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors">
                    Cancel
                </button>
                <button onclick="saveProfile()" class="flex-1 px-4 py-3 bg-primary text-white rounded-xl hover:bg-primary-dark transition-colors">
                    Save Changes
                </button>
            </div>
        </div>
    </div>
</div>

<script>
// Profile Modal Functions
function openProfileModal() {
    document.getElementById('profileModal').classList.remove('hidden');
    document.body.style.overflow = 'hidden';
}

function closeProfileModal() {
    document.getElementById('profileModal').classList.add('hidden');
    document.body.style.overflow = 'auto';
}

// Password toggle function
function togglePassword(inputId) {
    const input = document.getElementById(inputId);
    const button = input.nextElementSibling;
    const icon = button.querySelector('.material-symbols-outlined');
    
    if (input.type === 'password') {
        input.type = 'text';
        icon.textContent = 'visibility_off';
    } else {
        input.type = 'password';
        icon.textContent = 'visibility';
    }
}

// Profile photo preview
function previewProfilePhoto(input) {
    if (input.files && input.files[0]) {
        const reader = new FileReader();
        reader.onload = function(e) {
            document.getElementById('profilePreview').src = e.target.result;
        };
        reader.readAsDataURL(input.files[0]);
    }
}

// Save profile function
function saveProfile() {
    const form = document.getElementById('profileForm');
    const formData = new FormData(form);
    formData.append('action', 'update_profile');
    
    // Client-side validation
    const username = formData.get('username');
    const email = formData.get('email');
    const newPassword = formData.get('new_password');
    const confirmPassword = formData.get('confirm_password');
    
    // Validate username
    if (username && username.length < 3) {
        showNotification('Username must be at least 3 characters long', 'error');
        return;
    }
    
    if (username && !/^[a-zA-Z0-9_]+$/.test(username)) {
        showNotification('Username can only contain letters, numbers, and underscores', 'error');
        return;
    }
    
    // Validate email
    if (email && !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) {
        showNotification('Please enter a valid email address', 'error');
        return;
    }
    
    // Validate password
    if (newPassword && newPassword.length < 6) {
        showNotification('New password must be at least 6 characters long', 'error');
        return;
    }
    
    if (newPassword && newPassword !== confirmPassword) {
        showNotification('New passwords do not match', 'error');
        return;
    }
    
    // Show loading state
    const saveButton = document.querySelector('button[onclick="saveProfile()"]');
    const originalText = saveButton.textContent;
    saveButton.textContent = 'Saving...';
    saveButton.disabled = true;
    
    fetch('profile_update.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Show success message
            showNotification('Profile updated successfully!', 'success');
            closeProfileModal();
            
            // Update header avatar if changed
            if (data.avatar_url) {
                const headerAvatar = document.querySelector('img[alt="Admin Avatar"]');
                if (headerAvatar) {
                    headerAvatar.src = data.avatar_url;
                }
            }
            
            // Clear password fields for security
            document.getElementById('current_password').value = '';
            document.getElementById('new_password').value = '';
            document.getElementById('confirm_password').value = '';
        } else {
            showNotification(data.message || 'Error updating profile', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('Error updating profile', 'error');
    })
    .finally(() => {
        saveButton.textContent = originalText;
        saveButton.disabled = false;
    });
}

// Notification function
function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `fixed top-4 right-4 z-50 px-6 py-4 rounded-xl shadow-lg transition-all transform translate-x-full ${
        type === 'success' ? 'bg-green-500 text-white' : 
        type === 'error' ? 'bg-red-500 text-white' : 
        'bg-blue-500 text-white'
    }`;
    notification.textContent = message;
    
    document.body.appendChild(notification);
    
    // Animate in
    setTimeout(() => {
        notification.classList.remove('translate-x-full');
    }, 100);
    
    // Auto remove after 3 seconds
    setTimeout(() => {
        notification.classList.add('translate-x-full');
        setTimeout(() => {
            document.body.removeChild(notification);
        }, 300);
    }, 3000);
}

// Close modal when clicking outside
document.getElementById('profileModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeProfileModal();
    }
});

// Close modal with Escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeProfileModal();
    }
});
</script>
