<?php
session_start();
require_once __DIR__ . '/../includes/config.php';
require_once 'admin_auth.php';

$error = '';
$success = '';

// IP ban ekleme
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add_ban') {
    $ip_address = trim($_POST['ip_address'] ?? '');
    $reason = trim($_POST['reason'] ?? '');
    
    if (empty($ip_address)) {
        $error = 'IP adresi gereklidir.';
    } elseif (!filter_var($ip_address, FILTER_VALIDATE_IP)) {
        $error = 'Geçerli bir IP adresi girin.';
    } elseif (empty($reason)) {
        $error = 'Ban nedeni gereklidir.';
    } else {
        try {
            $stmt = $pdo->prepare('INSERT INTO banned_ips (ip_address, reason, banned_by) VALUES (?, ?, ?)');
            $stmt->execute([$ip_address, $reason, $_SESSION['user_id']]);
            $success = 'IP adresi başarıyla banlandı.';
        } catch (PDOException $e) {
            if ($e->getCode() == 23000) { // Duplicate entry
                $error = 'Bu IP adresi zaten banlı.';
            } else {
                $error = 'Ban işlemi sırasında hata: ' . $e->getMessage();
            }
        }
    }
}

// IP ban kaldırma
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'remove_ban') {
    $ban_id = intval($_POST['ban_id'] ?? 0);
    
    if ($ban_id > 0) {
        try {
            $stmt = $pdo->prepare('UPDATE banned_ips SET is_active = 0 WHERE id = ?');
            $stmt->execute([$ban_id]);
            $success = 'IP ban başarıyla kaldırıldı.';
        } catch (Exception $e) {
            $error = 'Ban kaldırma sırasında hata: ' . $e->getMessage();
        }
    }
}

// Filtreleme ve sayfalama
$perPage = 20;
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$offset = ($page - 1) * $perPage;

$whereArr = ['is_active = 1'];
$params = [];

if (isset($_GET['search']) && $_GET['search'] !== '') {
    $whereArr[] = '(ip_address LIKE :search OR reason LIKE :search)';
    $params[':search'] = '%' . $_GET['search'] . '%';
}

$whereSQL = 'WHERE ' . implode(' AND ', $whereArr);

// Toplam kayıt sayısı
$totalStmt = $pdo->prepare('SELECT COUNT(*) FROM banned_ips ' . $whereSQL);
$totalStmt->execute($params);
$totalRows = $totalStmt->fetchColumn();
$totalPages = ceil($totalRows / $perPage);

// Banlı IP'leri getir
$sql = '
    SELECT bi.*, u.username as banned_by_name 
    FROM banned_ips bi 
    LEFT JOIN user u ON bi.banned_by = u.id 
    ' . $whereSQL . ' 
    ORDER BY bi.banned_at DESC 
    LIMIT :limit OFFSET :offset
';
$stmt = $pdo->prepare($sql);
foreach ($params as $k => $v) $stmt->bindValue($k, $v);
$stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();
$banned_ips = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="utf-8"/>
    <meta content="width=device-width, initial-scale=1.0" name="viewport"/>
    <title>IP Ban Yönetimi - Admin Panel</title>
    <link href="https://fonts.googleapis.com" rel="preconnect"/>
    <link crossorigin="" href="https://fonts.gstatic.com" rel="preconnect"/>
    <link href="https://fonts.googleapis.com/css2?family=Manrope:wght@400;500;600;700;800&display=swap" rel="stylesheet"/>
    <script src="https://cdn.tailwindcss.com?plugins=forms,container-queries"></script>
    <link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined:opsz,wght,FILL,GRAD@20..48,100..700,0..1,-50..200" rel="stylesheet"/>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: "#1877f2",
                        "primary-light": "#e7f3ff",
                        background: "#f0f2f5",
                        surface: "#ffffff",
                        "text-primary": "#1c1e21",
                        "text-secondary": "#606770",
                        "line-light": "#e0e0e0",
                    },
                    fontFamily: {
                        display: ["Manrope", "sans-serif"],
                    },
                    borderRadius: {
                        DEFAULT: "0.75rem",
                        lg: "1rem",
                        xl: "1.5rem",
                        full: "9999px"
                    },
                    boxShadow: {
                        'soft': '0 4px 12px 0 rgb(0 0 0 / 0.07)',
                        'subtle': '0 1px 2px 0 rgb(0 0 0 / 0.05)',
                    }
                },
            },
        };
    </script>
    <style>
        .material-symbols-outlined {
            font-variation-settings: "FILL" 0, "wght" 400, "GRAD" 0, "opsz" 24;
        }
    </style>
</head>
<body class="bg-background font-display text-text-primary">
    <div class="flex min-h-screen">
        <?php include 'menu.php'; ?>
        <main class="flex-1 p-10 min-w-0">
            <header class="flex justify-between items-center mb-10">
                <div>
                    <h1 class="text-4xl font-extrabold text-text-primary">IP Ban Yönetimi</h1>
                    <p class="text-text-secondary mt-1">IP adreslerini banla ve yönet.</p>
                </div>
                <div class="flex items-center gap-4">
                    <button onclick="document.getElementById('addBanModal').style.display='block'" 
                        class="flex items-center gap-2 bg-red-600 text-white font-bold py-3 px-5 rounded-lg hover:bg-red-700 transition-colors">
                        <span class="material-symbols-outlined">block</span>
                        <span>IP Banla</span>
                    </button>
                </div>
            </header>

            <!-- Mesajlar -->
            <?php if (!empty($error)): ?>
            <div class="bg-red-50 border-l-4 border-red-400 p-4 mb-6">
                <div class="flex items-center">
                    <span class="material-symbols-outlined text-red-400 mr-2">error</span>
                    <p class="text-red-700"><?php echo htmlspecialchars($error); ?></p>
                </div>
            </div>
            <?php endif; ?>

            <?php if (!empty($success)): ?>
            <div class="bg-green-50 border-l-4 border-green-400 p-4 mb-6">
                <div class="flex items-center">
                    <span class="material-symbols-outlined text-green-400 mr-2">check_circle</span>
                    <p class="text-green-700"><?php echo htmlspecialchars($success); ?></p>
                </div>
            </div>
            <?php endif; ?>

            <!-- Arama ve Filtreler -->
            <div class="bg-surface p-6 rounded-xl border border-line-light shadow-soft mb-6">
                <form method="GET" class="flex justify-between items-center">
                    <div class="relative w-full max-w-md">
                        <span class="material-symbols-outlined absolute left-4 top-1/2 -translate-y-1/2 text-text-secondary">search</span>
                        <input name="search" value="<?php echo htmlspecialchars($_GET['search'] ?? ''); ?>" 
                            class="w-full bg-background border-transparent rounded-lg pl-12 pr-4 py-3 focus:ring-2 focus:ring-primary focus:border-transparent placeholder:text-text-secondary" 
                            placeholder="IP adresi veya ban nedeni ara..." type="text"/>
                    </div>
                    <button type="submit" class="flex items-center gap-2 text-text-secondary font-medium hover:text-text-primary">
                        <span class="material-symbols-outlined">search</span>
                        <span>Ara</span>
                    </button>
                </form>
            </div>

            <!-- IP Ban Listesi -->
            <?php if (empty($banned_ips)): ?>
            <div class="bg-white rounded-xl shadow-soft border border-line-light p-12 text-center">
                <div class="w-20 h-20 rounded-full bg-background mx-auto mb-4 flex items-center justify-center">
                    <span class="material-symbols-outlined text-4xl text-text-secondary">block</span>
                </div>
                <h3 class="text-xl font-bold text-text-primary mb-2">Banlı IP Bulunamadı</h3>
                <p class="text-text-secondary mb-6">Henüz banlı IP adresi yok. İlk IP banını eklemek için yukarıdaki butonu kullanabilirsin.</p>
            </div>
            <?php else: ?>

            <div class="overflow-x-auto bg-white rounded-xl shadow-soft border border-line-light">
                <table class="w-full text-left table-fixed min-w-[800px]">
                    <thead class="border-b border-line-light">
                        <tr class="bg-gradient-to-r from-red-50 to-red-100 border-b-2 border-red-200">
                            <th class="p-6 font-bold text-text-primary text-lg w-40">
                                <div class="flex items-center gap-3">
                                    <span class="material-symbols-outlined text-red-600">location_on</span>
                                    IP Adresi
                                </div>
                            </th>
                            <th class="p-6 font-bold text-text-primary text-lg">
                                <div class="flex items-center gap-3">
                                    <span class="material-symbols-outlined text-red-600">description</span>
                                    Ban Nedeni
                                </div>
                            </th>
                            <th class="p-6 font-bold text-text-primary text-lg w-40">
                                <div class="flex items-center gap-3">
                                    <span class="material-symbols-outlined text-red-600">person</span>
                                    Banlayana
                                </div>
                            </th>
                            <th class="p-6 font-bold text-text-primary text-lg w-40">
                                <div class="flex items-center gap-3">
                                    <span class="material-symbols-outlined text-red-600">schedule</span>
                                    Ban Tarihi
                                </div>
                            </th>
                            <th class="p-6 font-bold text-text-primary text-lg text-center w-32">
                                <div class="flex items-center justify-center gap-3">
                                    <span class="material-symbols-outlined text-red-600">settings</span>
                                    İşlemler
                                </div>
                            </th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($banned_ips as $ban): ?>
                        <tr class="border-b border-line-light hover:bg-gradient-to-r hover:from-red-50/50 hover:to-transparent transition-all duration-200">
                            <td class="p-6">
                                <div class="flex items-center gap-2">
                                    <span class="material-symbols-outlined text-red-500">block</span>
                                    <span class="font-mono font-bold text-text-primary"><?php echo htmlspecialchars($ban['ip_address']); ?></span>
                                </div>
                            </td>
                            <td class="p-6">
                                <p class="text-text-primary"><?php echo htmlspecialchars($ban['reason']); ?></p>
                            </td>
                            <td class="p-6">
                                <span class="text-text-secondary"><?php echo htmlspecialchars($ban['banned_by_name'] ?? 'Bilinmiyor'); ?></span>
                            </td>
                            <td class="p-6">
                                <div class="flex items-center gap-2 text-text-secondary">
                                    <span class="material-symbols-outlined text-sm">calendar_today</span>
                                    <span><?php echo date('d.m.Y H:i', strtotime($ban['banned_at'])); ?></span>
                                </div>
                            </td>
                            <td class="p-6">
                                <div class="flex justify-center">
                                    <form method="POST" class="inline" onsubmit="return confirm('Bu IP banını kaldırmak istediğinizden emin misiniz?');">
                                        <input type="hidden" name="action" value="remove_ban">
                                        <input type="hidden" name="ban_id" value="<?php echo $ban['id']; ?>">
                                        <button type="submit" 
                                            class="w-10 h-10 rounded-lg bg-green-100 text-green-600 flex items-center justify-center hover:bg-green-200 hover:scale-105 transition-all duration-200 border border-green-200" 
                                            title="Banı Kaldır">
                                            <span class="material-symbols-outlined text-lg">check_circle</span>
                                        </button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <!-- Sayfalama -->
            <?php if ($totalPages > 1): ?>
            <div class="flex justify-between items-center mt-6 pt-4 border-t border-line-light">
                <p class="text-sm text-text-secondary">
                    <?php echo $offset + 1; ?> - <?php echo min($offset + $perPage, $totalRows); ?> arası, 
                    toplam <?php echo $totalRows; ?> banlı IP
                </p>
                <div class="flex items-center gap-2">
                    <?php if ($page > 1): ?>
                        <a href="?page=<?php echo ($page - 1); ?>&search=<?php echo urlencode($_GET['search'] ?? ''); ?>" 
                            class="w-9 h-9 flex items-center justify-center rounded-lg border border-line-light hover:bg-background text-text-secondary">
                            <span class="material-symbols-outlined text-xl">chevron_left</span>
                        </a>
                    <?php endif; ?>
                    
                    <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                        <a href="?page=<?php echo $i; ?>&search=<?php echo urlencode($_GET['search'] ?? ''); ?>" 
                            class="w-9 h-9 flex items-center justify-center rounded-lg <?php echo $i == $page ? 'bg-primary text-white font-bold' : 'hover:bg-background text-text-secondary'; ?>">
                            <?php echo $i; ?>
                        </a>
                    <?php endfor; ?>
                    
                    <?php if ($page < $totalPages): ?>
                        <a href="?page=<?php echo ($page + 1); ?>&search=<?php echo urlencode($_GET['search'] ?? ''); ?>" 
                            class="w-9 h-9 flex items-center justify-center rounded-lg border border-line-light hover:bg-background text-text-secondary">
                            <span class="material-symbols-outlined text-xl">chevron_right</span>
                        </a>
                    <?php endif; ?>
                </div>
            </div>
            <?php endif; ?>

            <?php endif; ?>
        </main>
    </div>

    <!-- IP Ban Ekleme Modal -->
    <div id="addBanModal" style="display:none" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
        <div class="bg-white rounded-xl p-8 max-w-md w-full mx-4">
            <div class="flex justify-between items-center mb-6">
                <h3 class="text-xl font-bold text-text-primary">IP Adresi Banla</h3>
                <button onclick="document.getElementById('addBanModal').style.display='none'" 
                    class="text-text-secondary hover:text-text-primary">
                    <span class="material-symbols-outlined">close</span>
                </button>
            </div>
            
            <form method="POST" class="space-y-4">
                <input type="hidden" name="action" value="add_ban">
                
                <div>
                    <label for="ip_address" class="block text-sm font-medium text-text-primary mb-2">
                        IP Adresi *
                    </label>
                    <input id="ip_address" name="ip_address" type="text" required
                        class="w-full border border-line-light rounded-lg px-4 py-3 focus:ring-2 focus:ring-red-500 focus:border-transparent placeholder:text-text-secondary"
                        placeholder="192.168.1.1 veya 2001:db8::1"/>
                </div>
                
                <div>
                    <label for="reason" class="block text-sm font-medium text-text-primary mb-2">
                        Ban Nedeni *
                    </label>
                    <textarea id="reason" name="reason" rows="3" required
                        class="w-full border border-line-light rounded-lg px-4 py-3 focus:ring-2 focus:ring-red-500 focus:border-transparent placeholder:text-text-secondary"
                        placeholder="Bu IP adresinin neden banlandığını açıklayın..."></textarea>
                </div>
                
                <div class="flex gap-4 pt-4">
                    <button type="submit" 
                        class="flex-1 bg-red-600 text-white font-bold py-3 px-4 rounded-lg hover:bg-red-700 transition-colors">
                        IP'yi Banla
                    </button>
                    <button type="button" onclick="document.getElementById('addBanModal').style.display='none'"
                        class="flex-1 bg-gray-100 text-gray-700 font-bold py-3 px-4 rounded-lg hover:bg-gray-200 transition-colors">
                        İptal
                    </button>
                </div>
            </form>
        </div>
    </div>
</body>
</html>

