<?php
require_once 'admin_auth.php';
require_once '../includes/config.php';

// Handle menu item deletion
if (isset($_POST['delete_menu']) && isset($_POST['menu_id'])) {
    $menu_id = intval($_POST['menu_id']);
    $delete_query = "DELETE FROM menu_items WHERE id = ?";
    $stmt = $conn->prepare($delete_query);
    $stmt->bind_param("i", $menu_id);
    if ($stmt->execute()) {
        $success_message = "Menü öğesi başarıyla silindi!";
    } else {
        $error_message = "Menü öğesi silinirken hata oluştu.";
    }
}

// Handle menu item order update
if (isset($_POST['update_order'])) {
    $menu_type = $_POST['menu_type'] ?? 'footer';
    $orders = $_POST['menu_order'];
    foreach ($orders as $id => $order) {
        $update_query = "UPDATE menu_items SET sort_order = ? WHERE id = ?";
        $stmt = $conn->prepare($update_query);
        $stmt->bind_param("ii", $order, $id);
        $stmt->execute();
    }
    $success_message = "Menü sırası başarıyla güncellendi!";
}

// Handle footer menu setting update
if (isset($_POST['update_footer_setting'])) {
    $footer_menu_enabled = isset($_POST['footer_menu_enabled']) ? '1' : '0';
    
    // Update setting in database
    $stmt = $pdo->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");
    if ($stmt->execute(['footer_menu_enabled', $footer_menu_enabled])) {
        $success_message = "Footer menü ayarı başarıyla güncellendi!";
    } else {
        $error_message = "Footer menü ayarı güncellenirken hata oluştu.";
    }
}

// Get current menu type (only footer menu)
$current_menu_type = 'footer';

// Load settings
$settings = [];
$stmt = $pdo->query("SELECT setting_key, setting_value FROM settings");
foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// Check if menu_items table exists and has menu_type column
$table_exists = false;
$has_menu_type = false;

try {
    // Check if table exists
    $check_table = $conn->query("SHOW TABLES LIKE 'menu_items'");
    if ($check_table->num_rows > 0) {
        $table_exists = true;
        
        // Check if menu_type column exists
        $check_column = $conn->query("SHOW COLUMNS FROM menu_items LIKE 'menu_type'");
        if ($check_column->num_rows > 0) {
            $has_menu_type = true;
        }
    }
} catch (Exception $e) {
    // Table doesn't exist yet
}

$menus_result = null;
if ($table_exists && $has_menu_type) {
    // Fetch menu items by type
    $menus_query = "SELECT * FROM menu_items WHERE menu_type = ? ORDER BY sort_order ASC, id ASC";
    $stmt = $conn->prepare($menus_query);
    $stmt->bind_param("s", $current_menu_type);
    $stmt->execute();
    $menus_result = $stmt->get_result();
} elseif ($table_exists) {
    // Table exists but no menu_type column - fetch all items
    $menus_query = "SELECT * FROM menu_items ORDER BY sort_order ASC, id ASC";
    $menus_result = $conn->query($menus_query);
}

// Check for success message
if (isset($_GET['success'])) {
    $success_message = $_GET['success'];
}
?>

<?php include 'menu.php'; ?>

<div class="flex-1 bg-background min-h-screen">
    <div class="p-8">
        <?php
        $page_title = "Menü Yönetimi";
        $page_subtitle = "Header ve footer için navigasyon menü öğelerini yönetin";
        include 'includes/admin_header.php';
        ?>
        

        <?php if (!$table_exists): ?>
            <!-- Table Not Found Warning -->
            <div class="bg-yellow-50 border border-yellow-200 text-yellow-800 px-6 py-4 rounded-xl mb-6">
                <div class="flex items-center gap-2">
                    <span class="material-symbols-outlined">warning</span>
                    <strong>Veritabanı Tablosu Eksik</strong>
                </div>
                <p class="mt-2">menu_items tablosu mevcut değil. Lütfen sağlanan SQL kodunu kullanarak oluşturun.</p>
            </div>
        <?php elseif (!$has_menu_type): ?>
            <!-- Column Missing Warning -->
            <div class="bg-yellow-50 border border-yellow-200 text-yellow-800 px-6 py-4 rounded-xl mb-6">
                <div class="flex items-center gap-2">
                    <span class="material-symbols-outlined">warning</span>
                    <strong>Veritabanı Güncellemesi Gerekli</strong>
                </div>
                <p class="mt-2">menu_items tablosu mevcut ancak menu_type sütununa ihtiyaç var. Lütfen sağlanan SQL güncellemesini çalıştırın.</p>
            </div>
        <?php endif; ?>

        <!-- Menu Type Tabs -->
        <div class="flex gap-2 mb-6">
            <a href="?type=footer" class="inline-flex items-center gap-2 px-6 py-3 rounded-xl font-semibold transition-all duration-200 bg-primary text-white shadow-soft">
                <span class="material-symbols-outlined text-xl">web_asset</span>
                Footer Menüsü
            </a>
        </div>

        <!-- Footer Menu Settings -->
        <div class="bg-surface rounded-2xl shadow-soft border border-border-light p-6 mb-6">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-lg font-semibold text-text-primary">Footer Menü Ayarları</h3>
                <a href="menu_new.php?type=<?php echo $current_menu_type; ?>" class="inline-flex items-center gap-2 bg-primary hover:bg-primary-dark text-white font-semibold px-4 py-2 rounded-lg shadow-soft transition-all duration-200">
                    <span class="material-symbols-outlined text-lg">add</span>
                    Menü Öğesi Ekle
                </a>
            </div>
            <form method="POST" class="flex items-center gap-4">
                <div class="flex items-center gap-3">
                    <label class="text-text-secondary font-medium">Footer Menüsü Aktif:</label>
                    <label class="relative inline-flex items-center cursor-pointer">
                        <input type="checkbox" name="footer_menu_enabled" value="1" class="sr-only peer" <?php echo ($settings['footer_menu_enabled'] ?? '1') === '1' ? 'checked' : ''; ?>>
                        <div class="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-primary/20 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-primary"></div>
                    </label>
                </div>
                <button type="submit" name="update_footer_setting" class="bg-primary hover:bg-primary-dark text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors">
                    Ayarı Güncelle
                </button>
            </form>
        </div>

        <?php if (isset($success_message)): ?>
            <div class="bg-green-50 border border-green-200 text-green-700 px-6 py-4 rounded-xl mb-6">
                <div class="flex items-center gap-2">
                    <span class="material-symbols-outlined">check_circle</span>
                    <?php echo $success_message; ?>
                </div>
            </div>
        <?php endif; ?>

        <?php if (isset($error_message)): ?>
            <div class="bg-red-50 border border-red-200 text-red-700 px-6 py-4 rounded-xl mb-6">
                <div class="flex items-center gap-2">
                    <span class="material-symbols-outlined">error</span>
                    <?php echo $error_message; ?>
                </div>
            </div>
        <?php endif; ?>

        <!-- Menu Items -->
        <div class="bg-surface rounded-2xl shadow-soft border border-border-light overflow-hidden">
            <div class="p-6 border-b border-border-light">
                <div class="flex justify-between items-center">
                    <h2 class="text-xl font-semibold text-text-primary">Footer Menü Öğeleri</h2>
                    <div class="text-sm text-text-muted">Sıralamak için sürükleyin</div>
                </div>
            </div>
            
            <?php if ($menus_result && $menus_result->num_rows > 0): ?>
                <form method="POST" id="orderForm">
                    <?php if ($has_menu_type): ?>
                        <input type="hidden" name="menu_type" value="<?php echo $current_menu_type; ?>">
                    <?php endif; ?>
                    <div class="divide-y divide-border-light" id="menuItems">
                        <?php while($menu = $menus_result->fetch_assoc()): ?>
                            <div class="p-6 hover:bg-background transition-colors menu-item" data-id="<?php echo $menu['id']; ?>">
                                <div class="flex items-center justify-between">
                                    <div class="flex items-center gap-4">
                                        <div class="drag-handle cursor-move">
                                            <span class="material-symbols-outlined text-text-muted">drag_indicator</span>
                                        </div>
                                        <div class="flex-1">
                                            <div class="flex items-center gap-3 mb-2">
                                                <h3 class="font-semibold text-text-primary"><?php echo htmlspecialchars($menu['title']); ?></h3>
                                                <span class="inline-flex items-center gap-1 bg-blue-100 text-blue-700 px-2 py-1 rounded-lg text-xs font-medium">
                                                    Sıra: <?php echo $menu['sort_order']; ?>
                                                </span>
                                            </div>
                                            <div class="flex items-center gap-4 text-sm text-text-secondary">
                                                <span class="flex items-center gap-1">
                                                    <span class="material-symbols-outlined text-sm">link</span>
                                                    <?php echo htmlspecialchars($menu['url']); ?>
                                                </span>
                                                <?php if ($menu['target'] == '_blank'): ?>
                                                    <span class="flex items-center gap-1 text-orange-600">
                                                        <span class="material-symbols-outlined text-sm">open_in_new</span>
                                                        Yeni Sekme
                                                    </span>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="flex items-center gap-2">
                                        <input type="hidden" name="menu_order[<?php echo $menu['id']; ?>]" value="<?php echo $menu['sort_order']; ?>" class="order-input">
                                        
                                        <a href="menu_edit.php?id=<?php echo $menu['id']; ?>&type=<?php echo $current_menu_type; ?>" 
                                           class="inline-flex items-center gap-1 bg-blue-50 hover:bg-blue-100 text-blue-600 px-3 py-2 rounded-lg text-sm font-medium transition-colors">
                                            <span class="material-symbols-outlined text-base">edit</span>
                                            Düzenle
                                        </a>
                                        
                                        <button type="button" onclick="deleteMenuItem(<?php echo $menu['id']; ?>, '<?php echo htmlspecialchars($menu['title']); ?>')" 
                                                class="inline-flex items-center gap-1 bg-red-50 hover:bg-red-100 text-red-600 px-3 py-2 rounded-lg text-sm font-medium transition-colors">
                                            <span class="material-symbols-outlined text-base">delete</span>
                                            Sil
                                        </button>
                                    </div>
                                </div>
                            </div>
                        <?php endwhile; ?>
                    </div>
                    
                    <div class="p-6 bg-background border-t border-border-light">
                        <button type="submit" name="update_order" 
                                class="inline-flex items-center gap-2 bg-secondary hover:bg-orange-600 text-white font-semibold px-6 py-3 rounded-xl shadow-soft transition-all duration-200">
                            <span class="material-symbols-outlined text-xl">save</span>
                            Sırayı Kaydet
                        </button>
                    </div>
                </form>
            <?php elseif ($table_exists): ?>
                <div class="p-12 text-center">
                    <div class="w-16 h-16 bg-background rounded-2xl flex items-center justify-center mx-auto mb-4">
                        <span class="material-symbols-outlined text-2xl text-text-muted">menu</span>
                    </div>
                    <h3 class="text-xl font-semibold text-text-primary mb-2">Menü öğesi bulunamadı</h3>
                    <p class="text-text-secondary mb-6">İlk <?php echo $current_menu_type; ?> menü öğenizi oluşturun</p>
                    <?php if ($has_menu_type): ?>
                        <a href="menu_new.php?type=<?php echo $current_menu_type; ?>" class="inline-flex items-center gap-2 bg-primary hover:bg-primary-dark text-white font-semibold px-6 py-3 rounded-xl shadow-soft transition-all duration-200">
                            <span class="material-symbols-outlined text-xl">add</span>
                            İlk Menü Öğesini Oluştur
                        </a>
                    <?php else: ?>
                        <p class="text-red-600">Menü yönetimini etkinleştirmek için lütfen önce veritabanı tablosunu güncelleyin.</p>
                    <?php endif; ?>
                </div>
            <?php else: ?>
                <div class="p-12 text-center">
                    <div class="w-16 h-16 bg-red-100 rounded-2xl flex items-center justify-center mx-auto mb-4">
                        <span class="material-symbols-outlined text-2xl text-red-600">database</span>
                    </div>
                    <h3 class="text-xl font-semibold text-text-primary mb-2">Veritabanı Tablosu Eksik</h3>
                    <p class="text-text-secondary mb-6">Lütfen sağlanan SQL kodunu kullanarak menu_items tablosunu oluşturun.</p>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Delete confirmation modal -->
<form method="POST" id="deleteForm" class="hidden">
    <input type="hidden" name="menu_id" id="deleteMenuId">
    <input type="hidden" name="delete_menu" value="1">
</form>

<script>
function deleteMenuItem(menuId, menuTitle) {
    if (confirm(`"${menuTitle}" öğesini silmek istediğinizden emin misiniz? Bu işlem geri alınamaz.`)) {
        document.getElementById('deleteMenuId').value = menuId;
        document.getElementById('deleteForm').submit();
    }
}

// Simple drag and drop for reordering
let draggedElement = null;

document.querySelectorAll('.menu-item').forEach(item => {
    item.draggable = true;
    
    item.addEventListener('dragstart', function(e) {
        draggedElement = this;
        this.style.opacity = '0.5';
    });
    
    item.addEventListener('dragend', function(e) {
        this.style.opacity = '';
        draggedElement = null;
        updateOrderInputs();
    });
    
    item.addEventListener('dragover', function(e) {
        e.preventDefault();
    });
    
    item.addEventListener('drop', function(e) {
        e.preventDefault();
        if (draggedElement !== this) {
            const container = document.getElementById('menuItems');
            const draggedIndex = Array.from(container.children).indexOf(draggedElement);
            const dropIndex = Array.from(container.children).indexOf(this);
            
            if (draggedIndex < dropIndex) {
                container.insertBefore(draggedElement, this.nextSibling);
            } else {
                container.insertBefore(draggedElement, this);
            }
        }
    });
});

function updateOrderInputs() {
    const items = document.querySelectorAll('.menu-item');
    items.forEach((item, index) => {
        const orderInput = item.querySelector('.order-input');
        if (orderInput) {
            orderInput.value = index + 1;
        }
        const orderDisplay = item.querySelector('.bg-blue-100');
        if (orderDisplay) {
            orderDisplay.textContent = `Order: ${index + 1}`;
        }
    });
}
</script>

</body>
</html>

