<?php
require_once 'admin_auth.php';
require_once '../includes/config.php';

// Get page ID
$page_id = isset($_GET['id']) ? intval($_GET['id']) : 0;

if ($page_id <= 0) {
    header("Location: pages.php");
    exit();
}

// Fetch page data
$page_query = "SELECT * FROM pages WHERE id = ?";
$stmt = $conn->prepare($page_query);
$stmt->bind_param("i", $page_id);
$stmt->execute();
$page_result = $stmt->get_result();

if ($page_result->num_rows == 0) {
    header("Location: pages.php");
    exit();
}

$page = $page_result->fetch_assoc();

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $title = trim($_POST['title']);
    $slug = trim($_POST['slug']);
    $content = $_POST['content'];
    $meta_description = trim($_POST['meta_description']);
    $meta_keywords = trim($_POST['meta_keywords']);
    $status = $_POST['status'];
    $created_at = !empty($_POST['created_at']) ? $_POST['created_at'] : $page['created_at'];
    
    // Auto-generate slug if empty
    if (empty($slug)) {
        $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $title)));
    }
    
    // Validate required fields
    if (empty($title) || empty($content)) {
        $error_message = "Başlık ve içerik gerekli alanlardır.";
    } else {
        // Check if slug already exists (excluding current page)
        $check_slug = "SELECT id FROM pages WHERE slug = ? AND id != ?";
        $stmt = $conn->prepare($check_slug);
        $stmt->bind_param("si", $slug, $page_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $error_message = "Bu slug ile bir sayfa zaten mevcut. Lütfen farklı bir slug seçin.";
        } else {
            // Update page with custom date
            $update_query = "UPDATE pages SET title = ?, slug = ?, content = ?, meta_description = ?, meta_keywords = ?, status = ?, created_at = ?, updated_at = NOW() WHERE id = ?";
            $stmt = $conn->prepare($update_query);
            $stmt->bind_param("sssssssi", $title, $slug, $content, $meta_description, $meta_keywords, $status, $created_at, $page_id);
            
            if ($stmt->execute()) {
                header("Location: pages.php?success=Sayfa başarıyla güncellendi");
                exit();
            } else {
                $error_message = "Sayfa güncellenirken hata oluştu. Lütfen tekrar deneyin.";
            }
        }
    }
} else {
    // Pre-fill form with existing data
    $_POST = $page;
}
?>

<?php include 'menu.php'; ?>

<!-- Quill Editor CSS -->
<link href="https://cdn.quilljs.com/1.3.6/quill.snow.css" rel="stylesheet">
<style>
    .ql-editor {
        min-height: 350px;
        font-family: 'Inter', system-ui, -apple-system, sans-serif;
        font-size: 14px;
        line-height: 1.6;
    }
    .ql-toolbar {
        border-top: 1px solid #e5e7eb;
        border-left: 1px solid #e5e7eb;
        border-right: 1px solid #e5e7eb;
        border-bottom: none;
        background: #f9fafb;
    }
    .ql-container {
        border-bottom: 1px solid #e5e7eb;
        border-left: 1px solid #e5e7eb;
        border-right: 1px solid #e5e7eb;
        border-top: none;
        font-family: inherit;
    }
</style>

<div class="flex-1 bg-background min-h-screen">
    <div class="p-8">
        <!-- Header -->
        <div class="flex items-center gap-4 mb-8">
            <a href="pages.php" class="inline-flex items-center justify-center w-10 h-10 bg-surface hover:bg-surface-hover rounded-xl border border-border-light transition-colors">
                <span class="material-symbols-outlined text-text-secondary">arrow_back</span>
            </a>
            <div>
                <h1 class="text-3xl font-bold text-text-primary mb-2">Edit Page</h1>
                <p class="text-text-secondary">Modify page: <?php echo htmlspecialchars($page['title']); ?></p>
            </div>
        </div>

        <?php if (isset($error_message)): ?>
            <div class="bg-red-50 border border-red-200 text-red-700 px-6 py-4 rounded-xl mb-6">
                <div class="flex items-center gap-2">
                    <span class="material-symbols-outlined">error</span>
                    <?php echo $error_message; ?>
                </div>
            </div>
        <?php endif; ?>

        <!-- Form -->
        <form id="page-edit-form" method="POST" class="space-y-6">
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <!-- Main Content -->
                <div class="lg:col-span-2 space-y-6">
                    <!-- Title and Slug -->
                    <div class="bg-surface rounded-2xl shadow-soft border border-border-light p-6">
                        <h2 class="text-xl font-semibold text-text-primary mb-4">Page Details</h2>
                        
                        <div class="space-y-4">
                            <div>
                                <label for="title" class="block text-sm font-medium text-text-primary mb-2">Page Title *</label>
                                <input type="text" id="title" name="title" required 
                                       value="<?php echo htmlspecialchars($_POST['title']); ?>"
                                       class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors"
                                       placeholder="Enter page title">
                            </div>
                            
                            <div>
                                <label for="slug" class="block text-sm font-medium text-text-primary mb-2">URL Slug</label>
                                <div class="relative">
                                    <span class="absolute left-4 top-1/2 transform -translate-y-1/2 text-text-muted text-sm">/</span>
                                    <input type="text" id="slug" name="slug" 
                                           value="<?php echo htmlspecialchars($_POST['slug']); ?>"
                                           class="w-full pl-8 pr-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors"
                                           placeholder="page-slug">
                                </div>
                                <p class="text-xs text-text-muted mt-1">URL-friendly version of the title</p>
                            </div>
                        </div>
                    </div>

                    <!-- Content -->
                    <div class="bg-surface rounded-2xl shadow-soft border border-border-light p-6">
                        <h2 class="text-xl font-semibold text-text-primary mb-4">Page Content</h2>
                        
                        <div>
                            <label for="content" class="block text-sm font-medium text-text-primary mb-2">Content *</label>
                            <div class="border border-border-light rounded-xl overflow-hidden">
                                <div id="quill-editor" style="min-height: 400px;"><?php echo $_POST['content']; ?></div>
                            </div>
                            <textarea id="content" name="content" style="display: none;" required></textarea>
                            <p class="text-xs text-text-muted mt-1">Rich text editor with formatting options</p>
                        </div>
                    </div>

                    <!-- SEO Meta -->
                    <div class="bg-surface rounded-2xl shadow-soft border border-border-light p-6">
                        <h2 class="text-xl font-semibold text-text-primary mb-4">SEO Settings</h2>
                        
                        <div class="space-y-4">
                            <div>
                                <label for="meta_description" class="block text-sm font-medium text-text-primary mb-2">Meta Description</label>
                                <textarea id="meta_description" name="meta_description" rows="3"
                                          class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors"
                                          placeholder="Brief description for search engines (150-160 characters)"><?php echo htmlspecialchars($_POST['meta_description']); ?></textarea>
                            </div>
                            
                            <div>
                                <label for="meta_keywords" class="block text-sm font-medium text-text-primary mb-2">Meta Keywords</label>
                                <input type="text" id="meta_keywords" name="meta_keywords" 
                                       value="<?php echo htmlspecialchars($_POST['meta_keywords']); ?>"
                                       class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors"
                                       placeholder="keyword1, keyword2, keyword3">
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Sidebar -->
                <div class="space-y-6">
                    <!-- Publish -->
                    <div class="bg-surface rounded-2xl shadow-soft border border-border-light p-6">
                        <h3 class="text-lg font-semibold text-text-primary mb-4">Publish Settings</h3>
                        
                        <div class="space-y-4">
                            <div>
                                <label for="status" class="block text-sm font-medium text-text-primary mb-2">Status</label>
                                <select id="status" name="status" 
                                        class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors">
                                    <option value="draft" <?php echo ($_POST['status'] == 'draft') ? 'selected' : ''; ?>>Draft</option>
                                    <option value="published" <?php echo ($_POST['status'] == 'published') ? 'selected' : ''; ?>>Published</option>
                                </select>
                            </div>
                            
                            <div>
                                <label for="created_at" class="block text-sm font-medium text-text-primary mb-2">Publication Date</label>
                                <input type="datetime-local" id="created_at" name="created_at" 
                                       value="<?php echo date('Y-m-d\TH:i', strtotime($_POST['created_at'] ?? $page['created_at'])); ?>"
                                       class="w-full px-4 py-3 border border-border-light rounded-xl focus:ring-2 focus:ring-primary focus:border-primary transition-colors">
                                <p class="text-xs text-text-muted mt-1">Set custom publication date and time</p>
                            </div>
                        </div>
                        
                        <div class="flex flex-col gap-3 mt-6 pt-6 border-t border-border-light">
                            <button type="submit" onclick="updatePageContent()" 
                                    class="w-full bg-primary hover:bg-primary-dark text-white font-semibold py-3 px-4 rounded-xl transition-colors">
                                Update Page
                            </button>
                            <a href="pages.php" 
                               class="w-full text-center bg-surface-hover hover:bg-border-light text-text-secondary font-medium py-3 px-4 rounded-xl transition-colors">
                                Cancel
                            </a>
                        </div>
                    </div>

                    <!-- Page Info -->
                    <div class="bg-surface rounded-2xl shadow-soft border border-border-light p-6">
                        <h3 class="text-lg font-semibold text-text-primary mb-4">Page Information</h3>
                        <div class="space-y-3 text-sm">
                            <div class="flex justify-between">
                                <span class="text-text-muted">Created:</span>
                                <span class="text-text-secondary"><?php echo date('M j, Y', strtotime($page['created_at'])); ?></span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-text-muted">Updated:</span>
                                <span class="text-text-secondary"><?php echo date('M j, Y', strtotime($page['updated_at'])); ?></span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-text-muted">Current URL:</span>
                                <span class="text-text-secondary" id="preview-url">/<?php echo htmlspecialchars($page['slug']); ?></span>
                            </div>
                        </div>
                    </div>

                    <!-- Delete Page -->
                    <div class="bg-red-50 border border-red-200 rounded-2xl p-6">
                        <h3 class="text-lg font-semibold text-red-700 mb-2">Danger Zone</h3>
                        <p class="text-sm text-red-600 mb-4">Once you delete this page, there is no going back.</p>
                        <form method="POST" action="pages.php" onsubmit="return confirm('Are you sure you want to delete this page? This action cannot be undone.')">
                            <input type="hidden" name="page_id" value="<?php echo $page['id']; ?>">
                            <button type="submit" name="delete_page" 
                                    class="w-full bg-red-600 hover:bg-red-700 text-white font-semibold py-3 px-4 rounded-xl transition-colors">
                                Delete Page
                            </button>
                        </form>
                    </div>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Quill Editor JS -->
<script src="https://cdn.quilljs.com/1.3.6/quill.min.js"></script>
<script>
// Initialize Quill editor
var quill = new Quill('#quill-editor', {
    theme: 'snow',
    placeholder: 'Enter your page content here...',
    modules: {
        toolbar: [
            [{ 'header': [1, 2, 3, 4, 5, 6, false] }],
            [{ 'font': [] }],
            [{ 'size': ['small', false, 'large', 'huge'] }],
            ['bold', 'italic', 'underline', 'strike'],
            [{ 'color': [] }, { 'background': [] }],
            [{ 'script': 'sub'}, { 'script': 'super' }],
            ['blockquote', 'code-block'],
            [{ 'list': 'ordered'}, { 'list': 'bullet' }],
            [{ 'indent': '-1'}, { 'indent': '+1' }],
            [{ 'direction': 'rtl' }],
            [{ 'align': [] }],
            ['link', 'image', 'video'],
            ['clean']
        ]
    }
});

// Mevcut içeriği yükle (sayfa yüklendiğinde)
document.addEventListener('DOMContentLoaded', function() {
    var existingContent = document.querySelector('#quill-editor').innerHTML;
    if (existingContent.trim()) {
        quill.root.innerHTML = existingContent;
    }
});

// Update page content function
function updatePageContent() {
    var content = document.querySelector('#content');
    content.value = quill.root.innerHTML;
    
    // Debugging
    console.log('Quill content:', quill.root.innerHTML);
    console.log('Hidden textarea content:', content.value);
}

// Update hidden textarea when form is submitted
document.getElementById('page-edit-form').addEventListener('submit', function(e) {
    var content = document.querySelector('#content');
    content.value = quill.root.innerHTML;
    
    // Form doğrulaması
    if (quill.getText().trim().length === 0) {
        e.preventDefault();
        alert('Please enter page content.');
        return false;
    }
    
    console.log('Form submitted with content:', content.value);
});

// Update preview URL when slug changes
document.getElementById('slug').addEventListener('input', function() {
    document.getElementById('preview-url').textContent = '/' + this.value;
});
</script>

</body>
</html>

