<?php
require_once 'admin_auth.php';
require_once '../includes/config.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Geçersiz istek yöntemi']);
    exit;
}

if (!isset($_POST['action']) || $_POST['action'] !== 'update_profile') {
    echo json_encode(['success' => false, 'message' => 'Geçersiz işlem']);
    exit;
}

// Session kontrolü
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    echo json_encode(['success' => false, 'message' => 'Kimlik doğrulanmadı - Lütfen tekrar giriş yapın']);
    exit;
}

$admin_id = $_SESSION['admin_user_id'] ?? null;
if (!$admin_id) {
    echo json_encode(['success' => false, 'message' => 'Admin ID oturumda bulunamadı']);
    exit;
}

try {
    $pdo->beginTransaction();
    
    // Get current admin data
    $stmt = $pdo->prepare("SELECT * FROM user WHERE id = ? AND role = 'admin'");
    $stmt->execute([$admin_id]);
    $admin = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$admin) {
        throw new Exception('Admin bulunamadı');
    }
    
    $updates = [];
    $params = [];
    
    // Update username if provided and different
    if (!empty($_POST['username']) && trim($_POST['username']) !== $admin['username']) {
        $new_username = trim($_POST['username']);
        
        // Validate username
        if (strlen($new_username) < 3) {
            throw new Exception('Kullanıcı adı en az 3 karakter olmalı');
        }
        
        if (!preg_match('/^[a-zA-Z0-9_]+$/', $new_username)) {
            throw new Exception('Username can only contain letters, numbers, and underscores');
        }
        
        // Check if username is already taken
        $stmt = $pdo->prepare("SELECT id FROM user WHERE username = ? AND id != ?");
        $stmt->execute([$new_username, $admin_id]);
        if ($stmt->fetch()) {
            throw new Exception('Username already taken');
        }
        
        $updates[] = "username = ?";
        $params[] = $new_username;
    }
    
    // Update email if provided and different
    if (!empty($_POST['email']) && trim($_POST['email']) !== $admin['email']) {
        $new_email = trim($_POST['email']);
        
        // Validate email
        if (!filter_var($new_email, FILTER_VALIDATE_EMAIL)) {
            throw new Exception('Invalid email format');
        }
        
        // Check if email is already taken
        $stmt = $pdo->prepare("SELECT id FROM user WHERE email = ? AND id != ?");
        $stmt->execute([$new_email, $admin_id]);
        if ($stmt->fetch()) {
            throw new Exception('Email already taken');
        }
        
        $updates[] = "email = ?";
        $params[] = $new_email;
    }
    
    // Update password if provided
    if (!empty($_POST['new_password'])) {
        $new_password = trim($_POST['new_password']);
        
        // Verify current password
        if (empty($_POST['current_password'])) {
            throw new Exception('Current password is required to change password');
        }
        
        if (!password_verify($_POST['current_password'], $admin['password'])) {
            throw new Exception('Current password is incorrect');
        }
        
        if ($new_password !== trim($_POST['confirm_password'])) {
            throw new Exception('New passwords do not match');
        }
        
        if (strlen($new_password) < 6) {
            throw new Exception('New password must be at least 6 characters long');
        }
        
        if (strlen($new_password) > 128) {
            throw new Exception('New password is too long (maximum 128 characters)');
        }
        
        $updates[] = "password = ?";
        $params[] = password_hash($new_password, PASSWORD_DEFAULT);
    }
    
    // Handle profile photo upload
    $avatar_url = $admin['profile_photo'] ?? '/uploads/no-logo.png';
    
    if (isset($_FILES['profile_photo']) && $_FILES['profile_photo']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = __DIR__ . '/../uploads/';
        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        $maxSize = 5 * 1024 * 1024; // 5MB
        
        $fileType = $_FILES['profile_photo']['type'];
        $fileSize = $_FILES['profile_photo']['size'];
        
        if (!in_array($fileType, $allowedTypes)) {
            throw new Exception('Invalid file type. Only JPEG, PNG, GIF, and WebP are allowed.');
        }
        
        if ($fileSize > $maxSize) {
            throw new Exception('File size too large. Maximum 5MB allowed.');
        }
        
        // Generate unique filename
        $extension = pathinfo($_FILES['profile_photo']['name'], PATHINFO_EXTENSION);
        $filename = 'admin_avatar_' . $admin_id . '_' . time() . '.' . $extension;
        $filepath = $uploadDir . $filename;
        
        if (move_uploaded_file($_FILES['profile_photo']['tmp_name'], $filepath)) {
            // Delete old avatar if it exists and is not the default
            if (!empty($admin['profile_photo']) && $admin['profile_photo'] !== '/uploads/no-logo.png' && file_exists(__DIR__ . '/..' . $admin['profile_photo'])) {
                unlink(__DIR__ . '/..' . $admin['profile_photo']);
            }
            
            $avatar_url = '/uploads/' . $filename;
            $updates[] = "profile_photo = ?";
            $params[] = $avatar_url;
        } else {
            throw new Exception('Failed to upload profile photo');
        }
    }
    
    // Update database if there are changes
    if (!empty($updates)) {
        $params[] = $admin_id;
        // Role'ü kesinlikle değiştirme - sadece admin kalacak
        $sql = "UPDATE user SET " . implode(', ', $updates) . " WHERE id = ? AND role = 'admin'";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        // Eğer hiçbir satır etkilenmediyse, admin bulunamadı demektir
        if ($stmt->rowCount() === 0) {
            throw new Exception('Admin user not found or role changed');
        }
    }
    
    $pdo->commit();
    
    // Update session if username changed
    if (!empty($_POST['username']) && trim($_POST['username']) !== $admin['username']) {
        $_SESSION['admin_username'] = trim($_POST['username']);
    }
    
    // Update session if email changed
    if (!empty($_POST['email']) && trim($_POST['email']) !== $admin['email']) {
        $_SESSION['admin_email'] = trim($_POST['email']);
    }
    
    echo json_encode([
        'success' => true, 
        'message' => 'Profile updated successfully',
        'avatar_url' => $avatar_url,
        'updated_fields' => array_keys($updates)
    ]);
    
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollback();
    }
    
    echo json_encode([
        'success' => false, 
        'message' => $e->getMessage()
    ]);
}
?>
