<?php
/**
 * Secure Database Helper
 * SQL injection koruması ve güvenli veritabanı işlemleri
 */

class SecureDBHelper {
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Güvenli SELECT sorgusu
     */
    public function secureSelect($table, $columns = '*', $where = [], $params = []) {
        $sql = "SELECT " . $columns . " FROM " . $this->sanitizeTableName($table);
        
        if (!empty($where)) {
            $whereClause = [];
            foreach ($where as $column => $value) {
                $whereClause[] = $this->sanitizeColumnName($column) . " = ?";
                $params[] = $value;
            }
            $sql .= " WHERE " . implode(' AND ', $whereClause);
        }
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Güvenli INSERT sorgusu
     */
    public function secureInsert($table, $data) {
        $columns = array_keys($data);
        $sanitizedColumns = array_map([$this, 'sanitizeColumnName'], $columns);
        $placeholders = str_repeat('?,', count($columns) - 1) . '?';
        
        $sql = "INSERT INTO " . $this->sanitizeTableName($table) . 
               " (" . implode(', ', $sanitizedColumns) . ") VALUES (" . $placeholders . ")";
        
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute(array_values($data));
    }
    
    /**
     * Güvenli UPDATE sorgusu
     */
    public function secureUpdate($table, $data, $where, $whereParams = []) {
        $setClause = [];
        $params = [];
        
        foreach ($data as $column => $value) {
            $setClause[] = $this->sanitizeColumnName($column) . " = ?";
            $params[] = $value;
        }
        
        $whereClause = [];
        foreach ($where as $column => $value) {
            $whereClause[] = $this->sanitizeColumnName($column) . " = ?";
            $params[] = $value;
        }
        
        $sql = "UPDATE " . $this->sanitizeTableName($table) . 
               " SET " . implode(', ', $setClause) . 
               " WHERE " . implode(' AND ', $whereClause);
        
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute($params);
    }
    
    /**
     * Güvenli DELETE sorgusu
     */
    public function secureDelete($table, $where, $params = []) {
        $whereClause = [];
        foreach ($where as $column => $value) {
            $whereClause[] = $this->sanitizeColumnName($column) . " = ?";
            $params[] = $value;
        }
        
        $sql = "DELETE FROM " . $this->sanitizeTableName($table) . 
               " WHERE " . implode(' AND ', $whereClause);
        
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute($params);
    }
    
    /**
     * Tablo adını sanitize et
     */
    private function sanitizeTableName($table) {
        // Sadece alfanumerik karakterler ve alt çizgi
        return preg_replace('/[^a-zA-Z0-9_]/', '', $table);
    }
    
    /**
     * Sütun adını sanitize et
     */
    private function sanitizeColumnName($column) {
        // Sadece alfanumerik karakterler ve alt çizgi
        return preg_replace('/[^a-zA-Z0-9_]/', '', $column);
    }
    
    /**
     * Input verilerini sanitize et
     */
    public static function sanitizeInput($input) {
        if (is_array($input)) {
            return array_map([self::class, 'sanitizeInput'], $input);
        }
        return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
    }
    
    /**
     * XSS koruması için output sanitize et
     */
    public static function sanitizeOutput($output) {
        if (is_array($output)) {
            return array_map([self::class, 'sanitizeOutput'], $output);
        }
        return htmlspecialchars($output, ENT_QUOTES, 'UTF-8');
    }
}
?>




