<?php
require_once 'admin_auth.php';
require_once '../includes/config.php';
require_once '../includes/turkish_fix.php';

// Favicon sıfırlama işlemi
if (isset($_GET['reset_favicon']) && $_GET['reset_favicon'] === '1') {
    try {
        $stmt = $pdo->prepare('UPDATE settings SET setting_value = ? WHERE setting_key = ?');
        $stmt->execute(['', 'favicon_url']);
        
        // Remove all favicon files from uploads directory
        $uploadDir = __DIR__ . '/../uploads/';
        $faviconPatterns = [
            $uploadDir . 'favicon.*',
            $uploadDir . 'favicon_*.*'
        ];
        
        foreach ($faviconPatterns as $pattern) {
            foreach (glob($pattern) as $file) {
                if (file_exists($file)) {
                    unlink($file);
                }
            }
        }
        
        // Also remove favicon from root directory
        $rootFaviconPath = __DIR__ . '/../favicon.ico';
        if (file_exists($rootFaviconPath)) {
            unlink($rootFaviconPath);
        }
        
        $success_message = "Favicon ayarı başarıyla temizlendi. Yeni bir favicon yükleyebilirsiniz.";
    } catch (Exception $e) {
        $error_message = "Favicon ayarı temizlenirken hata oluştu: " . $e->getMessage();
    }
}

// Logo sıfırlama işlemi
if (isset($_GET['reset_logo']) && $_GET['reset_logo'] === '1') {
    try {
        $stmt = $pdo->prepare('UPDATE settings SET setting_value = ? WHERE setting_key = ?');
        $stmt->execute(['', 'logo_url']);
        $success_message = "Logo ayarı başarıyla temizlendi. Yeni bir logo yükleyebilirsiniz.";
    } catch (Exception $e) {
        $error_message = "Logo ayarı temizlenirken hata oluştu: " . $e->getMessage();
    }
}

// Ayar kategorileri ve varsayılan değerleri
$settings_categories = [
    'site' => [
        'title' => 'Site Ayarları',
        'icon' => 'language',
        'description' => 'Temel site bilgileri ve meta ayarları',
        'settings' => [
            'site_title' => [
                'type' => 'text', 
                'label' => 'Site Başlığı', 
                'placeholder' => 'Sitenizin ana başlığı', 
                'default' => '', 
                'help' => 'Tarayıcı sekmelerinde ve arama motorlarında görüntülenen site adı'
            ],
            'site_description' => [
                'type' => 'textarea', 
                'label' => 'Site Açıklaması', 
                'placeholder' => 'Siteniz hakkında kısa ve öz açıklama (150-160 karakter önerilir)', 
                'default' => '', 
                'help' => 'Arama motorlarında görüntülenen açıklama metni'
            ],
            'site_keywords' => [
                'type' => 'text', 
                'label' => 'Anahtar Kelimeler', 
                'placeholder' => 'website, directory, list, turkey', 
                'default' => '', 
                'help' => 'Virgülle ayrılmış anahtar kelimeler'
            ],
            'title_separator' => [
                'type' => 'text', 
                'label' => 'Başlık Ayırıcısı', 
                'placeholder' => '|', 
                'default' => '|', 
                'help' => 'Sayfa başlığı ve site adı arasındaki ayırıcı (örn.: Sayfa | Site Adı)'
            ],
            'copyright_text' => [
                'type' => 'text', 
                'label' => 'Telif Hakkı Metni', 
                'placeholder' => '© 2024 Site Adı. Tüm hakları saklıdır.', 
                'default' => '© 2024 Site Adı. Tüm hakları saklıdır.',
                'help' => 'Footer alanında görüntülenen telif hakkı bilgisi'
            ],
            'site_slogan' => [
                'type' => 'text', 
                'label' => 'Site Sloganı', 
                'placeholder' => 'En iyi siteleri keşfedin', 
                'default' => '', 
                'help' => 'Site başlığının altında görüntülenen kısa slogan'
            ],
            'hero_title' => [
                'type' => 'text', 
                'label' => 'Ana Sayfa Başlığı', 
                'placeholder' => 'İnternetin En İyilerini Keşfedin', 
                'default' => 'İnternetin En İyilerini Keşfedin',
                'help' => 'Ana sayfada görüntülenen büyük başlık'
            ],
            'hero_subtitle' => [
                'type' => 'textarea', 
                'label' => 'Ana Sayfa Alt Başlığı', 
                'placeholder' => 'Kanka, en kaliteli ve güvenilir web sitelerini bir araya getiren modern bir dizin platformudur.', 
                'default' => 'Kanka, en kaliteli ve güvenilir web sitelerini bir araya getiren modern bir dizin platformudur.',
                'help' => 'Ana sayfa başlığının altında görüntülenen açıklama metni'
            ],
            'gradient_start_color' => [
                'type' => 'color', 
                'label' => 'Gradient Başlangıç Rengi', 
                'default' => '#38BDF8',
                'help' => 'Gradient geçişinin başlangıç rengi'
            ],
            'gradient_end_color' => [
                'type' => 'color', 
                'label' => 'Gradient Bitiş Rengi', 
                'default' => '#A78BFA',
                'help' => 'Gradient geçişinin bitiş rengi'
            ],
            'footer_menu_enabled' => [
                'type' => 'checkbox', 
                'label' => 'Footer Menü Aktif', 
                'default' => '1',
                'help' => 'Sitede footer menüsünün görüntülenmesini sağlar'
            ],
            'terms_of_service_url' => [
                'type' => 'text', 
                'label' => 'Kullanım Şartları URL', 
                'placeholder' => '/kullanim-sartlari', 
                'default' => '/kullanim-sartlari',
                'help' => 'Kayıt sayfasında Kullanım Şartları linkinin URL\'si'
            ],
            'privacy_policy_url' => [
                'type' => 'text', 
                'label' => 'Gizlilik Politikası URL', 
                'placeholder' => '/gizlilik-politikasi', 
                'default' => '/gizlilik-politikasi',
                'help' => 'Kayıt sayfasında Gizlilik Politikası linkinin URL\'si'
            ],
        ]
    ],
    'appearance' => [
        'title' => 'Görünüm',
        'icon' => 'palette',
        'description' => 'Logo, renk ve tema ayarları',
        'settings' => [
            'logo_type' => [
                'type' => 'select', 
                'label' => 'Logo Türü', 
                'options' => ['text' => 'Text Logo', 'image' => 'Image Logo'], 
                'default' => 'text', 
                'help' => 'Sitenizin üstte nasıl görüneceğini belirleyin'
            ],
            'logo_text' => [
                'type' => 'text', 
                'label' => 'Metin Logosu', 
                'placeholder' => 'Site Adı', 
                'default' => '', 
                'help' => 'Metin logosu seçildiğinde görüntülenecek metin'
            ],
            'logo_upload' => [
                'type' => 'file', 
                'label' => 'Logo Dosyası', 
                'accept' => 'image/*', 
                'default' => '', 
                'help' => 'Logoyu PNG, JPG veya SVG formatında yükleyin (önerilen boyut: 200x50px)'
            ],
            'favicon_upload' => [
                'type' => 'file', 
                'label' => 'Favicon Dosyası', 
                'accept' => '.ico,.png,.jpg,.jpeg,.gif', 
                'default' => '', 
                'help' => 'Tarayıcı sekmesinde görüntülenen küçük ikon (16x16 veya 32x32 piksel)'
            ],
            'primary_color' => [
                'type' => 'color', 
                'label' => 'Ana Renk', 
                'default' => '#3b82f6',
                'help' => 'Sitede kullanılan ana renk'
            ],
            'theme' => [
                'type' => 'select', 
                'label' => 'Tema', 
                'options' => ['light' => 'Light Theme', 'dark' => 'Dark Theme'], 
                'default' => 'light',
                'help' => 'Sitenin genel renk teması'
            ],
            'topvoted_title' => [
                'type' => 'text', 
                'label' => 'En Çok Oylanan Sayfa Başlığı', 
                'default' => 'Top Voted Sites',
                'help' => 'En çok oylanan siteler sayfasının başlığı'
            ],
            'topvoted_subtitle' => [
                'type' => 'textarea', 
                'label' => 'En Çok Oylanan Sayfa Açıklaması', 
                'default' => 'Discover the most popular and highly rated websites in our directory, ranked by community votes.',
                'help' => 'En çok oylanan siteler sayfasının açıklama metni'
            ],
            'newly_added_title' => [
                'type' => 'text', 
                'label' => 'Yeni Eklenen Sayfa Başlığı', 
                'default' => 'Latest Added Sites',
                'help' => 'Yeni eklenen siteler sayfasının başlığı'
            ],
            'newly_added_subtitle' => [
                'type' => 'textarea', 
                'label' => 'Yeni Eklenen Sayfa Açıklaması', 
                'default' => 'Discover amazing websites that our community has recently discovered and added.',
                'help' => 'Yeni eklenen siteler sayfasının açıklama metni'
            ],
        ]
    ],
    'content' => [
        'title' => 'İçerik Ayarları',
        'icon' => 'view_list',
        'description' => 'Sayfa başına görüntülenen içerik miktarları',
        'settings' => [
            'site_count' => [
                'type' => 'number', 
                'label' => 'Genel Site Sayısı (Sayfa Başına)', 
                'min' => 5, 
                'max' => 100, 
                'default' => 12, 
                'help' => 'Kategori ve arama sayfalarında görüntülenecek site sayısı'
            ],
            'newly_added_page_count' => [
                'type' => 'number', 
                'label' => 'Yeni Eklenen Sayfa Sayısı', 
                'min' => 6, 
                'max' => 50, 
                'default' => 12, 
                'help' => 'Yeni eklenen siteler sayfasında görüntülenecek site sayısı'
            ],
            'topvoted_page_count' => [
                'type' => 'number', 
                'label' => 'Popüler Siteler Sayfa Sayısı', 
                'min' => 6, 
                'max' => 50, 
                'default' => 12, 
                'help' => 'En çok oylanan siteler sayfasında görüntülenecek site sayısı'
            ],
            'newly_added_count' => [
                'type' => 'number', 
                'label' => 'Ana Sayfa - Yeni Siteler', 
                'min' => 2, 
                'max' => 20, 
                'default' => 4, 
                'help' => 'Ana sayfada "Yeni Eklenen" bölümünde görüntülenecek site sayısı'
            ],
            'top_voted_count' => [
                'type' => 'number', 
                'label' => 'Ana Sayfa - Popüler Siteler', 
                'min' => 2, 
                'max' => 20, 
                'default' => 6, 
                'help' => 'Ana sayfada "En Çok Oylanan" bölümünde görüntülenecek site sayısı'
            ],
            'show_newly_added' => [
                'type' => 'checkbox', 
                'label' => 'Yeni Eklenen Bölümü Göster', 
                'default' => '1',
                'help' => 'Ana sayfada yeni eklenen siteler bölümünü etkinleştir'
            ],
            'show_top_voted' => [
                'type' => 'checkbox', 
                'label' => 'Popüler Siteler Bölümünü Göster', 
                'default' => '1',
                'help' => 'Ana sayfada en çok oylanan siteler bölümünü etkinleştir'
            ],
        ]
    ],
    'seo' => [
        'title' => 'SEO Optimizasyonu',
        'icon' => 'search',
        'description' => 'Arama motoru optimizasyonu ayarları',
        'settings' => [
            'seo_title_separator' => [
                'type' => 'text', 
                'label' => 'SEO Başlık Ayırıcısı', 
                'placeholder' => ' - ', 
                'default' => ' | ', 
                'help' => 'Separator character between page title and site name'
            ],
            'seo_auto_meta' => [
                'type' => 'checkbox', 
                'label' => 'Otomatik Meta Açıklama', 
                'default' => '1', 
                'help' => 'Generate automatic meta description from site descriptions'
            ],
            'meta_home_title' => [
                'type' => 'text', 
                'label' => 'Ana Sayfa Özel Başlığı', 
                'placeholder' => 'Boş bırakılırsa site başlığı kullanılır', 
                'default' => '',
                'help' => 'Set a custom title for homepage'
            ],
            'link_rel_type' => [
                'type' => 'select', 
                'label' => 'Site Bağlantıları SEO Değeri', 
                'options' => ['nofollow' => 'NoFollow (No SEO value)', 'dofollow' => 'DoFollow (Has SEO value)'], 
                'default' => 'nofollow', 
                'help' => 'Whether added site links will be followed by search engines'
            ],
            'description_link_rel_type' => [
                'type' => 'select', 
                'label' => 'Açıklama Bağlantıları SEO Değeri', 
                'options' => ['nofollow' => 'NoFollow (No SEO value)', 'dofollow' => 'DoFollow (Has SEO value)'], 
                'default' => 'nofollow', 
                'help' => 'SEO value of links in site descriptions'
            ],
            'seo_robots' => [
                'type' => 'text', 
                'label' => 'Robots Meta Etiketi', 
                'placeholder' => 'index, follow', 
                'default' => 'index, follow', 
                'help' => 'General robots directive for search engines'
            ],
        ]
    ],
    'security' => [
        'title' => 'Güvenlik',
        'icon' => 'security',
        'description' => 'Site güvenliği ve kullanıcı ayarları',
        'settings' => [
            'user_registration' => [
                'type' => 'checkbox', 
                'label' => 'Kullanıcı Kaydına İzin Ver', 
                'default' => '1',
                'help' => 'Allow new users to register on the site'
            ],
            'site_submission' => [
                'type' => 'checkbox', 
                'label' => 'Site Gönderimini Etkinleştir', 
                'default' => '1',
                'help' => 'Allow users to add new sites'
            ],
            'require_approval' => [
                'type' => 'checkbox', 
                'label' => 'Site Onayı Gerekli', 
                'default' => '1',
                'help' => 'Added sites should be published after admin approval'
            ],
            'max_sites_per_user' => [
                'type' => 'number', 
                'label' => 'Kullanıcı Başına Maksimum Site', 
                'min' => 1, 
                'max' => 100, 
                'default' => 10,
                'help' => 'Maximum number of sites a user can add'
            ],
            'require_email_verification' => [
                'type' => 'checkbox', 
                'label' => 'E-posta Doğrulaması Gerekli', 
                'default' => '0',
                'help' => 'New users must verify their email addresses'
            ],
        ]
    ],
    'email' => [
        'title' => 'E-posta Ayarları',
        'icon' => 'email',
        'description' => 'E-posta gönderme ve doğrulama ayarları',
        'settings' => [
            'smtp_enabled' => [
                'type' => 'checkbox', 
                'label' => 'SMTP Kullan', 
                'default' => '0', 
                'help' => 'E-posta gönderme için SMTP sunucusu kullan'
            ],
            'smtp_host' => [
                'type' => 'text', 
                'label' => 'SMTP Sunucusu', 
                'placeholder' => 'mail.server.com', 
                'default' => '',
                'help' => 'SMTP server address'
            ],
            'smtp_port' => [
                'type' => 'number', 
                'label' => 'SMTP Portu', 
                'min' => 1, 
                'max' => 65535, 
                'default' => 587,
                'help' => 'SMTP port number (usually 587 or 465)'
            ],
            'smtp_username' => [
                'type' => 'text', 
                'label' => 'SMTP Kullanıcı Adı', 
                'placeholder' => 'mail@server.com', 
                'default' => '',
                'help' => 'SMTP username (usually email address)'
            ],
            'smtp_password' => [
                'type' => 'password', 
                'label' => 'SMTP Şifresi', 
                'default' => '',
                'help' => 'SMTP password (use app password for Gmail)'
            ],
            'smtp_encryption' => [
                'type' => 'select', 
                'label' => 'Şifreleme', 
                'options' => ['tls' => 'TLS', 'ssl' => 'SSL', 'none' => 'None'], 
                'default' => 'tls',
                'help' => 'SMTP connection encryption type'
            ],
            'mail_from_email' => [
                'type' => 'text', 
                'label' => 'Gönderen E-posta', 
                'placeholder' => 'noreply@yoursite.com', 
                'default' => '',
                'help' => 'E-postaların gönderileceği e-posta adresi'
            ],
            'mail_from_name' => [
                'type' => 'text', 
                'label' => 'Gönderen Adı', 
                'placeholder' => 'Site Adı', 
                'default' => '',
                'help' => 'E-postalarda görüntülenen gönderen adı'
            ],
        ]
    ],
    'indexing' => [
        'title' => 'İndeksleme ve Sitemap',
        'icon' => 'travel_explore',
        'description' => 'Sitemap ve robots.txt ayarları',
        'settings' => [
            'sitemap_enabled' => [
                'type' => 'checkbox', 
                'label' => 'Otomatik Sitemap Oluşturma', 
                'default' => '1', 
                'help' => 'Automatically update sitemap.xml when site is added'
            ],
            'sitemap_auto_update' => [
                'type' => 'checkbox', 
                'label' => 'Otomatik Güncelleme', 
                'default' => '1', 
                'help' => 'Sitemap is automatically updated whenever a site is added/updated'
            ],
            'sitemap_include_sites' => [
                'type' => 'checkbox', 
                'label' => 'Site Detaylarını Dahil Et', 
                'default' => '1', 
                'help' => 'Add site detail pages to sitemap'
            ],
            'sitemap_include_pages' => [
                'type' => 'checkbox', 
                'label' => 'Sayfaları Dahil Et', 
                'default' => '1', 
                'help' => 'Add static pages to sitemap (About, Contact etc.)'
            ],
            'sitemap_include_blog_categories' => [
                'type' => 'checkbox', 
                'label' => 'Blog Kategorilerini Dahil Et', 
                'default' => '1', 
                'help' => 'Add blog category pages to sitemap'
            ],
            'sitemap_include_blogs' => [
                'type' => 'checkbox', 
                'label' => 'Blog Yazılarını Dahil Et', 
                'default' => '1', 
                'help' => 'Add blog posts to sitemap'
            ],
            'sitemap_include_categories' => [
                'type' => 'checkbox', 
                'label' => 'Kategorileri Dahil Et', 
                'default' => '1', 
                'help' => 'Add category pages to sitemap'
            ],
            'sitemap_changefreq' => [
                'type' => 'select', 
                'label' => 'Değişim Sıklığı', 
                'options' => [
                    'always' => 'Always', 
                    'hourly' => 'Hourly', 
                    'daily' => 'Daily', 
                    'weekly' => 'Weekly', 
                    'monthly' => 'Monthly', 
                    'yearly' => 'Yearly', 
                    'never' => 'Never'
                ], 
                'default' => 'weekly', 
                'help' => 'Change frequency of pages in sitemap'
            ],
            'sitemap_priority_home' => [
                'type' => 'number', 
                'label' => 'Ana Sayfa Önceliği', 
                'min' => 0.1, 
                'max' => 1.0, 
                'step' => 0.1, 
                'default' => 1.0, 
                'help' => 'Sitemap priority for homepage (0.1-1.0)'
            ],
            'sitemap_priority_categories' => [
                'type' => 'number', 
                'label' => 'Kategori Sayfası Önceliği', 
                'min' => 0.1, 
                'max' => 1.0, 
                'step' => 0.1, 
                'default' => 0.8, 
                'help' => 'Sitemap priority for category pages'
            ],
            'sitemap_priority_sites' => [
                'type' => 'number', 
                'label' => 'Site Detay Önceliği', 
                'min' => 0.1, 
                'max' => 1.0, 
                'step' => 0.1, 
                'default' => 0.6, 
                'help' => 'Sitemap priority for site detail pages'
            ],
            'sitemap_priority_blogs' => [
                'type' => 'number', 
                'label' => 'Blog Yazıları Önceliği', 
                'min' => 0.1, 
                'max' => 1.0, 
                'step' => 0.1, 
                'default' => 0.7, 
                'help' => 'Sitemap priority for blog posts'
            ],
            'robots_enabled' => [
                'type' => 'checkbox', 
                'label' => 'Otomatik Robots.txt Oluşturma', 
                'default' => '1', 
                'help' => 'Create robots.txt file automatically'
            ],
            'robots_allow_all' => [
                'type' => 'checkbox', 
                'label' => 'Tüm Robotlara İzin Ver', 
                'default' => '1', 
                'help' => 'Allow all search engines'
            ],
            'robots_custom_rules' => [
                'type' => 'textarea', 
                'label' => 'Özel Robot Kuralları', 
                'placeholder' => "Disallow: /admin/\nDisallow: /tmp/\nDisallow: /private/", 
                'default' => '', 
                'help' => 'Additional robots.txt rules (one rule per line)'
            ],
        ]
    ],
    'email' => [
        'title' => 'E-posta Ayarları',
        'icon' => 'email',
        'description' => 'SMTP ve e-posta gönderme ayarları',
        'settings' => [
            'smtp_enabled' => [
                'type' => 'checkbox', 
                'label' => 'SMTP Kullan', 
                'default' => '0', 
                'help' => 'E-posta gönderme için SMTP sunucusu kullan'
            ],
            'smtp_host' => [
                'type' => 'text', 
                'label' => 'SMTP Sunucusu', 
                'placeholder' => 'mail.usasportstvs.com', 
                'default' => '', 
                'help' => 'SMTP server address (e.g.: mail.usasportstvs.com)'
            ],
            'smtp_port' => [
                'type' => 'number', 
                'label' => 'SMTP Portu', 
                'min' => 1, 
                'max' => 65535, 
                'default' => 587, 
                'help' => 'SMTP connection port (usually 587 or 465)'
            ],
            'smtp_encryption' => [
                'type' => 'select', 
                'label' => 'Güvenlik Türü', 
                'options' => ['tls' => 'TLS', 'ssl' => 'SSL', 'none' => 'None'], 
                'default' => 'tls',
                'help' => 'Encryption type for SMTP connection'
            ],
            'smtp_username' => [
                'type' => 'text', 
                'label' => 'SMTP Kullanıcı Adı', 
                'placeholder' => 'info@usasportstvs.com', 
                'default' => '', 
                'help' => 'Email address for SMTP authentication'
            ],
            'smtp_password' => [
                'type' => 'password', 
                'label' => 'SMTP Şifresi', 
                'placeholder' => 'Email password', 
                'default' => '', 
                'help' => 'Password of the email account'
            ],
            'mail_from_email' => [
                'type' => 'email', 
                'label' => 'Gönderen E-posta', 
                'placeholder' => 'info@usasportstvs.com', 
                'default' => '', 
                'help' => 'E-postaların gönderileceği e-posta adresi'
            ],
            'mail_from_name' => [
                'type' => 'text', 
                'label' => 'Gönderen Adı', 
                'placeholder' => 'Site Adı', 
                'default' => '', 
                'help' => 'E-postalarda görüntülenen gönderen adı'
            ],
            'password_reset_enabled' => [
                'type' => 'checkbox', 
                'label' => 'Şifre Sıfırlamayı Etkinleştir', 
                'default' => '1', 
                'help' => 'Allow users to reset their passwords'
            ],
            'password_reset_token_expire' => [
                'type' => 'number', 
                'label' => 'Sıfırlama Bağlantısı Geçerliliği (Dakika)', 
                'min' => 5, 
                'max' => 1440, 
                'default' => 60, 
                'help' => 'How many minutes the password reset link will be valid'
            ],
        ]
    ],
    'ads' => [
        'title' => 'Reklam Ayarları',
        'icon' => 'campaign',
        'description' => 'Reklam kodlarınızı yönetin ve tüm sayfalarda görüntüleyin',
        'settings' => [
            'ads_enabled' => [
                'type' => 'checkbox', 
                'label' => 'Reklamları Aktif Et', 
                'default' => '1', 
                'help' => 'Site genelinde reklamları göster/gizle'
            ],
            'ads_management_link' => [
                'type' => 'custom', 
                'label' => 'Reklam Yönetimi', 
                'default' => '', 
                'help' => 'Detaylı reklam yönetimi için buraya tıklayın',
                'custom_html' => '<a href="ads_settings.php" class="inline-flex items-center gap-2 bg-primary hover:bg-blue-600 text-white px-6 py-3 rounded-xl font-semibold transition-all duration-200"><span class="material-symbols-outlined">campaign</span>Reklam Ayarları</a>'
            ],
        ]
    ]
];

// Mevcut ayarları veritabanından çek
$current_settings = [];
$settings_result = $pdo->query('SELECT setting_key, setting_value FROM settings');
if ($settings_result) {
    while ($row = $settings_result->fetch(PDO::FETCH_ASSOC)) {
        $current_settings[$row['setting_key']] = $row['setting_value'];
    }
}

// Tüm ayarları bir dizide topla
$all_settings = [];
foreach ($settings_categories as $category) {
    foreach ($category['settings'] as $key => $config) {
        $default_value = $config['default'] ?? '';
        $all_settings[$key] = $current_settings[$key] ?? $default_value;
    }
}

// File upload alanları için özel değerleri ekle (bunlar $settings_categories'de tanımlanmamış)
$all_settings['logo_url'] = $current_settings['logo_url'] ?? '';
$all_settings['favicon_url'] = $current_settings['favicon_url'] ?? '';

// Reset logo/favicon işlemleri
if (isset($_GET['reset_logo'])) {
    try {
        $pdo->beginTransaction();
        
        // Delete logo file
        if (!empty($all_settings['logo_url'])) {
            $logoPath = __DIR__ . '/../' . ltrim($all_settings['logo_url'], '/');
            if (file_exists($logoPath)) {
                unlink($logoPath);
            }
        }
        
        // Remove logo_url from database
        $stmt = $pdo->prepare('DELETE FROM settings WHERE setting_key = ?');
        $stmt->execute(['logo_url']);
        
        $pdo->commit();
        header('Location: ' . $_SERVER['PHP_SELF'] . '?success=logo_reset');
        exit;
    } catch (Exception $e) {
        $pdo->rollback();
        header('Location: ' . $_SERVER['PHP_SELF'] . '?error=logo_reset_failed');
        exit;
    }
}

if (isset($_GET['reset_favicon'])) {
    try {
        $pdo->beginTransaction();
        
        // Delete favicon file
        if (!empty($all_settings['favicon_url'])) {
            $faviconPath = __DIR__ . '/../' . ltrim($all_settings['favicon_url'], '/');
            if (file_exists($faviconPath)) {
                unlink($faviconPath);
            }
        }
        
        // Remove favicon_url from database
        $stmt = $pdo->prepare('DELETE FROM settings WHERE setting_key = ?');
        $stmt->execute(['favicon_url']);
        
        $pdo->commit();
        header('Location: ' . $_SERVER['PHP_SELF'] . '?success=favicon_reset');
        exit;
    } catch (Exception $e) {
        $pdo->rollback();
        header('Location: ' . $_SERVER['PHP_SELF'] . '?error=favicon_reset_failed');
        exit;
    }
}

// AJAX istekleri işleme
if (isset($_POST['ajax_action'])) {
    header('Content-Type: application/json');
    
    try {
        if ($_POST['ajax_action'] === 'generate_sitemap') {
            $result = generateSitemap($pdo, $all_settings);
            echo json_encode($result);
        } elseif ($_POST['ajax_action'] === 'generate_robots') {
            $result = generateRobotsTxt($pdo, $all_settings);
            echo json_encode($result);
        } else {
            echo json_encode(['success' => false, 'message' => 'Geçersiz işlem!']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Hata: ' . $e->getMessage()]);
    }
    exit;
}

// Form gönderildiğinde
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $pdo->beginTransaction();
        
        // File upload handling
        $uploadedFiles = [];
        if (!empty($_FILES)) {
            foreach ($_FILES as $fieldName => $file) {
                if ($file['error'] === UPLOAD_ERR_OK) {
                    $uploadDir = __DIR__ . '/../uploads/';
                    if (!is_dir($uploadDir)) {
                        mkdir($uploadDir, 0755, true);
                    }
                    
                    $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
                    $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'svg', 'ico'];
                    
                    if (in_array($fileExtension, $allowedExtensions)) {
                        // Special handling for favicon and logo files
                        if ($fieldName === 'favicon_upload') {
                            $fileName = 'favicon_' . time() . '.' . $fileExtension; // Unique filename
                        } elseif ($fieldName === 'logo_upload') {
                            $fileName = 'logo_' . time() . '.' . $fileExtension;
                        } else {
                            $fileName = $fieldName . '_' . time() . '.' . $fileExtension;
                        }
                        
                        $filePath = $uploadDir . $fileName;
                        
                        // Delete old files if exists
                        if ($fieldName === 'favicon_upload') {
                            // Remove all old favicon files
                            $faviconPatterns = [
                                $uploadDir . 'favicon.*',
                                $uploadDir . 'favicon_*.*'
                            ];
                            
                            foreach ($faviconPatterns as $pattern) {
                                foreach (glob($pattern) as $oldFile) {
                                    if (file_exists($oldFile)) {
                                        unlink($oldFile);
                                    }
                                }
                            }
                            
                            // Also remove from root directory
                            $rootFaviconPath = __DIR__ . '/../favicon.ico';
                            if (file_exists($rootFaviconPath)) {
                                unlink($rootFaviconPath);
                            }
                        } elseif ($fieldName === 'logo_upload') {
                            // Delete old logo files
                            $oldLogoPattern = $uploadDir . 'logo_*';
                            foreach (glob($oldLogoPattern) as $oldFile) {
                                if (file_exists($oldFile)) {
                                    unlink($oldFile);
                                }
                            }
                        }
                        
                        if (move_uploaded_file($file['tmp_name'], $filePath)) {
                            $uploadedFiles[$fieldName] = '/uploads/' . $fileName;
                        }
                    }
                }
            }
        }
        
        // Update settings
        foreach ($settings_categories as $category) {
            foreach ($category['settings'] as $key => $config) {
                $current_value = $all_settings[$key];
                
                if (isset($uploadedFiles[$key])) {
                    $value = $uploadedFiles[$key];
                    // Update logo_url if logo_upload was uploaded
                    if ($key === 'logo_upload') {
                        $stmt = $pdo->prepare('INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)');
                        $stmt->execute(['logo_url', $value]);
                        $all_settings['logo_url'] = $value;
                    }
                    // Update favicon_url if favicon_upload was uploaded
                    if ($key === 'favicon_upload') {
                        $stmt = $pdo->prepare('INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)');
                        $stmt->execute(['favicon_url', $value]);
                        $all_settings['favicon_url'] = $value;
                        
                        // Also update the favicon in the root directory with cache busting
                        $rootFaviconPath = __DIR__ . '/../favicon.ico';
                        $uploadedFaviconPath = __DIR__ . '/../' . ltrim($value, '/');
                        if (file_exists($uploadedFaviconPath)) {
                            // Remove old favicon.ico first
                            if (file_exists($rootFaviconPath)) {
                                unlink($rootFaviconPath);
                            }
                            // Copy new favicon
                            copy($uploadedFaviconPath, $rootFaviconPath);
                            
                            // Add cache busting parameter to URL
                            $cacheBustingUrl = $value . '?v=' . time();
                            $stmt = $pdo->prepare('UPDATE settings SET setting_value = ? WHERE setting_key = ?');
                            $stmt->execute([$cacheBustingUrl, 'favicon_url']);
                            $all_settings['favicon_url'] = $cacheBustingUrl;
                        }
                    }
                } else {
                    $value = $_POST[$key] ?? $current_value;
                }
                
                // Handle file upload fields specially
                if ($config['type'] === 'file') {
                    // For file fields, only update if a new file was uploaded
                    if (isset($uploadedFiles[$key])) {
                        // New file was uploaded, value is already set above
                        $stmt = $pdo->prepare('INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)');
                        $stmt->execute([$key, $value]);
                        $all_settings[$key] = $value;
                    }
                    // If no new file uploaded, keep the existing value (don't update database)
                    // But make sure the current value is preserved in $all_settings
                    $all_settings[$key] = $current_value;
                    continue;
                }
                
                $stmt = $pdo->prepare('INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)');
                $stmt->execute([$key, $value]);
                $all_settings[$key] = $value;
            }
        }
        
        $pdo->commit();
        $success_message = "Ayarlar başarıyla güncellendi!";
        
    } catch (Exception $e) {
        $pdo->rollback();
        $error_message = 'Ayarlar kaydedilirken hata oluştu: ' . $e->getMessage();
    }
}
?>

<?php include 'menu.php'; ?>

<div class="flex-1 bg-background min-h-screen">
    <div class="p-8">
        <?php
        $page_title = "Site Settings";
        $page_subtitle = "Manage your site's general settings";
        include 'includes/admin_header.php';
        ?>
        
        <!-- Action Buttons -->
        <div class="flex gap-3 mb-8">
            <button onclick="resetForm()" class="inline-flex items-center gap-2 bg-surface hover:bg-gray-50 text-text-secondary border border-gray-200 font-semibold px-6 py-3 rounded-xl shadow-soft transition-all duration-200">
                <span class="material-symbols-outlined text-xl">refresh</span>
                Sıfırla
            </button>
            <button form="settingsForm" type="submit" class="inline-flex items-center gap-2 bg-primary hover:bg-blue-600 text-white font-semibold px-6 py-3 rounded-xl shadow-soft transition-all duration-200">
                <span class="material-symbols-outlined text-xl">save</span>
                Kaydet
            </button>
        </div>

        <?php if (isset($success_message)): ?>
            <div class="bg-green-50 border border-green-200 text-green-800 px-6 py-4 rounded-xl mb-6">
                <div class="flex items-center gap-2">
                    <span class="material-symbols-outlined">check_circle</span>
                    <?php echo safeEcho($success_message); ?>
                </div>
            </div>
        <?php endif; ?>

        <?php if (isset($_GET['success'])): ?>
            <?php if ($_GET['success'] === 'logo_reset'): ?>
                <div class="bg-green-50 border border-green-200 text-green-800 px-6 py-4 rounded-xl mb-6">
                    <div class="flex items-center gap-2">
                        <span class="material-symbols-outlined">check_circle</span>
                        Logo successfully deleted!
                    </div>
                </div>
            <?php elseif ($_GET['success'] === 'favicon_reset'): ?>
                <div class="bg-green-50 border border-green-200 text-green-800 px-6 py-4 rounded-xl mb-6">
                    <div class="flex items-center gap-2">
                        <span class="material-symbols-outlined">check_circle</span>
                        Favicon successfully deleted!
                    </div>
                </div>
            <?php endif; ?>
        <?php endif; ?>

        <?php if (isset($_GET['error'])): ?>
            <?php if ($_GET['error'] === 'logo_reset_failed'): ?>
                <div class="bg-red-50 border border-red-200 text-red-700 px-6 py-4 rounded-xl mb-6">
                    <div class="flex items-center gap-2">
                        <span class="material-symbols-outlined">error</span>
                        Error occurred while deleting logo!
                    </div>
                </div>
            <?php elseif ($_GET['error'] === 'favicon_reset_failed'): ?>
                <div class="bg-red-50 border border-red-200 text-red-700 px-6 py-4 rounded-xl mb-6">
                    <div class="flex items-center gap-2">
                        <span class="material-symbols-outlined">error</span>
                        Error occurred while deleting favicon!
                    </div>
                </div>
            <?php endif; ?>
        <?php endif; ?>

        <?php if (isset($error_message)): ?>
            <div class="bg-red-50 border border-red-200 text-red-700 px-6 py-4 rounded-xl mb-6">
                <div class="flex items-center gap-2">
                    <span class="material-symbols-outlined">error</span>
                    <?php echo safeEcho($error_message); ?>
                </div>
            </div>
        <?php endif; ?>

        <!-- Settings Navigation -->
        <div class="flex gap-2 mb-6 overflow-x-auto">
            <?php foreach ($settings_categories as $cat_key => $category): ?>
                <button onclick="showCategory('<?php echo $cat_key; ?>')" 
                        class="settings-tab flex items-center gap-2 px-6 py-3 rounded-xl font-semibold transition-all duration-200 whitespace-nowrap <?php echo $cat_key === 'site' ? 'bg-primary text-white shadow-soft' : 'bg-surface hover:bg-gray-50 text-text-secondary'; ?>" 
                        data-category="<?php echo $cat_key; ?>">
                    <span class="material-symbols-outlined text-xl"><?php echo $category['icon']; ?></span>
                    <?php echo safeEcho($category['title']); ?>
                </button>
            <?php endforeach; ?>
        </div>

        <!-- Settings Form -->
        <form id="settingsForm" method="POST" enctype="multipart/form-data" class="space-y-6">
            <?php foreach ($settings_categories as $cat_key => $category): ?>
                <div class="settings-category <?php echo $cat_key !== 'site' ? 'hidden' : ''; ?>" data-category="<?php echo $cat_key; ?>">
                    <div class="bg-surface rounded-2xl shadow-soft border border-gray-100 overflow-hidden">
                        <div class="p-6 border-b border-gray-100">
                            <div class="flex items-center gap-3 mb-2">
                                <div class="w-10 h-10 bg-primary/10 rounded-xl flex items-center justify-center">
                                    <span class="material-symbols-outlined text-primary text-xl"><?php echo $category['icon']; ?></span>
                                </div>
                                <div>
                                    <h2 class="text-xl font-semibold text-text-primary"><?php echo safeEcho($category['title']); ?></h2>
                                    <p class="text-sm text-text-secondary"><?php echo safeEcho($category['description'] ?? ''); ?></p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="p-6">
                            <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                                <?php foreach ($category['settings'] as $setting_key => $config): ?>
                                    <div class="<?php echo ($config['type'] === 'textarea') ? 'lg:col-span-2' : ''; ?>">
                                        <label for="<?php echo $setting_key; ?>" class="block text-sm font-medium text-text-primary mb-2">
                                            <?php echo safeEcho($config['label']); ?>
                                            <?php if (isset($config['help'])): ?>
                                                <span class="ml-1 text-text-secondary cursor-help" title="<?php echo htmlspecialchars($config['help']); ?>">
                                                    <span class="material-symbols-outlined text-sm">info</span>
                                                </span>
                                            <?php endif; ?>
                                        </label>
                                        
                                        <?php 
                                        $current_value = $all_settings[$setting_key] ?? $config['default'] ?? '';
                                        
                                        switch ($config['type']):
                                            case 'text':
                                        ?>
                                            <input type="text" 
                                                   id="<?php echo $setting_key; ?>" 
                                                   name="<?php echo $setting_key; ?>" 
                                                   value="<?php echo htmlspecialchars($current_value); ?>"
                                                   placeholder="<?php echo htmlspecialchars($config['placeholder'] ?? ''); ?>"
                                                   class="w-full px-4 py-3 rounded-xl border border-gray-200 focus:ring-2 focus:ring-primary focus:border-primary transition-all"
                                                   <?php if ($setting_key === 'logo_text' && ($all_settings['logo_type'] ?? 'text') === 'image'): ?>style="display: none;"<?php endif; ?>>
                                        <?php break; case 'textarea': ?>
                                            <textarea id="<?php echo $setting_key; ?>" 
                                                      name="<?php echo $setting_key; ?>" 
                                                      rows="3"
                                                      placeholder="<?php echo htmlspecialchars($config['placeholder'] ?? ''); ?>"
                                                      class="w-full px-4 py-3 rounded-xl border border-gray-200 focus:ring-2 focus:ring-primary focus:border-primary transition-all resize-none"><?php echo htmlspecialchars($current_value); ?></textarea>
                                        <?php break; case 'number': ?>
                                            <input type="number" 
                                                   id="<?php echo $setting_key; ?>" 
                                                   name="<?php echo $setting_key; ?>" 
                                                   value="<?php echo htmlspecialchars($current_value); ?>"
                                                   min="<?php echo $config['min'] ?? ''; ?>"
                                                   max="<?php echo $config['max'] ?? ''; ?>"
                                                   step="<?php echo $config['step'] ?? '1'; ?>"
                                                   class="w-full px-4 py-3 rounded-xl border border-gray-200 focus:ring-2 focus:ring-primary focus:border-primary transition-all">
                                        <?php break; case 'select': ?>
                                            <select id="<?php echo $setting_key; ?>" 
                                                    name="<?php echo $setting_key; ?>" 
                                                    class="w-full px-4 py-3 rounded-xl border border-gray-200 focus:ring-2 focus:ring-primary focus:border-primary transition-all">
                                                <?php foreach ($config['options'] as $value => $label): ?>
                                                    <option value="<?php echo htmlspecialchars($value); ?>" <?php echo ($current_value == $value) ? 'selected' : ''; ?>>
                                                        <?php echo safeEcho($label); ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                        <?php break; case 'checkbox': ?>
                                            <div class="flex items-center">
                                                <input type="hidden" name="<?php echo $setting_key; ?>" value="0">
                                                <input type="checkbox" 
                                                       id="<?php echo $setting_key; ?>" 
                                                       name="<?php echo $setting_key; ?>" 
                                                       value="1" 
                                                       <?php echo ($current_value == '1') ? 'checked' : ''; ?>
                                                       class="w-5 h-5 text-primary border-gray-300 rounded focus:ring-primary">
                                                <span class="ml-2 text-sm text-text-secondary">Active</span>
                                            </div>
                                        <?php break; case 'color': ?>
                                            <div class="flex gap-3">
                                                <input type="color" 
                                                       id="<?php echo $setting_key; ?>" 
                                                       name="<?php echo $setting_key; ?>" 
                                                       value="<?php echo htmlspecialchars($current_value); ?>"
                                                       class="w-16 h-12 rounded-lg border border-gray-200 cursor-pointer">
                                                <input type="text" 
                                                       value="<?php echo htmlspecialchars($current_value); ?>"
                                                       class="flex-1 px-4 py-3 rounded-xl border border-gray-200 focus:ring-2 focus:ring-primary focus:border-primary transition-all"
                                                       readonly>
                                            </div>
                                        <?php break; case 'file': ?>
                                            <div class="space-y-3">
                                                <input type="file" 
                                                       id="<?php echo $setting_key; ?>" 
                                                       name="<?php echo $setting_key; ?>" 
                                                       accept="<?php echo $config['accept'] ?? ''; ?>"
                                                       class="w-full px-4 py-3 rounded-xl border border-gray-200 focus:ring-2 focus:ring-primary focus:border-primary transition-all file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-semibold file:bg-primary file:text-white hover:file:bg-blue-600"
                                                       <?php if ($setting_key === 'logo_upload' && ($all_settings['logo_type'] ?? 'text') === 'text'): ?>style="display: none;"<?php endif; ?>>
                                                
                                                <!-- Current Logo Display -->
                                                <?php if ($setting_key === 'logo_upload' && !empty($all_settings['logo_url'])): ?>
                                                    <div class="mt-3 p-4 bg-gradient-to-r from-blue-50 to-indigo-50 dark:from-gray-800 dark:to-gray-700 rounded-xl border border-blue-200 dark:border-gray-600">
                                                        <div class="flex items-center gap-4">
                                                            <div class="flex-shrink-0">
                                                                <img src="<?php echo htmlspecialchars($all_settings['logo_url']); ?>" alt="Current Logo" class="h-10 max-w-40 object-contain bg-white dark:bg-gray-600 p-2 rounded-lg shadow-sm">
                                                            </div>
                                                            <div class="flex-1 min-w-0">
                                                                <div class="flex items-center gap-2 mb-1">
                                                                    <i class="fas fa-check-circle text-green-500 text-sm"></i>
                                                                    <span class="text-sm font-semibold text-gray-800 dark:text-gray-200">Current Logo Active</span>
                                                                </div>
                                                                <p class="text-xs text-gray-600 dark:text-gray-400 truncate"><?php echo htmlspecialchars(basename($all_settings['logo_url'])); ?></p>
                                                            </div>
                                                            <div class="flex-shrink-0">
                                                                <a href="?reset_logo=1" class="inline-flex items-center gap-1 px-3 py-1.5 text-xs font-medium text-red-600 dark:text-red-400 bg-red-50 dark:bg-red-900/20 hover:bg-red-100 dark:hover:bg-red-900/30 rounded-lg transition-colors">
                                                                    <i class="fas fa-trash-alt"></i>
                                                                    Delete
                                                                </a>
                                                            </div>
                                                        </div>
                                                    </div>
                                                <?php endif; ?>
                                                
                                                <!-- Current Favicon Display -->
                                                <?php if ($setting_key === 'favicon_upload' && !empty($all_settings['favicon_url'])): ?>
                                                    <div class="mt-3 p-4 bg-gradient-to-r from-green-50 to-emerald-50 dark:from-gray-800 dark:to-gray-700 rounded-xl border border-green-200 dark:border-gray-600">
                                                        <div class="flex items-center gap-4">
                                                            <div class="flex-shrink-0">
                                                                <img src="<?php echo htmlspecialchars($all_settings['favicon_url']); ?>" alt="Current Favicon" class="w-8 h-8 bg-white dark:bg-gray-600 p-1 rounded-lg shadow-sm">
                                                            </div>
                                                            <div class="flex-1 min-w-0">
                                                                <div class="flex items-center gap-2 mb-1">
                                                                    <i class="fas fa-check-circle text-green-500 text-sm"></i>
                                                                    <span class="text-sm font-semibold text-gray-800 dark:text-gray-200">Current Favicon Active</span>
                                                                </div>
                                                                <p class="text-xs text-gray-600 dark:text-gray-400 truncate"><?php echo htmlspecialchars(basename($all_settings['favicon_url'])); ?></p>
                                                            </div>
                                                            <div class="flex-shrink-0">
                                                                <a href="?reset_favicon=1" class="inline-flex items-center gap-1 px-3 py-1.5 text-xs font-medium text-red-600 dark:text-red-400 bg-red-50 dark:bg-red-900/20 hover:bg-red-100 dark:hover:bg-red-900/30 rounded-lg transition-colors">
                                                                    <i class="fas fa-trash-alt"></i>
                                                                    Delete
                                                                </a>
                                                            </div>
                                                        </div>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        <?php break; case 'password': ?>
                                            <input type="password" 
                                                   id="<?php echo $setting_key; ?>" 
                                                   name="<?php echo $setting_key; ?>" 
                                                   value="<?php echo htmlspecialchars($current_value); ?>"
                                                   placeholder="<?php echo htmlspecialchars($config['placeholder'] ?? ''); ?>"
                                                   class="w-full px-4 py-3 rounded-xl border border-gray-200 focus:ring-2 focus:ring-primary focus:border-primary transition-all">
                                        <?php break; case 'email': ?>
                                            <input type="email" 
                                                   id="<?php echo $setting_key; ?>" 
                                                   name="<?php echo $setting_key; ?>" 
                                                   value="<?php echo htmlspecialchars($current_value); ?>"
                                                   placeholder="<?php echo htmlspecialchars($config['placeholder'] ?? ''); ?>"
                                                   class="w-full px-4 py-3 rounded-xl border border-gray-200 focus:ring-2 focus:ring-primary focus:border-primary transition-all">
                                        <?php break; case 'custom': ?>
                                            <?php echo $config['custom_html'] ?? ''; ?>
                                        <?php endswitch; ?>
                                        
                                        <?php if (isset($config['help'])): ?>
                                            <p class="mt-2 text-sm text-text-secondary"><?php echo safeEcho($config['help']); ?></p>
                                        <?php endif; ?>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                            
                            <?php if ($cat_key === 'indexing'): ?>
                                <!-- İndeksleme Özel Eylemleri -->
                                <div class="mt-6 pt-6 border-t border-gray-100">
                                    <h3 class="text-lg font-semibold text-text-primary mb-4">Manual Operations</h3>
                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div class="bg-blue-50 rounded-xl p-4">
                                            <div class="flex items-center gap-3 mb-3">
                                                <div class="w-10 h-10 bg-blue-100 rounded-lg flex items-center justify-center">
                                                    <span class="material-symbols-outlined text-blue-600">map</span>
                                                </div>
                                                <div>
                                                    <h4 class="font-semibold text-gray-900">Generate Sitemap</h4>
                                                    <p class="text-sm text-gray-600">Manually create XML sitemap file</p>
                                                </div>
                                            </div>
                                            <button type="button" onclick="generateSitemapAjax()" class="w-full bg-blue-600 hover:bg-blue-700 text-white font-medium py-2 px-4 rounded-lg transition-colors">
                                                <span class="material-symbols-outlined text-sm mr-1">refresh</span>
                                                Generate Sitemap
                                            </button>
                                        </div>
                                        
                                        <div class="bg-green-50 rounded-xl p-4">
                                            <div class="flex items-center gap-3 mb-3">
                                                <div class="w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center">
                                                    <span class="material-symbols-outlined text-green-600">smart_toy</span>
                                                </div>
                                                <div>
                                                    <h4 class="font-semibold text-gray-900">Generate Robots.txt</h4>
                                                    <p class="text-sm text-gray-600">Manually create robots.txt file</p>
                                                </div>
                                            </div>
                                            <button type="button" onclick="generateRobotsAjax()" class="w-full bg-green-600 hover:bg-green-700 text-white font-medium py-2 px-4 rounded-lg transition-colors">
                                                <span class="material-symbols-outlined text-sm mr-1">refresh</span>
                                                Generate Robots.txt
                                            </button>
                                        </div>
                                    </div>
                                    
                                    <!-- Sonuç Mesaj Alanı -->
                                    <div id="indexingResults" class="mt-4 hidden">
                                        <div class="bg-gray-50 rounded-lg p-4">
                                            <div class="flex items-center gap-2">
                                                <span class="material-symbols-outlined text-gray-600">info</span>
                                                <span id="indexingMessage" class="text-sm text-gray-700"></span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
            
            <!-- Fixed Save Button -->
            <div class="sticky bottom-8 flex justify-end">
                <button type="submit" class="inline-flex items-center gap-2 bg-primary hover:bg-blue-600 text-white font-semibold px-8 py-4 rounded-xl shadow-lg transition-all duration-200 transform hover:scale-105">
                    <span class="material-symbols-outlined text-xl">save</span>
                    Ayarları Kaydet
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function showCategory(categoryKey) {
    // Hide all categories
    document.querySelectorAll('.settings-category').forEach(cat => {
        cat.classList.add('hidden');
    });
    
    // Remove active class from all tabs
    document.querySelectorAll('.settings-tab').forEach(tab => {
        tab.classList.remove('bg-primary', 'text-white', 'shadow-soft');
        tab.classList.add('bg-surface', 'hover:bg-gray-50', 'text-text-secondary');
    });
    
    // Show selected category
    document.querySelector(`[data-category="${categoryKey}"].settings-category`).classList.remove('hidden');
    
    // Add active class to selected tab
    const activeTab = document.querySelector(`[data-category="${categoryKey}"].settings-tab`);
    activeTab.classList.remove('bg-surface', 'hover:bg-gray-50', 'text-text-secondary');
    activeTab.classList.add('bg-primary', 'text-white', 'shadow-soft');
}

function resetForm() {
    if (confirm('Are you sure you want to cancel all changes?')) {
        document.getElementById('settingsForm').reset();
    }
}

// Color input synchronization
document.querySelectorAll('input[type="color"]').forEach(colorInput => {
    const textInput = colorInput.nextElementSibling;
    
    colorInput.addEventListener('change', function() {
        textInput.value = this.value;
    });
    
    textInput.addEventListener('input', function() {
        if (/^#[0-9A-F]{6}$/i.test(this.value)) {
            colorInput.value = this.value;
        }
    });
});

// Logo type change handler
document.addEventListener('DOMContentLoaded', function() {
    const logoTypeSelect = document.getElementById('logo_type');
    const logoTextInput = document.getElementById('logo_text');
    const logoUploadInput = document.getElementById('logo_upload');
    
    if (logoTypeSelect) {
        logoTypeSelect.addEventListener('change', function() {
            if (this.value === 'image') {
                // Show logo upload field
                if (logoUploadInput) {
                    logoUploadInput.style.display = 'block';
                }
                // Hide logo text field
                if (logoTextInput) {
                    logoTextInput.style.display = 'none';
                }
            } else {
                // Show logo text field
                if (logoTextInput) {
                    logoTextInput.style.display = 'block';
                }
                // Hide logo upload field
                if (logoUploadInput) {
                    logoUploadInput.style.display = 'none';
                }
            }
        });
        
        // Trigger change event on page load
        logoTypeSelect.dispatchEvent(new Event('change'));
    }
    
    // Logo upload preview
    if (logoUploadInput) {
        logoUploadInput.addEventListener('change', function() {
            const file = this.files[0];
            if (file) {
                // Show preview of selected file
                const reader = new FileReader();
                reader.onload = function(e) {
                    // You can add preview functionality here if needed
                    console.log('Logo file selected:', file.name);
                };
                reader.readAsDataURL(file);
            }
        });
    }
});

// Form validation
document.getElementById('settingsForm').addEventListener('submit', function(e) {
    const requiredFields = this.querySelectorAll('input[required], textarea[required], select[required]');
    let isValid = true;
    
    requiredFields.forEach(field => {
        if (!field.value.trim()) {
            field.classList.add('border-red-500');
            isValid = false;
        } else {
            field.classList.remove('border-red-500');
        }
    });
    
    if (!isValid) {
        e.preventDefault();
        alert('Please fill in all required fields.');
    }
});

// AJAX fonksiyonları
function generateSitemapAjax() {
    const button = event.target;
    const originalText = button.innerHTML;
    button.innerHTML = '<span class="material-symbols-outlined text-sm mr-1 animate-spin">refresh</span>Generating...';
    button.disabled = true;
    
    fetch('', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'ajax_action=generate_sitemap'
    })
    .then(response => response.json())
    .then(data => {
        showIndexingResult(data.message, data.success ? 'success' : 'error');
        button.innerHTML = originalText;
        button.disabled = false;
    })
    .catch(error => {
        showIndexingResult('Bir hata oluştu: ' + error.message, 'error');
        button.innerHTML = originalText;
        button.disabled = false;
    });
}

function generateRobotsAjax() {
    const button = event.target;
    const originalText = button.innerHTML;
    button.innerHTML = '<span class="material-symbols-outlined text-sm mr-1 animate-spin">refresh</span>Generating...';
    button.disabled = true;
    
    fetch('', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'ajax_action=generate_robots'
    })
    .then(response => response.json())
    .then(data => {
        showIndexingResult(data.message, data.success ? 'success' : 'error');
        button.innerHTML = originalText;
        button.disabled = false;
    })
    .catch(error => {
        showIndexingResult('Bir hata oluştu: ' + error.message, 'error');
        button.innerHTML = originalText;
        button.disabled = false;
    });
}

function showIndexingResult(message, type) {
    const resultsDiv = document.getElementById('indexingResults');
    const messageSpan = document.getElementById('indexingMessage');
    
    resultsDiv.className = 'mt-4';
    if (type === 'success') {
        resultsDiv.innerHTML = `
            <div class="bg-green-50 border border-green-200 rounded-lg p-4">
                <div class="flex items-center gap-2">
                    <span class="material-symbols-outlined text-green-600">check_circle</span>
                    <span class="text-sm text-green-700">${message}</span>
                </div>
            </div>
        `;
    } else {
        resultsDiv.innerHTML = `
            <div class="bg-red-50 border border-red-200 rounded-lg p-4">
                <div class="flex items-center gap-2">
                    <span class="material-symbols-outlined text-red-600">error</span>
                    <span class="text-sm text-red-700">${message}</span>
                </div>
            </div>
        `;
    }
    
    // 5 saniye sonra mesajı gizle
    setTimeout(() => {
        resultsDiv.style.opacity = '0';
        setTimeout(() => {
            resultsDiv.classList.add('hidden');
            resultsDiv.style.opacity = '1';
        }, 300);
    }, 5000);
}
</script>

<?php
/**
 * Sitemap oluşturma fonksiyonu
 */
function generateSitemap($pdo, $settings) {
    global $conn; // MySQLi bağlantısını kullan
    
    try {
        $sitemap = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $sitemap .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
        
        $base_url = 'https://' . $_SERVER['HTTP_HOST'];
        $changefreq = $settings['sitemap_changefreq'] ?? 'weekly';
        
        // Ana sayfa
        $home_priority = $settings['sitemap_priority_home'] ?? '1.0';
        $sitemap .= "  <url>\n";
        $sitemap .= "    <loc>{$base_url}/</loc>\n";
        $sitemap .= "    <lastmod>" . date('Y-m-d') . "</lastmod>\n";
        $sitemap .= "    <changefreq>{$changefreq}</changefreq>\n";
        $sitemap .= "    <priority>{$home_priority}</priority>\n";
        $sitemap .= "  </url>\n";
        
        // Kategoriler
        if (($settings['sitemap_include_categories'] ?? '1') === '1') {
            $category_priority = $settings['sitemap_priority_categories'] ?? '0.8';
            $categories_result = $conn->query("SELECT name, slug FROM categories");
            if ($categories_result && $categories_result->num_rows > 0) {
                while ($category = $categories_result->fetch_assoc()) {
                    $sitemap .= "  <url>\n";
                    $sitemap .= "    <loc>{$base_url}/category/{$category['slug']}</loc>\n";
                    $sitemap .= "    <lastmod>" . date('Y-m-d') . "</lastmod>\n";
                    $sitemap .= "    <changefreq>{$changefreq}</changefreq>\n";
                    $sitemap .= "    <priority>{$category_priority}</priority>\n";
                    $sitemap .= "  </url>\n";
                }
            }
        }
        
        // Site detayları - ZORUNLU EKLE
        if (($settings['sitemap_include_sites'] ?? '1') === '1') {
            $site_priority = $settings['sitemap_priority_sites'] ?? '0.6';
            
            // Onaylı siteleri al - updated_at kolonu olmayabilir
            $sites_query = "SELECT id, site_name, site_url";
            
            // slug kolonu var mı kontrol et
            $check_slug = $conn->query("SHOW COLUMNS FROM site_submissions LIKE 'slug'");
            if ($check_slug && $check_slug->num_rows > 0) {
                $sites_query .= ", slug";
            }
            
            // updated_at kolonu var mı kontrol et
            $check_updated = $conn->query("SHOW COLUMNS FROM site_submissions LIKE 'updated_at'");
            if ($check_updated && $check_updated->num_rows > 0) {
                $sites_query .= ", updated_at";
            }
            
            // created_at kolonu var mı kontrol et  
            $check_created = $conn->query("SHOW COLUMNS FROM site_submissions LIKE 'created_at'");
            if ($check_created && $check_created->num_rows > 0) {
                $sites_query .= ", created_at";
            }
            
            $sites_query .= " FROM site_submissions WHERE status = 'approved' ORDER BY id DESC";
            
            $sites_result = $conn->query($sites_query);
            
            if ($sites_result && $sites_result->num_rows > 0) {
                while ($site = $sites_result->fetch_assoc()) {
                    $sitemap .= "  <url>\n";
                    
                    // SEO URL oluştur
                    if (isset($site['slug']) && !empty($site['slug'])) {
                        // SEO URL formatı: /site/site-slug
                        $sitemap .= "    <loc>{$base_url}/site/{$site['slug']}</loc>\n";
                    } else {
                        // Slug yoksa site adından oluştur
                        $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $site['site_name'])));
                        $slug = trim($slug, '-');
                        if (!empty($slug)) {
                            $sitemap .= "    <loc>{$base_url}/site/{$slug}</loc>\n";
                        } else {
                            // Son çare olarak ID kullan
                            $sitemap .= "    <loc>{$base_url}/sitedetails.php?id={$site['id']}</loc>\n";
                        }
                    }
                    
                    // Son güncelleme tarihi
                    if (isset($site['updated_at']) && !empty($site['updated_at']) && $site['updated_at'] != '0000-00-00 00:00:00') {
                        $sitemap .= "    <lastmod>" . date('Y-m-d', strtotime($site['updated_at'])) . "</lastmod>\n";
                    } elseif (isset($site['created_at']) && !empty($site['created_at']) && $site['created_at'] != '0000-00-00 00:00:00') {
                        $sitemap .= "    <lastmod>" . date('Y-m-d', strtotime($site['created_at'])) . "</lastmod>\n";
                    } else {
                        $sitemap .= "    <lastmod>" . date('Y-m-d') . "</lastmod>\n";
                    }
                    
                    $sitemap .= "    <changefreq>{$changefreq}</changefreq>\n";
                    $sitemap .= "    <priority>{$site_priority}</priority>\n";
                    $sitemap .= "  </url>\n";
                }
            }
        }
        
        // Statik sayfalar
        if (($settings['sitemap_include_pages'] ?? '1') === '1') {
            $page_priority = $settings['sitemap_priority_sites'] ?? '0.6';
            
            // Pages tablosundan sayfaları al
            $pages_result = $conn->query("SELECT slug, updated_at FROM pages WHERE status = 'published'");
            if ($pages_result && $pages_result->num_rows > 0) {
                while ($page = $pages_result->fetch_assoc()) {
                    $sitemap .= "  <url>\n";
                    $sitemap .= "    <loc>{$base_url}/page/{$page['slug']}</loc>\n";
                    
                    if (!empty($page['updated_at']) && $page['updated_at'] != '0000-00-00 00:00:00') {
                        $sitemap .= "    <lastmod>" . date('Y-m-d', strtotime($page['updated_at'])) . "</lastmod>\n";
                    } else {
                        $sitemap .= "    <lastmod>" . date('Y-m-d') . "</lastmod>\n";
                    }
                    
                    $sitemap .= "    <changefreq>{$changefreq}</changefreq>\n";
                    $sitemap .= "    <priority>{$page_priority}</priority>\n";
                    $sitemap .= "  </url>\n";
                }
            }
        }
        
        // Blog kategorileri
        if (($settings['sitemap_include_blog_categories'] ?? '1') === '1') {
            $blog_category_priority = $settings['sitemap_priority_categories'] ?? '0.8';
            
            $blog_categories_result = $conn->query("SELECT slug FROM blog_categories");
            if ($blog_categories_result && $blog_categories_result->num_rows > 0) {
                while ($blog_category = $blog_categories_result->fetch_assoc()) {
                    $sitemap .= "  <url>\n";
                    $sitemap .= "    <loc>{$base_url}/blog/category/{$blog_category['slug']}</loc>\n";
                    $sitemap .= "    <lastmod>" . date('Y-m-d') . "</lastmod>\n";
                    $sitemap .= "    <changefreq>{$changefreq}</changefreq>\n";
                    $sitemap .= "    <priority>{$blog_category_priority}</priority>\n";
                    $sitemap .= "  </url>\n";
                }
            }
        }
        
        // Blog yazıları
        if (($settings['sitemap_include_blogs'] ?? '1') === '1') {
            $blog_priority = $settings['sitemap_priority_blogs'] ?? '0.7';
            
            $blogs_result = $conn->query("SELECT slug, published_at, updated_at FROM blog_posts WHERE status = 'published'");
            if ($blogs_result && $blogs_result->num_rows > 0) {
                while ($blog = $blogs_result->fetch_assoc()) {
                    $sitemap .= "  <url>\n";
                    $sitemap .= "    <loc>{$base_url}/blog/{$blog['slug']}</loc>\n";
                    
                    // Son güncelleme tarihi
                    if (!empty($blog['updated_at']) && $blog['updated_at'] != '0000-00-00 00:00:00') {
                        $sitemap .= "    <lastmod>" . date('Y-m-d', strtotime($blog['updated_at'])) . "</lastmod>\n";
                    } elseif (!empty($blog['published_at']) && $blog['published_at'] != '0000-00-00 00:00:00') {
                        $sitemap .= "    <lastmod>" . date('Y-m-d', strtotime($blog['published_at'])) . "</lastmod>\n";
                    } else {
                        $sitemap .= "    <lastmod>" . date('Y-m-d') . "</lastmod>\n";
                    }
                    
                    $sitemap .= "    <changefreq>{$changefreq}</changefreq>\n";
                    $sitemap .= "    <priority>{$blog_priority}</priority>\n";
                    $sitemap .= "  </url>\n";
                }
            }
        }
        
        $sitemap .= '</urlset>';
        
        // Sitemap dosyasını yaz
        $sitemap_path = __DIR__ . '/../sitemap.xml';
        if (file_put_contents($sitemap_path, $sitemap)) {
            return ['success' => true, 'message' => 'Sitemap generated successfully!', 'path' => '/sitemap.xml'];
        } else {
            return ['success' => false, 'message' => 'Sitemap file could not be written.'];
        }
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'Error generating sitemap: ' . $e->getMessage()];
    }
}

/**
 * Robots.txt oluşturma fonksiyonu
 */
function generateRobotsTxt($pdo, $settings) {
    try {
        $robots = '';
        
        if (($settings['robots_allow_all'] ?? '1') === '1') {
            $robots .= "User-agent: *\n";
            $robots .= "Allow: /\n\n";
        } else {
            $robots .= "User-agent: *\n";
            $robots .= "Disallow: /\n\n";
        }
        
        // Özel kurallar
        $custom_rules = $settings['robots_custom_rules'] ?? '';
        if (!empty($custom_rules)) {
            $robots .= trim($custom_rules) . "\n\n";
        }
        
        // Varsayılan yasaklı alanlar
        $robots .= "# Admin ve özel alanlar\n";
        $robots .= "Disallow: /admin/\n";
        $robots .= "Disallow: /config.php\n";
        $robots .= "Disallow: /includes/\n";
        $robots .= "Disallow: /*.php$\n\n";
        
        // Sitemap referansı
        if (($settings['sitemap_enabled'] ?? '1') === '1') {
            $base_url = 'https://' . $_SERVER['HTTP_HOST'];
            $robots .= "# Sitemap\n";
            $robots .= "Sitemap: {$base_url}/sitemap.xml\n";
        }
        
        // Robots.txt dosyasını yaz
        $robots_path = __DIR__ . '/../robots.txt';
        if (file_put_contents($robots_path, $robots)) {
            return ['success' => true, 'message' => 'Robots.txt generated successfully!', 'path' => '/robots.txt'];
        } else {
            return ['success' => false, 'message' => 'Robots.txt file could not be written.'];
        }
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'Error generating robots.txt: ' . $e->getMessage()];
    }
}

/**
 * Site eklendiğinde/güncellendiğinde sitemap otomatik güncelle
 */
function autoUpdateSitemap($pdo) {
    global $conn; // MySQLi bağlantısını kullan
    try {
        // Settings'i al
        $settings = [];
        $settings_result = $conn->query('SELECT setting_key, setting_value FROM settings');
        if ($settings_result) {
            while ($row = $settings_result->fetch_assoc()) {
                $settings[$row['setting_key']] = $row['setting_value'];
            }
        }
        
        // Otomatik güncelleme aktif mi kontrol et
        if (($settings['sitemap_auto_update'] ?? '1') === '1' && ($settings['sitemap_enabled'] ?? '1') === '1') {
            generateSitemap($pdo, $settings);
            return true;
        }
        return false;
    } catch (Exception $e) {
        error_log('Sitemap auto-update failed: ' . $e->getMessage());
        return false;
    }
}
?>

