<?php
require_once 'admin_auth.php';
require_once '../includes/config.php';
require_once '../includes/turkish_fix.php';

// Favicon sıfırlama işlemi
if (isset($_GET['reset_favicon']) && $_GET['reset_favicon'] === '1') {
    try {
        $stmt = $pdo->prepare('UPDATE settings SET setting_value = ? WHERE setting_key = ?');
        $stmt->execute(['', 'favicon_url']);
        $success_message = "Favicon ayarı başarıyla temizlendi. Yeni favicon yükleyebilirsiniz.";
    } catch (Exception $e) {
        $error_message = "Favicon ayarı temizlenirken hata oluştu: " . $e->getMessage();
    }
}

// Ayar kategorileri ve varsayılan değerleri
$settings_categories = [
    'site' => [
        'title' => 'Site Ayarları',
        'icon' => 'language',
        'description' => 'Temel site bilgileri ve meta ayarları',
        'settings' => [
            'site_title' => [
                'type' => 'text', 
                'label' => 'Site Başlığı', 
                'placeholder' => 'Sitenizin ana başlığı', 
                'default' => '', 
                'help' => 'Tarayıcı sekmesinde ve arama motorlarında görünen site adı'
            ],
            'site_description' => [
                'type' => 'textarea', 
                'label' => 'Site Açıklaması', 
                'placeholder' => 'Siteniz hakkında kısa ve öz açıklama (150-160 karakter önerilir)', 
                'default' => '', 
                'help' => 'Arama motorlarında görünen açıklama metni'
            ],
            'site_keywords' => [
                'type' => 'text', 
                'label' => 'Anahtar Kelimeler', 
                'placeholder' => 'web sitesi, dizin, listesi, türkiye', 
                'default' => '', 
                'help' => 'Virgülle ayrılmış anahtar kelimeler'
            ],
            'title_separator' => [
                'type' => 'text', 
                'label' => 'Başlık Ayırıcısı', 
                'placeholder' => '|', 
                'default' => '|', 
                'help' => 'Sayfa başlığı ile site adı arasındaki ayırıcı (örn: Sayfa | Site Adı)'
            ],
            'copyright_text' => [
                'type' => 'text', 
                'label' => 'Telif Hakkı Metni', 
                'placeholder' => '© 2024 Site Adı. Tüm hakları saklıdır.', 
                'default' => '© 2024 Site Adı. Tüm hakları saklıdır.',
                'help' => 'Footer alanında görünen telif hakkı bilgisi'
            ],
            'site_slogan' => [
                'type' => 'text', 
                'label' => 'Site Sloganı', 
                'placeholder' => 'En iyi siteleri keşfedin', 
                'default' => '', 
                'help' => 'Site başlığının altında görünen kısa slogan'
            ],
        ]
    ],
    'appearance' => [
        'title' => 'Görünüm',
        'icon' => 'palette',
        'description' => 'Logo, renk ve tema ayarları',
        'settings' => [
            'logo_type' => [
                'type' => 'select', 
                'label' => 'Logo Türü', 
                'options' => ['text' => 'Metin Logo', 'image' => 'Resim Logo'], 
                'default' => 'text', 
                'help' => 'Sitenizin üst kısmında nasıl görüneceğini belirleyin'
            ],
            'logo_text' => [
                'type' => 'text', 
                'label' => 'Metin Logo', 
                'placeholder' => 'Site Adı', 
                'default' => '', 
                'help' => 'Metin logo seçildiğinde gösterilecek yazı'
            ],
            'logo_upload' => [
                'type' => 'file', 
                'label' => 'Logo Dosyası', 
                'accept' => 'image/*', 
                'default' => '', 
                'help' => 'PNG, JPG veya SVG formatında logo yükleyin (önerilen boyut: 200x50px)'
            ],
            'logo_url' => [
                'type' => 'text', 
                'label' => 'Logo URL\'si', 
                'placeholder' => '/uploads/logo.png', 
                'default' => '', 
                'help' => 'Yüklenen logo dosyasının tam yolu'
            ],
            'favicon_upload' => [
                'type' => 'file', 
                'label' => 'Favicon Dosyası', 
                'accept' => '.ico,.png,.jpg,.jpeg,.gif', 
                'default' => '', 
                'help' => 'Tarayıcı sekmesinde görünen küçük ikon (16x16 veya 32x32 piksel)'
            ],
            'favicon_url' => [
                'type' => 'text', 
                'label' => 'Favicon URL\'si', 
                'placeholder' => '/uploads/favicon.ico', 
                'default' => '', 
                'help' => 'Yüklenen favicon dosyasının tam yolu'
            ],
            'primary_color' => [
                'type' => 'color', 
                'label' => 'Ana Renk', 
                'default' => '#3b82f6',
                'help' => 'Site genelinde kullanılan ana renk'
            ],
            'theme' => [
                'type' => 'select', 
                'label' => 'Tema', 
                'options' => ['light' => 'Açık Tema', 'dark' => 'Koyu Tema'], 
                'default' => 'light',
                'help' => 'Sitenin genel renk teması'
            ],
        ]
    ],
    'content' => [
        'title' => 'İçerik Ayarları',
        'icon' => 'view_list',
        'description' => 'Sayfa başına görünen içerik miktarları',
        'settings' => [
            'site_count' => [
                'type' => 'number', 
                'label' => 'Genel Site Sayısı (Sayfa Başına)', 
                'min' => 5, 
                'max' => 100, 
                'default' => 12, 
                'help' => 'Kategori ve arama sayfalarında gösterilecek site sayısı'
            ],
            'newly_added_page_count' => [
                'type' => 'number', 
                'label' => 'Yeni Eklenen Sayfa Sayısı', 
                'min' => 6, 
                'max' => 50, 
                'default' => 12, 
                'help' => 'Yeni eklenen siteler sayfasında gösterilecek site sayısı'
            ],
            'topvoted_page_count' => [
                'type' => 'number', 
                'label' => 'Popüler Siteler Sayfa Sayısı', 
                'min' => 6, 
                'max' => 50, 
                'default' => 12, 
                'help' => 'En çok oylanan siteler sayfasında gösterilecek site sayısı'
            ],
            'newly_added_count' => [
                'type' => 'number', 
                'label' => 'Ana Sayfa - Yeni Siteler', 
                'min' => 2, 
                'max' => 20, 
                'default' => 4, 
                'help' => 'Ana sayfada "Yeni Eklenen" bölümünde gösterilecek site sayısı'
            ],
            'top_voted_count' => [
                'type' => 'number', 
                'label' => 'Ana Sayfa - Popüler Siteler', 
                'min' => 2, 
                'max' => 20, 
                'default' => 6, 
                'help' => 'Ana sayfada "En Çok Oylanan" bölümünde gösterilecek site sayısı'
            ],
            'show_newly_added' => [
                'type' => 'checkbox', 
                'label' => 'Yeni Eklenen Bölümünü Göster', 
                'default' => '1',
                'help' => 'Ana sayfada yeni eklenen siteler bölümünü aktif et'
            ],
            'show_top_voted' => [
                'type' => 'checkbox', 
                'label' => 'Popüler Siteler Bölümünü Göster', 
                'default' => '1',
                'help' => 'Ana sayfada en çok oylanan siteler bölümünü aktif et'
            ],
        ]
    ],
    'seo' => [
        'title' => 'SEO Optimizasyonu',
        'icon' => 'search',
        'description' => 'Arama motoru optimizasyon ayarları',
        'settings' => [
            'seo_title_separator' => [
                'type' => 'text', 
                'label' => 'SEO Başlık Ayırıcısı', 
                'placeholder' => ' - ', 
                'default' => ' | ', 
                'help' => 'Sayfa başlığı ile site adı arasındaki ayırıcı karakter'
            ],
            'seo_auto_meta' => [
                'type' => 'checkbox', 
                'label' => 'Otomatik Meta Açıklama', 
                'default' => '1', 
                'help' => 'Site açıklamalarından otomatik meta description oluştur'
            ],
            'meta_home_title' => [
                'type' => 'text', 
                'label' => 'Ana Sayfa Özel Başlığı', 
                'placeholder' => 'Boş bırakırsanız site başlığı kullanılır', 
                'default' => '',
                'help' => 'Ana sayfa için özel bir başlık belirleyin'
            ],
            'link_rel_type' => [
                'type' => 'select', 
                'label' => 'Site Linklerinin SEO Değeri', 
                'options' => ['nofollow' => 'NoFollow (SEO değeri yok)', 'dofollow' => 'DoFollow (SEO değeri var)'], 
                'default' => 'nofollow', 
                'help' => 'Eklenen site linklerinin arama motorları tarafından takip edilip edilmeyeceği'
            ],
            'description_link_rel_type' => [
                'type' => 'select', 
                'label' => 'Açıklama Linklerinin SEO Değeri', 
                'options' => ['nofollow' => 'NoFollow (SEO değeri yok)', 'dofollow' => 'DoFollow (SEO değeri var)'], 
                'default' => 'nofollow', 
                'help' => 'Site açıklamalarındaki linklerin SEO değeri'
            ],
            'seo_robots' => [
                'type' => 'text', 
                'label' => 'Robots Meta Etiketi', 
                'placeholder' => 'index, follow', 
                'default' => 'index, follow', 
                'help' => 'Arama motorları için genel robots direktifi'
            ],
        ]
    ],
    'security' => [
        'title' => 'Güvenlik',
        'icon' => 'security',
        'description' => 'Site güvenliği ve kullanıcı ayarları',
        'settings' => [
            'user_registration' => [
                'type' => 'checkbox', 
                'label' => 'Kullanıcı Kaydına İzin Ver', 
                'default' => '1',
                'help' => 'Yeni kullanıcıların siteye kayıt olmasına izin ver'
            ],
            'site_submission' => [
                'type' => 'checkbox', 
                'label' => 'Site Gönderimini Aktif Et', 
                'default' => '1',
                'help' => 'Kullanıcıların yeni site eklemesine izin ver'
            ],
            'require_approval' => [
                'type' => 'checkbox', 
                'label' => 'Site Onayı Gerekli', 
                'default' => '1',
                'help' => 'Eklenen siteler admin onayından sonra yayınlansın'
            ],
            'max_sites_per_user' => [
                'type' => 'number', 
                'label' => 'Kullanıcı Başına Maksimum Site', 
                'min' => 1, 
                'max' => 100, 
                'default' => 10,
                'help' => 'Bir kullanıcının ekleyebileceği maksimum site sayısı'
            ],
        ]
    ]
];

// Mevcut ayarları veritabanından çek
$current_settings = [];
$stmt = $pdo->query('SELECT setting_key, setting_value FROM settings');
foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
    $current_settings[$row['setting_key']] = $row['setting_value'];
}

// Tüm ayarları bir dizide topla
$all_settings = [];
foreach ($settings_categories as $category) {
    foreach ($category['settings'] as $key => $config) {
        $default_value = $config['default'] ?? '';
        $all_settings[$key] = $current_settings[$key] ?? $default_value;
    }
}

// Eski kod uyumluluğu için settings_keys alias
$settings_keys = $all_settings;

$success = false;
$error = '';

// AJAX istekleri için sitemap ve robots generator
if (isset($_POST['action']) && in_array($_POST['action'], ['generate_sitemap', 'generate_robots'])) {
    header('Content-Type: application/json');
    
    try {
        if ($_POST['action'] === 'generate_sitemap') {
            $result = generateSitemap($pdo, $all_settings);
            echo json_encode(['success' => true, 'message' => 'Sitemap başarıyla oluşturuldu!', 'details' => $result]);
        } elseif ($_POST['action'] === 'generate_robots') {
            $result = generateRobotsTxt($pdo, $all_settings);
            echo json_encode(['success' => true, 'message' => 'Robots.txt başarıyla oluşturuldu!', 'details' => $result]);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Hata: ' . $e->getMessage()]);
    }
    exit;
}

// Form gönderildiğinde
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $pdo->beginTransaction();
        
        // File upload handling
        $uploadedFiles = [];
        if (!empty($_FILES)) {
            foreach ($_FILES as $fieldName => $file) {
                if ($file['error'] === UPLOAD_ERR_OK) {
                    $uploadDir = __DIR__ . '/../uploads/';
                    if (!is_dir($uploadDir)) {
                        mkdir($uploadDir, 0755, true);
                    }
                    
                    $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
                    $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'svg', 'ico'];
                    
                    if (in_array($fileExtension, $allowedExtensions)) {
                        // Special handling for favicon files
                        if ($fieldName === 'favicon_upload') {
                            $fileName = 'favicon.' . $fileExtension;
                        } else {
                            $fileName = $fieldName . '_' . time() . '.' . $fileExtension;
                        }
                        
                        $filePath = $uploadDir . $fileName;
                        
                        // Delete old favicon if exists
                        if ($fieldName === 'favicon_upload') {
                            $oldFaviconPattern = $uploadDir . 'favicon.*';
                            foreach (glob($oldFaviconPattern) as $oldFile) {
                                if (file_exists($oldFile)) {
                                    unlink($oldFile);
                                }
                            }
                        }
                        
                        if (move_uploaded_file($file['tmp_name'], $filePath)) {
                            $uploadedFiles[$fieldName] = '/uploads/' . $fileName;
                            // Debug: favicon upload success
                            if ($fieldName === 'favicon_upload') {
                                error_log("Favicon uploaded successfully: " . $filePath);
                                error_log("Favicon URL will be: /uploads/" . $fileName);
                            }
                        } else {
                            // Debug: favicon upload failed
                            if ($fieldName === 'favicon_upload') {
                                error_log("Favicon upload failed for: " . $filePath);
                            }
                        }
                    } else {
                        if ($fieldName === 'favicon_upload') {
                            error_log("Favicon upload failed: invalid extension " . $fileExtension);
                        }
                    }
                }
            }
        }
        
        foreach ($all_settings as $key => $current_value) {
            if ($key === 'show_newly_added' || $key === 'show_top_voted' || $key === 'seo_auto_meta') {
                $value = isset($_POST[$key]) ? '1' : '0';
            } elseif (isset($uploadedFiles[$key])) {
                // Use uploaded file path
                $value = $uploadedFiles[$key];
                // Update logo_url if logo_upload was uploaded
                if ($key === 'logo_upload') {
                    $stmt = $pdo->prepare('INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)');
                    $stmt->execute(['logo_url', $value]);
                    $all_settings['logo_url'] = $value;
                }
                // Update favicon_url if favicon_upload was uploaded
                if ($key === 'favicon_upload') {
                    $stmt = $pdo->prepare('INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)');
                    $stmt->execute(['favicon_url', $value]);
                    $all_settings['favicon_url'] = $value;
                    error_log("Favicon URL updated to: " . $value);
                }
            } else {
                $value = $_POST[$key] ?? $current_value;
            }
            
            $stmt = $pdo->prepare('INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)');
            $stmt->execute([$key, $value]);
            $all_settings[$key] = $value;
        }
        
        // Eski kod uyumluluğu için settings_keys'i güncelle
        $settings_keys = $all_settings;
        
        $pdo->commit();
        $success = true;
        
    } catch (Exception $e) {
        $pdo->rollback();
        $error_message = 'Ayarlar kaydedilirken hata oluştu: ' . $e->getMessage();
    }
}
?>

<?php include 'menu.php'; ?>

<div class="flex-1 bg-background min-h-screen">
    <div class="p-8">
        <!-- Header -->
        <div class="flex justify-between items-center mb-8">
            <div>
                <h1 class="text-3xl font-bold text-text-primary mb-2">Site Ayarları</h1>
                <p class="text-text-secondary">Sitenizin genel ayarlarını yönetin</p>
            </div>
            <div class="flex gap-3">
                <button onclick="resetForm()" class="inline-flex items-center gap-2 bg-surface hover:bg-gray-50 text-text-secondary border border-gray-200 font-semibold px-6 py-3 rounded-xl shadow-soft transition-all duration-200">
                    <span class="material-symbols-outlined text-xl">refresh</span>
                    Sıfırla
                </button>
                <button form="settingsForm" type="submit" class="inline-flex items-center gap-2 bg-primary hover:bg-blue-600 text-white font-semibold px-6 py-3 rounded-xl shadow-soft transition-all duration-200">
                    <span class="material-symbols-outlined text-xl">save</span>
                    Kaydet
                </button>
            </div>
        </div>

        <?php if (isset($success_message)): ?>
            <div class="bg-green-50 border border-green-200 text-green-800 px-6 py-4 rounded-xl mb-6">
                <div class="flex items-center gap-2">
                    <span class="material-symbols-outlined">check_circle</span>
                    <?php echo safeEcho($success_message); ?>
                </div>
            </div>
        <?php endif; ?>

        <?php if (isset($error_message)): ?>
            <div class="bg-red-50 border border-red-200 text-red-700 px-6 py-4 rounded-xl mb-6">
                <div class="flex items-center gap-2">
                    <span class="material-symbols-outlined">error</span>
                    <?php echo safeEcho($error_message); ?>
                </div>
            </div>
        <?php endif; ?>

        <!-- Settings Navigation -->
        <div class="flex gap-2 mb-6 overflow-x-auto">
            <?php foreach ($settings_categories as $cat_key => $category): ?>
                <button onclick="showCategory('<?php echo $cat_key; ?>')" 
                        class="settings-tab flex items-center gap-2 px-6 py-3 rounded-xl font-semibold transition-all duration-200 whitespace-nowrap <?php echo $cat_key === 'site' ? 'bg-primary text-white shadow-soft' : 'bg-surface hover:bg-gray-50 text-text-secondary'; ?>" 
                        data-category="<?php echo $cat_key; ?>">
                    <span class="material-symbols-outlined text-xl"><?php echo $category['icon']; ?></span>
                    <?php echo safeEcho($category['title']); ?>
                </button>
            <?php endforeach; ?>
        </div>

        <!-- Settings Form -->
        <form id="settingsForm" method="POST" enctype="multipart/form-data" class="space-y-6">
            },
        };
    </script>
    <style>
        .material-symbols-outlined {
            font-variation-settings: "FILL" 0, "wght" 400, "GRAD" 0, "opsz" 24;
        }
        .active .material-symbols-outlined {
            font-variation-settings: 'FILL' 1;
        }
        
        /* Tab animasyonları */
        .tab-content {
            opacity: 0;
            transform: translateY(10px);
            transition: all 0.3s ease;
            display: none;
        }
        .tab-content.active {
            opacity: 1;
            transform: translateY(0);
            display: block;
        }
        
        /* Tab butonları için hover efekti */
        .tab-btn {
            transition: all 0.2s ease;
        }
    </style>
</head>
<body class="bg-background font-display text-text-primary">
    <div class="flex min-h-screen">
        <?php include 'menu.php'; ?>
        
        <main class="flex-1 p-10 min-w-0">
            <!-- Header -->
            <header class="flex justify-between items-center mb-10">
                <div>
                    <h1 class="text-4xl font-extrabold text-text-primary">Site Ayarları</h1>
                    <p class="text-text-secondary mt-1">Sitenizi yönetin ve özelleştirin.</p>
                </div>
                <div class="flex items-center gap-4">
                    <button type="submit" form="settings-form" class="flex items-center gap-2 bg-primary text-white font-bold py-3 px-5 rounded-lg hover:bg-primary/90 transition-colors">
                        <span class="material-symbols-outlined">save</span>
                        <span>Kaydet</span>
                    </button>
                </div>
            </header>

            <!-- Success/Error Messages -->
            <?php if ($success): ?>
            <div class="bg-green-50 border-l-4 border-green-400 p-4 mb-6">
                <div class="flex items-center">
                    <span class="material-symbols-outlined text-green-400 mr-2">check_circle</span>
                    <p class="text-green-700 font-medium"><?php echo is_string($success) ? htmlspecialchars($success) : 'Ayarlar başarıyla kaydedildi!'; ?></p>
                </div>
            </div>
            <?php endif; ?>

            <?php if (!empty($error)): ?>
            <div class="bg-red-50 border-l-4 border-red-400 p-4 mb-6">
                <div class="flex items-center">
                    <span class="material-symbols-outlined text-red-400 mr-2">error</span>
                    <p class="text-red-700 font-medium"><?php echo htmlspecialchars($error); ?></p>
                </div>
            </div>
            <?php endif; ?>

            <!-- Favicon Status Check -->
            <?php 
            $favicon_url = $all_settings['favicon_url'] ?? '';
            if (!empty($favicon_url) && !file_exists(__DIR__ . '/..' . $favicon_url)): 
            ?>
            <div class="bg-yellow-50 border-l-4 border-yellow-400 p-4 mb-6">
                <div class="flex items-center justify-between">
                    <div class="flex items-center">
                        <span class="material-symbols-outlined text-yellow-400 mr-2">warning</span>
                        <div>
                            <p class="text-yellow-700 font-medium">Favicon dosyası bulunamadı!</p>
                            <p class="text-yellow-600 text-sm">Database'de kayıtlı favicon: <code><?php echo htmlspecialchars($favicon_url); ?></code></p>
                        </div>
                    </div>
                    <a href="?reset_favicon=1" class="bg-yellow-600 hover:bg-yellow-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors">
                        Favicon Ayarını Temizle
                    </a>
                </div>
            </div>
            <?php endif; ?>

            <div class="flex gap-8">
                <!-- Sol Sidebar - Tab Navigation -->
                <div class="w-80">
                    <div class="bg-surface rounded-xl border border-line-light shadow-soft p-6 sticky top-8">
                        <h3 class="text-lg font-bold text-text-primary mb-4">Ayar Kategorileri</h3>
                        <nav class="space-y-2">
                            <?php foreach ($settings_categories as $key => $category): ?>
                            <button type="button" onclick="showTab('<?php echo $key; ?>', event)" 
                                class="tab-btn w-full flex items-center gap-3 px-4 py-3 rounded-lg text-left transition-colors <?php echo $key === 'site' ? 'bg-primary-light text-primary font-semibold' : 'hover:bg-background text-text-secondary'; ?>"
                                data-tab="<?php echo $key; ?>">
                                <span class="material-symbols-outlined"><?php echo $category['icon']; ?></span>
                                <span><?php echo $category['title']; ?></span>
                            </button>
                            <?php endforeach; ?>
                        </nav>
                    </div>
                </div>

                <!-- Sağ Content - Form Sections -->
                <div class="flex-1">
                    <form id="settings-form" method="POST" enctype="multipart/form-data" class="space-y-8">
                        <?php foreach ($settings_categories as $category_key => $category): ?>
                        <div id="tab-<?php echo $category_key; ?>" class="tab-content <?php echo $category_key === 'site' ? 'active' : ''; ?>">
                            <div class="bg-surface rounded-xl border border-line-light shadow-soft p-8">
                                <div class="flex items-center gap-3 mb-8">
                                    <div class="w-12 h-12 bg-primary-light rounded-lg flex items-center justify-center">
                                        <span class="material-symbols-outlined text-primary text-2xl"><?php echo $category['icon']; ?></span>
                                    </div>
                                    <div>
                                        <h2 class="text-2xl font-bold text-text-primary"><?php echo $category['title']; ?></h2>
                                        <p class="text-text-secondary">Bu kategoriye ait ayarları yapılandırın</p>
                                    </div>
                                </div>

                                <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                                    <?php foreach ($category['settings'] as $setting_key => $config): ?>
                                    <div class="<?php echo in_array($config['type'], ['textarea', 'file']) ? 'lg:col-span-2' : ''; ?>">
                                        <label class="block text-sm font-medium text-text-primary mb-2" for="<?php echo $setting_key; ?>">
                                            <?php echo $config['label']; ?>
                                            <?php if (isset($config['required']) && $config['required']): ?>
                                                <span class="text-red-500">*</span>
                                            <?php endif; ?>
                                        </label>
                                        
                                        <?php if ($config['type'] === 'text'): ?>
                                            <input type="text" 
                                                id="<?php echo $setting_key; ?>" 
                                                name="<?php echo $setting_key; ?>" 
                                                value="<?php echo htmlspecialchars($all_settings[$setting_key]); ?>"
                                                placeholder="<?php echo $config['placeholder'] ?? ''; ?>"
                                                class="w-full border border-line-light rounded-lg px-4 py-3 focus:ring-2 focus:ring-primary focus:border-transparent">
                                        
                                        <?php elseif ($config['type'] === 'textarea'): ?>
                                            <textarea id="<?php echo $setting_key; ?>" 
                                                name="<?php echo $setting_key; ?>" 
                                                rows="3"
                                                placeholder="<?php echo $config['placeholder'] ?? ''; ?>"
                                                class="w-full border border-line-light rounded-lg px-4 py-3 focus:ring-2 focus:ring-primary focus:border-transparent"><?php echo htmlspecialchars($all_settings[$setting_key]); ?></textarea>
                                        
                                        <?php elseif ($config['type'] === 'file'): ?>
                                            <div class="space-y-3">
                                                <input type="file" 
                                                    id="<?php echo $setting_key; ?>" 
                                                    name="<?php echo $setting_key; ?>" 
                                                    accept="<?php echo $config['accept'] ?? ''; ?>"
                                                    class="w-full border border-line-light rounded-lg px-4 py-3 focus:ring-2 focus:ring-primary focus:border-transparent file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:bg-primary file:text-white hover:file:bg-primary-dark">
                                                <?php if (!empty($all_settings[$setting_key])): ?>
                                                    <div class="flex items-center gap-3 p-3 bg-green-50 border border-green-200 rounded-lg">
                                                        <span class="material-symbols-outlined text-green-600">check_circle</span>
                                                        <span class="text-green-700 text-sm">Mevcut dosya: <?php echo basename($all_settings[$setting_key]); ?></span>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        
                                        <?php elseif ($config['type'] === 'number'): ?>
                                            <input type="number" 
                                                id="<?php echo $setting_key; ?>" 
                                                name="<?php echo $setting_key; ?>" 
                                                value="<?php echo htmlspecialchars($all_settings[$setting_key]); ?>"
                                                min="<?php echo $config['min'] ?? ''; ?>" 
                                                max="<?php echo $config['max'] ?? ''; ?>"
                                                class="w-full border border-line-light rounded-lg px-4 py-3 focus:ring-2 focus:ring-primary focus:border-transparent">
                                        
                                        <?php elseif ($config['type'] === 'color'): ?>
                                            <div class="flex items-center gap-3">
                                                <input type="color" 
                                                    id="<?php echo $setting_key; ?>" 
                                                    name="<?php echo $setting_key; ?>" 
                                                    value="<?php echo htmlspecialchars($all_settings[$setting_key]); ?>"
                                                    class="w-16 h-12 border border-line-light rounded-lg cursor-pointer">
                                                <input type="text" 
                                                    value="<?php echo htmlspecialchars($all_settings[$setting_key]); ?>"
                                                    readonly
                                                    class="flex-1 border border-line-light rounded-lg px-4 py-3 bg-background">
                                            </div>
                                        
                                        <?php elseif ($config['type'] === 'select'): ?>
                                            <select id="<?php echo $setting_key; ?>" 
                                                name="<?php echo $setting_key; ?>"
                                                class="w-full border border-line-light rounded-lg px-4 py-3 focus:ring-2 focus:ring-primary focus:border-transparent">
                                                <?php foreach ($config['options'] as $value => $label): ?>
                                                    <option value="<?php echo $value; ?>" <?php echo ($all_settings[$setting_key] == $value) ? 'selected' : ''; ?>>
                                                        <?php echo $label; ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                        
                                        <?php elseif ($config['type'] === 'checkbox'): ?>
                                            <div class="flex items-center gap-3">
                                                <input type="checkbox" 
                                                    id="<?php echo $setting_key; ?>" 
                                                    name="<?php echo $setting_key; ?>" 
                                                    value="1"
                                                    <?php echo ($all_settings[$setting_key] == '1') ? 'checked' : ''; ?>
                                                    class="w-4 h-4 text-primary border-gray-300 rounded focus:ring-primary">
                                                <label for="<?php echo $setting_key; ?>" class="text-sm text-text-secondary">
                                                    Etkinleştir
                                                </label>
                                            </div>
                                        <?php endif; ?>
                                        
                                        <?php if (isset($config['help'])): ?>
                                            <p class="text-xs text-text-secondary mt-2 flex items-start gap-2">
                                                <span class="material-symbols-outlined text-blue-500" style="font-size: 16px;">info</span>
                                                <span><?php echo $config['help']; ?></span>
                                            </p>
                                        <?php endif; ?>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                                
                                <!-- İndexleme Ayarları için özel butonlar -->
                                <?php if ($category_key === 'indexing'): ?>
                                <div class="mt-8 pt-6 border-t border-line-light">
                                    <h3 class="text-lg font-semibold text-text-primary mb-4 flex items-center gap-2">
                                        <span class="material-symbols-outlined text-blue-500">build</span>
                                        Manuel İşlemler
                                    </h3>
                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <!-- Sitemap Generate Button -->
                                        <div class="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-6">
                                            <div class="flex items-center gap-3 mb-3">
                                                <span class="material-symbols-outlined text-blue-600 text-2xl">map</span>
                                                <h4 class="font-semibold text-blue-800 dark:text-blue-200">Sitemap Oluştur</h4>
                                            </div>
                                            <p class="text-sm text-blue-700 dark:text-blue-300 mb-4">
                                                Tüm sayfaları içeren sitemap.xml dosyasını manuel olarak oluşturun.
                                            </p>
                                            <button type="button" 
                                                onclick="generateFile('sitemap')" 
                                                class="w-full bg-blue-600 hover:bg-blue-700 text-white font-semibold py-2 px-4 rounded-lg transition-colors duration-200 flex items-center justify-center gap-2">
                                                <span class="material-symbols-outlined">refresh</span>
                                                <span>Sitemap Oluştur</span>
                                            </button>
                                        </div>
                                        
                                        <!-- Robots.txt Generate Button -->
                                        <div class="bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg p-6">
                                            <div class="flex items-center gap-3 mb-3">
                                                <span class="material-symbols-outlined text-green-600 text-2xl">smart_toy</span>
                                                <h4 class="font-semibold text-green-800 dark:text-green-200">Robots.txt Oluştur</h4>
                                            </div>
                                            <p class="text-sm text-green-700 dark:text-green-300 mb-4">
                                                Arama motorları için robots.txt dosyasını manuel olarak oluşturun.
                                            </p>
                                            <button type="button" 
                                                onclick="generateFile('robots')" 
                                                class="w-full bg-green-600 hover:bg-green-700 text-white font-semibold py-2 px-4 rounded-lg transition-colors duration-200 flex items-center justify-center gap-2">
                                                <span class="material-symbols-outlined">smart_toy</span>
                                                <span>Robots.txt Oluştur</span>
                                            </button>
                                        </div>
                                    </div>
                                    
                                    <!-- Status Area -->
                                    <div id="generation-status" class="mt-4 hidden">
                                        <div class="bg-gray-50 dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg p-4">
                                            <div id="status-content" class="text-sm"></div>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </form>
                </div>
            </div>
        </main>
    </div>

    <script>
        // Tab switching function
        function showTab(tabName, event) {
            if (event) {
                event.preventDefault();
                event.stopPropagation();
            }
            
            // Tüm tab butonlarını inaktif yap
            const tabButtons = document.querySelectorAll('.tab-btn');
            if (tabButtons.length > 0) {
                tabButtons.forEach(btn => {
                    btn.classList.remove('bg-primary-light', 'text-primary', 'font-semibold');
                    btn.classList.add('hover:bg-background', 'text-text-secondary');
                });
            }
            
            // Aktif tab butonunu işaretle
            const activeBtn = document.querySelector(`[data-tab="${tabName}"]`);
            if (activeBtn) {
                activeBtn.classList.add('bg-primary-light', 'text-primary', 'font-semibold');
                activeBtn.classList.remove('hover:bg-background', 'text-text-secondary');
            }
            
            // Tüm tab contentleri gizle
            const tabContents = document.querySelectorAll('.tab-content');
            if (tabContents.length > 0) {
                tabContents.forEach(content => {
                    content.classList.remove('active');
                    content.style.display = 'none';
                });
            }
            
            // Aktif tab contentini göster
            const activeContent = document.getElementById(`tab-${tabName}`);
            if (activeContent) {
                activeContent.style.display = 'block';
                activeContent.classList.add('active');
            }
            
            return false;
        }

        // Sayfa yüklendiğinde çalışacak kodlar
        document.addEventListener('DOMContentLoaded', function() {
            try {
                // Renk picker için real-time güncelleme
                const colorInputs = document.querySelectorAll('input[type="color"]');
                if (colorInputs.length > 0) {
                    colorInputs.forEach(colorInput => {
                        colorInput.addEventListener('change', function() {
                            const textInput = this.parentElement.querySelector('input[type="text"]');
                            if (textInput) {
                                textInput.value = this.value;
                            }
                        });
                    });
                }

                // Form gönderildiğinde loading durumu
                const settingsForm = document.getElementById('settings-form');
                if (settingsForm) {
                    settingsForm.addEventListener('submit', function() {
                        const submitBtn = document.querySelector('button[type="submit"]');
                        if (submitBtn) {
                            submitBtn.innerHTML = '<span class="material-symbols-outlined animate-spin mr-2">refresh</span>Kaydediliyor...';
                            submitBtn.disabled = true;
                        }
                    });
                }

                // İlk tab'ı ayarla ve diğerlerini gizle
                const tabContents = document.querySelectorAll('.tab-content');
                if (tabContents.length > 0) {
                    tabContents.forEach((content, index) => {
                        if (index !== 0) {
                            content.style.display = 'none';
                            content.classList.remove('active');
                        }
                    });
                }
            } catch (error) {
                console.log('Settings page JavaScript initialization completed');
            }
        });
        
        // Sitemap ve Robots.txt oluşturma fonksiyonu
        window.generateFile = function(type) {
            const button = event.target.closest('button');
            const originalContent = button.innerHTML;
            const statusDiv = document.getElementById('generation-status');
            const statusContent = document.getElementById('status-content');
            
            // Loading durumu
            button.disabled = true;
            button.innerHTML = '<span class="material-symbols-outlined animate-spin">refresh</span> Oluşturuluyor...';
            
            // Status alanını göster
            statusDiv.classList.remove('hidden');
            statusContent.innerHTML = '<div class="flex items-center gap-2 text-blue-600"><span class="material-symbols-outlined animate-spin">refresh</span> ' + 
                                     (type === 'sitemap' ? 'Sitemap oluşturuluyor...' : 'Robots.txt oluşturuluyor...') + '</div>';
            
            // AJAX isteği
            fetch(window.location.href, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'action=generate_' + type
            })
            .then(response => response.json())
            .then(data => {
                button.disabled = false;
                button.innerHTML = originalContent;
                
                if (data.success) {
                    statusContent.innerHTML = `
                        <div class="flex items-start gap-3">
                            <span class="material-symbols-outlined text-green-600 mt-0.5">check_circle</span>
                            <div>
                                <div class="font-semibold text-green-800 dark:text-green-200">${data.message}</div>
                                <div class="text-sm text-gray-600 dark:text-gray-400 mt-2">
                                    <div><strong>Dosya:</strong> ${data.details.file}</div>
                                    <div><strong>Boyut:</strong> ${data.details.size}</div>
                                    ${data.details.urls ? '<div><strong>URL Sayısı:</strong> ' + data.details.urls + '</div>' : ''}
                                    ${data.details.lines ? '<div><strong>Satır Sayısı:</strong> ' + data.details.lines + '</div>' : ''}
                                </div>
                            </div>
                        </div>
                    `;
                } else {
                    statusContent.innerHTML = `
                        <div class="flex items-center gap-2 text-red-600">
                            <span class="material-symbols-outlined">error</span>
                            <span>${data.message}</span>
                        </div>
                    `;
                }
                
                // 5 saniye sonra gizle
                setTimeout(() => {
                    statusDiv.classList.add('hidden');
                }, 5000);
            })
            .catch(error => {
                button.disabled = false;
                button.innerHTML = originalContent;
                statusContent.innerHTML = `
                    <div class="flex items-center gap-2 text-red-600">
                        <span class="material-symbols-outlined">error</span>
                        <span>Bir hata oluştu: ${error.message}</span>
                    </div>
                `;
            });
        };
    </script>
</body>
</html>

<?php
/**
 * Sitemap.xml oluşturur
 */
function generateSitemap($pdo, $settings) {
    $baseUrl = 'https://' . $_SERVER['HTTP_HOST'];
    $changefreq = $settings['sitemap_changefreq'] ?? 'weekly';
    $priorityHome = $settings['sitemap_priority_home'] ?? 1.0;
    $priorityCategories = $settings['sitemap_priority_categories'] ?? 0.8;
    $prioritySites = $settings['sitemap_priority_sites'] ?? 0.6;
    
    $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
    $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
    
    // Ana sayfa
    $xml .= "  <url>\n";
    $xml .= "    <loc>{$baseUrl}/</loc>\n";
    $xml .= "    <lastmod>" . date('Y-m-d\TH:i:s+00:00') . "</lastmod>\n";
    $xml .= "    <changefreq>{$changefreq}</changefreq>\n";
    $xml .= "    <priority>{$priorityHome}</priority>\n";
    $xml .= "  </url>\n";
    
    // Sabit sayfalar
    $staticPages = [
        '/newly-added' => 'Newly Added Sites',
        '/topvoted' => 'Top Voted Sites', 
        '/categories' => 'Categories',
        '/login' => 'Login',
        '/register' => 'Register',
        '/submitsite' => 'Submit Site'
    ];
    
    foreach ($staticPages as $url => $title) {
        $xml .= "  <url>\n";
        $xml .= "    <loc>{$baseUrl}{$url}</loc>\n";
        $xml .= "    <lastmod>" . date('Y-m-d\TH:i:s+00:00') . "</lastmod>\n";
        $xml .= "    <changefreq>{$changefreq}</changefreq>\n";
        $xml .= "    <priority>0.7</priority>\n";
        $xml .= "  </url>\n";
    }
    
    // Kategoriler
    if (($settings['sitemap_include_categories'] ?? '1') === '1') {
        $stmt = $pdo->query("SELECT name, slug FROM categories ORDER BY name");
        while ($category = $stmt->fetch(PDO::FETCH_ASSOC)) {
            // Slug varsa kullan, yoksa name'den oluştur
            if (!empty($category['slug'])) {
                $categorySlug = $category['slug'];
            } else {
                // Türkçe karakterleri değiştir ve slug oluştur
                $categoryName = $category['name'];
                $turkish = ['ç', 'Ç', 'ğ', 'Ğ', 'ı', 'I', 'İ', 'ö', 'Ö', 'ş', 'Ş', 'ü', 'Ü'];
                $english = ['c', 'C', 'g', 'G', 'i', 'I', 'I', 'o', 'O', 's', 'S', 'u', 'U'];
                $categoryName = str_replace($turkish, $english, $categoryName);
                $categorySlug = strtolower(preg_replace('/[^a-zA-Z0-9]+/', '-', trim($categoryName)));
            }
            
            $xml .= "  <url>\n";
            $xml .= "    <loc>{$baseUrl}/category/{$categorySlug}</loc>\n";
            $xml .= "    <lastmod>" . date('Y-m-d\TH:i:s+00:00') . "</lastmod>\n";
            $xml .= "    <changefreq>{$changefreq}</changefreq>\n";
            $xml .= "    <priority>{$priorityCategories}</priority>\n";
            $xml .= "  </url>\n";
        }
    }
    
    // Site detayları
    if (($settings['sitemap_include_sites'] ?? '1') === '1') {
        // URLHelper'ı include et
        $urlHelperPath = __DIR__ . '/../url_helper.php';
        if (file_exists($urlHelperPath)) {
            require_once $urlHelperPath;
        }
        
        $stmt = $pdo->query("SELECT site_name, olusturulma_tarihi FROM site_submissions WHERE status = 'approved' ORDER BY olusturulma_tarihi DESC");
        while ($site = $stmt->fetch(PDO::FETCH_ASSOC)) {
            // URLHelper varsa kullan, yoksa basit URL oluştur
            if (class_exists('URLHelper')) {
                $siteUrl = URLHelper::getSiteURL($site['site_name']);
            } else {
                // Fallback: basit URL oluştur
                $siteSlug = strtolower(preg_replace('/[^a-zA-Z0-9]+/', '-', trim($site['site_name'])));
                $siteUrl = "/site/" . $siteSlug;
            }
            
            $lastmod = $site['olusturulma_tarihi'] ? date('Y-m-d\TH:i:s+00:00', strtotime($site['olusturulma_tarihi'])) : date('Y-m-d\TH:i:s+00:00');
            
            $xml .= "  <url>\n";
            $xml .= "    <loc>{$baseUrl}{$siteUrl}</loc>\n";
            $xml .= "    <lastmod>{$lastmod}</lastmod>\n";
            $xml .= "    <changefreq>{$changefreq}</changefreq>\n";
            $xml .= "    <priority>{$prioritySites}</priority>\n";
            $xml .= "  </url>\n";
        }
    }
    
    $xml .= '</urlset>';
    
    // Sitemap dosyasını yaz
    $sitemapPath = __DIR__ . '/../sitemap.xml';
    if (file_put_contents($sitemapPath, $xml) === false) {
        throw new Exception('Sitemap dosyası yazılamadı');
    }
    
    // URL sayısını hesapla
    $urlCount = substr_count($xml, '<url>');
    
    return [
        'file' => 'sitemap.xml',
        'size' => number_format(strlen($xml)) . ' bytes',
        'urls' => $urlCount,
        'path' => $sitemapPath
    ];
}

/**
 * Robots.txt oluşturur
 */
function generateRobotsTxt($pdo, $settings) {
    $baseUrl = 'https://' . $_SERVER['HTTP_HOST'];
    $allowAll = ($settings['robots_allow_all'] ?? '1') === '1';
    $customRules = $settings['robots_custom_rules'] ?? '';
    
    $robots = "# Robots.txt - Generated automatically\n";
    $robots .= "# " . date('Y-m-d H:i:s') . "\n\n";
    
    if ($allowAll) {
        $robots .= "User-agent: *\n";
        $robots .= "Allow: /\n\n";
        
        // Yasaklı dizinler
        $disallowDirs = [
            '/admin/',
            '/config.php',
            '/uploads/temp/',
            '/*.sql$',
            '/tmp/',
            '/logs/'
        ];
        
        foreach ($disallowDirs as $dir) {
            $robots .= "Disallow: {$dir}\n";
        }
        
        $robots .= "\n";
    } else {
        $robots .= "User-agent: *\n";
        $robots .= "Disallow: /\n\n";
    }
    
    // Özel kurallar
    if (!empty($customRules)) {
        $robots .= "# Custom Rules\n";
        $robots .= $customRules . "\n\n";
    }
    
    // Sitemap referansı
    if (($settings['sitemap_enabled'] ?? '1') === '1') {
        $robots .= "# Sitemap\n";
        $robots .= "Sitemap: {$baseUrl}/sitemap.xml\n";
    }
    
    // Robots dosyasını yaz
    $robotsPath = __DIR__ . '/../robots.txt';
    if (file_put_contents($robotsPath, $robots) === false) {
        throw new Exception('Robots.txt dosyası yazılamadı');
    }
    
    return [
        'file' => 'robots.txt',
        'size' => number_format(strlen($robots)) . ' bytes',
        'lines' => substr_count($robots, "\n"),
        'path' => $robotsPath
    ];
}

/**
 * Site eklendiğinde/güncellendiğinde sitemap otomatik güncelle
 */
function autoUpdateSitemap($pdo) {
    try {
        // Settings'i al
        $stmt = $pdo->query('SELECT setting_key, setting_value FROM settings');
        $settings = [];
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
            $settings[$row['setting_key']] = $row['setting_value'];
        }
        
        // Otomatik güncelleme aktif mi kontrol et
        if (($settings['sitemap_auto_update'] ?? '1') === '1' && ($settings['sitemap_enabled'] ?? '1') === '1') {
            // URLHelper'ı yükle
            $urlHelperPath = __DIR__ . '/../url_helper.php';
            if (file_exists($urlHelperPath)) {
                require_once $urlHelperPath;
            }
            
            generateSitemap($pdo, $settings);
            return true;
        }
        return false;
    } catch (Exception $e) {
        error_log('Sitemap auto-update failed: ' . $e->getMessage());
        return false;
    }
}
?>


