<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
require_once __DIR__ . '/../includes/config.php';
if (!isset($pdo)) require_once __DIR__ . '/../includes/config.php';

// Toplu silme işlemi
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_delete'])) {
    $site_ids = $_POST['site_ids'] ?? [];
    
    if (!empty($site_ids) && is_array($site_ids)) {
        try {
            // Güvenlik için sadece sayısal ID'leri al
            $site_ids = array_filter($site_ids, 'is_numeric');
            $site_ids = array_map('intval', $site_ids);
            
            if (!empty($site_ids)) {
                $placeholders = str_repeat('?,', count($site_ids) - 1) . '?';
                
                // Transaction başlat
                $pdo->beginTransaction();
                
                try {
                    // Önce site_votes tablosundaki ilişkili kayıtları sil
                    $sql_votes = "DELETE FROM site_votes WHERE site_id IN ($placeholders)";
                    $stmt_votes = $pdo->prepare($sql_votes);
                    $stmt_votes->execute($site_ids);
                    
                    // Sonra site_submissions tablosundaki kayıtları sil
                    $sql_sites = "DELETE FROM site_submissions WHERE id IN ($placeholders)";
                    $stmt_sites = $pdo->prepare($sql_sites);
                    $stmt_sites->execute($site_ids);
                    
                    // Transaction'ı commit et
                    $pdo->commit();
                    
                    $deleted_count = $stmt_sites->rowCount();
                    $_SESSION['bulk_delete_success'] = $deleted_count;
                    
                } catch (PDOException $e) {
                    // Hata durumunda transaction'ı rollback et
                    $pdo->rollback();
                    throw $e;
                }
            }
        } catch (PDOException $e) {
            $_SESSION['bulk_delete_error'] = 'Veritabanı hatası: ' . $e->getMessage();
        }
    } else {
        $_SESSION['bulk_delete_error'] = 'Silme için site seçilmedi.';
    }
    
    // Redirect to avoid resubmission
    header('Location: sites.php');
    exit;
}

// Sayfalama ayarları
$perPage = isset($_GET['perPage']) ? intval($_GET['perPage']) : 5;
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$offset = ($page - 1) * $perPage;
?>
<!DOCTYPE html>
<html lang="en"><head>
<meta charset="utf-8"/>
<meta content="width=device-width, initial-scale=1.0" name="viewport"/>
<title>Siteler Yönetimi</title>
<link href="https://fonts.googleapis.com" rel="preconnect"/>
<link crossorigin="" href="https://fonts.gstatic.com" rel="preconnect"/>
<link href="https://fonts.googleapis.com/css2?family=Manrope:wght@400;500;600;700;800&amp;display=swap" rel="stylesheet"/>
<script src="https://cdn.tailwindcss.com?plugins=forms,container-queries"></script>
<link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined:opsz,wght,FILL,GRAD@20..48,100..700,0..1,-50..200" rel="stylesheet"/>
<script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: "#1877f2", // A modern, friendly blue
                        "primary-light": "#e7f3ff",
                        background: "#f0f2f5",
                        surface: "#ffffff",
                        "text-primary": "#1c1e21",
                        "text-secondary": "#606770",
                        "line-light": "#e0e0e0",
                    },
                    fontFamily: {
                        display: ["Manrope", "sans-serif"],
                    },
                    borderRadius: {
                        DEFAULT: "0.75rem",
                        lg: "1rem",
                        xl: "1.5rem",
                        full: "9999px"
                    },
                    boxShadow: {
                        'soft': '0 4px 12px 0 rgb(0 0 0 / 0.07)',
                        'subtle': '0 1px 2px 0 rgb(0 0 0 / 0.05)',
                    }
                },
            },
        };
    </script>
<style>
        .material-symbols-outlined {
            font-variation-settings: "FILL" 0, "wght" 400, "GRAD" 0, "opsz" 24;
        }
        .active .material-symbols-outlined {
            font-variation-settings: 'FILL' 1;
        }
    </style>
</head>
<body class="bg-background font-display text-text-primary">
<div class="flex min-h-screen">
<?php include 'menu.php'; ?>
<main class="flex-1 p-10 min-w-0">
<?php
$page_title = "Siteler Yönetimi";
$page_subtitle = "Dizininizdeki siteleri yönetin, ekleyin ve düzenleyin.";
include 'includes/admin_header.php';
?>

<!-- Flash Messages -->
<?php if (isset($_SESSION['bulk_delete_success'])): ?>
<div class="mb-6 p-4 bg-green-100 border border-green-200 rounded-lg flex items-center">
    <span class="material-symbols-outlined text-green-600 mr-3">check_circle</span>
    <div>
        <p class="text-green-800 font-medium">Toplu silme başarılı!</p>
        <p class="text-green-700 text-sm"><?php echo $_SESSION['bulk_delete_success']; ?> site silindi.</p>
    </div>
</div>
<?php 
unset($_SESSION['bulk_delete_success']);
endif; 
?>

<?php if (isset($_SESSION['bulk_delete_error'])): ?>
<div class="mb-6 p-4 bg-red-100 border border-red-200 rounded-lg flex items-center">
    <span class="material-symbols-outlined text-red-600 mr-3">error</span>
    <div>
        <p class="text-red-800 font-medium">Toplu silme başarısız!</p>
        <p class="text-red-700 text-sm"><?php echo htmlspecialchars($_SESSION['bulk_delete_error']); ?></p>
    </div>
</div>
<?php 
unset($_SESSION['bulk_delete_error']);
endif; 
?>

<!-- Search and Filter Form -->
<form method="GET" class="flex items-center gap-4 bg-surface rounded-xl shadow-soft p-6 mb-6 border border-line-light">
    <div class="flex items-center gap-2 flex-1 min-w-[220px] max-w-md">
        <div class="relative w-full">
            <span class="material-symbols-outlined absolute left-4 top-1/2 -translate-y-1/2 text-text-secondary">search</span>
            <input class="w-full pl-12 pr-4 py-3 bg-background border-none rounded-lg focus:ring-2 focus:ring-primary" name="search" value="<?php echo htmlspecialchars($_GET['search'] ?? ''); ?>" placeholder="İsme veya URL'ye göre ara..." type="text"/>
        </div>
        <button type="submit" class="px-4 py-2 rounded-lg bg-primary text-white font-bold">Ara</button>
    </div>
    <select name="status" class="px-4 py-2 bg-background rounded-lg text-text-secondary font-medium hover:bg-line-light w-44" onchange="this.form.submit()">
        <option value="all" <?php if(!isset($_GET['status'])||$_GET['status']=='all') echo 'selected'; ?>>Durum: Tümü</option>
        <option value="pending" <?php if(isset($_GET['status'])&&$_GET['status']=='pending') echo 'selected'; ?>>Beklemede</option>
        <option value="approved" <?php if(isset($_GET['status'])&&$_GET['status']=='approved') echo 'selected'; ?>>Onaylandı</option>
        <option value="rejected" <?php if(isset($_GET['status'])&&$_GET['status']=='rejected') echo 'selected'; ?>>Reddedildi</option>
    </select>
    <select name="sort" class="px-4 py-2 bg-background rounded-lg text-text-secondary font-medium hover:bg-line-light w-52" onchange="this.form.submit()">
        <option value="date" <?php if(!isset($_GET['sort'])||$_GET['sort']=='date') echo 'selected'; ?>>Sırala: Eklenme Tarihi</option>
        <option value="name" <?php if(isset($_GET['sort'])&&$_GET['sort']=='name') echo 'selected'; ?>>Sırala: İsim</option>
        <option value="status" <?php if(isset($_GET['sort'])&&$_GET['sort']=='status') echo 'selected'; ?>>Sırala: Durum</option>
    </select>
    <select name="perPage" class="px-4 py-2 bg-background rounded-lg text-text-secondary font-medium hover:bg-line-light" onchange="this.form.submit()">
        <?php $perPageOptions = [5,10,20,50,100,200,500,1000];
        $selectedPerPage = isset($_GET['perPage']) ? intval($_GET['perPage']) : 5;
        foreach($perPageOptions as $opt) {
            echo '<option value="'.$opt.'"'.($selectedPerPage==$opt?' selected':'').'>'.$opt.' / sayfa</option>';
        }
        ?>
    </select>
    <input type="hidden" name="page" value="<?php echo $page; ?>">
</form>

<!-- Bulk Actions -->
<div id="bulkActions" class="hidden bg-red-50 border border-red-200 rounded-xl p-4 mb-6">
    <div class="flex items-center justify-between">
        <div class="flex items-center gap-3">
            <span class="material-symbols-outlined text-red-600">delete</span>
            <span class="font-semibold text-red-800">Toplu İşlemler</span>
            <span id="selectedCount" class="bg-red-100 text-red-700 px-2 py-1 rounded-full text-sm font-medium">0 seçildi</span>
        </div>
        <div class="flex items-center gap-3">
            <button onclick="clearSelection()" class="px-4 py-2 bg-gray-100 hover:bg-gray-200 text-gray-700 rounded-lg font-medium">Temizle</button>
            <button onclick="confirmBulkDelete()" class="px-4 py-2 bg-red-600 hover:bg-red-700 text-white rounded-lg font-medium">Seçilenleri Sil</button>
        </div>
    </div>
</div>

<div class="overflow-x-auto bg-white rounded-xl shadow-soft border border-line-light">
<table class="w-full text-left table-fixed min-w-[1200px]">
<thead class="border-b border-line-light">
<tr class="bg-gradient-to-r from-primary/5 to-primary/10 border-b-2 border-primary/20">
    <th class="p-6 font-bold text-text-primary text-lg w-16">
        <div class="flex items-center gap-3">
            <input id="selectAll" class="rounded border-gray-300 text-primary focus:ring-primary scale-125" type="checkbox" onchange="toggleAll()"/>
            <span class="material-symbols-outlined text-primary">select_all</span>
        </div>
    </th>
    <th class="p-6 font-bold text-text-primary text-lg w-20">
        <div class="flex items-center gap-3">
            <span class="material-symbols-outlined text-primary">image</span>
            Logo
        </div>
    </th>
    <th class="p-6 font-bold text-text-primary text-lg w-48">
        <div class="flex items-center gap-3">
            <span class="material-symbols-outlined text-primary">language</span>
            Site Adı
        </div>
    </th>
    <th class="p-6 font-bold text-text-primary text-lg w-32">
        <div class="flex items-center gap-3">
            <span class="material-symbols-outlined text-primary">category</span>
            Kategori
        </div>
    </th>
    <th class="p-6 font-bold text-text-primary text-lg w-32">
        <div class="flex items-center gap-3">
            <span class="material-symbols-outlined text-primary">schedule</span>
            Eklenme Tarihi
        </div>
    </th>
    <th class="p-6 font-bold text-text-primary text-lg text-center w-28">
        <div class="flex items-center justify-center gap-3">
            <span class="material-symbols-outlined text-primary">flag</span>
            Durum
        </div>
    </th>
    <th class="p-6 font-bold text-text-primary text-lg w-32">
        <div class="flex items-center gap-3">
            <span class="material-symbols-outlined text-primary">person</span>
            Kullanıcı
        </div>
    </th>
    <th class="p-6 font-bold text-text-primary text-lg w-32">
        <div class="flex items-center gap-3">
            <span class="material-symbols-outlined text-primary">location_on</span>
            IP Adresi
        </div>
    </th>
    <th class="p-6 font-bold text-text-primary text-lg text-center w-32">
        <div class="flex items-center justify-center gap-3">
            <span class="material-symbols-outlined text-primary">check_circle</span>
            Onay
        </div>
    </th>
    <th class="p-6 font-bold text-text-primary text-lg text-center w-40">
        <div class="flex items-center justify-center gap-3">
            <span class="material-symbols-outlined text-primary">settings</span>
            İşlemler
        </div>
    </th>
</tr>
</thead>
<tbody>

<?php
// Toplam kayıt sayısı
$totalStmt = $pdo->query('SELECT COUNT(*) FROM site_submissions');
$totalRows = $totalStmt->fetchColumn();
$totalPages = ceil($totalRows / $perPage);

// Filtre ve sıralama
$whereArr = [];
$params = [];
if (isset($_GET['status']) && $_GET['status'] != 'all') {
    $whereArr[] = 's.status = :status';
    $params[':status'] = $_GET['status'];
}
if (isset($_GET['search']) && $_GET['search'] !== '') {
    $whereArr[] = '(s.site_name LIKE :search OR s.site_url LIKE :search)';
    $params[':search'] = '%' . $_GET['search'] . '%';
}
$whereSQL = count($whereArr) ? ('WHERE ' . implode(' AND ', $whereArr)) : '';
$orderSQL = 'ORDER BY s.id DESC';
if (isset($_GET['sort'])) {
    if ($_GET['sort'] == 'name') $orderSQL = 'ORDER BY s.site_name ASC';
    elseif ($_GET['sort'] == 'status') $orderSQL = 'ORDER BY s.status ASC';
    else $orderSQL = 'ORDER BY s.id DESC';
}
$sql = 'SELECT s.*, c.name as category_name, u.username FROM site_submissions s LEFT JOIN categories c ON s.category_id = c.id LEFT JOIN user u ON s.user_id = u.id ' . $whereSQL . ' ' . $orderSQL . ' LIMIT :limit OFFSET :offset';
$stmt = $pdo->prepare($sql);
foreach ($params as $key => $val) {
    $stmt->bindValue($key, $val);
}
$stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();

while ($site = $stmt->fetch(PDO::FETCH_ASSOC)) {
    echo '<tr class="border-b border-line-light hover:bg-gradient-to-r hover:from-primary/5 hover:to-transparent transition-all duration-200 group" data-site-id="'.$site['id'].'">';
    
    // Checkbox with ID
    echo '<td class="p-6">';
    echo '<div class="flex items-center gap-4">';
    echo '<input class="site-checkbox rounded border-gray-300 text-primary focus:ring-primary scale-125" type="checkbox" value="'.$site['id'].'" onchange="updateBulkActions()"/>';
    echo '<div class="text-text-secondary text-sm">#'.$site['id'].'</div>';
    echo '</div>';
    echo '</td>';
    
    // Logo sütunu
    echo '<td class="p-6">';
    $logoUrl = $site['site_logo'] ?? '';
    if (!empty($logoUrl)) {
        // Clean up any existing full URLs
        $logoUrl = str_replace('https://usasportstvs.com/uploads/', '', $logoUrl);
        $logoUrl = str_replace('https://usasportstvs.com/', '', $logoUrl);
        
        // Remove /admin/uploads/ prefix if exists
        if (str_starts_with($logoUrl, '/admin/uploads/')) {
            $logoUrl = str_replace('/admin/uploads/', '', $logoUrl);
        }
        
        // Remove admin/uploads/ prefix if exists  
        if (str_starts_with($logoUrl, 'admin/uploads/')) {
            $logoUrl = str_replace('admin/uploads/', '', $logoUrl);
        }
        
        // Remove uploads/ prefix if exists
        if (str_starts_with($logoUrl, 'uploads/')) {
            $logoUrl = str_replace('uploads/', '', $logoUrl);
        }
        
        // Remove /uploads/ prefix if exists
        if (str_starts_with($logoUrl, '/uploads/')) {
            $logoUrl = str_replace('/uploads/', '', $logoUrl);
        }
        
        // Create final URL
        $logoUrl = 'https://usasportstvs.com/uploads/' . $logoUrl;
        
        echo '<div class="w-12 h-12 rounded-xl overflow-hidden shadow-soft border border-line-light">';
        echo '<img src="'.htmlspecialchars($logoUrl).'" alt="Logo" class="w-full h-full object-contain bg-gray-50" loading="lazy" onerror="this.onerror=null;this.src=\'https://usasportstvs.com/uploads/no-logo.png\';this.alt=\'No Logo\';">';
        echo '</div>';
    } else {
        echo '<div class="w-12 h-12 rounded-xl bg-gray-100 flex items-center justify-center border border-line-light">';
        echo '<span class="material-symbols-outlined text-gray-400">image</span>';
        echo '</div>';
    }
    echo '</td>';
    
    // Site Name
    echo '<td class="p-6">';
    echo '<div class="font-bold text-text-primary text-lg group-hover:text-primary transition-colors site-name-cell">'.htmlspecialchars($site['site_name']).'</div>';
    echo '</td>';
    
    // Category
    echo '<td class="p-6">';
    echo '<div class="inline-flex items-center gap-2 bg-background px-3 py-1 rounded-lg site-category-cell">';
    echo '<span class="material-symbols-outlined text-text-secondary text-sm">tag</span>';
    echo '<span class="text-text-secondary">'.htmlspecialchars($site['category_name'] ?? $site['category_id']).'</span>';
    echo '</div>';
    echo '</td>';
    
    // Date
    echo '<td class="p-6">';
    echo '<div class="text-text-secondary site-date-cell">'.htmlspecialchars($site['olusturulma_tarihi']).'</div>';
    echo '</td>';
    
    // Status
    echo '<td class="p-6 text-center site-status-cell">';
    $status = strtolower(trim($site['status']));
    if ($status === 'approved' || $status === 'onaylandı') {
        echo '<span class="inline-flex items-center gap-2 px-4 py-2 text-sm font-bold text-green-700 bg-green-100 rounded-full shadow-soft">';
        echo '<span class="material-symbols-outlined text-xs">check_circle</span>';
        echo 'Onaylandı</span>';
    } elseif ($status === 'pending' || $status === 'bekliyor') {
        echo '<span class="inline-flex items-center gap-2 px-4 py-2 text-sm font-bold text-yellow-700 bg-yellow-100 rounded-full shadow-soft">';
        echo '<span class="material-symbols-outlined text-xs">schedule</span>';
        echo 'Beklemede</span>';
    } elseif ($status === 'rejected' || $status === 'reddedildi') {
        echo '<span class="inline-flex items-center gap-2 px-4 py-2 text-sm font-bold text-red-700 bg-red-100 rounded-full shadow-soft">';
        echo '<span class="material-symbols-outlined text-xs">cancel</span>';
        echo 'Reddedildi</span>';
    } else {
        echo '<span class="inline-flex items-center gap-2 px-4 py-2 text-sm font-bold text-gray-700 bg-gray-100 rounded-full shadow-soft">';
        echo '<span class="material-symbols-outlined text-xs">help</span>';
        echo 'Unknown</span>';
    }
    echo '</td>';
    
    // User
    echo '<td class="p-6">';
    echo '<div class="inline-flex items-center gap-2 bg-background px-3 py-1 rounded-lg site-user-cell">';
    echo '<span class="material-symbols-outlined text-text-secondary text-sm">person</span>';
    echo '<span class="text-text-secondary">'.htmlspecialchars($site['username'] ?? $site['user_id']).'</span>';
    echo '</div>';
    echo '</td>';
    
    // IP
    echo '<td class="p-6">';
    echo '<div class="inline-flex items-center gap-2 bg-background px-3 py-1 rounded-lg site-ip-cell">';
    echo '<span class="material-symbols-outlined text-text-secondary text-sm">location_on</span>';
    echo '<span class="text-text-secondary font-mono text-sm">'.htmlspecialchars($site['ip']).'</span>';
    echo '</div>';
    echo '</td>';
    
    // Approval Buttons
    echo '<td class="p-6 text-center">';
    echo '<div class="flex items-center justify-center gap-3">';
    echo '<form class="onay-form" data-action="approve" data-id="'.$site['id'].'" style="display:inline;">';
    echo '<button type="submit" class="w-12 h-12 rounded-xl bg-green-100 text-green-700 flex items-center justify-center hover:bg-green-200 hover:scale-105 transition-all duration-200 shadow-soft" title="Onayla">';
    echo '<span class="material-symbols-outlined text-xl">check_circle</span>';
    echo '</button>';
    echo '</form>';
    echo '<form class="onay-form" data-action="disapprove" data-id="'.$site['id'].'" style="display:inline;">';
    echo '<button type="submit" class="w-12 h-12 rounded-xl bg-yellow-100 text-yellow-700 flex items-center justify-center hover:bg-yellow-200 hover:scale-105 transition-all duration-200 shadow-soft" title="Onaydan Kaldır">';
    echo '<span class="material-symbols-outlined text-xl">cancel</span>';
    echo '</button>';
    echo '</form>';
    echo '</div>';
    echo '</td>';
    
    // Actions
    echo '<td class="p-4">';
    echo '<div class="flex items-center gap-2">';
    echo '<a href="site_view.php?id='.$site['id'].'" class="w-9 h-9 rounded-full bg-green-100 text-green-600 flex items-center justify-center hover:bg-green-200" title="View Site">';
    echo '<span class="material-symbols-outlined text-xl">visibility</span>';
    echo '</a>';
    echo '<a href="site_edit_form.php?id='.$site['id'].'" class="w-9 h-9 rounded-full bg-blue-100 text-blue-600 flex items-center justify-center hover:bg-blue-200" title="Edit Site">';
    echo '<span class="material-symbols-outlined text-xl">edit</span>';
    echo '</a>';
    echo '<a href="site_delete.php?id='.$site['id'].'" onclick="return confirmDelete(\''.$site['site_name'].'\', '.$site['id'].');" class="w-9 h-9 rounded-full bg-red-100 text-red-600 flex items-center justify-center hover:bg-red-200" title="Delete Site">';
    echo '<span class="material-symbols-outlined text-xl">delete</span>';
    echo '</a>';
    echo '</div></td>';
    echo '</tr>';
}
?>
</tbody>
</table>
</div>

<!-- Pagination -->
<?php if ($totalPages > 1): ?>
<div class="flex items-center justify-center gap-2 mt-6">
    <?php if ($page > 1): ?>
        <a href="?page=<?php echo ($page - 1); ?>&<?php echo http_build_query(array_filter(['search' => $_GET['search'] ?? '', 'status' => $_GET['status'] ?? '', 'sort' => $_GET['sort'] ?? '', 'perPage' => $_GET['perPage'] ?? ''])); ?>" class="flex items-center justify-center w-10 h-10 rounded-lg bg-background hover:bg-line-light transition-colors">
            <span class="material-symbols-outlined">chevron_left</span>
        </a>
    <?php endif; ?>
    
    <?php
    $startPage = max(1, $page - 2);
    $endPage = min($totalPages, $page + 2);
    
    if ($startPage > 1): ?>
        <a href="?page=1&<?php echo http_build_query(array_filter(['search' => $_GET['search'] ?? '', 'status' => $_GET['status'] ?? '', 'sort' => $_GET['sort'] ?? '', 'perPage' => $_GET['perPage'] ?? ''])); ?>" class="flex items-center justify-center w-10 h-10 rounded-lg bg-background hover:bg-line-light transition-colors font-medium">1</a>
        <?php if ($startPage > 2): ?>
            <span class="px-2 text-text-secondary">...</span>
        <?php endif; ?>
    <?php endif; ?>
    
    <?php for ($i = $startPage; $i <= $endPage; $i++): ?>
        <a href="?page=<?php echo $i; ?>&<?php echo http_build_query(array_filter(['search' => $_GET['search'] ?? '', 'status' => $_GET['status'] ?? '', 'sort' => $_GET['sort'] ?? '', 'perPage' => $_GET['perPage'] ?? ''])); ?>" class="flex items-center justify-center w-10 h-10 rounded-lg <?php echo $i == $page ? 'bg-primary text-white' : 'bg-background hover:bg-line-light'; ?> transition-colors font-medium">
            <?php echo $i; ?>
        </a>
    <?php endfor; ?>
    
    <?php if ($endPage < $totalPages): ?>
        <?php if ($endPage < $totalPages - 1): ?>
            <span class="px-2 text-text-secondary">...</span>
        <?php endif; ?>
        <a href="?page=<?php echo $totalPages; ?>&<?php echo http_build_query(array_filter(['search' => $_GET['search'] ?? '', 'status' => $_GET['status'] ?? '', 'sort' => $_GET['sort'] ?? '', 'perPage' => $_GET['perPage'] ?? ''])); ?>" class="flex items-center justify-center w-10 h-10 rounded-lg bg-background hover:bg-line-light transition-colors font-medium"><?php echo $totalPages; ?></a>
    <?php endif; ?>
    
    <?php if ($page < $totalPages): ?>
        <a href="?page=<?php echo ($page + 1); ?>&<?php echo http_build_query(array_filter(['search' => $_GET['search'] ?? '', 'status' => $_GET['status'] ?? '', 'sort' => $_GET['sort'] ?? '', 'perPage' => $_GET['perPage'] ?? ''])); ?>" class="flex items-center justify-center w-10 h-10 rounded-lg bg-background hover:bg-line-light transition-colors">
            <span class="material-symbols-outlined">chevron_right</span>
        </a>
    <?php endif; ?>
</div>
<?php endif; ?>

</main>
</div>

<script>
// Onay butonları - Form submit
document.querySelectorAll('.onay-form').forEach(function(form) {
    form.addEventListener('submit', function(e) {
        e.preventDefault();
        var siteId = this.getAttribute('data-id');
        var action = this.getAttribute('data-action');
        
        // Action'ı status'a çevir
        var status = '';
        if (action === 'approve') {
            status = 'approved';
        } else if (action === 'disapprove') {
            status = 'rejected';
        } else {
            status = 'pending';
        }
        
        // Form oluştur ve submit et
        var submitForm = document.createElement('form');
        submitForm.method = 'POST';
        submitForm.action = 'update_site_status.php';
        submitForm.style.display = 'none';
        
        var siteIdInput = document.createElement('input');
        siteIdInput.type = 'hidden';
        siteIdInput.name = 'site_id';
        siteIdInput.value = siteId;
        submitForm.appendChild(siteIdInput);
        
        var statusInput = document.createElement('input');
        statusInput.type = 'hidden';
        statusInput.name = 'status';
        statusInput.value = status;
        submitForm.appendChild(statusInput);
        
        document.body.appendChild(submitForm);
        submitForm.submit();
    });
});

// Bulk operations JavaScript
function toggleAll() {
    const masterCheckbox = document.getElementById('selectAll');
    const siteCheckboxes = document.querySelectorAll('.site-checkbox');
    
    siteCheckboxes.forEach(checkbox => {
        checkbox.checked = masterCheckbox.checked;
    });
    
    updateBulkActions();
}

function updateBulkActions() {
    const checkedBoxes = document.querySelectorAll('.site-checkbox:checked');
    const bulkActionsDiv = document.getElementById('bulkActions');
    const selectedCountSpan = document.getElementById('selectedCount');
    const masterCheckbox = document.getElementById('selectAll');
    
    if (checkedBoxes.length > 0) {
        bulkActionsDiv.classList.remove('hidden');
        selectedCountSpan.textContent = checkedBoxes.length + ' selected';
    } else {
        bulkActionsDiv.classList.add('hidden');
    }
    
    // Update master checkbox state
    const allCheckboxes = document.querySelectorAll('.site-checkbox');
    if (checkedBoxes.length === allCheckboxes.length && allCheckboxes.length > 0) {
        masterCheckbox.checked = true;
        masterCheckbox.indeterminate = false;
    } else if (checkedBoxes.length > 0) {
        masterCheckbox.checked = false;
        masterCheckbox.indeterminate = true;
    } else {
        masterCheckbox.checked = false;
        masterCheckbox.indeterminate = false;
    }
}

function clearSelection() {
    const allCheckboxes = document.querySelectorAll('.site-checkbox, #selectAll');
    allCheckboxes.forEach(checkbox => {
        checkbox.checked = false;
        checkbox.indeterminate = false;
    });
    updateBulkActions();
}

function confirmBulkDelete() {
    const checkedBoxes = document.querySelectorAll('.site-checkbox:checked');
    
    if (checkedBoxes.length === 0) {
        alert('Please select sites to delete!');
        return;
    }
    
    if (confirm(`Are you sure you want to delete ${checkedBoxes.length} selected sites? This action cannot be undone!`)) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.style.display = 'none';
        
        // Add bulk delete flag
        const bulkDeleteInput = document.createElement('input');
        bulkDeleteInput.type = 'hidden';
        bulkDeleteInput.name = 'bulk_delete';
        bulkDeleteInput.value = '1';
        form.appendChild(bulkDeleteInput);
        
        // Add selected site IDs
        checkedBoxes.forEach(checkbox => {
            const input = document.createElement('input');
            input.type = 'hidden';
            input.name = 'site_ids[]';
            input.value = checkbox.value;
            form.appendChild(input);
        });
        
        document.body.appendChild(form);
        form.submit();
    }
}

// Modern delete confirmation
function confirmDelete(siteName, siteId) {
    const modal = document.createElement('div');
    modal.className = 'fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50';
    modal.innerHTML = `
        <div class="bg-white rounded-lg p-6 max-w-md w-full mx-4 transform scale-95 transition-transform duration-200">
            <div class="flex items-center mb-4">
                <span class="material-symbols-outlined text-red-500 text-3xl mr-3">warning</span>
                <h3 class="text-lg font-semibold text-gray-900">Confirm Deletion</h3>
            </div>
            <p class="text-gray-600 mb-6">
                Are you sure you want to delete <strong>"${siteName}"</strong>?<br>
                <span class="text-sm text-red-600">This action cannot be undone.</span>
            </p>
            <div class="flex gap-3">
                <button onclick="closeDeleteModal()" class="flex-1 px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50">
                    Cancel
                </button>
                <button onclick="proceedDelete(${siteId})" class="flex-1 px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700">
                    Delete Site
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Animate in
    setTimeout(() => {
        modal.querySelector('.bg-white').style.transform = 'scale(1)';
    }, 10);
    
    // Store modal reference
    window.currentDeleteModal = modal;
    
    return false; // Prevent default link behavior
}

function closeDeleteModal() {
    if (window.currentDeleteModal) {
        const modal = window.currentDeleteModal;
        modal.querySelector('.bg-white').style.transform = 'scale(0.95)';
        setTimeout(() => {
            document.body.removeChild(modal);
        }, 200);
        window.currentDeleteModal = null;
    }
}

function proceedDelete(siteId) {
    closeDeleteModal();
    
    // Show loading indicator
    const loader = document.createElement('div');
    loader.className = 'fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50';
    loader.innerHTML = `
        <div class="bg-white rounded-lg p-6 text-center">
            <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-3"></div>
            <p class="text-gray-600">Deleting site...</p>
        </div>
    `;
    document.body.appendChild(loader);
    
    // Proceed with deletion
    window.location.href = `site_delete.php?id=${siteId}`;
}

// Close modal on escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape' && window.currentDeleteModal) {
        closeDeleteModal();
    }
});
</script>
</body>
</html>

