<?php
session_start();
require_once __DIR__ . '/../includes/config.php';
require_once 'admin_auth.php';

if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header('Location: users.php?error=invalid_user');
    exit;
}

$user_id = intval($_GET['id']);

// Kullanıcıyı getir
$stmt = $pdo->prepare('SELECT * FROM user WHERE id = ? AND is_banned = 1');
$stmt->execute([$user_id]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$user) {
    header('Location: users.php?error=user_not_found_or_not_banned');
    exit;
}

$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $reason = trim($_POST['reason'] ?? '');
    
    if (empty($reason)) {
        $error = 'Unban nedeni gereklidir.';
    } else {
        try {
            $pdo->beginTransaction();
            
            // Kullanıcının banını kaldır
            $stmt = $pdo->prepare('UPDATE user SET is_banned = 0 WHERE id = ?');
            $stmt->execute([$user_id]);
            
            // Ban geçmişine ekle
            $stmt = $pdo->prepare('INSERT INTO ban_history (user_id, ip_address, action, reason, admin_id) VALUES (?, ?, ?, ?, ?)');
            $stmt->execute([$user_id, $user['ip'], 'unban', $reason, $_SESSION['user_id']]);
            
            $pdo->commit();
            header('Location: users.php?success=user_unbanned');
            exit;
            
        } catch (Exception $e) {
            $pdo->rollback();
            $error = 'Unban işlemi sırasında hata: ' . $e->getMessage();
        }
    }
}

// Ban geçmişini getir
$stmt = $pdo->prepare('
    SELECT bh.*, u.username as admin_name 
    FROM ban_history bh 
    LEFT JOIN user u ON bh.admin_id = u.id 
    WHERE bh.user_id = ? 
    ORDER BY bh.created_at DESC 
    LIMIT 5
');
$stmt->execute([$user_id]);
$ban_history = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="utf-8"/>
    <meta content="width=device-width, initial-scale=1.0" name="viewport"/>
    <title>Kullanıcı Ban Kaldır - Admin Panel</title>
    <link href="https://fonts.googleapis.com" rel="preconnect"/>
    <link crossorigin="" href="https://fonts.gstatic.com" rel="preconnect"/>
    <link href="https://fonts.googleapis.com/css2?family=Manrope:wght@400;500;600;700;800&display=swap" rel="stylesheet"/>
    <script src="https://cdn.tailwindcss.com?plugins=forms,container-queries"></script>
    <link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined:opsz,wght,FILL,GRAD@20..48,100..700,0..1,-50..200" rel="stylesheet"/>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: "#1877f2",
                        "primary-light": "#e7f3ff",
                        background: "#f0f2f5",
                        surface: "#ffffff",
                        "text-primary": "#1c1e21",
                        "text-secondary": "#606770",
                        "line-light": "#e0e0e0",
                    },
                    fontFamily: {
                        display: ["Manrope", "sans-serif"],
                    },
                    borderRadius: {
                        DEFAULT: "0.75rem",
                        lg: "1rem",
                        xl: "1.5rem",
                        full: "9999px"
                    },
                    boxShadow: {
                        'soft': '0 4px 12px 0 rgb(0 0 0 / 0.07)',
                        'subtle': '0 1px 2px 0 rgb(0 0 0 / 0.05)',
                    }
                },
            },
        };
    </script>
    <style>
        .material-symbols-outlined {
            font-variation-settings: "FILL" 0, "wght" 400, "GRAD" 0, "opsz" 24;
        }
    </style>
</head>
<body class="bg-background font-display text-text-primary">
    <div class="flex min-h-screen">
        <?php include 'menu.php'; ?>
        <main class="flex-1 p-10 min-w-0">
            <header class="flex justify-between items-center mb-10">
                <div>
                    <h1 class="text-4xl font-extrabold text-text-primary">Ban Kaldır</h1>
                    <p class="text-text-secondary mt-1">Kullanıcının banını kaldır ve erişimini yeniden aktifleştir.</p>
                </div>
                <a href="users.php" class="flex items-center gap-2 bg-gray-100 text-gray-700 font-bold py-3 px-5 rounded-lg hover:bg-gray-200 transition-colors">
                    <span class="material-symbols-outlined">arrow_back</span>
                    <span>Geri Dön</span>
                </a>
            </header>

            <div class="max-w-4xl">
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    <!-- Sol Kolon: Kullanıcı Bilgileri ve Unban Formu -->
                    <div class="space-y-6">
                        <!-- Kullanıcı Bilgileri -->
                        <div class="bg-surface p-6 rounded-xl border border-line-light shadow-soft">
                            <h2 class="text-xl font-bold text-text-primary mb-4 flex items-center gap-2">
                                <span class="material-symbols-outlined text-green-600">person_check</span>
                                Banlı Kullanıcı
                            </h2>
                            <div class="flex items-center gap-4">
                                <img alt="User Avatar" class="w-16 h-16 rounded-full object-cover border-2 border-red-200" 
                                     src="<?php echo htmlspecialchars($user['profile_photo'] ? '../' . $user['profile_photo'] : '../uploads/no-logo.png'); ?>"/>
                                <div>
                                    <h3 class="font-bold text-lg text-text-primary"><?php echo htmlspecialchars($user['username']); ?></h3>
                                    <p class="text-text-secondary"><?php echo htmlspecialchars($user['email']); ?></p>
                                    <p class="text-sm text-text-secondary">IP: <?php echo htmlspecialchars($user['ip'] ?? 'Bilinmiyor'); ?></p>
                                    <div class="mt-2">
                                        <span class="inline-flex items-center gap-1 px-2 py-1 text-xs font-medium rounded-full bg-red-100 text-red-700">
                                            <span class="material-symbols-outlined text-xs">block</span>
                                            Banlı
                                        </span>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Unban Formu -->
                        <div class="bg-surface p-6 rounded-xl border border-line-light shadow-soft">
                            <h2 class="text-xl font-bold text-text-primary mb-6 flex items-center gap-2">
                                <span class="material-symbols-outlined text-green-600">check_circle</span>
                                Ban Kaldırma
                            </h2>

                            <?php if (!empty($error)): ?>
                            <div class="bg-red-50 border-l-4 border-red-400 p-4 mb-6">
                                <div class="flex items-center">
                                    <span class="material-symbols-outlined text-red-400 mr-2">error</span>
                                    <p class="text-red-700"><?php echo htmlspecialchars($error); ?></p>
                                </div>
                            </div>
                            <?php endif; ?>

                            <form method="POST" class="space-y-6">
                                <div>
                                    <label for="reason" class="block text-sm font-medium text-text-primary mb-2">
                                        Unban Nedeni *
                                    </label>
                                    <textarea id="reason" name="reason" rows="4" required
                                        class="w-full border border-line-light rounded-lg px-4 py-3 focus:ring-2 focus:ring-green-500 focus:border-transparent placeholder:text-text-secondary"
                                        placeholder="Bu kullanıcının banının neden kaldırıldığını açıklayın..."><?php echo htmlspecialchars($_POST['reason'] ?? ''); ?></textarea>
                                </div>

                                <div class="bg-green-50 border border-green-200 rounded-lg p-4">
                                    <div class="flex items-start gap-3">
                                        <span class="material-symbols-outlined text-green-600 mt-0.5">info</span>
                                        <div>
                                            <h4 class="font-medium text-green-800 mb-1">Bilgi</h4>
                                            <ul class="text-sm text-green-700 space-y-1">
                                                <li>• Kullanıcı tekrar normal şekilde siteye erişebilecek</li>
                                                <li>• Yeni site ekleyebilecek ve tüm özellikleri kullanabilecek</li>
                                                <li>• Bu işlem geri alınabilir</li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>

                                <div class="flex gap-4">
                                    <button type="submit" 
                                        class="flex items-center gap-2 bg-green-600 text-white font-bold py-3 px-6 rounded-lg hover:bg-green-700 transition-colors">
                                        <span class="material-symbols-outlined">check_circle</span>
                                        <span>Banı Kaldır</span>
                                    </button>
                                    <a href="users.php" 
                                        class="flex items-center gap-2 bg-gray-100 text-gray-700 font-bold py-3 px-6 rounded-lg hover:bg-gray-200 transition-colors">
                                        <span class="material-symbols-outlined">cancel</span>
                                        <span>İptal</span>
                                    </a>
                                </div>
                            </form>
                        </div>
                    </div>

                    <!-- Sağ Kolon: Ban Geçmişi -->
                    <div>
                        <div class="bg-surface p-6 rounded-xl border border-line-light shadow-soft">
                            <h2 class="text-xl font-bold text-text-primary mb-6 flex items-center gap-2">
                                <span class="material-symbols-outlined text-blue-600">history</span>
                                Ban Geçmişi
                            </h2>

                            <?php if (empty($ban_history)): ?>
                                <p class="text-text-secondary text-center py-8">Henüz ban geçmişi bulunmuyor.</p>
                            <?php else: ?>
                                <div class="space-y-4">
                                    <?php foreach ($ban_history as $record): ?>
                                    <div class="border border-line-light rounded-lg p-4">
                                        <div class="flex items-start justify-between mb-2">
                                            <div class="flex items-center gap-2">
                                                <span class="material-symbols-outlined <?php echo $record['action'] === 'ban' ? 'text-red-600' : 'text-green-600'; ?>">
                                                    <?php echo $record['action'] === 'ban' ? 'block' : 'check_circle'; ?>
                                                </span>
                                                <span class="font-semibold <?php echo $record['action'] === 'ban' ? 'text-red-600' : 'text-green-600'; ?>">
                                                    <?php echo $record['action'] === 'ban' ? 'Banlandı' : 'Ban Kaldırıldı'; ?>
                                                </span>
                                            </div>
                                            <span class="text-sm text-text-secondary">
                                                <?php echo date('d.m.Y H:i', strtotime($record['created_at'])); ?>
                                            </span>
                                        </div>
                                        <p class="text-sm text-text-primary mb-2"><?php echo htmlspecialchars($record['reason']); ?></p>
                                        <p class="text-xs text-text-secondary">
                                            Admin: <?php echo htmlspecialchars($record['admin_name'] ?? 'Bilinmiyor'); ?>
                                        </p>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>
</body>
</html>

