<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
require_once 'admin_auth.php';
require_once __DIR__ . '/../includes/config.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Geçersiz istek yöntemi']);
    exit;
}

// Session kontrolü
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    echo json_encode(['success' => false, 'message' => 'Kimlik doğrulanmadı - Lütfen tekrar giriş yapın']);
    exit;
}

try {
    // Form verilerini al
    $username = trim($_POST['username'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $role = $_POST['role'] ?? 'user';
    $status = $_POST['status'] ?? 'active';
    
    // Debug: Log received data
    error_log("User creation attempt - Username: $username, Email: $email, Role: $role, Status: $status");
    
    // Admin username kısıtlaması
    $restricted_usernames = [
        'admin', 'administrator', 'moderator', 'mod', 'root', 'superadmin', 'super', 'owner', 'founder',
        'ceo', 'manager', 'staff', 'support', 'help', 'system', 'bot', 'api', 'test', 'demo',
        'guest', 'user', 'member', 'visitor', 'anonymous', 'null', 'undefined', 'www', 'mail',
        'ftp', 'news', 'info', 'contact', 'about', 'privacy', 'terms', 'legal', 'security',
        'webmaster', 'hostmaster', 'postmaster', 'noreply', 'no-reply', 'donotreply',
        'sofi', 'sofia', 'sofie', 'sophie', 'sophia', 'sofya', 'sofija', 'sofia', 'sofie',
        'admin123', 'admin1', 'admin2', 'admin3', 'admin4', 'admin5', 'root123', 'root1',
        'moderator123', 'mod123', 'superadmin123', 'super123', 'owner123', 'founder123',
        'ceo123', 'manager123', 'staff123', 'support123', 'help123', 'system123', 'bot123',
        'api123', 'test123', 'demo123', 'guest123', 'user123', 'member123', 'visitor123',
        'anonymous123', 'null123', 'undefined123', 'www123', 'mail123', 'ftp123', 'news123',
        'info123', 'contact123', 'about123', 'privacy123', 'terms123', 'legal123', 'security123',
        'webmaster123', 'hostmaster123', 'postmaster123', 'noreply123', 'no-reply123', 'donotreply123'
    ];
    
    // Validasyon
    if (empty($username)) {
        throw new Exception('Kullanıcı adı gerekli');
    }
    
    $username_lower = strtolower($username);
    if (in_array($username_lower, $restricted_usernames)) {
        throw new Exception('Bu kullanıcı adı kısıtlı ve kullanılamaz');
    }
    
    if (strlen($username) < 3) {
        throw new Exception('Kullanıcı adı en az 3 karakter olmalı');
    }
    
    if (strlen($username) > 20) {
        throw new Exception('Kullanıcı adı en fazla 20 karakter olmalı');
    }
    
    if (!preg_match('/^[a-zA-Z0-9_]+$/', $username)) {
        throw new Exception('Kullanıcı adı sadece harf, rakam ve alt çizgi içerebilir');
    }
    
    if (strpos($username_lower, 'admin') !== false || strpos($username_lower, 'mod') !== false || 
        strpos($username_lower, 'root') !== false || strpos($username_lower, 'super') !== false) {
        throw new Exception('Kullanıcı adı admin, mod, root, super gibi yetki kelimeleri içeremez');
    }
    
    if (empty($email)) {
        throw new Exception('E-posta gerekli');
    }
    
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        throw new Exception('Geçersiz e-posta formatı');
    }
    
    if (empty($password)) {
        throw new Exception('Şifre gerekli');
    }
    
    if (strlen($password) < 6) {
        throw new Exception('Şifre en az 6 karakter olmalı');
    }
    
    if (strlen($password) > 128) {
        throw new Exception('Şifre çok uzun (maksimum 128 karakter)');
    }
    
    if (!in_array($role, ['user', 'admin'])) {
        throw new Exception('Geçersiz rol');
    }
    
    if (!in_array($status, ['active', 'inactive'])) {
        throw new Exception('Geçersiz durum');
    }
    
    // Username ve email unique kontrolü
    $stmt = $pdo->prepare("SELECT id, username, email FROM user WHERE username = ? OR email = ?");
    $stmt->execute([$username, $email]);
    $existing = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($existing) {
        if ($existing['username'] === $username) {
            throw new Exception('Kullanıcı adı zaten mevcut');
        }
        if ($existing['email'] === $email) {
            throw new Exception('E-posta zaten mevcut');
        }
    }
    
    // IP adresini al
    $ip = $_SERVER['REMOTE_ADDR'] ?? '';
    
    // Default avatar
    $default_avatar = "/uploads/no-logo.png";
    
    // Transaction başlat
    $pdo->beginTransaction();
    
    try {
        // Kullanıcıyı oluştur
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        $is_banned = ($status === 'inactive') ? 1 : 0;
        $stmt = $pdo->prepare("
            INSERT INTO user (username, email, password, ip, role, profile_photo, email_verified, is_banned) 
            VALUES (?, ?, ?, ?, ?, ?, 1, ?)
        ");
        
        $result = $stmt->execute([
            $username,
            $email,
            $hashedPassword,
            $ip,
            $role,
            $default_avatar,
            $is_banned
        ]);
        
        if (!$result) {
            throw new Exception('Veritabanında kullanıcı oluşturulamadı');
        }
        
        $userId = $pdo->lastInsertId();
        
        // Transaction'ı commit et
        $pdo->commit();
        
        // Debug: Log successful creation
        error_log("User created successfully - ID: $userId, Username: $username");
        
        echo json_encode([
            'success' => true, 
            'message' => "Kullanıcı '$username' başarıyla oluşturuldu!",
            'user_id' => $userId
        ]);
        
    } catch (Exception $e) {
        $pdo->rollback();
        throw $e;
    }
    
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollback();
    }
    
    // Debug: Log error
    error_log("User creation failed: " . $e->getMessage());
    
    echo json_encode([
        'success' => false, 
        'message' => $e->getMessage()
    ]);
}
?>
