<?php
require_once 'admin_auth.php';
require_once '../includes/config.php';

// CSRF kontrolü sadece POST istekleri için
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    CSRFHelper::checkToken();
}

// Initialize variables
$user = null;
$error = '';
$page_title = 'Kullanıcıyı Sil';

// Get user ID from URL
if (isset($_GET['id'])) {
    $user_id = (int)$_GET['id'];
    
    try {
        // Check if user table exists
        $stmt = $pdo->query("SHOW TABLES LIKE 'user'");
        if ($stmt->rowCount() == 0) {
            $_SESSION['error'] = 'Kullanıcı tablosu bulunamadı. Lütfen yönetici ile iletişime geçin.';
            header('Location: users.php');
            exit;
        }
        
        // Get user details
        $stmt = $pdo->prepare("SELECT id, username, email, role, is_banned FROM user WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            $_SESSION['error'] = 'Kullanıcı bulunamadı!';
            header('Location: users.php');
            exit;
        }
    } catch (Exception $e) {
        $_SESSION['error'] = 'Veritabanı hatası: ' . $e->getMessage();
        header('Location: users.php');
        exit;
    }
    
    // Admin kullanıcılar da silinebilir
} else {
    $_SESSION['error'] = 'Geçersiz kullanıcı ID!';
    header('Location: users.php');
    exit;
}

// Handle deletion
if ($_POST && isset($_POST['confirm_delete']) && $_POST['confirm_delete'] === 'yes') {
    try {
        // Debug: Log the deletion attempt
        error_log("User deletion attempt for ID: " . $user_id);
        
        // Check if user table exists
        $stmt = $pdo->query("SHOW TABLES LIKE 'user'");
        if ($stmt->rowCount() == 0) {
            $_SESSION['error'] = 'Kullanıcı tablosu bulunamadı. Lütfen yönetici ile iletişime geçin.';
            header('Location: users.php');
            exit;
        }
        
        // Start transaction
        $pdo->beginTransaction();
        
        // Delete user votes first (foreign key constraint) - if table exists
        try {
            $stmt = $pdo->query("SHOW TABLES LIKE 'site_votes'");
            if ($stmt->rowCount() > 0) {
                $stmt = $pdo->prepare("DELETE FROM site_votes WHERE user_id = ?");
                $stmt->execute([$user_id]);
                error_log("Deleted votes for user ID: " . $user_id);
            }
        } catch (Exception $e) {
            error_log("Error deleting votes: " . $e->getMessage());
            // Continue with deletion even if votes can't be deleted
        }
        
        // Delete user sites - if table exists
        try {
            $stmt = $pdo->query("SHOW TABLES LIKE 'site_submissions'");
            if ($stmt->rowCount() > 0) {
                $stmt = $pdo->prepare("DELETE FROM site_submissions WHERE user_id = ?");
                $stmt->execute([$user_id]);
                error_log("Deleted sites for user ID: " . $user_id);
            }
        } catch (Exception $e) {
            error_log("Error deleting sites: " . $e->getMessage());
            // Continue with deletion even if sites can't be deleted
        }
        
        // Delete the user
        $stmt = $pdo->prepare("DELETE FROM user WHERE id = ?");
        $result = $stmt->execute([$user_id]);
        
        if ($result && $stmt->rowCount() > 0) {
            // Commit transaction
            $pdo->commit();
            error_log("Successfully deleted user ID: " . $user_id);
            $_SESSION['success'] = "User '{$user['username']}' has been successfully deleted!";
        } else {
            $pdo->rollback();
            error_log("Failed to delete user ID: " . $user_id . " - No rows affected");
            $_SESSION['error'] = "Failed to delete user. User may not exist or may have already been deleted.";
        }
        
        header('Location: users.php');
        exit;
        
    } catch (PDOException $e) {
        if (isset($pdo) && $pdo->inTransaction()) {
            $pdo->rollback();
        }
        $error = 'Error deleting user: ' . $e->getMessage();
    } catch (Exception $e) {
        if (isset($pdo) && $pdo->inTransaction()) {
            $pdo->rollback();
        }
        $error = 'Error deleting user: ' . $e->getMessage();
    }
}
// Include meta head function
include '../includes/meta_head.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - Admin Panel</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined" rel="stylesheet">
    <?php echo generateMetaHead($page_title, "Delete user {$user['username']} from the admin panel"); ?>
    <style>
        .shake {
            animation: shake 0.5s;
        }
        @keyframes shake {
            0%, 100% { transform: translateX(0); }
            25% { transform: translateX(-5px); }
            75% { transform: translateX(5px); }
        }
        .pulse-border {
            animation: pulse-border 2s infinite;
        }
        @keyframes pulse-border {
            0%, 100% { border-color: #ef4444; }
            50% { border-color: #dc2626; }
        }
        .main-content {
            /* Centered content */
        }
        .content-wrapper {
            /* Centered content */
        }
    </style>
</head>
<?php include 'menu.php'; ?>

<div class="flex-1 ml-80 flex items-center justify-center min-h-screen">
    <main class="p-4 main-content w-full">
        <div class="max-w-4xl mx-auto content-wrapper">
            <!-- Header -->
            <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-4">
                <div class="flex items-center justify-between">
                    <div class="flex items-center">
                        <span class="material-symbols-outlined text-red-500 text-3xl mr-3">person_remove</span>
                        <div>
                            <h1 class="text-2xl font-bold text-gray-900">Delete User</h1>
                            <p class="text-gray-600">Permanently remove user from the system</p>
                        </div>
                    </div>
                    <a href="users.php" class="inline-flex items-center px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition-colors">
                        <span class="material-symbols-outlined mr-2">arrow_back</span>
                        Back to Users
                    </a>
                </div>
            </div>

            <?php if ($error): ?>
            <div class="bg-red-50 border border-red-200 rounded-lg p-4 mb-6 shake">
                <div class="flex items-center">
                    <span class="material-symbols-outlined text-red-500 mr-2">error</span>
                    <p class="text-red-800"><?php echo htmlspecialchars($error); ?></p>
                </div>
            </div>
            <?php endif; ?>

            <!-- User Details and Warning in Grid -->
            <div class="grid grid-cols-1 lg:grid-cols-2 gap-4 mb-4">
                <!-- User Details Card -->
                <div class="bg-white rounded-lg shadow-sm border border-red-200 pulse-border p-4">
                    <div class="flex items-center mb-3">
                        <span class="material-symbols-outlined text-orange-500 text-xl mr-2">warning</span>
                        <h2 class="text-lg font-semibold text-gray-900">User Information</h2>
                    </div>
                    
                    <div class="space-y-3">
                        <div class="flex items-center">
                            <span class="material-symbols-outlined text-gray-400 mr-2 text-sm">person</span>
                            <div>
                                <p class="text-xs text-gray-500">Username</p>
                                <p class="font-semibold text-gray-900 text-sm"><?php echo htmlspecialchars($user['username']); ?></p>
                            </div>
                        </div>
                        
                        <div class="flex items-center">
                            <span class="material-symbols-outlined text-gray-400 mr-2 text-sm">email</span>
                            <div>
                                <p class="text-xs text-gray-500">Email</p>
                                <p class="font-semibold text-gray-900 text-sm"><?php echo htmlspecialchars($user['email']); ?></p>
                            </div>
                        </div>
                        
                        <div class="flex items-center">
                            <span class="material-symbols-outlined text-gray-400 mr-2 text-sm">admin_panel_settings</span>
                            <div>
                                <p class="text-xs text-gray-500">Role</p>
                                <span class="px-2 py-1 text-xs font-semibold rounded-full <?php echo $user['role'] === 'admin' ? 'bg-purple-100 text-purple-800' : 'bg-blue-100 text-blue-800'; ?>">
                                    <?php echo ucfirst($user['role']); ?>
                                </span>
                            </div>
                        </div>
                        
                        <div class="flex items-center">
                            <span class="material-symbols-outlined text-gray-400 mr-2 text-sm">toggle_on</span>
                            <div>
                                <p class="text-xs text-gray-500">Status</p>
                                <span class="px-2 py-1 text-xs font-semibold rounded-full <?php echo $user['is_banned'] == 0 ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>">
                                    <?php echo $user['is_banned'] == 0 ? 'Active' : 'Banned'; ?>
                                </span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Warning Card -->
                <div class="bg-red-50 border border-red-200 rounded-lg p-4">
                    <div class="flex items-start">
                        <span class="material-symbols-outlined text-red-500 text-xl mr-2 flex-shrink-0">dangerous</span>
                        <div>
                            <h3 class="text-lg font-semibold text-red-900 mb-2">⚠️ Warning</h3>
                            <div class="text-red-800 space-y-1 text-sm">
                                <p class="font-medium">Deleting this user will:</p>
                                <ul class="list-disc list-inside ml-3 space-y-1">
                                    <li>Permanently remove the user account</li>
                                    <li>Delete all sites submitted by this user</li>
                                    <li>Remove all votes cast by this user</li>
                                    <li><strong>Cannot be undone</strong></li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Confirmation Form -->
            <form method="POST" action="" class="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
                <?php echo CSRFHelper::getTokenField(); ?>
                <h3 class="text-lg font-semibold text-gray-900 mb-4 flex items-center">
                    <span class="material-symbols-outlined text-red-500 mr-2">check_circle</span>
                    Confirm Deletion
                </h3>
                
                <div class="mb-6">
                    <label class="flex items-start space-x-3 cursor-pointer">
                        <input type="checkbox" id="confirm-checkbox" class="mt-1 rounded border-gray-300 text-red-600 focus:ring-red-500" required>
                        <span class="text-gray-700">
                            I understand that deleting user <strong>"<?php echo htmlspecialchars($user['username']); ?>"</strong> 
                            will permanently remove their account and all associated data. This action cannot be undone.
                        </span>
                    </label>
                </div>
                
                <div class="flex gap-4">
                    <a href="users.php" class="flex-1 inline-flex items-center justify-center px-6 py-3 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition-colors">
                        <span class="material-symbols-outlined mr-2">cancel</span>
                        Cancel
                    </a>
                    <button type="submit" name="confirm_delete" value="yes" id="delete-btn" 
                            class="flex-1 inline-flex items-center justify-center px-6 py-3 bg-red-600 text-white rounded-lg hover:bg-red-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                            disabled>
                        <span class="material-symbols-outlined mr-2">delete_forever</span>
                        Delete User
                    </button>
                </div>
            </form>
        </div>
    </main>
</div>

<script>
// Enable delete button only when checkbox is checked
document.getElementById('confirm-checkbox').addEventListener('change', function() {
    document.getElementById('delete-btn').disabled = !this.checked;
});

// Add extra confirmation on form submit
document.querySelector('form').addEventListener('submit', function(e) {
    if (!confirm(`Are you absolutely sure you want to delete user "${<?php echo json_encode($user['username']); ?>}"? This action cannot be undone!`)) {
        e.preventDefault();
    }
});
</script>

</body>
</html>

