<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
include '../includes/config.php';

// Admin kontrolü
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: login.php');
    exit();
}

// Kullanıcı ID kontrolü
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header('Location: users.php');
    exit();
}

$user_id = (int)$_GET['id'];

// Form gönderildi mi kontrol et
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username']);
    $email = trim($_POST['email']);
    $role = $_POST['role'];
    
    $errors = [];
    
    // Validasyon
    if (empty($username)) {
        $errors[] = "Kullanıcı adı gerekli.";
    } else {
        // Username başka kullanıcıda var mı kontrol et
        $stmt = $pdo->prepare("SELECT id FROM user WHERE username = ? AND id != ?");
        $stmt->execute([$username, $user_id]);
        if ($stmt->fetch()) {
            $errors[] = "Kullanıcı adı zaten mevcut.";
        }
    }
    
    if (empty($email)) {
        $errors[] = "E-posta gerekli.";
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = "Geçersiz e-posta formatı.";
    } else {
        // Email başka kullanıcıda var mı kontrol et
        $stmt = $pdo->prepare("SELECT id FROM user WHERE email = ? AND id != ?");
        $stmt->execute([$email, $user_id]);
        if ($stmt->fetch()) {
            $errors[] = "E-posta zaten mevcut.";
        }
    }
    
    if (!in_array($role, ['admin', 'kullanici'])) {
        $errors[] = "Geçersiz rol seçildi.";
    }
    
    // Şifre değiştirilecek mi?
    $password_update = "";
    if (!empty($_POST['password'])) {
        if (strlen($_POST['password']) < 6) {
            $errors[] = "Şifre en az 6 karakter olmalı.";
        } else {
            $password_update = ", password = ?";
            $password_hash = password_hash($_POST['password'], PASSWORD_DEFAULT);
        }
    }
    
    // Profil fotoğrafı yüklendi mi?
    $profile_photo_update = "";
    if (isset($_FILES['profile_photo']) && $_FILES['profile_photo']['error'] === UPLOAD_ERR_OK) {
        $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
        $file_type = $_FILES['profile_photo']['type'];
        
        if (!in_array($file_type, $allowed_types)) {
            $errors[] = "Geçersiz dosya türü. Sadece JPEG, PNG ve GIF dosyalarına izin verilir.";
        } elseif ($_FILES['profile_photo']['size'] > 5 * 1024 * 1024) { // 5MB limit
            $errors[] = "Dosya boyutu çok büyük. Maksimum 5MB izin verilir.";
        } else {
            $upload_dir = '../uploads/';
            $file_extension = pathinfo($_FILES['profile_photo']['name'], PATHINFO_EXTENSION);
            $new_filename = 'profile_' . $user_id . '_' . time() . '.' . $file_extension;
            $upload_path = $upload_dir . $new_filename;
            
            if (move_uploaded_file($_FILES['profile_photo']['tmp_name'], $upload_path)) {
                $profile_photo_update = ", profile_photo = ?";
                $profile_photo_name = $new_filename;
            } else {
                $errors[] = "Profil fotoğrafı yüklenemedi.";
            }
        }
    }
    
    // Hata yoksa güncelle
    if (empty($errors)) {
        $sql = "UPDATE user SET username = ?, email = ?, role = ?" . $password_update . $profile_photo_update . " WHERE id = ?";
        $params = [$username, $email, $role];
        
        if (!empty($_POST['password'])) {
            $params[] = $password_hash;
        }
        
        if (!empty($profile_photo_name)) {
            $params[] = $profile_photo_name;
        }
        
        $params[] = $user_id;
        
        $stmt = $pdo->prepare($sql);
        if ($stmt->execute($params)) {
            header('Location: users.php?success=User updated successfully');
            exit();
        } else {
            $errors[] = "Failed to update user.";
        }
    }
}

// Kullanıcı bilgilerini çek
$stmt = $pdo->prepare("SELECT * FROM user WHERE id = ?");
$stmt->execute([$user_id]);
$user = $stmt->fetch();

if (!$user) {
    header('Location: users.php?error=User not found');
    exit();
}
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kullanıcı Düzenle - Admin Panel</title>
    <link href="https://fonts.googleapis.com" rel="preconnect"/>
    <link crossorigin="" href="https://fonts.gstatic.com" rel="preconnect"/>
    <link href="https://fonts.googleapis.com/css2?family=Manrope:wght@400;500;600;700;800&amp;display=swap" rel="stylesheet"/>
    <script src="https://cdn.tailwindcss.com?plugins=forms,container-queries"></script>
    <link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined:opsz,wght,FILL,GRAD@20..48,100..700,0..1,-50..200" rel="stylesheet"/>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: "#1877f2",
                        "primary-light": "#e7f3ff",
                        background: "#f0f2f5",
                        surface: "#ffffff",
                        "text-primary": "#1c1e21",
                        "text-secondary": "#606770",
                        "line-light": "#e0e0e0",
                    },
                    fontFamily: {
                        display: ["Manrope", "sans-serif"],
                    },
                    borderRadius: {
                        DEFAULT: "0.75rem",
                        lg: "1rem",
                        xl: "1.5rem",
                        full: "9999px"
                    },
                    boxShadow: {
                        'soft': '0 4px 12px 0 rgb(0 0 0 / 0.07)',
                        'subtle': '0 1px 2px 0 rgb(0 0 0 / 0.05)',
                    }
                },
            },
        };
    </script>
    <style>
        .material-symbols-outlined {
            font-variation-settings: "FILL" 0, "wght" 400, "GRAD" 0, "opsz" 24;
        }
        .active .material-symbols-outlined {
            font-variation-settings: 'FILL' 1;
        }
    </style>
</head>
<body class="bg-background font-display text-text-primary">
<div class="flex min-h-screen">
<?php include 'menu.php'; ?>
<main class="flex-1 p-10">
    <div class="max-w-2xl mx-auto">
            <!-- Header -->
            <div class="flex items-center justify-between mb-6">
                <div>
                    <h1 class="text-2xl font-bold text-gray-800">Kullanıcı Düzenle</h1>
                    <p class="text-gray-600">Kullanıcı bilgilerini güncelleyin</p>
                </div>
                <a href="users.php" class="bg-gray-500 text-white px-4 py-2 rounded-lg hover:bg-gray-600 flex items-center gap-2">
                    <span class="material-symbols-outlined">arrow_back</span>
                    Kullanıcılara Dön
                </a>
            </div>

            <!-- Error Messages -->
            <?php if (!empty($errors)): ?>
            <div class="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
                <h3 class="text-red-800 font-medium mb-2">Lütfen aşağıdaki hataları düzeltin:</h3>
                <ul class="text-red-700 text-sm space-y-1">
                    <?php foreach ($errors as $error): ?>
                    <li>• <?php echo htmlspecialchars($error); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endif; ?>

            <!-- Edit Form -->
            <div class="bg-white rounded-lg shadow-sm border">
                <div class="p-6 border-b">
                    <h2 class="text-xl font-semibold text-gray-800">Kullanıcı Bilgileri</h2>
                </div>
                
                <form method="POST" enctype="multipart/form-data" class="p-6 space-y-6">
                    <!-- Current Profile Photo -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Mevcut Profil Fotoğrafı</label>
                        <div class="flex items-center gap-4">
                            <?php if (!empty($user['profile_photo'])): ?>
                                <?php
                                // Profil fotoğrafı yolunu düzenle
                                $photo_path = $user['profile_photo'];
                                // Eğer yol uploads/ ile başlamıyorsa, ../uploads/ ekle
                                if (!str_starts_with($photo_path, 'uploads/') && !str_starts_with($photo_path, '../uploads/')) {
                                    $photo_path = '../uploads/' . $photo_path;
                                }
                                // Eğer /admin ile başlıyorsa düzelt
                                if (str_starts_with($photo_path, '/admin/')) {
                                    $photo_path = str_replace('/admin/', '../', $photo_path);
                                }
                                // Eğer uploads/uploads/ gibi çift yol varsa düzelt
                                $photo_path = str_replace('uploads/uploads/', 'uploads/', $photo_path);
                                $photo_path = str_replace('../uploads/uploads/', '../uploads/', $photo_path);
                                ?>
                                <img src="<?php echo htmlspecialchars($photo_path); ?>" 
                                     alt="Profile Photo" class="w-16 h-16 rounded-full object-cover border-2 border-gray-200">
                                <span class="text-sm text-gray-600">Mevcut fotoğraf</span>
                            <?php else: ?>
                                <div class="w-16 h-16 rounded-full bg-gray-300 flex items-center justify-center border-2 border-gray-200">
                                    <span class="material-symbols-outlined text-gray-500">person</span>
                                </div>
                                <span class="text-sm text-gray-600">Fotoğraf yüklenmedi</span>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Profile Photo Upload -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Yeni Profil Fotoğrafı (İsteğe Bağlı)</label>
                        <input type="file" name="profile_photo" accept="image/*" 
                               class="block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-medium file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100">
                        <p class="text-xs text-gray-500 mt-1">Maksimum dosya boyutu: 5MB. Desteklenen formatlar: JPEG, PNG, GIF</p>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Username -->
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Kullanıcı Adı *</label>
                            <input type="text" name="username" required 
                                   value="<?php echo htmlspecialchars($user['username']); ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                        </div>

                        <!-- Email -->
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">E-posta *</label>
                            <input type="email" name="email" required 
                                   value="<?php echo htmlspecialchars($user['email']); ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                        </div>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Role -->
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Rol *</label>
                            <select name="role" required 
                                    class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                                <option value="kullanici" <?php echo $user['role'] === 'kullanici' ? 'selected' : ''; ?>>Kullanıcı</option>
                                <option value="admin" <?php echo $user['role'] === 'admin' ? 'selected' : ''; ?>>Admin</option>
                            </select>
                        </div>
                    </div>

                    <!-- Password -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Yeni Şifre (Mevcut şifreyi korumak için boş bırakın)</label>
                        <input type="password" name="password" 
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                        <p class="text-xs text-gray-500 mt-1">Minimum 6 karakter</p>
                    </div>

                    <!-- User Info -->
                    <div class="bg-gray-50 rounded-lg p-4">
                        <h3 class="font-medium text-gray-800 mb-2">Ek Bilgiler</h3>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                            <div>
                                <span class="text-gray-600">Kullanıcı ID:</span>
                                <span class="text-gray-800 font-medium"><?php echo $user['id']; ?></span>
                            </div>
                            <div>
                                <span class="text-gray-600">Kayıt Tarihi:</span>
                                <span class="text-gray-800 font-medium"><?php echo date('d.m.Y H:i', strtotime($user['kayit_tarihi'])); ?></span>
                            </div>
                            <div>
                                <span class="text-gray-600">IP Adresi:</span>
                                <span class="text-gray-800 font-medium"><?php echo htmlspecialchars($user['ip'] ?? 'N/A'); ?></span>
                            </div>
                        </div>
                    </div>

                    <!-- Buttons -->
                    <div class="flex gap-4 pt-4">
                        <button type="submit" 
                                class="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 flex items-center gap-2">
                            <span class="material-symbols-outlined">save</span>
                            Kullanıcıyı Güncelle
                        </button>
                        <a href="user_view.php?id=<?php echo $user['id']; ?>" 
                           class="bg-gray-500 text-white px-6 py-2 rounded-lg hover:bg-gray-600 flex items-center gap-2">
                            <span class="material-symbols-outlined">visibility</span>
                            Kullanıcıyı Görüntüle
                        </a>
                    </div>
                </form>
            </div>
        </div>
    </div>
</main>
</div>
</body>
</html>

