<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
require_once 'admin_auth.php';
require_once __DIR__ . '/../includes/config.php';

// Toplu silme işlemi
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_delete'])) {
    $user_ids = $_POST['user_ids'] ?? [];
    
    if (!empty($user_ids) && is_array($user_ids)) {
        try {
            // Güvenlik için sadece sayısal ID'leri al
            $user_ids = array_filter($user_ids, 'is_numeric');
            $user_ids = array_map('intval', $user_ids);
            
            if (!empty($user_ids)) {
                // Admin kullanıcılar da silinebilir
                // Placeholder'ları oluştur
                $placeholders = str_repeat('?,', count($user_ids) - 1) . '?';
                
                // Transaction başlat
                $pdo->beginTransaction();
                
                try {
                    // Önce site_votes tablosundaki ilişkili kayıtları sil
                    $sql_votes = "DELETE FROM site_votes WHERE user_id IN ($placeholders)";
                    $stmt_votes = $pdo->prepare($sql_votes);
                    $stmt_votes->execute($user_ids);
                    
                    // Sonra site_submissions tablosundaki ilişkili kayıtları sil
                    $sql_sites = "DELETE FROM site_submissions WHERE user_id IN ($placeholders)";
                    $stmt_sites = $pdo->prepare($sql_sites);
                    $stmt_sites->execute($user_ids);
                    
                    // Son olarak user tablosundaki kayıtları sil
                    $sql_users = "DELETE FROM user WHERE id IN ($placeholders)";
                    $stmt_users = $pdo->prepare($sql_users);
                    $stmt_users->execute($user_ids);
                    
                    // Transaction'ı commit et
                    $pdo->commit();
                    
                    $deleted_count = $stmt_users->rowCount();
                    $_SESSION['bulk_delete_success'] = $deleted_count;
                    
                } catch (PDOException $e) {
                    // Hata durumunda transaction'ı rollback et
                    $pdo->rollback();
                    throw $e;
                }
            }
        } catch (PDOException $e) {
            $_SESSION['bulk_delete_error'] = 'Veritabanı hatası: ' . $e->getMessage();
        }
    } else {
        $_SESSION['bulk_delete_error'] = 'Silme için kullanıcı seçilmedi.';
    }
    
    // Yeniden gönderimi önlemek için yönlendir
    header('Location: users.php');
    exit;
}

$perPage = isset($_GET['perPage']) ? intval($_GET['perPage']) : 10;
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$offset = ($page - 1) * $perPage;

$whereArr = [];
$params = [];

// Arama filtresi
if (isset($_GET['search']) && $_GET['search'] !== '') {
    $whereArr[] = '(username LIKE :search OR email LIKE :search)';
    $params[':search'] = '%' . $_GET['search'] . '%';
}

// Durum filtresi - status kolonu olmadığı için role kullanıyoruz
if (isset($_GET['status']) && $_GET['status'] !== '') {
    if ($_GET['status'] === 'admin') {
        $whereArr[] = 'role = :status';
        $params[':status'] = 'admin';
    } elseif ($_GET['status'] === 'user') {
        $whereArr[] = 'role = :status';
        $params[':status'] = 'kullanici';
    }
}

$whereSQL = count($whereArr) ? ('WHERE ' . implode(' AND ', $whereArr)) : '';

// Sıralama
$orderSQL = 'ORDER BY kayit_tarihi DESC';
if (isset($_GET['sort'])) {
    switch ($_GET['sort']) {
        case 'kayit_tarihi_asc':
            $orderSQL = 'ORDER BY kayit_tarihi ASC';
            break;
        case 'username_asc':
            $orderSQL = 'ORDER BY username ASC';
            break;
        case 'username_desc':
            $orderSQL = 'ORDER BY username DESC';
            break;
        default:
            $orderSQL = 'ORDER BY kayit_tarihi DESC';
    }
}

// Toplam sayısını al
$countSQL = "SELECT COUNT(*) FROM user $whereSQL";
$countStmt = $pdo->prepare($countSQL);
$countStmt->execute($params);
$totalUsers = $countStmt->fetchColumn();

// Sayfalama hesaplamaları
$totalPages = ceil($totalUsers / $perPage);

// Kullanıcıları al
$sql = "SELECT id, username, email, role, kayit_tarihi FROM user $whereSQL $orderSQL LIMIT :limit OFFSET :offset";
$stmt = $pdo->prepare($sql);

foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value);
}
$stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();
$users = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="en"><head>
<meta charset="utf-8"/>
<meta content="width=device-width, initial-scale=1.0" name="viewport"/>
<title>User Management</title>
<link href="https://fonts.googleapis.com" rel="preconnect"/>
<link crossorigin="" href="https://fonts.gstatic.com" rel="preconnect"/>
<link href="https://fonts.googleapis.com/css2?family=Manrope:wght@400;500;600;700;800&amp;display=swap" rel="stylesheet"/>
<script src="https://cdn.tailwindcss.com?plugins=forms,container-queries"></script>
<link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined:opsz,wght,FILL,GRAD@20..48,100..700,0..1,-50..200" rel="stylesheet"/>
<script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: "#1877f2", // A modern, friendly blue
                        "primary-light": "#e7f3ff",
                        background: "#f0f2f5",
                        surface: "#ffffff",
                        "text-primary": "#1c1e21",
                        "text-secondary": "#606770",
                        "line-light": "#e0e0e0",
                    },
                    fontFamily: {
                        display: ["Manrope", "sans-serif"],
                    },
                    borderRadius: {
                        DEFAULT: "0.75rem",
                        lg: "1rem",
                        xl: "1.5rem",
                        full: "9999px"
                    },
                    boxShadow: {
                        'soft': '0 4px 12px 0 rgb(0 0 0 / 0.07)',
                        'subtle': '0 1px 2px 0 rgb(0 0 0 / 0.05)',
                    }
                },
            },
        };
    </script>
<style>
        .material-symbols-outlined {
            font-variation-settings: "FILL" 0, "wght" 400, "GRAD" 0, "opsz" 24;
        }
        .active .material-symbols-outlined {
            font-variation-settings: 'FILL' 1;
        }
    </style>
</head>
<body class="bg-background font-display text-text-primary">
<div class="flex min-h-screen">
<?php include 'menu.php'; ?>
<main class="flex-1 p-10 min-w-0">
<?php
$page_title = "Users";
$page_subtitle = "Manage all users in the directory.";
include 'includes/admin_header.php';
?>

<!-- Action Buttons -->
<div class="flex gap-3 mb-8">
    <button onclick="openNewUserModal()" class="flex items-center gap-2 bg-primary text-white font-bold py-3 px-5 rounded-lg hover:bg-primary/90 transition-colors">
        <span class="material-symbols-outlined">add</span>
        <span>New User</span>
    </button>
</div>
<div class="bg-surface p-6 rounded-xl border border-line-light shadow-soft">
<form method="GET" class="flex justify-between items-center mb-6">
<div class="relative w-full max-w-md">
<span class="material-symbols-outlined absolute left-4 top-1/2 -translate-y-1/2 text-text-secondary">search</span>
<input name="search" value="<?php echo htmlspecialchars($_GET['search'] ?? ''); ?>" class="w-full bg-background border-transparent rounded-lg pl-12 pr-4 py-3 focus:ring-2 focus:ring-primary focus:border-transparent placeholder:text-text-secondary" placeholder="Kullanıcıları isim veya email ile ara..." type="text"/>
</div>
<div class="flex items-center gap-4">
<button type="submit" class="flex items-center gap-2 text-text-secondary font-medium hover:text-text-primary">
<span class="material-symbols-outlined">search</span>
<span>Ara</span>
</button>
<select name="status" class="px-4 py-2 bg-background rounded-lg text-text-secondary font-medium hover:bg-line-light" onchange="this.form.submit()">
<option value="">Tüm Kullanıcılar</option>
<option value="admin" <?php echo ($_GET['status'] ?? '') == 'admin' ? 'selected' : ''; ?>>Adminler</option>
<option value="user" <?php echo ($_GET['status'] ?? '') == 'user' ? 'selected' : ''; ?>>Kullanıcılar</option>
</select>
<select name="sort" class="px-4 py-2 bg-background rounded-lg text-text-secondary font-medium hover:bg-line-light" onchange="this.form.submit()">
<option value="kayit_tarihi_desc" <?php echo ($_GET['sort'] ?? '') == 'kayit_tarihi_desc' ? 'selected' : ''; ?>>En Yeni Önce</option>
<option value="kayit_tarihi_asc" <?php echo ($_GET['sort'] ?? '') == 'kayit_tarihi_asc' ? 'selected' : ''; ?>>En Eski Önce</option>
<option value="username_asc" <?php echo ($_GET['sort'] ?? '') == 'username_asc' ? 'selected' : ''; ?>>İsim A-Z</option>
<option value="username_desc" <?php echo ($_GET['sort'] ?? '') == 'username_desc' ? 'selected' : ''; ?>>İsim Z-A</option>
</select>
<select name="perPage" class="px-4 py-2 bg-background rounded-lg text-text-secondary font-medium hover:bg-line-light" onchange="this.form.submit()">
<?php $perPageOptions = [5,10,20,50,100];
$selectedPerPage = isset($_GET['perPage']) ? intval($_GET['perPage']) : 10;
foreach($perPageOptions as $opt) {
    echo '<option value="'.$opt.'"'.($selectedPerPage==$opt?' selected':'').'>'.$opt.' / sayfa</option>';
}
?>
</select>
</div>
</form>

<!-- Success/Error Messages -->
<?php if (isset($_GET['success'])): ?>
    <?php if ($_GET['success'] === 'user_banned'): ?>
        <div class="bg-red-50 border-l-4 border-red-400 p-4 mb-6">
            <div class="flex items-center">
                <span class="material-symbols-outlined text-red-400 mr-2">block</span>
                <p class="text-red-700">Kullanıcı başarıyla banlandı.</p>
            </div>
        </div>
    <?php elseif ($_GET['success'] === 'user_unbanned'): ?>
        <div class="bg-green-50 border-l-4 border-green-400 p-4 mb-6">
            <div class="flex items-center">
                <span class="material-symbols-outlined text-green-400 mr-2">check_circle</span>
                <p class="text-green-700">Kullanıcının banı başarıyla kaldırıldı.</p>
            </div>
        </div>
    <?php endif; ?>
<?php endif; ?>

<?php if (isset($_GET['error'])): ?>
    <?php if ($_GET['error'] === 'cannot_ban_admin'): ?>
        <div class="bg-yellow-50 border-l-4 border-yellow-400 p-4 mb-6">
            <div class="flex items-center">
                <span class="material-symbols-outlined text-yellow-400 mr-2">warning</span>
                <p class="text-yellow-700">Admin kullanıcıları banlanamaz.</p>
            </div>
        </div>
    <?php elseif ($_GET['error'] === 'user_not_found'): ?>
        <div class="bg-red-50 border-l-4 border-red-400 p-4 mb-6">
            <div class="flex items-center">
                <span class="material-symbols-outlined text-red-400 mr-2">error</span>
                <p class="text-red-700">Kullanıcı bulunamadı.</p>
            </div>
        </div>
    <?php endif; ?>
<?php endif; ?>

<!-- Flash Messages -->
<?php if (isset($_SESSION['bulk_delete_success'])): ?>
<div class="mb-6 p-4 bg-green-100 border border-green-200 rounded-lg flex items-center">
    <span class="material-symbols-outlined text-green-600 mr-3">check_circle</span>
    <div>
        <p class="text-green-800 font-medium">Toplu silme başarılı!</p>
        <p class="text-green-700 text-sm"><?php echo $_SESSION['bulk_delete_success']; ?> kullanıcı silindi.</p>
    </div>
</div>
<?php 
unset($_SESSION['bulk_delete_success']);
endif; 
?>

<?php if (isset($_SESSION['bulk_delete_error'])): ?>
<div class="mb-6 p-4 bg-red-100 border border-red-200 rounded-lg flex items-center">
    <span class="material-symbols-outlined text-red-600 mr-3">error</span>
    <div>
        <p class="text-red-800 font-medium">Toplu silme başarısız!</p>
        <p class="text-red-700 text-sm"><?php echo htmlspecialchars($_SESSION['bulk_delete_error']); ?></p>
    </div>
</div>
<?php 
unset($_SESSION['bulk_delete_error']);
endif; 
?>

<!-- General Success Messages -->
<?php if (isset($_SESSION['success'])): ?>
<div class="mb-6 p-4 bg-green-100 border border-green-200 rounded-lg flex items-center">
    <span class="material-symbols-outlined text-green-600 mr-3">check_circle</span>
    <div>
        <p class="text-green-800 font-medium">Başarılı!</p>
        <p class="text-green-700 text-sm"><?php echo htmlspecialchars($_SESSION['success']); ?></p>
    </div>
</div>
<?php 
unset($_SESSION['success']);
endif; 
?>

<!-- General Error Messages -->
<?php if (isset($_SESSION['error'])): ?>
<div class="mb-6 p-4 bg-red-100 border border-red-200 rounded-lg flex items-center">
    <span class="material-symbols-outlined text-red-600 mr-3">error</span>
    <div>
        <p class="text-red-800 font-medium">Hata!</p>
        <p class="text-red-700 text-sm"><?php echo htmlspecialchars($_SESSION['error']); ?></p>
    </div>
</div>
<?php 
unset($_SESSION['error']);
endif; 
?>

<!-- Bulk Actions -->
<div class="mb-4 flex items-center justify-between">
    <div class="flex items-center gap-4">
        <button id="bulk-delete-btn" onclick="bulkDelete()" disabled 
                class="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors disabled:bg-gray-300 disabled:cursor-not-allowed flex items-center gap-2">
            <span class="material-symbols-outlined text-sm">delete</span>
            Seçilenleri Sil
        </button>
        <span id="selected-count" class="text-sm text-gray-600">0 kullanıcı seçildi</span>
    </div>
</div>
<?php
require_once __DIR__ . '/../includes/config.php';

$perPage = isset($_GET['perPage']) ? intval($_GET['perPage']) : 10;
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$offset = ($page - 1) * $perPage;

$whereArr = [];
$params = [];

// Arama filtresi
if (isset($_GET['search']) && $_GET['search'] !== '') {
    $whereArr[] = '(username LIKE :search OR email LIKE :search)';
    $params[':search'] = '%' . $_GET['search'] . '%';
}

// Durum filtresi - status kolonu olmadığı için role kullanıyoruz
if (isset($_GET['status']) && $_GET['status'] !== '') {
    if ($_GET['status'] === 'admin') {
        $whereArr[] = 'role = :status';
        $params[':status'] = 'admin';
    } elseif ($_GET['status'] === 'user') {
        $whereArr[] = 'role = :status';
        $params[':status'] = 'kullanici';
    }
}

$whereSQL = count($whereArr) ? ('WHERE ' . implode(' AND ', $whereArr)) : '';

// Sıralama
$orderSQL = 'ORDER BY kayit_tarihi DESC';
if (isset($_GET['sort'])) {
    switch($_GET['sort']) {
        case 'kayit_tarihi_asc':
            $orderSQL = 'ORDER BY kayit_tarihi ASC';
            break;
        case 'username_asc':
            $orderSQL = 'ORDER BY username ASC';
            break;
        case 'username_desc':
            $orderSQL = 'ORDER BY username DESC';
            break;
        default:
            $orderSQL = 'ORDER BY kayit_tarihi DESC';
    }
}

// Toplam kullanıcı sayısı
$totalStmt = $pdo->prepare('SELECT COUNT(*) FROM user ' . $whereSQL);
$totalStmt->execute($params);
$totalRows = $totalStmt->fetchColumn();
$totalPages = ceil($totalRows / $perPage);

// Kullanıcıları çek - ip adresi ve ban durumu eklendi
$sql = 'SELECT id, username, email, kayit_tarihi, profile_photo, role, ip, is_banned FROM user ' . $whereSQL . ' ' . $orderSQL . ' LIMIT :limit OFFSET :offset';
$stmt = $pdo->prepare($sql);
foreach ($params as $k => $v) $stmt->bindValue($k, $v);
$stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();
$users = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Her kullanıcı için eklediği site sayısını çek
$userIds = array_column($users, 'id');
$siteCounts = [];
if (!empty($userIds)) {
    $siteCountStmt = $pdo->prepare('SELECT user_id, COUNT(*) as site_count FROM site_submissions WHERE user_id IN (' . implode(',', array_fill(0, count($userIds), '?')) . ') GROUP BY user_id');
    $siteCountStmt->execute($userIds);
    $siteCounts = array_column($siteCountStmt->fetchAll(PDO::FETCH_ASSOC), 'site_count', 'user_id');
}

function getStatusBadge($status) {
    switch(strtolower($status)) {
        case 'active':
            return '<span class="px-3 py-1 text-sm font-medium rounded-full bg-green-100 text-green-700">Aktif</span>';
        case 'pending':
            return '<span class="px-3 py-1 text-sm font-medium rounded-full bg-yellow-100 text-yellow-700">Beklemede</span>';
        case 'banned':
            return '<span class="px-3 py-1 text-sm font-medium rounded-full bg-red-100 text-red-700">Banlı</span>';
        default:
            return '<span class="px-3 py-1 text-sm font-medium rounded-full bg-gray-100 text-gray-700">' . htmlspecialchars(ucfirst($status)) . '</span>';
    }
}
?>

<?php if (empty($users)): ?>
<div class="bg-white rounded-xl shadow-soft border border-line-light p-12 text-center">
    <div class="w-20 h-20 rounded-full bg-background mx-auto mb-4 flex items-center justify-center">
        <span class="material-symbols-outlined text-4xl text-text-secondary">person_off</span>
    </div>
    <h3 class="text-xl font-bold text-text-primary mb-2">Kullanıcı Bulunamadı</h3>
    <p class="text-text-secondary mb-6">Arama kriterlerinize uygun kullanıcı bulunamadı. Filtrelerinizi ayarlamayı deneyin.</p>
    <a href="?" class="inline-flex items-center gap-2 bg-primary text-white font-bold py-3 px-6 rounded-lg hover:bg-primary/90 transition-colors">
        <span class="material-symbols-outlined">refresh</span>
        <span>Filtreleri Sıfırla</span>
    </a>
</div>
<?php else: ?>

<div class="overflow-x-auto bg-white rounded-xl shadow-soft border border-line-light">
<table class="w-full text-left table-fixed min-w-[1200px]">
<thead class="border-b border-line-light">
<tr class="bg-gradient-to-r from-primary/5 to-primary/10 border-b-2 border-primary/20">
<th class="p-6 font-bold text-text-primary text-lg w-16">
    <input type="checkbox" id="select-all" class="w-4 h-4 text-primary bg-gray-100 border-gray-300 rounded focus:ring-primary focus:ring-2" onchange="toggleSelectAll()">
</th>
<th class="p-6 font-bold text-text-primary text-lg w-64">
    <div class="flex items-center gap-3">
        <span class="material-symbols-outlined text-primary">person</span>
        Kullanıcı
    </div>
</th>
<th class="p-6 font-bold text-text-primary text-lg w-32">
    <div class="flex items-center gap-3">
        <span class="material-symbols-outlined text-primary">admin_panel_settings</span>
        Rol
    </div>
</th>
<th class="p-6 font-bold text-text-primary text-lg w-40">
    <div class="flex items-center gap-3">
        <span class="material-symbols-outlined text-primary">location_on</span>
        IP Adresi
    </div>
</th>
<th class="p-6 font-bold text-text-primary text-lg w-32">
    <div class="flex items-center gap-3">
        <span class="material-symbols-outlined text-primary">language</span>
        Eklenen Siteler
    </div>
</th>
<th class="p-6 font-bold text-text-primary text-lg w-36">
    <div class="flex items-center gap-3">
        <span class="material-symbols-outlined text-primary">schedule</span>
        Katılım Tarihi
    </div>
</th>
<th class="p-6 font-bold text-text-primary text-lg text-center w-40">
    <div class="flex items-center justify-center gap-3">
        <span class="material-symbols-outlined text-primary">settings</span>
        İşlemler
    </div>
</th>
</tr>
</thead>
<tbody>
<?php foreach ($users as $user): ?>
<tr class="border-b border-line-light hover:bg-gradient-to-r hover:from-primary/5 hover:to-transparent transition-all duration-200 group">
<td class="p-6">
    <input type="checkbox" class="user-checkbox w-4 h-4 text-primary bg-gray-100 border-gray-300 rounded focus:ring-primary focus:ring-2" 
           value="<?php echo $user['id']; ?>" onchange="updateBulkActions()">
</td>
<td class="p-6">
<div class="flex items-center gap-4">
<img alt="User Avatar" class="w-12 h-12 rounded-full object-cover border-2 border-gray-100 group-hover:border-primary/30 transition-colors" src="<?php echo htmlspecialchars($user['profile_photo'] ? '../' . $user['profile_photo'] : '../uploads/no-logo.png'); ?>"/>
<div>
<div class="flex items-center gap-2">
    <p class="font-bold text-text-primary text-lg group-hover:text-primary transition-colors"><?php echo htmlspecialchars($user['username']); ?></p>
    <?php if ($user['is_banned']): ?>
        <span class="inline-flex items-center gap-1 px-2 py-0.5 text-xs font-medium rounded-full bg-red-100 text-red-700 border border-red-200">
            <span class="material-symbols-outlined text-xs">block</span>
            Banlı
        </span>
    <?php endif; ?>
</div>
<p class="text-sm text-text-secondary flex items-center gap-1">
    <span class="material-symbols-outlined text-xs">email</span>
    <?php echo htmlspecialchars($user['email']); ?>
</p>
</div>
</div>
</td>
<td class="p-6">
<?php 
$role = $user['role'] == 'kullanici' ? 'User' : ucfirst($user['role'] ?? 'User');
$roleColor = $role == 'Admin' ? 'bg-purple-100 text-purple-700 border-purple-200' : 'bg-blue-100 text-blue-700 border-blue-200';
?>
<span class="inline-flex items-center gap-1 px-3 py-1.5 text-sm font-medium rounded-full border <?php echo $roleColor; ?>">
    <span class="material-symbols-outlined text-xs"><?php echo $role == 'Admin' ? 'admin_panel_settings' : 'person'; ?></span>
    <?php echo $role; ?>
</span>
</td>
<td class="p-6">
<span class="font-mono text-sm bg-gray-100 px-3 py-1.5 rounded-lg text-text-secondary border">
    <?php echo htmlspecialchars($user['ip'] ?? 'N/A'); ?>
</span>
</td>
<td class="p-6">
<div class="flex items-center gap-2">
    <span class="material-symbols-outlined text-primary text-lg">language</span>
    <span class="font-bold text-lg text-text-primary"><?php echo intval($siteCounts[$user['id']] ?? 0); ?></span>
    <span class="text-text-secondary text-sm">sites</span>
</div>
</td>
<td class="p-6">
<div class="flex items-center gap-2 text-text-secondary">
    <span class="material-symbols-outlined text-sm">calendar_today</span>
    <span class="font-medium"><?php echo date('M d, Y', strtotime($user['kayit_tarihi'])); ?></span>
</div>
</td>
<td class="p-6">
<div class="flex items-center justify-center gap-2">
<a href="user_view.php?id=<?php echo $user['id']; ?>" class="w-10 h-10 rounded-lg bg-green-100 text-green-600 flex items-center justify-center hover:bg-green-200 hover:scale-105 transition-all duration-200 border border-green-200" title="Kullanıcıyı Görüntüle">
<span class="material-symbols-outlined text-lg">visibility</span>
</a>
<a href="user_edit.php?id=<?php echo $user['id']; ?>" class="w-10 h-10 rounded-lg bg-blue-100 text-blue-600 flex items-center justify-center hover:bg-blue-200 hover:scale-105 transition-all duration-200 border border-blue-200" title="Kullanıcıyı Düzenle">
<span class="material-symbols-outlined text-lg">edit</span>
</a>
<?php if ($user['is_banned']): ?>
    <a href="unban_user.php?id=<?php echo $user['id']; ?>" class="w-10 h-10 rounded-lg bg-orange-100 text-orange-600 flex items-center justify-center hover:bg-orange-200 hover:scale-105 transition-all duration-200 border border-orange-200" title="Banı Kaldır">
    <span class="material-symbols-outlined text-lg">check_circle</span>
    </a>
<?php else: ?>
    <a href="ban_user.php?id=<?php echo $user['id']; ?>" class="w-10 h-10 rounded-lg bg-red-100 text-red-600 flex items-center justify-center hover:bg-red-200 hover:scale-105 transition-all duration-200 border border-red-200" title="Kullanıcıyı Banla">
    <span class="material-symbols-outlined text-lg">block</span>
    </a>
<?php endif; ?>
<a href="user_delete.php?id=<?php echo $user['id']; ?>" class="w-10 h-10 rounded-lg bg-gray-100 text-gray-600 flex items-center justify-center hover:bg-gray-200 hover:scale-105 transition-all duration-200 border border-gray-200" title="Delete User">
<span class="material-symbols-outlined text-lg">delete</span>
</a>
</div>
</td>
</tr>
<?php endforeach; ?>
</tbody>
</table>
</div>

<!-- Pagination -->
<?php if ($totalPages > 1): ?>
<div class="flex justify-between items-center mt-6 pt-4 border-t border-line-light">
<p class="text-sm text-text-secondary">Showing <?php echo $offset + 1; ?> to <?php echo min($offset + $perPage, $totalRows); ?> of <?php echo $totalRows; ?> users</p>
<div class="flex items-center gap-2">
    <?php if ($page > 1): ?>
        <a href="?page=<?php echo ($page - 1); ?>&<?php echo http_build_query(array_filter(['search' => $_GET['search'] ?? '', 'status' => $_GET['status'] ?? '', 'sort' => $_GET['sort'] ?? '', 'perPage' => $_GET['perPage'] ?? ''])); ?>" class="w-9 h-9 flex items-center justify-center rounded-lg border border-line-light hover:bg-background text-text-secondary">
            <span class="material-symbols-outlined text-xl">chevron_left</span>
        </a>
    <?php endif; ?>
    
    <?php
    $startPage = max(1, $page - 2);
    $endPage = min($totalPages, $page + 2);
    
    if ($startPage > 1): ?>
        <a href="?page=1&<?php echo http_build_query(array_filter(['search' => $_GET['search'] ?? '', 'status' => $_GET['status'] ?? '', 'sort' => $_GET['sort'] ?? '', 'perPage' => $_GET['perPage'] ?? ''])); ?>" class="w-9 h-9 flex items-center justify-center rounded-lg hover:bg-background text-text-secondary">1</a>
        <?php if ($startPage > 2): ?>
            <span class="text-text-secondary">...</span>
        <?php endif; ?>
    <?php endif; ?>
    
    <?php for ($i = $startPage; $i <= $endPage; $i++): ?>
        <a href="?page=<?php echo $i; ?>&<?php echo http_build_query(array_filter(['search' => $_GET['search'] ?? '', 'status' => $_GET['status'] ?? '', 'sort' => $_GET['sort'] ?? '', 'perPage' => $_GET['perPage'] ?? ''])); ?>" class="w-9 h-9 flex items-center justify-center rounded-lg <?php echo $i == $page ? 'bg-primary text-white font-bold' : 'hover:bg-background text-text-secondary'; ?>">
            <?php echo $i; ?>
        </a>
    <?php endfor; ?>
    
    <?php if ($endPage < $totalPages): ?>
        <?php if ($endPage < $totalPages - 1): ?>
            <span class="text-text-secondary">...</span>
        <?php endif; ?>
        <a href="?page=<?php echo $totalPages; ?>&<?php echo http_build_query(array_filter(['search' => $_GET['search'] ?? '', 'status' => $_GET['status'] ?? '', 'sort' => $_GET['sort'] ?? '', 'perPage' => $_GET['perPage'] ?? ''])); ?>" class="w-9 h-9 flex items-center justify-center rounded-lg hover:bg-background text-text-secondary"><?php echo $totalPages; ?></a>
    <?php endif; ?>
    
    <?php if ($page < $totalPages): ?>
        <a href="?page=<?php echo ($page + 1); ?>&<?php echo http_build_query(array_filter(['search' => $_GET['search'] ?? '', 'status' => $_GET['status'] ?? '', 'sort' => $_GET['sort'] ?? '', 'perPage' => $_GET['perPage'] ?? ''])); ?>" class="w-9 h-9 flex items-center justify-center rounded-lg border border-line-light hover:bg-background text-text-secondary">
            <span class="material-symbols-outlined text-xl">chevron_right</span>
        </a>
    <?php endif; ?>
</div>
</div>
<?php endif; ?>

<?php endif; ?>

</div>
</main>
</div>

<script>
// Checkbox selection functions
function toggleSelectAll() {
    const selectAllCheckbox = document.getElementById('select-all');
    const userCheckboxes = document.querySelectorAll('.user-checkbox');
    
    userCheckboxes.forEach(checkbox => {
        checkbox.checked = selectAllCheckbox.checked;
    });
    
    updateBulkActions();
}

function updateBulkActions() {
    const userCheckboxes = document.querySelectorAll('.user-checkbox');
    const checkedBoxes = document.querySelectorAll('.user-checkbox:checked');
    const selectAllCheckbox = document.getElementById('select-all');
    const bulkDeleteBtn = document.getElementById('bulk-delete-btn');
    const selectedCount = document.getElementById('selected-count');
    
    // Update select all checkbox state
    if (checkedBoxes.length === 0) {
        selectAllCheckbox.indeterminate = false;
        selectAllCheckbox.checked = false;
    } else if (checkedBoxes.length === userCheckboxes.length) {
        selectAllCheckbox.indeterminate = false;
        selectAllCheckbox.checked = true;
    } else {
        selectAllCheckbox.indeterminate = true;
    }
    
    // Update bulk actions
    bulkDeleteBtn.disabled = checkedBoxes.length === 0;
    selectedCount.textContent = `${checkedBoxes.length} users selected`;
}

// Bulk delete function
function bulkDelete() {
    const checkedBoxes = document.querySelectorAll('.user-checkbox:checked');
    
    if (checkedBoxes.length === 0) {
        alert('Please select users to delete.');
        return;
    }
    
    const userNames = Array.from(checkedBoxes).map(checkbox => {
        const row = checkbox.closest('tr');
        return row.querySelector('td:nth-child(2) .font-bold').textContent;
    });
    
    const modal = document.createElement('div');
    modal.className = 'fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50';
    modal.innerHTML = `
        <div class="bg-white rounded-lg p-6 max-w-md w-full mx-4 transform scale-95 transition-transform duration-200">
            <div class="flex items-center mb-4">
                <span class="material-symbols-outlined text-red-500 text-3xl mr-3">warning</span>
                <h3 class="text-lg font-semibold text-gray-900">Confirm Bulk Deletion</h3>
            </div>
            <p class="text-gray-600 mb-4">
                Are you sure you want to delete <strong>${checkedBoxes.length} users</strong>?
            </p>
            <div class="max-h-32 overflow-y-auto mb-4 p-3 bg-gray-50 rounded border">
                <ul class="text-sm text-gray-700">
                    ${userNames.map(name => `<li>• ${name}</li>`).join('')}
                </ul>
            </div>
            <p class="text-sm text-red-600 mb-6">
                <strong>Warning:</strong> This action cannot be undone. All selected users will be permanently deleted.
            </p>
            <div class="flex gap-3">
                <button onclick="closeBulkDeleteModal()" class="flex-1 px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50">
                    Cancel
                </button>
                <button onclick="proceedBulkDelete()" class="flex-1 px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700">
                    Delete Users
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Animate in
    setTimeout(() => {
        modal.querySelector('.bg-white').style.transform = 'scale(1)';
    }, 10);
    
    // Store modal reference
    window.currentBulkDeleteModal = modal;
}

function closeBulkDeleteModal() {
    if (window.currentBulkDeleteModal) {
        const modal = window.currentBulkDeleteModal;
        modal.querySelector('.bg-white').style.transform = 'scale(0.95)';
        setTimeout(() => {
            document.body.removeChild(modal);
        }, 200);
        window.currentBulkDeleteModal = null;
    }
}

function proceedBulkDelete() {
    closeBulkDeleteModal();
    
    const checkedBoxes = document.querySelectorAll('.user-checkbox:checked');
    
    // Show loading indicator
    const loader = document.createElement('div');
    loader.className = 'fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50';
    loader.innerHTML = `
        <div class="bg-white rounded-lg p-6 text-center">
            <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-3"></div>
            <p class="text-gray-600">Deleting users...</p>
        </div>
    `;
    document.body.appendChild(loader);
    
    // Create form and submit
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = 'users.php';
    
    const bulkDeleteInput = document.createElement('input');
    bulkDeleteInput.type = 'hidden';
    bulkDeleteInput.name = 'bulk_delete';
    bulkDeleteInput.value = '1';
    form.appendChild(bulkDeleteInput);
    
    // Add selected user IDs
    checkedBoxes.forEach(checkbox => {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'user_ids[]';
        input.value = checkbox.value;
        form.appendChild(input);
    });
    
    document.body.appendChild(form);
    form.submit();
}

// Close modal on escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape' && window.currentBulkDeleteModal) {
        closeBulkDeleteModal();
    }
});

// New User Modal Functions
function openNewUserModal() {
    const modal = document.getElementById('new-user-modal');
    modal.classList.remove('hidden');
    setTimeout(() => {
        modal.querySelector('.modal-content').classList.remove('scale-95', 'opacity-0');
        modal.querySelector('.modal-content').classList.add('scale-100', 'opacity-100');
    }, 50);
}

function closeNewUserModal() {
    const modal = document.getElementById('new-user-modal');
    modal.querySelector('.modal-content').classList.remove('scale-100', 'opacity-100');
    modal.querySelector('.modal-content').classList.add('scale-95', 'opacity-0');
    setTimeout(() => {
        modal.classList.add('hidden');
        // Reset form
        document.getElementById('new-user-form').reset();
        document.getElementById('new-user-message').classList.add('hidden');
    }, 300);
}

function togglePasswordVisibility(fieldId) {
    const field = document.getElementById(fieldId);
    const icon = field.nextElementSibling.querySelector('.material-symbols-outlined');
    if (field.type === 'password') {
        field.type = 'text';
        icon.textContent = 'visibility';
    } else {
        field.type = 'password';
        icon.textContent = 'visibility_off';
    }
}

// Password visibility toggle function
function togglePasswordVisibility(inputId) {
    const input = document.getElementById(inputId);
    const button = input.nextElementSibling;
    const icon = button.querySelector('.material-symbols-outlined');
    
    if (input.type === 'password') {
        input.type = 'text';
        icon.textContent = 'visibility';
    } else {
        input.type = 'password';
        icon.textContent = 'visibility_off';
    }
}

// Handle new user form submission
document.addEventListener('DOMContentLoaded', function() {
    const newUserForm = document.getElementById('new-user-form');
    if (newUserForm) {
        newUserForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const form = e.target;
            const formData = new FormData(form);
            const messageDiv = document.getElementById('new-user-message');
            const submitButton = form.querySelector('button[type="submit"]');
            
            // Show loading state
            const originalText = submitButton.textContent;
            submitButton.textContent = 'Creating...';
            submitButton.disabled = true;
            
            // Clear previous messages
            messageDiv.classList.add('hidden');
            messageDiv.classList.remove('bg-green-100', 'text-green-800', 'bg-red-100', 'text-red-800');
            messageDiv.innerHTML = '';
            
            // Client-side validation
            const username = formData.get('username');
            const email = formData.get('email');
            const password = formData.get('password');
            
            // Username validation
            const usernameError = validateAdminUsername(username);
            if (usernameError) {
                showNewUserMessage(usernameError, 'error');
                submitButton.textContent = originalText;
                submitButton.disabled = false;
                return;
            }
            
            if (!email || !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) {
                showNewUserMessage('Please enter a valid email address', 'error');
                submitButton.textContent = originalText;
                submitButton.disabled = false;
                return;
            }
            
            if (!password || password.length < 6) {
                showNewUserMessage('Password must be at least 6 characters long', 'error');
                submitButton.textContent = originalText;
                submitButton.disabled = false;
                return;
            }
            
            fetch('user_create.php', {
                method: 'POST',
                body: formData
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.json();
            })
            .then(data => {
                if (data.success) {
                    showNewUserMessage(data.message, 'success');
                    // Reset form
                    form.reset();
                    // Close modal after 2 seconds
                    setTimeout(() => {
                        closeNewUserModal();
                        // Reload page to show new user
                        window.location.reload();
                    }, 2000);
                } else {
                    showNewUserMessage(data.message || 'Error creating user', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNewUserMessage('An unexpected error occurred. Please try again.', 'error');
            })
            .finally(() => {
                submitButton.textContent = originalText;
                submitButton.disabled = false;
            });
        });
    }
});

// Admin username validation function
function validateAdminUsername(username) {
    const restrictedUsernames = [
        'admin', 'administrator', 'moderator', 'mod', 'root', 'superadmin', 'super', 'owner', 'founder',
        'ceo', 'manager', 'staff', 'support', 'help', 'system', 'bot', 'api', 'test', 'demo',
        'guest', 'user', 'member', 'visitor', 'anonymous', 'null', 'undefined', 'www', 'mail',
        'ftp', 'news', 'info', 'contact', 'about', 'privacy', 'terms', 'legal', 'security',
        'webmaster', 'hostmaster', 'postmaster', 'noreply', 'no-reply', 'donotreply',
        'sofi', 'sofia', 'sofie', 'sophie', 'sophia', 'sofya', 'sofija', 'sofia', 'sofie',
        'admin123', 'admin1', 'admin2', 'admin3', 'admin4', 'admin5', 'root123', 'root1',
        'moderator123', 'mod123', 'superadmin123', 'super123', 'owner123', 'founder123',
        'ceo123', 'manager123', 'staff123', 'support123', 'help123', 'system123', 'bot123',
        'api123', 'test123', 'demo123', 'guest123', 'user123', 'member123', 'visitor123',
        'anonymous123', 'null123', 'undefined123', 'www123', 'mail123', 'ftp123', 'news123',
        'info123', 'contact123', 'about123', 'privacy123', 'terms123', 'legal123', 'security123',
        'webmaster123', 'hostmaster123', 'postmaster123', 'noreply123', 'no-reply123', 'donotreply123'
    ];
    
    if (!username || username.trim() === '') {
        return 'Kullanıcı adı gerekli';
    }
    
    const usernameLower = username.toLowerCase();
    if (restrictedUsernames.includes(usernameLower)) {
        return 'Bu kullanıcı adı kısıtlı ve kullanılamaz';
    }
    
    if (username.length < 3) {
        return 'Kullanıcı adı en az 3 karakter olmalı';
    }
    
    if (username.length > 20) {
        return 'Kullanıcı adı en fazla 20 karakter olmalı';
    }
    
    if (!/^[a-zA-Z0-9_]+$/.test(username)) {
        return 'Kullanıcı adı sadece harf, rakam ve alt çizgi içerebilir';
    }
    
    if (usernameLower.includes('admin') || usernameLower.includes('mod') || 
        usernameLower.includes('root') || usernameLower.includes('super')) {
        return 'Kullanıcı adı admin, mod, root, super gibi yetki kelimeleri içeremez';
    }
    
    return null; // Valid
}

// Helper function to show messages
function showNewUserMessage(message, type) {
    const messageDiv = document.getElementById('new-user-message');
    messageDiv.classList.remove('hidden');
    messageDiv.classList.remove('bg-green-100', 'text-green-800', 'bg-red-100', 'text-red-800');
    
    if (type === 'success') {
        messageDiv.classList.add('bg-green-100', 'text-green-800');
        messageDiv.innerHTML = `<div class="flex items-center gap-2"><span class="material-symbols-outlined">check_circle</span>${message}</div>`;
    } else {
        messageDiv.classList.add('bg-red-100', 'text-red-800');
        messageDiv.innerHTML = `<div class="flex items-center gap-2"><span class="material-symbols-outlined">error</span>${message}</div>`;
    }
}
</script>

<!-- New User Modal -->
<div id="new-user-modal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
    <div class="modal-content bg-white dark:bg-gray-800 rounded-2xl shadow-xl w-full max-w-md p-8 relative transform transition-all scale-95 opacity-0">
        <button type="button" onclick="closeNewUserModal()" class="absolute top-4 right-4 text-gray-400 hover:text-gray-600 dark:hover:text-gray-200">
            <span class="material-symbols-outlined">close</span>
        </button>
        <h2 class="text-2xl font-bold text-gray-900 dark:text-white mb-6">Add New User</h2>
        
        <form id="new-user-form" method="POST" enctype="multipart/form-data">
            <div id="new-user-message" class="mb-4 p-3 rounded-lg text-sm hidden"></div>
            
            <div class="space-y-5">
                <!-- Username -->
                <div>
                    <label for="new_username" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Username</label>
                    <input type="text" id="new_username" name="username" required class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white transition-colors">
                </div>
                
                <!-- Email -->
                <div>
                    <label for="new_email" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Email</label>
                    <input type="email" id="new_email" name="email" required class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white transition-colors">
                </div>
                
                <!-- Password -->
                <div>
                    <label for="new_password" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Password</label>
                    <div class="relative">
                        <input type="password" id="new_password" name="password" required class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white transition-colors">
                        <button type="button" onclick="togglePasswordVisibility('new_password')" class="absolute inset-y-0 right-0 pr-3 flex items-center text-gray-400 dark:text-gray-500">
                            <span class="material-symbols-outlined text-lg">visibility_off</span>
                        </button>
                    </div>
                </div>
                
                <!-- Role -->
                <div>
                    <label for="new_role" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Role</label>
                    <select id="new_role" name="role" class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white transition-colors">
                        <option value="user">User</option>
                        <option value="admin">Admin</option>
                    </select>
                </div>
                
                <!-- Status -->
                <div>
                    <label for="new_status" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Status</label>
                    <select id="new_status" name="status" class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white transition-colors">
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                    </select>
                </div>
                
                <button type="submit" class="w-full bg-primary hover:bg-primary-dark text-white font-semibold py-3 px-4 rounded-xl transition-colors">
                    Create User
                </button>
            </div>
        </form>
    </div>
</div>

</body></html>

