<?php
// forgot_password.php - Password reset request
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once '../includes/config.php';
require_once '../utils/email_helper.php';
require_once '../includes/turkish_fix.php';

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = trim($_POST['email'] ?? '');
    
    if (empty($email)) {
        $error = 'E-posta adresi gerekli.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Lütfen geçerli bir e-posta adresi girin.';
    } else {
        try {
            // Find user
            $stmt = $pdo->prepare('SELECT id, username, email FROM user WHERE email = ?');
            $stmt->execute([$email]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($user) {
                // Create reset token
                $resetToken = bin2hex(random_bytes(32));
                $expireTime = date('Y-m-d H:i:s', strtotime('+60 minutes')); // Valid for 1 hour
                
                // Save token to database
                $stmt = $pdo->prepare('
                    INSERT INTO password_reset_tokens (user_id, token, expires_at, created_at, used) 
                    VALUES (?, ?, ?, NOW(), 0)
                    ON DUPLICATE KEY UPDATE 
                    token = VALUES(token), 
                    expires_at = VALUES(expires_at), 
                    created_at = NOW(),
                    used = 0
                ');
                $stmt->execute([$user['id'], $resetToken, $expireTime]);
                
                // Send email
                $emailSent = sendPasswordResetEmail($user['email'], $resetToken, $user['username']);
                
                if ($emailSent) {
                    $message = 'Şifre sıfırlama bağlantısı e-posta adresinize gönderildi. Lütfen e-postanızı kontrol edin.';
                } else {
                    $error = 'E-posta gönderilirken bir hata oluştu. Lütfen daha sonra tekrar deneyin.';
                }
            } else {
                // Security: show same message even if email doesn't exist
                $message = 'Bu e-posta adresi ile bir hesap varsa, şifre sıfırlama bağlantısı gönderilmiştir.';
            }
        } catch (PDOException $e) {
            error_log('Password reset error: ' . $e->getMessage());
            $error = 'Bir hata oluştu. Lütfen daha sonra tekrar deneyin.';
        }
    }
}
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Şifremi Unuttum</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined:opsz,wght,FILL,GRAD@20..48,100..700,0..1,-50..200" rel="stylesheet">
    <style>
        .material-symbols-outlined {
            font-variation-settings: 'FILL' 0, 'wght' 400, 'GRAD' 0, 'opsz' 24;
        }
    </style>
</head>
<body class="bg-gray-50 min-h-screen flex items-center justify-center">
    <div class="max-w-md w-full bg-white rounded-lg shadow-md p-8">
        <div class="text-center mb-8">
            <div class="inline-flex items-center justify-center w-16 h-16 bg-blue-100 rounded-full mb-4">
                <span class="material-symbols-outlined text-blue-600 text-2xl">lock_reset</span>
            </div>
            <h1 class="text-2xl font-bold text-gray-900">Şifremi Unuttum</h1>
            <p class="text-gray-600 mt-2">E-posta adresinizi girin, size şifre sıfırlama bağlantısı gönderelim.</p>
        </div>

        <?php if ($message): ?>
            <div class="mb-6 p-4 bg-green-100 border border-green-200 rounded-lg">
                <div class="flex items-center">
                    <span class="material-symbols-outlined text-green-600 mr-3">check_circle</span>
                    <p class="text-green-800"><?php echo htmlspecialchars($message); ?></p>
                </div>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="mb-6 p-4 bg-red-100 border border-red-200 rounded-lg">
                <div class="flex items-center">
                    <span class="material-symbols-outlined text-red-600 mr-3">error</span>
                    <p class="text-red-800"><?php echo htmlspecialchars($error); ?></p>
                </div>
            </div>
        <?php endif; ?>

        <form method="POST" class="space-y-6">
            <div>
                <label for="email" class="block text-sm font-medium text-gray-700 mb-2">
                    E-posta Adresi
                </label>
                <div class="relative">
                    <span class="material-symbols-outlined absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">email</span>
                    <input 
                        type="email" 
                        id="email" 
                        name="email" 
                        required
                        value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>"
                        class="w-full pl-12 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="example@email.com"
                    >
                </div>
            </div>

            <button 
                type="submit" 
                class="w-full bg-blue-600 text-white py-3 px-4 rounded-lg hover:bg-blue-700 focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 transition duration-200 font-medium"
            >
                Sıfırlama Bağlantısı Gönder
            </button>
        </form>

        <div class="mt-6 text-center">
            <a href="/login" class="text-blue-600 hover:text-blue-800 text-sm font-medium">
                ← Giriş sayfasına dön
            </a>
        </div>
    </div>
</body>
</html>