<?php
// reset_password.php - Password Reset
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once '../includes/config.php';
require_once '../includes/turkish_fix.php';

$message = '';
$error = '';
$validToken = false;
$token = $_GET['token'] ?? '';

// Token validation
if (!empty($token)) {
    try {
        $stmt = $pdo->prepare('
            SELECT prt.*, u.email, u.username 
            FROM password_reset_tokens prt 
            JOIN user u ON prt.user_id = u.id 
            WHERE prt.token = ? AND prt.expires_at > ? AND prt.used = 0
        ');
        $stmt->execute([$token, date('Y-m-d H:i:s')]);
        $resetData = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($resetData) {
            $validToken = true;
        } else {
            // Check token details for better error message
            $stmt = $pdo->prepare('SELECT * FROM password_reset_tokens WHERE token = ?');
            $stmt->execute([$token]);
            $tokenData = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($tokenData) {
                if ($tokenData['used'] == 1) {
                    $error = 'Bu sıfırlama bağlantısı zaten kullanılmış.';
                } elseif ($tokenData['expires_at'] < date('Y-m-d H:i:s')) {
                    $error = 'Bu sıfırlama bağlantısının süresi dolmuş.';
                } else {
                    $error = 'Geçersiz sıfırlama bağlantısı.';
                }
            } else {
                $error = 'Geçersiz veya süresi dolmuş sıfırlama bağlantısı.';
            }
        }
    } catch (PDOException $e) {
        error_log('Token validation error: ' . $e->getMessage());
        $error = 'Bir hata oluştu. Lütfen daha sonra tekrar deneyin.';
    }
} else {
    $error = 'Sıfırlama tokeni bulunamadı.';
}

// Password reset process
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $validToken) {
    $newPassword = $_POST['password'] ?? '';
    $confirmPassword = $_POST['confirm_password'] ?? '';
    
    if (empty($newPassword)) {
        $error = 'Yeni şifre gereklidir.';
    } elseif (strlen($newPassword) < 6) {
        $error = 'Şifre en az 6 karakter olmalıdır.';
    } elseif ($newPassword !== $confirmPassword) {
        $error = 'Şifreler eşleşmiyor.';
    } else {
        try {
            // Update password
            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
            $stmt = $pdo->prepare('UPDATE user SET password = ? WHERE id = ?');
            $stmt->execute([$hashedPassword, $resetData['user_id']]);
            
            // Mark token as used
            $stmt = $pdo->prepare('UPDATE password_reset_tokens SET used = 1 WHERE token = ?');
            $stmt->execute([$token]);
            
            $message = 'Şifreniz başarıyla güncellendi. Artık yeni şifrenizle giriş yapabilirsiniz.';
            $validToken = false; // Hide form
            
        } catch (PDOException $e) {
            error_log('Password update error: ' . $e->getMessage());
            $error = 'Şifreniz güncellenirken bir hata oluştu.';
        }
    }
}
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Şifre Sıfırla</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined:opsz,wght,FILL,GRAD@20..48,100..700,0..1,-50..200" rel="stylesheet">
    <style>
        .material-symbols-outlined {
            font-variation-settings: 'FILL' 0, 'wght' 400, 'GRAD' 0, 'opsz' 24;
        }
    </style>
</head>
<body class="bg-gray-50 min-h-screen flex items-center justify-center">
    <div class="max-w-md w-full bg-white rounded-lg shadow-md p-8">
        <div class="text-center mb-8">
            <div class="inline-flex items-center justify-center w-16 h-16 bg-blue-100 rounded-full mb-4">
                <span class="material-symbols-outlined text-blue-600 text-2xl">lock_reset</span>
            </div>
            <h1 class="text-2xl font-bold text-gray-900">Şifre Sıfırla</h1>
            <?php if ($validToken): ?>
                <p class="text-gray-600 mt-2">Hesabınız için yeni bir şifre belirleyin.</p>
            <?php endif; ?>
        </div>

        <?php if ($message): ?>
            <div class="mb-6 p-4 bg-green-100 border border-green-200 rounded-lg">
                <div class="flex items-center">
                    <span class="material-symbols-outlined text-green-600 mr-3">check_circle</span>
                    <p class="text-green-800"><?php echo htmlspecialchars($message); ?></p>
                </div>
                <div class="mt-4 text-center">
                    <a href="/login" class="inline-flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition duration-200">
                        <span class="material-symbols-outlined mr-2">login</span>
                        Giriş Yap
                    </a>
                </div>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="mb-6 p-4 bg-red-100 border border-red-200 rounded-lg">
                <div class="flex items-center">
                    <span class="material-symbols-outlined text-red-600 mr-3">error</span>
                    <p class="text-red-800"><?php echo htmlspecialchars($error); ?></p>
                </div>
                <?php if (!$validToken): ?>
                    <div class="mt-4 text-center">
                        <a href="/forgot-password" class="text-blue-600 hover:text-blue-800 text-sm font-medium">
                            Yeni sıfırlama bağlantısı iste
                        </a>
                    </div>
                <?php endif; ?>
            </div>
        <?php endif; ?>

        <?php if ($validToken): ?>
            <form method="POST" class="space-y-6">
                <div>
                    <label for="password" class="block text-sm font-medium text-gray-700 mb-2">
                        Yeni Şifre
                    </label>
                    <div class="relative">
                        <span class="material-symbols-outlined absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">lock</span>
                        <input 
                            type="password" 
                            id="password" 
                            name="password" 
                            required
                            minlength="6"
                            class="w-full pl-12 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                            placeholder="En az 6 karakter"
                        >
                    </div>
                </div>

                <div>
                    <label for="confirm_password" class="block text-sm font-medium text-gray-700 mb-2">
                        Şifreyi Onayla
                    </label>
                    <div class="relative">
                        <span class="material-symbols-outlined absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">lock</span>
                        <input 
                            type="password" 
                            id="confirm_password" 
                            name="confirm_password" 
                            required
                            minlength="6"
                            class="w-full pl-12 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                            placeholder="Şifrenizi tekrar girin"
                        >
                    </div>
                </div>

                <button 
                    type="submit" 
                    class="w-full bg-blue-600 text-white py-3 px-4 rounded-lg hover:bg-blue-700 focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 transition duration-200 font-medium"
                >
                    Şifreyi Güncelle
                </button>
            </form>
        <?php endif; ?>

        <div class="mt-6 text-center">
            <a href="/login" class="text-blue-600 hover:text-blue-800 text-sm font-medium">
                ← Giriş sayfasına dön
            </a>
        </div>
    </div>

    <script>
        // Şifre eşleşme kontrolü
        document.getElementById('confirm_password')?.addEventListener('input', function() {
            const password = document.getElementById('password').value;
            const confirmPassword = this.value;
            
            if (confirmPassword && password !== confirmPassword) {
                this.setCustomValidity('Şifreler eşleşmiyor');
            } else {
                this.setCustomValidity('');
            }
        });
    </script>
</body>
</html>