<?php
require_once '../includes/config.php';
require_once '../includes/turkish_fix.php';

// Kategorileri al
$categories = [];
try {
    $stmt = $pdo->query("SELECT id, name FROM categories ORDER BY name");
    $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    // Kategoriler yok
}

// Mevcut seçili kategorileri al
$selectedCategories = [];
try {
    $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = 'bot_target_categories'");
    $stmt->execute();
    $value = $stmt->fetchColumn();
    if ($value) {
        $selectedCategories = array_map('intval', explode(',', $value));
    }
} catch (Exception $e) {
    // Setting yok
}

// Sites per category ayarını al
$sitesPerCategory = 3;
try {
    $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = 'bot_sites_per_category'");
    $stmt->execute();
    $value = $stmt->fetchColumn();
    if ($value !== false) {
        $sitesPerCategory = max(1, intval($value));
    }
} catch (Exception $e) {
    // Setting yok
}

// Bot ayarlarını al
function getBotSetting($pdo, $key, $default = '') {
    try {
        $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
        $stmt->execute(['bot_' . $key]);
        $result = $stmt->fetchColumn();
        return $result !== false ? $result : $default;
    } catch (Exception $e) {
        return $default;
    }
}

// POST işlemi
if ($_POST) {
    if (isset($_POST['action'])) {
        if ($_POST['action'] === 'save_settings') {
            // Kategori ayarlarını kaydet
            $newCategories = $_POST['categories'] ?? [];
            $newCategories = array_map('intval', $newCategories);
            $newCategories = array_filter($newCategories);
            
            $sitesPerCat = max(1, intval($_POST['sites_per_category'] ?? 3));
            
            // Ayarları kaydet
            $stmt = $pdo->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?");
            $categoryValue = implode(',', $newCategories);
            $stmt->execute(['bot_target_categories', $categoryValue, $categoryValue]);
            $stmt->execute(['bot_sites_per_category', $sitesPerCat, $sitesPerCat]);
            
            $selectedCategories = $newCategories;
            $sitesPerCategory = $sitesPerCat;
            $success_message = "Kategori ayarları başarıyla kaydedildi!";
            
        } elseif ($_POST['action'] === 'run_bot') {
            // Bot'u çalıştır
            if (!empty($selectedCategories)) {
                header('Location: run_category_bot_batch.php');
                exit;
            } else {
                $error_message = "Lütfen en az bir kategori seçin!";
            }
        }
    } elseif (isset($_POST['set_language'])) {
        // Dil ayarını kaydet
        $language = $_POST['language'];
        if (in_array($language, ['tr', 'en'])) {
            $stmt = $pdo->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?");
            $stmt->execute(['bot_content_language', $language, $language]);
            header('Location: category_selector.php?language_updated=1');
            exit;
        }
    }
}

// Site ayarlarını al
$stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = 'site_name'");
$stmt->execute();
$site_name = $stmt->fetchColumn() ?: 'Site Directory';
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kategori Seçici | <?php echo htmlspecialchars($site_name); ?></title>
    <link href="https://fonts.googleapis.com/css2?family=Public+Sans:wght@400;500;600;700&display=swap" rel="stylesheet">
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    fontFamily: {
                        'sans': ['Public Sans', 'ui-sans-serif', 'system-ui'],
                    }
                }
            }
        }
    </script>
</head>
<body class="bg-gray-50 font-sans">
    <div class="max-w-6xl mx-auto p-6">
        <!-- Header -->
        <div class="bg-white rounded-lg shadow-sm p-6 mb-6">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-2xl font-bold text-gray-900 mb-2">🎯 Gelişmiş Kategori Bazlı Bot</h1>
                    <p class="text-gray-600">Birden fazla kategori seçin ve toplu olarak site ekleyin</p>
                </div>
                <a href="index.php" class="bg-gray-500 hover:bg-gray-600 text-white px-4 py-2 rounded-lg font-medium transition-colors">
                    ← Geri Dön
                </a>
            </div>
        </div>

        <!-- Language Selection -->
        <div class="bg-white rounded-lg shadow-sm p-6 mb-6">
            <div class="flex items-center justify-between">
                <div>
                    <h3 class="text-lg font-semibold text-gray-900 mb-2">🌍 İçerik Dili Seçimi</h3>
                    <p class="text-gray-600">Bot hangi dilde içerik çekeceğini seçin</p>
                </div>
                <div class="flex gap-3">
                    <button type="button" onclick="setLanguage('tr')" 
                            class="language-btn px-6 py-3 rounded-xl font-semibold transition-all <?php echo (getBotSetting($pdo, 'content_language', 'tr') === 'tr') ? 'bg-blue-600 text-white shadow-lg' : 'bg-gray-200 text-gray-700 hover:bg-gray-300'; ?>">
                        🇹🇷 Türkçe
                    </button>
                    <button type="button" onclick="setLanguage('en')" 
                            class="language-btn px-6 py-3 rounded-xl font-semibold transition-all <?php echo (getBotSetting($pdo, 'content_language', 'tr') === 'en') ? 'bg-blue-600 text-white shadow-lg' : 'bg-gray-200 text-gray-700 hover:bg-gray-300'; ?>">
                        🇺🇸 English
                    </button>
                </div>
            </div>
        </div>

        <?php if (isset($success_message)): ?>
        <div class="bg-green-50 border border-green-200 rounded-lg p-4 mb-6">
            <div class="flex items-center">
                <svg class="h-5 w-5 text-green-400 mr-3" viewBox="0 0 20 20" fill="currentColor">
                    <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd" />
                </svg>
                <p class="text-green-800 font-medium"><?php echo $success_message; ?></p>
            </div>
        </div>
        <?php endif; ?>

        <?php if (isset($_GET['language_updated'])): ?>
        <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6">
            <div class="flex items-center">
                <svg class="h-5 w-5 text-blue-400 mr-3" viewBox="0 0 20 20" fill="currentColor">
                    <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd" />
                </svg>
                <p class="text-blue-800 font-medium">Dil ayarı güncellendi!</p>
            </div>
        </div>
        <?php endif; ?>

        <?php if (isset($error_message)): ?>
        <div class="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
            <div class="flex items-center">
                <svg class="h-5 w-5 text-red-400 mr-3" viewBox="0 0 20 20" fill="currentColor">
                    <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clip-rule="evenodd" />
                </svg>
                <p class="text-red-800 font-medium"><?php echo $error_message; ?></p>
            </div>
        </div>
        <?php endif; ?>

        <form method="POST" class="space-y-6">
            <input type="hidden" name="action" value="save_settings">
            
            <!-- Ayarlar -->
            <div class="bg-white rounded-lg shadow-sm p-6">
                <h2 class="text-lg font-semibold text-gray-900 mb-4">Bot Ayarları</h2>
                
                <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Kategori Başına Site Sayısı</label>
                        <input type="number" name="sites_per_category" value="<?php echo $sitesPerCategory; ?>" 
                               min="1" max="10"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                        <p class="text-xs text-gray-500 mt-1">Her kategori için kaç site ekleneceği</p>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Seçili Kategori Sayısı</label>
                        <div class="text-2xl font-bold text-blue-600" id="selected-count"><?php echo count($selectedCategories); ?></div>
                        <p class="text-xs text-gray-500 mt-1">Toplam seçili kategori</p>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Tahmini Toplam Site</label>
                        <div class="text-2xl font-bold text-green-600" id="estimated-total"><?php echo count($selectedCategories) * $sitesPerCategory; ?></div>
                        <p class="text-xs text-gray-500 mt-1">Eklenecek toplam site sayısı</p>
                    </div>
                </div>
            </div>

            <!-- Hızlı Seçim -->
            <div class="bg-white rounded-lg shadow-sm p-6">
                <h2 class="text-lg font-semibold text-gray-900 mb-4">Hızlı Seçim</h2>
                <div class="flex flex-wrap gap-3">
                    <button type="button" onclick="selectAll()" 
                            class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors">
                        🔥 Tümünü Seç
                    </button>
                    <button type="button" onclick="selectNone()" 
                            class="bg-gray-500 hover:bg-gray-600 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors">
                        ❌ Hiçbirini Seçme
                    </button>
                    <button type="button" onclick="selectPopular()" 
                            class="bg-purple-600 hover:bg-purple-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors">
                        ⭐ Popüler Kategoriler
                    </button>
                    <button type="button" onclick="selectBusiness()" 
                            class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors">
                        💼 İş Kategorileri
                    </button>
                </div>
            </div>

            <!-- Kategori Seçimi -->
            <div class="bg-white rounded-lg shadow-sm p-6">
                <h2 class="text-lg font-semibold text-gray-900 mb-4">Kategori Seçimi</h2>
                
                <?php if (empty($categories)): ?>
                <p class="text-gray-500">Henüz kategori tanımlanmamış.</p>
                <?php else: ?>
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                    <?php foreach ($categories as $category): ?>
                    <label class="flex items-center p-4 border-2 border-gray-200 rounded-lg hover:border-blue-300 transition-colors cursor-pointer category-item"
                           data-category-id="<?php echo $category['id']; ?>">
                        <input type="checkbox" name="categories[]" value="<?php echo $category['id']; ?>" 
                               <?php echo in_array($category['id'], $selectedCategories) ? 'checked' : ''; ?>
                               class="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500 category-checkbox">
                        <div class="ml-3 flex-grow">
                            <p class="text-sm font-medium text-gray-900"><?php safeEcho($category['name']); ?></p>
                            <p class="text-xs text-gray-500">ID: <?php echo $category['id']; ?></p>
                        </div>
                    </label>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </div>

            <!-- Aksiyon Butonları -->
            <div class="bg-white rounded-lg shadow-sm p-6">
                <div class="flex flex-wrap gap-4 justify-center">
                    <button type="submit" name="action" value="save_settings"
                            class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-3 rounded-lg font-medium transition-colors">
                        💾 Ayarları Kaydet
                    </button>
                    
                    <?php if (!empty($selectedCategories)): ?>
                    <button type="submit" name="action" value="run_bot"
                            class="bg-green-600 hover:bg-green-700 text-white px-6 py-3 rounded-lg font-medium transition-colors">
                        🚀 Bot'u Çalıştır (<?php echo count($selectedCategories); ?> Kategori)
                    </button>
                    <?php endif; ?>
                    
                    <a href="logs.php" 
                       class="bg-gray-600 hover:bg-gray-700 text-white px-6 py-3 rounded-lg font-medium transition-colors">
                        📋 Bot Logları
                    </a>
                </div>
            </div>
        </form>
    </div>

    <script>
        // Dil değiştirme fonksiyonu
        function setLanguage(language) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.style.display = 'none';
            
            const input = document.createElement('input');
            input.type = 'hidden';
            input.name = 'language';
            input.value = language;
            
            const actionInput = document.createElement('input');
            actionInput.type = 'hidden';
            actionInput.name = 'set_language';
            actionInput.value = '1';
            
            form.appendChild(input);
            form.appendChild(actionInput);
            document.body.appendChild(form);
            form.submit();
        }
        
        // Popular categories (most common ones)
        const popularCategories = [9, 8, 14, 6, 18, 3, 11]; // Tech, Business, Education, Health, Shopping, Tourism, Food
        
        // Business related categories
        const businessCategories = [8, 9, 10, 12, 5]; // Business, Tech, Law, Real Estate, Industry
        
        function selectAll() {
            document.querySelectorAll('.category-checkbox').forEach(checkbox => {
                checkbox.checked = true;
            });
            updateCounters();
            updateItemStyles();
        }
        
        function selectNone() {
            document.querySelectorAll('.category-checkbox').forEach(checkbox => {
                checkbox.checked = false;
            });
            updateCounters();
            updateItemStyles();
        }
        
        function selectPopular() {
            selectNone();
            popularCategories.forEach(catId => {
                const checkbox = document.querySelector(`input[value="${catId}"]`);
                if (checkbox) {
                    checkbox.checked = true;
                }
            });
            updateCounters();
            updateItemStyles();
        }
        
        function selectBusiness() {
            selectNone();
            businessCategories.forEach(catId => {
                const checkbox = document.querySelector(`input[value="${catId}"]`);
                if (checkbox) {
                    checkbox.checked = true;
                }
            });
            updateCounters();
            updateItemStyles();
        }
        
        function updateCounters() {
            const checkedCount = document.querySelectorAll('.category-checkbox:checked').length;
            const sitesPerCategory = parseInt(document.querySelector('input[name="sites_per_category"]').value) || 3;
            
            document.getElementById('selected-count').textContent = checkedCount;
            document.getElementById('estimated-total').textContent = checkedCount * sitesPerCategory;
        }
        
        function updateItemStyles() {
            document.querySelectorAll('.category-item').forEach(item => {
                const checkbox = item.querySelector('.category-checkbox');
                if (checkbox.checked) {
                    item.classList.add('border-blue-500', 'bg-blue-50');
                    item.classList.remove('border-gray-200');
                } else {
                    item.classList.remove('border-blue-500', 'bg-blue-50');
                    item.classList.add('border-gray-200');
                }
            });
        }
        
        // Event listeners
        document.addEventListener('DOMContentLoaded', function() {
            // İlk yüklemede stilleri güncelle
            updateItemStyles();
            
            // Checkbox değişikliklerini dinle
            document.querySelectorAll('.category-checkbox').forEach(checkbox => {
                checkbox.addEventListener('change', function() {
                    updateCounters();
                    updateItemStyles();
                });
            });
            
            // Sites per category değişikliğini dinle
            document.querySelector('input[name="sites_per_category"]').addEventListener('input', function() {
                updateCounters();
            });
        });
    </script>
</body>
</html>