<?php
class GoogleSearchBot {
    private $pdo;
    private $apiKey;
    private $searchEngineId;
    private $analyzer;
    private $contentLanguage;
    
    public function __construct($pdo, $apiKey, $searchEngineId, $analyzer) {
        $this->pdo = $pdo;
        $this->apiKey = $apiKey;
        $this->searchEngineId = $searchEngineId;
        $this->analyzer = $analyzer;
        $this->contentLanguage = $this->getContentLanguage();
    }
    
    /**
     * İçerik dilini al
     */
    private function getContentLanguage() {
        try {
            $stmt = $this->pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = 'bot_content_language'");
            $stmt->execute();
            $language = $stmt->fetchColumn();
            return $language ?: 'tr'; // Varsayılan Türkçe
        } catch (Exception $e) {
            return 'tr'; // Hata durumunda Türkçe
        }
    }
    
    /**
     * Kategori bazlı site arama
     */
    public function searchSitesByCategory($categoryId, $count = 5) {
        $sites = [];
        $keywords = $this->getCategoryKeywords($categoryId);
        
        // Seçilen dile göre anahtar kelimeleri kullan
        if ($this->contentLanguage === 'en') {
            $allKeywords = $keywords['english'];
        } else {
            $allKeywords = $keywords['turkish'];
        }
        
        // Rastgele arama terimleri ekle (dil desteği ile)
        $randomTerms = $this->getRandomSearchTerms($categoryId);
        $allKeywords = array_merge($allKeywords, $randomTerms);
        
        shuffle($allKeywords);
        
        // API ile arama - Genişletilmiş versiyon
        if (!empty($this->apiKey) && !empty($this->searchEngineId)) {
            // Daha fazla arama terimi kullan
            $searchTerms = array_slice($allKeywords, 0, min(15, count($allKeywords)));
            
            foreach ($searchTerms as $keyword) {
                // Her arama terimi için farklı sayfa numaraları dene
                $startIndexes = [1, 11, 21, 31, 41]; // Farklı sayfa numaraları
                shuffle($startIndexes);
                
                foreach (array_slice($startIndexes, 0, 2) as $startIndex) {
                    $results = $this->performGoogleSearch($keyword, min(5, $count), $startIndex);
                    if (!empty($results)) {
                        $sites = array_merge($sites, $results);
                    }
                    
                    if (count($sites) >= $count * 2) { // Daha fazla sonuç al
                        break 2;
                    }
                    
                    usleep(300000); // Daha hızlı rate limiting
                }
                
                if (count($sites) >= $count * 2) {
                    break;
                }
            }
        }
        
        // Yetersizse fallback sitelerden ekle
        if (count($sites) < $count) {
            $fallbackSites = $this->getCategoryFallbackSites($categoryId, $count - count($sites));
            $sites = array_merge($sites, $fallbackSites);
        }
        
        // Duplicates kaldır ve limit uygula - Gelişmiş versiyon
        $uniqueSites = [];
        $seenDomains = [];
        $seenTitles = [];
        
        foreach ($sites as $site) {
            $domain = parse_url($site['url'], PHP_URL_HOST);
            $title = strtolower(trim($site['title'] ?? ''));
            
            // Domain ve title duplicate kontrolü
            if (!in_array($domain, $seenDomains) && !in_array($title, $seenTitles) && $this->isValidSite($site['url'])) {
                $uniqueSites[] = $site;
                $seenDomains[] = $domain;
                $seenTitles[] = $title;
                
                if (count($uniqueSites) >= $count) {
                    break;
                }
            }
        }
        
        // Eğer yeterli unique site yoksa, daha esnek duplicate kontrolü yap
        if (count($uniqueSites) < $count) {
            $flexibleSites = [];
            $seenDomainsFlexible = [];
            
            foreach ($sites as $site) {
                $domain = parse_url($site['url'], PHP_URL_HOST);
                
                // Sadece domain kontrolü (daha esnek)
                if (!in_array($domain, $seenDomainsFlexible) && $this->isValidSite($site['url'])) {
                    $flexibleSites[] = $site;
                    $seenDomainsFlexible[] = $domain;
                    
                    if (count($flexibleSites) >= $count) {
                        break;
                    }
                }
            }
            
            $uniqueSites = $flexibleSites;
        }
        
        return $uniqueSites;
    }
    
    /**
     * Kategoriye özgü fallback siteler getir
     */
    private function getCategoryFallbackSites($categoryId, $count) {
        $allFallbackSites = $this->getFallbackSites(100); // Tüm siteleri al
        
        // Kategoriye göre filtrele
        $categorySites = array_filter($allFallbackSites, function($site) use ($categoryId) {
            return isset($site['category_id']) && $site['category_id'] == $categoryId;
        });
        
        // Eğer kategoriye özgü site yoksa genel sitelerden al
        if (empty($categorySites)) {
            $categorySites = $allFallbackSites;
        }
        
        shuffle($categorySites);
        return array_slice($categorySites, 0, $count);
    }
    
    /**
     * Tüm kategorilere site ekle
     */
    public function populateAllCategories($sitesPerCategory = 3) {
        $categories = range(1, 19); // 19 kategori
        $results = [];
        
        foreach ($categories as $categoryId) {
            $this->log("Kategori $categoryId için siteler aranıyor...");
            
            $sites = $this->searchSitesByCategory($categoryId, $sitesPerCategory);
            $successCount = 0;
            $errorCount = 0;
            
            foreach ($sites as $siteData) {
                $result = $this->processSiteWithCategory($siteData, $categoryId);
                
                if ($result['success']) {
                    $successCount++;
                    $this->log("✅ Kategori $categoryId - " . $result['site_name'], 'success');
                } else {
                    $errorCount++;
                    $this->log("❌ Kategori $categoryId - " . $siteData['url'] . " - " . $result['error'], 'error');
                }
                
                sleep(1); // Rate limiting
            }
            
            $results[$categoryId] = [
                'success' => $successCount,
                'errors' => $errorCount
            ];
            
            // Kategoriler arası bekleme
            sleep(2);
        }
        
        return $results;
    }
    
    /**
     * Siteyi belirli kategoriye ekler
     */
    public function processSiteWithCategory($siteData, $forceCategoryId = null) {
        try {
            // Site bilgilerini analiz et
            $siteInfo = $this->analyzer->analyzeSite($siteData['url']);
            
            if (!$siteInfo['success']) {
                return [
                    'success' => false,
                    'error' => 'Site analizi başarısız: ' . $siteInfo['error']
                ];
            }
            
            // AI ile içerik üret
            $content = $this->analyzer->generateContent($siteInfo);
            
            // Logo çek
            $logoUrl = $this->analyzer->extractLogo($siteInfo['url']);
            error_log("GoogleSearchBot: Logo extracted: " . ($logoUrl ?: 'No logo found'));
            
            // Kategori seç (zorla verilmişse onu kullan, yoksa akıllı seçim)
            $categoryId = $forceCategoryId ?: $this->analyzer->getSmartCategory($siteInfo);
            
            // Aynı URL zaten var mı kontrol et
            $checkStmt = $this->pdo->prepare("SELECT id FROM site_submissions WHERE site_url = ?");
            $checkStmt->execute([$siteInfo['url']]);
            if ($checkStmt->fetch()) {
                error_log("GoogleSearchBot: URL already exists, skipping: " . $siteInfo['url']);
                return false; // URL zaten mevcut, ekleme
            }
            
            // Veritabanına ekle
            $stmt = $this->pdo->prepare("
                INSERT INTO site_submissions (
                    site_name, site_url, contact_mail, short_description, 
                    long_description, category_id, logo_url, status, submitted_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, 'approved', NOW())
            ");
            
            $success = $stmt->execute([
                $siteInfo['title'],
                $siteInfo['url'],
                $siteInfo['email'],
                $content['short_description'],
                $content['long_description'],
                $categoryId,
                $logoUrl ?? '' // Çekilen logo URL'i
            ]);
            
            if ($success) {
                $siteId = $this->pdo->lastInsertId();
                
                // Eski site_logo kolonunu da doldur (geriye uyumluluk için)
                if (!empty($siteInfo['logo'])) {
                    $this->pdo->prepare("UPDATE site_submissions SET site_logo = ? WHERE id = ?")
                           ->execute([$siteInfo['logo'], $siteId]);
                }
                
                return [
                    'success' => true,
                    'site_id' => $siteId,
                    'site_name' => $siteInfo['title'],
                    'url' => $siteInfo['url'],
                    'category_id' => $categoryId
                ];
            } else {
                return [
                    'success' => false,
                    'error' => 'Veritabanına ekleme hatası'
                ];
            }
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'İşlem hatası: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Kategori bazlı arama anahtar kelimeleri - Genişletilmiş versiyon
     */
    private function getCategoryKeywords($categoryId) {
        $categoryKeywords = [
            1 => [ // Foreign Language
                'turkish' => ['dil kursu', 'yabancı dil', 'çeviri hizmeti', 'dil öğrenme', 'İngilizce kursu', 'Almanca kursu', 'tercüman', 'dil eğitimi', 'konuşma kursu', 'Fransızca öğren', 'İspanyolca kurs', 'Japonca eğitim', 'Çince öğrenme', 'Rusça kursu', 'Arapça eğitim', 'dil okulu', 'online dil kursu', 'dil sertifikası', 'TOEFL hazırlık', 'IELTS kursu'],
                'english' => ['language course', 'foreign language learning', 'translation services', 'language school', 'English tutoring', 'language exchange', 'interpreter services', 'linguistic training', 'French learning', 'Spanish course', 'German lessons', 'Italian language', 'Portuguese classes', 'Chinese learning', 'Japanese course', 'Korean language', 'Russian lessons', 'Arabic learning', 'online language course', 'language certification', 'TOEFL preparation', 'IELTS training', 'language app', 'bilingual education', 'multilingual services']
            ],
            2 => [ // Transport
                'turkish' => ['ulaşım hizmetleri', 'havayolu şirketi', 'otobüs firması', 'tren işletmesi', 'kargo şirketi', 'nakliye firması', 'bilet rezervasyon', 'lojistik', 'taksi hizmeti', 'metro sistemi', 'deniz yolu', 'karayolu taşımacılığı', 'hava kargo', 'express kargo', 'şehir içi ulaşım', 'intercity otobüs', 'yük taşımacılığı', 'gemi seferleri', 'feribot hizmeti'],
                'english' => ['transportation services', 'airline company', 'bus services', 'railway company', 'cargo shipping', 'freight services', 'ticket booking platform', 'logistics company', 'taxi service', 'metro system', 'shipping company', 'road transport', 'air cargo', 'express delivery', 'urban transport', 'intercity bus', 'freight transport', 'ferry service', 'cruise line', 'car rental', 'bike sharing', 'ride sharing', 'public transport', 'delivery service', 'courier service']
            ],
            3 => [ // Tourism & Travel
                'turkish' => ['turizm acentesi', 'seyahat rehberi', 'otel rezervasyon', 'tatil köyleri', 'gezi planlaması', 'tur operatörü', 'konaklama hizmetleri'],
                'english' => ['travel agency', 'tourism guide', 'hotel booking', 'vacation resorts', 'travel planning', 'tour operator', 'accommodation services', 'destination guide']
            ],
            4 => [ // Sports & Teams
                'turkish' => ['spor kulübü', 'fitness merkezi', 'spor ekipmanları', 'antrenman programı', 'futbol takımı', 'basketbol kulübü', 'spor akademisi', 'yüzme havuzu', 'tenis kulübü', 'golf sahası', 'voleybol takımı', 'hokey kulübü', 'boks salonu', 'dövüş sanatları', 'yoga stüdyosu', 'pilates merkezi', 'crossfit box', 'futbol okulu', 'basketbol okulu', 'spor mağazası', 'spor beslenme', 'spor fizyoterapisi', 'atletizm kulübü', 'koşu kulübü', 'bisiklet kulübü'],
                'english' => ['sports club', 'fitness center', 'sports equipment', 'training programs', 'football team', 'basketball club', 'sports academy', 'athletic training', 'swimming pool', 'tennis club', 'golf course', 'volleyball team', 'hockey club', 'boxing gym', 'martial arts', 'yoga studio', 'pilates center', 'crossfit box', 'football school', 'basketball school', 'sports store', 'sports nutrition', 'sports physiotherapy', 'NFL team', 'NBA team', 'MLB team', 'NHL team', 'soccer club', 'rugby team', 'cricket club', 'baseball team', 'swimming club', 'running club', 'cycling club', 'triathlon', 'marathon training', 'sports medicine', 'athletic performance', 'personal trainer', 'group fitness', 'weight training', 'cardio workout']
            ],
            5 => [ // Industry
                'turkish' => ['sanayi şirketi', 'imalat firması', 'inşaat şirketi', 'mühendislik hizmetleri', 'fabrika üretimi', 'endüstriyel çözümler'],
                'english' => ['industrial company', 'manufacturing firm', 'construction company', 'engineering services', 'factory production', 'industrial solutions', 'machinery supplier']
            ],
            6 => [ // Health & Life
                'turkish' => ['sağlık merkezi', 'tıp merkezi', 'hastane hizmetleri', 'doktor muayenehanesi', 'klinik', 'wellness merkezi', 'sağlıklı yaşam'],
                'english' => ['health center', 'medical clinic', 'hospital services', 'doctor practice', 'healthcare provider', 'wellness center', 'healthy living', 'medical care']
            ],
            7 => [ // Culture & Art
                'turkish' => ['sanat galerisi', 'müze', 'kültür merkezi', 'sanat atölyesi', 'müzik okulu', 'sanatsal etkinlik', 'kültürel aktivite'],
                'english' => ['art gallery', 'museum', 'cultural center', 'art studio', 'music school', 'artistic events', 'cultural activities', 'creative arts']
            ],
            8 => [ // Business
                'turkish' => ['işletme danışmanlığı', 'şirket hizmetleri', 'iş geliştirme', 'girişimcilik', 'ticari hizmetler', 'kurumsal çözümler'],
                'english' => ['business consulting', 'corporate services', 'business development', 'entrepreneurship', 'commercial services', 'enterprise solutions', 'business advisory']
            ],
            9 => [ // Internet & Tech
                'turkish' => ['teknoloji şirketi', 'yazılım geliştirme', 'web tasarım', 'bilgisayar hizmetleri', 'mobil uygulama', 'dijital çözümler'],
                'english' => ['technology company', 'software development', 'web design', 'computer services', 'mobile app development', 'digital solutions', 'IT services', 'tech startup']
            ],
            10 => [ // Law & Consulting
                'turkish' => ['hukuk bürosu', 'avukatlık hizmetleri', 'mali müşavirlik', 'hukuki danışmanlık', 'adalet hizmetleri', 'yasal danışmanlık'],
                'english' => ['law firm', 'legal services', 'financial consulting', 'legal advisory', 'attorney services', 'legal consultation', 'justice services']
            ],
            11 => [ // Food & Agri
                'turkish' => ['restoran', 'yemek hizmetleri', 'tarım işletmesi', 'çiftlik ürünleri', 'gıda üretimi', 'mutfak hizmetleri', 'organik tarım'],
                'english' => ['restaurant', 'food services', 'agricultural business', 'farm products', 'food production', 'catering services', 'organic farming', 'culinary services']
            ],
            12 => [ // Real Estate
                'turkish' => ['emlak ofisi', 'gayrimenkul hizmetleri', 'ev satışı', 'daire kiralama', 'inşaat projesi', 'konut gelişimi'],
                'english' => ['real estate agency', 'property services', 'house sales', 'apartment rental', 'construction project', 'residential development', 'property management']
            ],
            13 => [ // Entertainment
                'turkish' => ['eğlence merkezi', 'sinema', 'oyun merkezi', 'müzik prodüksiyonu', 'etkinlik organizasyonu', 'show hizmetleri'],
                'english' => ['entertainment center', 'cinema', 'gaming center', 'music production', 'event organization', 'show services', 'entertainment industry']
            ],
            14 => [ // Education
                'turkish' => ['eğitim kurumu', 'okul', 'üniversite', 'kurs merkezi', 'öğretim hizmetleri', 'akademik program', 'eğitim danışmanlığı'],
                'english' => ['educational institution', 'school', 'university', 'training center', 'teaching services', 'academic program', 'educational consulting']
            ],
            15 => [ // Religion & Ethics
                'turkish' => ['dini kurum', 'camii', 'kilise', 'din hizmetleri', 'manevi danışmanlık', 'ibadet yeri', 'dini eğitim'],
                'english' => ['religious institution', 'mosque', 'church', 'religious services', 'spiritual counseling', 'place of worship', 'religious education']
            ],
            16 => [ // Government
                'turkish' => ['devlet kurumu', 'belediye hizmetleri', 'kamu hizmetleri', 'resmi kurum', 'hükümet ajansı', 'kamu yönetimi'],
                'english' => ['government agency', 'municipal services', 'public services', 'official institution', 'government department', 'public administration']
            ],
            17 => [ // Press & Media
                'turkish' => ['haber ajansı', 'gazete', 'medya şirketi', 'basın kuruluşu', 'televizyon kanalı', 'radyo istasyonu', 'habercilik'],
                'english' => ['news agency', 'newspaper', 'media company', 'press organization', 'television channel', 'radio station', 'journalism', 'broadcasting']
            ],
            18 => [ // Shopping
                'turkish' => ['alışveriş merkezi', 'online mağaza', 'e-ticaret sitesi', 'satış mağazası', 'perakende satış', 'ürün satışı'],
                'english' => ['shopping center', 'online store', 'e-commerce site', 'retail store', 'retail sales', 'product sales', 'marketplace', 'shopping platform']
            ],
            19 => [ // Other Websites
                'turkish' => ['çeşitli hizmetler', 'genel bilgi', 'karma içerik', 'farklı kategoriler'],
                'english' => ['various services', 'general information', 'mixed content', 'different categories', 'miscellaneous']
            ]
        ];
        
        return $categoryKeywords[$categoryId] ?? $categoryKeywords[19];
    }
    
    /**
     * Kategoriye özgü rastgele arama terimleri
     */
    private function getRandomSearchTerms($categoryId) {
        $randomTerms = [
            1 => [
                'english' => ['language learning app', 'online translator', 'language exchange', 'bilingual website', 'multilingual service', 'language certification', 'TOEFL preparation', 'IELTS training', 'language school', 'conversation practice'],
                'turkish' => ['dil öğrenme uygulaması', 'çeviri sitesi', 'dil değişimi', 'çok dilli hizmet', 'dil sertifikası', 'TOEFL hazırlık', 'IELTS kursu', 'dil okulu', 'konuşma pratiği', 'dil kursu']
            ],
            2 => [
                'english' => ['transportation app', 'delivery service', 'shipping company', 'logistics platform', 'transport booking', 'freight services', 'cargo tracking', 'express delivery', 'logistics management'],
                'turkish' => ['ulaşım uygulaması', 'kargo hizmeti', 'nakliye platformu', 'lojistik şirketi', 'ulaşım rezervasyonu', 'yük taşımacılığı', 'kargo takibi', 'hızlı teslimat', 'lojistik yönetimi']
            ],
            3 => [
                'english' => ['travel booking', 'hotel reservation', 'vacation rental', 'tour guide', 'travel blog', 'destination guide', 'travel planning', 'adventure tourism', 'luxury travel'],
                'turkish' => ['seyahat rezervasyonu', 'otel rezervasyonu', 'tatil kiralama', 'seyahat blogu', 'hedef rehberi', 'seyahat planlaması', 'macera turizmi', 'lüks seyahat', 'tur operatörü']
            ],
            4 => [
                'english' => ['sports news', 'fitness app', 'athletic training', 'sports equipment', 'team management', 'sports academy', 'personal trainer', 'gym membership', 'sports nutrition'],
                'turkish' => ['spor haberleri', 'fitness uygulaması', 'atletik antrenman', 'takım yönetimi', 'spor akademisi', 'kişisel antrenör', 'spor salonu üyeliği', 'spor beslenme', 'futbol okulu']
            ],
            5 => [
                'english' => ['industrial automation', 'manufacturing software', 'construction management', 'engineering consulting', 'factory equipment', 'production line', 'quality control', 'industrial design'],
                'turkish' => ['endüstriyel otomasyon', 'imalat yazılımı', 'inşaat yönetimi', 'mühendislik danışmanlığı', 'fabrika ekipmanı', 'üretim hattı', 'kalite kontrol', 'endüstriyel tasarım']
            ],
            6 => [
                'english' => ['health monitoring', 'medical app', 'wellness program', 'fitness tracker', 'health insurance', 'medical care', 'healthcare provider', 'wellness center'],
                'turkish' => ['sağlık takibi', 'tıbbi uygulama', 'wellness programı', 'sağlık sigortası', 'tıbbi bakım', 'sağlık hizmeti sağlayıcısı', 'wellness merkezi', 'hastane hizmetleri']
            ],
            7 => [
                'english' => ['art gallery', 'cultural center', 'museum guide', 'art education', 'creative workshop', 'art exhibition', 'cultural heritage', 'art auction'],
                'turkish' => ['sanat galerisi', 'kültür merkezi', 'müze rehberi', 'sanat eğitimi', 'yaratıcı atölye', 'sanat sergisi', 'kültürel miras', 'sanat müzayedesı']
            ],
            8 => [
                'english' => ['business software', 'corporate training', 'management consulting', 'startup accelerator', 'business intelligence', 'business advisory', 'corporate services'],
                'turkish' => ['iş yazılımı', 'kurumsal eğitim', 'yönetim danışmanlığı', 'startup hızlandırıcı', 'iş zekası', 'iş danışmanlığı', 'kurumsal hizmetler']
            ],
            9 => [
                'english' => ['software development', 'tech startup', 'digital marketing', 'cloud computing', 'data analytics', 'artificial intelligence', 'machine learning', 'cybersecurity'],
                'turkish' => ['yazılım geliştirme', 'teknoloji startup', 'dijital pazarlama', 'bulut bilişim', 'veri analizi', 'yapay zeka', 'makine öğrenmesi', 'siber güvenlik']
            ],
            10 => [
                'english' => ['legal advice', 'law firm', 'legal document', 'court filing', 'legal research', 'attorney services', 'legal consultation', 'justice services'],
                'turkish' => ['hukuki danışmanlık', 'hukuk bürosu', 'hukuki belge', 'mahkeme dosyası', 'hukuki araştırma', 'avukatlık hizmetleri', 'hukuki danışmanlık', 'adalet hizmetleri']
            ],
            11 => [
                'english' => ['restaurant booking', 'food delivery', 'recipe app', 'cooking class', 'food blog', 'culinary services', 'organic food', 'local produce'],
                'turkish' => ['restoran rezervasyonu', 'yemek siparişi', 'tarif uygulaması', 'yemek blogu', 'mutfak hizmetleri', 'organik gıda', 'yerel ürün', 'tarım işletmesi']
            ],
            12 => [
                'english' => ['property listing', 'real estate app', 'home buying', 'rental property', 'property management', 'real estate agent', 'property investment'],
                'turkish' => ['emlak ilanı', 'emlak uygulaması', 'ev satın alma', 'kiralık emlak', 'emlak yönetimi', 'emlak acentesi', 'emlak yatırımı']
            ],
            13 => [
                'english' => ['movie streaming', 'entertainment news', 'gaming platform', 'music streaming', 'show booking', 'entertainment industry', 'live streaming'],
                'turkish' => ['film streaming', 'eğlence haberleri', 'oyun platformu', 'müzik streaming', 'show rezervasyonu', 'eğlence endüstrisi', 'canlı yayın']
            ],
            14 => [
                'english' => ['online course', 'educational app', 'learning platform', 'academic resource', 'study tool', 'educational institution', 'training center'],
                'turkish' => ['online kurs', 'eğitim uygulaması', 'öğrenme platformu', 'akademik kaynak', 'eğitim kurumu', 'kurs merkezi', 'öğretim hizmetleri']
            ],
            15 => [
                'english' => ['religious service', 'spiritual guidance', 'faith community', 'religious education', 'worship service', 'religious institution'],
                'turkish' => ['dini hizmet', 'manevi rehberlik', 'inanç topluluğu', 'dini eğitim', 'ibadet hizmeti', 'dini kurum']
            ],
            16 => [
                'english' => ['government service', 'public information', 'official website', 'citizen portal', 'public record', 'government agency'],
                'turkish' => ['devlet hizmeti', 'kamu bilgisi', 'resmi website', 'vatandaş portalı', 'kamu kaydı', 'devlet kurumu']
            ],
            17 => [
                'english' => ['news website', 'media outlet', 'journalism platform', 'news aggregator', 'press release', 'broadcasting'],
                'turkish' => ['haber sitesi', 'medya kuruluşu', 'gazetecilik platformu', 'haber toplayıcı', 'basın açıklaması', 'yayıncılık']
            ],
            18 => [
                'english' => ['online shopping', 'e-commerce platform', 'retail store', 'shopping app', 'marketplace', 'shopping center'],
                'turkish' => ['online alışveriş', 'e-ticaret platformu', 'perakende mağaza', 'alışveriş uygulaması', 'alışveriş merkezi', 'ürün satışı']
            ],
            19 => [
                'english' => ['general service', 'utility app', 'productivity tool', 'lifestyle app', 'general information', 'miscellaneous'],
                'turkish' => ['genel hizmet', 'yardımcı uygulama', 'verimlilik aracı', 'yaşam tarzı uygulaması', 'genel bilgi', 'çeşitli hizmetler']
            ]
        ];
        
        $categoryTerms = $randomTerms[$categoryId] ?? $randomTerms[19];
        
        // Seçilen dile göre terimleri döndür
        if ($this->contentLanguage === 'en') {
            return $categoryTerms['english'] ?? [];
        } else {
            return $categoryTerms['turkish'] ?? [];
        }
    }

    /**
     * Rastgele web siteleri arar
     */
    public function searchRandomSites($keywords, $count = 10) {
        $sites = [];
        
        // Her durumda en az fallback siteler döndürülsün
        if (empty($this->apiKey) || empty($this->searchEngineId)) {
            return $this->getFallbackSites($count);
        }
        
        // Dil desteği ile anahtar kelimeleri filtrele
        $keywordList = array_map('trim', explode(',', $keywords));
        $filteredKeywords = $this->filterKeywordsByLanguage($keywordList);
        
        // Log: Dil filtreleme sonucu
        error_log("Dil Filtreleme - Seçilen dil: " . $this->contentLanguage);
        error_log("Dil Filtreleme - Toplam anahtar kelime: " . count($keywordList));
        error_log("Dil Filtreleme - Filtrelenmiş anahtar kelime: " . count($filteredKeywords));
        error_log("Dil Filtreleme - İlk 5 filtrelenmiş kelime: " . implode(', ', array_slice($filteredKeywords, 0, 5)));
        
        if (empty($filteredKeywords)) {
            // Eğer filtrelenmiş anahtar kelime yoksa, tüm anahtar kelimeleri kullan
            $filteredKeywords = $keywordList;
            error_log("Dil Filtreleme - Filtrelenmiş kelime yok, tüm kelimeler kullanılıyor");
        }
        
        for ($i = 0; $i < $count; $i++) {
            $randomKeyword = $filteredKeywords[array_rand($filteredKeywords)];
            $results = $this->performGoogleSearch($randomKeyword, 1, mt_rand(1, 10));
            
            if (!empty($results)) {
                $sites = array_merge($sites, $results);
            }
            
            // Rate limiting
            usleep(500000); // 0.5 saniye bekle
        }
        
        // Eğer API'den hiç site gelmezse fallback siteler döndür
        if (empty($sites)) {
            return $this->getFallbackSites($count);
        }
        
        // Duplicates kaldır
        $uniqueSites = [];
        $seenDomains = [];
        
        foreach ($sites as $site) {
            $domain = parse_url($site['url'], PHP_URL_HOST);
            if (!in_array($domain, $seenDomains)) {
                $uniqueSites[] = $site;
                $seenDomains[] = $domain;
            }
        }
        
        // Eğer unique site sayısı az ise fallback sitelerle tamamla
        if (count($uniqueSites) < $count) {
            $fallbackSites = $this->getFallbackSites($count - count($uniqueSites));
            $uniqueSites = array_merge($uniqueSites, $fallbackSites);
        }
        
        return array_slice($uniqueSites, 0, $count);
    }
    
    /**
     * Anahtar kelimeleri dile göre filtrele
     */
    private function filterKeywordsByLanguage($keywords) {
        // Dil bazlı anahtar kelime listeleri
        $turkishKeywords = [
            'eğitim', 'kurs', 'öğrenme', 'ders', 'okul', 'üniversite', 'akademi', 'eğitim merkezi',
            'sağlık', 'hastane', 'klinik', 'doktor', 'eczane', 'tedavi', 'terapi', 'fitness',
            'teknoloji', 'yazılım', 'bilgisayar', 'internet', 'web tasarım', 'mobil uygulama',
            'iş', 'şirket', 'kurumsal', 'danışmanlık', 'muhasebe', 'hukuk', 'finans',
            'eğlence', 'sinema', 'müzik', 'oyun', 'spor', 'futbol', 'basketbol',
            'seyahat', 'otel', 'tatil', 'turizm', 'rehber', 'uçak bileti', 'araç kiralama',
            'alışveriş', 'mağaza', 'e-ticaret', 'moda', 'elektronik', 'kitap',
            'yemek', 'restoran', 'kafe', 'tarif', 'mutfak', 'gıda',
            'emlak', 'ev', 'daire', 'satış', 'kiralık', 'gayrimenkul',
            'haber', 'gazete', 'medya', 'blog', 'forum', 'topluluk',
            'dil öğrenme', 'türkçe ders', 'ingilizce kurs', 'fransızca', 'almanca',
            'hava yolu', 'otobüs', 'tren', 'kargo', 'nakliye', 'teslimat',
            'tatil köyü', 'dağ evi', 'şehir turu', 'kültür turu', 'yemek turu',
            'futbol takımı', 'basketbol kulübü', 'tenis akademisi', 'yüzme havuzu',
            'endüstri', 'üretim', 'mühendislik', 'fabrika', 'kalite kontrol',
            'sağlık hizmeti', 'wellness merkezi', 'beslenme', 'mental sağlık',
            'müze', 'sanat galerisi', 'kültür merkezi', 'müzik mekanı', 'tiyatro',
            'kurumsal web sitesi', 'profesyonel hizmet', 'startup', 'iş dizini',
            'yazılım geliştirme', 'IT hizmeti', 'web tasarım', 'mobil uygulama',
            'hukuk bürosu', 'avukat', 'hukuki danışmanlık', 'iş hukuku',
            'restoran rezervasyon', 'yemek siparişi', 'tarım şirketi', 'organik gıda',
            'emlak ilanı', 'emlak acentesi', 'ev satışı', 'daire kiralama',
            'sinema', 'oyun platformu', 'müzik streaming', 'konser mekanı',
            'online kurs', 'üniversite', 'öğrenme platformu', 'eğitim kaynağı'
        ];
        
        $englishKeywords = [
            'education', 'course', 'learning', 'school', 'university', 'academy', 'training',
            'health', 'hospital', 'clinic', 'doctor', 'pharmacy', 'treatment', 'therapy', 'fitness',
            'technology', 'software', 'computer', 'internet', 'web design', 'mobile app',
            'business', 'company', 'corporate', 'consulting', 'accounting', 'law', 'finance',
            'entertainment', 'movie', 'music', 'game', 'sports', 'football', 'basketball',
            'travel', 'hotel', 'vacation', 'tourism', 'guide', 'flight booking', 'car rental',
            'shopping', 'store', 'e-commerce', 'fashion', 'electronics', 'book',
            'food', 'restaurant', 'cafe', 'recipe', 'kitchen', 'grocery',
            'real estate', 'house', 'apartment', 'sale', 'rental', 'property',
            'news', 'newspaper', 'media', 'blog', 'forum', 'community',
            'language learning', 'english course', 'spanish lessons', 'french tutor', 'german class',
            'airline booking', 'bus schedule', 'train tickets', 'shipping company', 'logistics',
            'beach resort', 'mountain lodge', 'city tour', 'cultural tour', 'food tour',
            'football team', 'basketball club', 'tennis academy', 'swimming pool', 'gym',
            'manufacturing', 'industrial equipment', 'engineering firm', 'factory automation',
            'healthcare provider', 'wellness center', 'nutrition advice', 'mental health',
            'museum exhibition', 'art gallery', 'cultural center', 'music venue', 'theater',
            'corporate website', 'professional service', 'startup company', 'business directory',
            'software development', 'IT service', 'web design', 'mobile app', 'cloud computing',
            'law firm', 'legal service', 'attorney', 'business law', 'legal advice',
            'restaurant booking', 'food delivery', 'agricultural company', 'organic food',
            'property listing', 'real estate agent', 'home buying', 'apartment rental',
            'movie theater', 'gaming platform', 'music streaming', 'concert venue',
            'online course', 'university', 'learning platform', 'educational resource'
        ];
        
        // Seçilen dile göre anahtar kelime listesini kullan
        if ($this->contentLanguage === 'tr') {
            return $turkishKeywords;
        } elseif ($this->contentLanguage === 'en') {
            return $englishKeywords;
        } else {
            // Varsayılan olarak tüm anahtar kelimeleri kullan
            return $keywords;
        }
    }
    
    /**
     * Google Custom Search API çağrısı
     */
    private function performGoogleSearch($query, $num = 10, $start = 1) {
        // Search Engine ID formatını düzelt
        $searchEngineId = $this->searchEngineId;
        if (strpos($searchEngineId, 'cx=') === 0) {
            $searchEngineId = substr($searchEngineId, 3); // cx= kısmını kaldır
        }
        
        $url = "https://www.googleapis.com/customsearch/v1?" . http_build_query([
            'key' => $this->apiKey,
            'cx' => $searchEngineId,
            'q' => $query,
            'num' => min($num, 10),
            'start' => $start,
            'safe' => 'off'
        ]);
        
        try {
            $context = stream_context_create([
                'http' => [
                    'timeout' => 30,
                    'user_agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
                ]
            ]);
            
            $response = @file_get_contents($url, false, $context);
            if ($response === false) {
                // API hatası durumunda fallback siteler döndür
                return $this->getFallbackSites(5);
            }
            
            $data = json_decode($response, true);
            
            if (!isset($data['items'])) {
                // API'de veri yoksa fallback siteler döndür
                return $this->getFallbackSites(5);
            }
            
            $results = [];
            foreach ($data['items'] as $item) {
                // Spam ve zararlı siteleri filtrele
                if ($this->isValidSite($item['link'])) {
                    $results[] = [
                        'url' => $item['link'],
                        'title' => $item['title'] ?? '',
                        'snippet' => $item['snippet'] ?? ''
                    ];
                }
            }
            
            // Eğer filtreleme sonrası hiç site kalmadıysa fallback döndür
            if (empty($results)) {
                return $this->getFallbackSites(3);
            }
            
            return $results;
            
        } catch (Exception $e) {
            // Hata durumunda fallback siteler döndür
            return $this->getFallbackSites(5);
        }
    }
    
    /**
     * Site geçerlilik kontrolü
     */
    private function isValidSite($url) {
        $domain = parse_url($url, PHP_URL_HOST);
        
        // Kara liste
        $blacklist = [
            'facebook.com', 'twitter.com', 'instagram.com', 'youtube.com',
            'linkedin.com', 'pinterest.com', 'reddit.com', 'tiktok.com',
            'wikipedia.org', 'amazon.com', 'ebay.com', 'alibaba.com',
            'google.com', 'bing.com', 'yahoo.com', 'baidu.com',
            'porn', 'xxx', 'adult', 'casino', 'gambling', 'bet'
        ];
        
        foreach ($blacklist as $blocked) {
            if (strpos($domain, $blocked) !== false) {
                return false;
            }
        }
        
        // Sadece HTTP/HTTPS
        if (!preg_match('/^https?:\/\//', $url)) {
            return false;
        }
        
        // Daha önce eklenmiş mi kontrol et
        try {
            $stmt = $this->pdo->prepare("SELECT id FROM site_submissions WHERE site_url = ?");
            $stmt->execute([$url]);
            $exists = $stmt->fetchColumn() !== false;
            
            // Eğer site zaten varsa false döndür
            if ($exists) {
                return false;
            }
            
            return true;
        } catch (Exception $e) {
            // Veritabanı hatası durumunda site'ı geçerli say
            return true;
        }
    }
    
    /**
     * API olmadığında fallback siteler
     */
    private function getFallbackSites($count) {
        $fallbackSites = [
            // 1. Foreign Language
            ['url' => 'https://duolingo.com', 'title' => 'Duolingo', 'snippet' => 'Language learning platform with gamified lessons', 'category_id' => 1],
            ['url' => 'https://babbel.com', 'title' => 'Babbel', 'snippet' => 'Conversation-focused language learning', 'category_id' => 1],
            ['url' => 'https://lingoda.com', 'title' => 'Lingoda', 'snippet' => 'Online language school with certified teachers', 'category_id' => 1],
            ['url' => 'https://berlitz.com.tr', 'title' => 'Berlitz', 'snippet' => 'Dil eğitimi ve çeviri hizmetleri', 'category_id' => 1],
            
            // 2. Transport
            ['url' => 'https://flightaware.com', 'title' => 'FlightAware', 'snippet' => 'Flight tracking and aviation data', 'category_id' => 2],
            ['url' => 'https://rome2rio.com', 'title' => 'Rome2Rio', 'snippet' => 'Travel planning and booking platform', 'category_id' => 2],
            ['url' => 'https://turkishairlines.com', 'title' => 'Turkish Airlines', 'snippet' => 'Türkiye\'nin milli havayolu şirketi', 'category_id' => 2],
            ['url' => 'https://tcdd.gov.tr', 'title' => 'TCDD', 'snippet' => 'Türkiye Cumhuriyeti Devlet Demiryolları', 'category_id' => 2],
            
            // 3. Tourism & Travel
            ['url' => 'https://booking.com', 'title' => 'Booking.com', 'snippet' => 'Hotel and accommodation booking', 'category_id' => 3],
            ['url' => 'https://airbnb.com', 'title' => 'Airbnb', 'snippet' => 'Online marketplace for lodging', 'category_id' => 3],
            ['url' => 'https://tripadvisor.com', 'title' => 'TripAdvisor', 'snippet' => 'Travel reviews and booking', 'category_id' => 3],
            ['url' => 'https://etstur.com', 'title' => 'ETS Tur', 'snippet' => 'Tatil ve seyahat acentesi', 'category_id' => 3],
            ['url' => 'https://jollytur.com', 'title' => 'Jolly Tur', 'snippet' => 'Türkiye\'nin lider turizm şirketi', 'category_id' => 3],
            
            // 4. Sports & Teams
            ['url' => 'https://espn.com', 'title' => 'ESPN', 'snippet' => 'Sports news and live scores', 'category_id' => 4],
            ['url' => 'https://strava.com', 'title' => 'Strava', 'snippet' => 'Social fitness tracking app', 'category_id' => 4],
            ['url' => 'https://myfitnesspal.com', 'title' => 'MyFitnessPal', 'snippet' => 'Calorie counter and diet tracker', 'category_id' => 4],
            ['url' => 'https://galatasaray.org', 'title' => 'Galatasaray', 'snippet' => 'Galatasaray Spor Kulübü resmi sitesi', 'category_id' => 4],
            ['url' => 'https://fenerbahce.org', 'title' => 'Fenerbahçe', 'snippet' => 'Fenerbahçe Spor Kulübü resmi sitesi', 'category_id' => 4],
            
            // 5. Industry
            ['url' => 'https://caterpillar.com', 'title' => 'Caterpillar', 'snippet' => 'Industrial equipment manufacturer', 'category_id' => 5],
            ['url' => 'https://ge.com', 'title' => 'General Electric', 'snippet' => 'Industrial technology company', 'category_id' => 5],
            ['url' => 'https://tusas.com', 'title' => 'TUSAŞ', 'snippet' => 'Türk havacılık ve uzay sanayii', 'category_id' => 5],
            ['url' => 'https://aselsan.com.tr', 'title' => 'ASELSAN', 'snippet' => 'Savunma sanayii ve elektronik sistemler', 'category_id' => 5],
            
            // 6. Health & Life
            ['url' => 'https://webmd.com', 'title' => 'WebMD', 'snippet' => 'Medical information and health advice', 'category_id' => 6],
            ['url' => 'https://headspace.com', 'title' => 'Headspace', 'snippet' => 'Meditation and mindfulness app', 'category_id' => 6],
            ['url' => 'https://acibadem.com.tr', 'title' => 'Acıbadem', 'snippet' => 'Özel hastaneler grubu', 'category_id' => 6],
            ['url' => 'https://memorial.com.tr', 'title' => 'Memorial', 'snippet' => 'Sağlık hizmetleri ve hastaneler', 'category_id' => 6],
            
            // 7. Culture & Art
            ['url' => 'https://met.org', 'title' => 'Metropolitan Museum', 'snippet' => 'World-famous art museum', 'category_id' => 7],
            ['url' => 'https://louvre.fr', 'title' => 'Louvre Museum', 'snippet' => 'Iconic French art museum', 'category_id' => 7],
            ['url' => 'https://muze.gov.tr', 'title' => 'Türkiye Müzeleri', 'snippet' => 'Türkiye müzeleri ve kültür varlıkları', 'category_id' => 7],
            ['url' => 'https://istanbulmodern.org', 'title' => 'Istanbul Modern', 'snippet' => 'Çağdaş sanat müzesi', 'category_id' => 7],
            
            // 8. Business
            ['url' => 'https://linkedin.com', 'title' => 'LinkedIn', 'snippet' => 'Professional networking platform', 'category_id' => 8],
            ['url' => 'https://glassdoor.com', 'title' => 'Glassdoor', 'snippet' => 'Job search and company reviews', 'category_id' => 8],
            ['url' => 'https://forbes.com', 'title' => 'Forbes', 'snippet' => 'Business news and entrepreneur stories', 'category_id' => 8],
            ['url' => 'https://tobb.org.tr', 'title' => 'TOBB', 'snippet' => 'Türkiye Odalar ve Borsalar Birliği', 'category_id' => 8],
            
            // 9. Internet & Tech
            ['url' => 'https://github.com', 'title' => 'GitHub', 'snippet' => 'Development platform inspired by the way you work', 'category_id' => 9],
            ['url' => 'https://stackoverflow.com', 'title' => 'Stack Overflow', 'snippet' => 'Programming Q&A platform for developers', 'category_id' => 9],
            ['url' => 'https://techcrunch.com', 'title' => 'TechCrunch', 'snippet' => 'Technology news and startup information', 'category_id' => 9],
            ['url' => 'https://webrazzi.com', 'title' => 'Webrazzi', 'snippet' => 'Teknoloji ve girişim haberleri', 'category_id' => 9],
            ['url' => 'https://shiftdelete.net', 'title' => 'ShiftDelete', 'snippet' => 'Teknoloji haberleri ve incelemeleri', 'category_id' => 9],
            
            // 10. Law & Consulting
            ['url' => 'https://martindale.com', 'title' => 'Martindale-Hubbell', 'snippet' => 'Lawyer directory and legal resources', 'category_id' => 10],
            ['url' => 'https://lexisnexis.com', 'title' => 'LexisNexis', 'snippet' => 'Legal research and business intelligence', 'category_id' => 10],
            ['url' => 'https://barobirlik.org.tr', 'title' => 'Türkiye Barolar Birliği', 'snippet' => 'Avukatlar ve hukuki hizmetler', 'category_id' => 10],
            
            // 11. Food & Agri
            ['url' => 'https://yelp.com', 'title' => 'Yelp', 'snippet' => 'Local business reviews and ratings', 'category_id' => 11],
            ['url' => 'https://zomato.com', 'title' => 'Zomato', 'snippet' => 'Restaurant discovery and food delivery', 'category_id' => 11],
            ['url' => 'https://yemeksepeti.com', 'title' => 'Yemeksepeti', 'snippet' => 'Yemek siparişi ve restoran rehberi', 'category_id' => 11],
            ['url' => 'https://getir.com', 'title' => 'Getir', 'snippet' => 'Hızlı teslimat ve market alışverişi', 'category_id' => 11],
            
            // 12. Real Estate
            ['url' => 'https://zillow.com', 'title' => 'Zillow', 'snippet' => 'Real estate listings and home values', 'category_id' => 12],
            ['url' => 'https://realtor.com', 'title' => 'Realtor.com', 'snippet' => 'Property search and real estate listings', 'category_id' => 12],
            ['url' => 'https://sahibinden.com', 'title' => 'Sahibinden', 'snippet' => 'Emlak ilanları ve ikinci el eşya', 'category_id' => 12],
            ['url' => 'https://hurriyetemlak.com', 'title' => 'Hürriyet Emlak', 'snippet' => 'Emlak ilanları ve gayrimenkul', 'category_id' => 12],
            
            // 13. Entertainment
            ['url' => 'https://netflix.com', 'title' => 'Netflix', 'snippet' => 'Streaming entertainment service', 'category_id' => 13],
            ['url' => 'https://imdb.com', 'title' => 'IMDb', 'snippet' => 'Movie and TV show database', 'category_id' => 13],
            ['url' => 'https://twitch.tv', 'title' => 'Twitch', 'snippet' => 'Live streaming platform for gamers', 'category_id' => 13],
            ['url' => 'https://sinema.com', 'title' => 'Sinema.com', 'snippet' => 'Film haberleri ve sinema rehberi', 'category_id' => 13],
            
            // 14. Education
            ['url' => 'https://coursera.org', 'title' => 'Coursera', 'snippet' => 'Online courses from top universities', 'category_id' => 14],
            ['url' => 'https://udemy.com', 'title' => 'Udemy', 'snippet' => 'Online learning platform with courses', 'category_id' => 14],
            ['url' => 'https://khanacademy.org', 'title' => 'Khan Academy', 'snippet' => 'Free online courses and practice', 'category_id' => 14],
            ['url' => 'https://yok.gov.tr', 'title' => 'YÖK', 'snippet' => 'Yükseköğretim Kurulu', 'category_id' => 14],
            ['url' => 'https://meb.gov.tr', 'title' => 'MEB', 'snippet' => 'Milli Eğitim Bakanlığı', 'category_id' => 14],
            
            // 15. Religion & Ethics
            ['url' => 'https://diyanet.gov.tr', 'title' => 'Diyanet İşleri Başkanlığı', 'snippet' => 'Din işleri ve rehberlik hizmetleri', 'category_id' => 15],
            ['url' => 'https://vatican.va', 'title' => 'Vatican', 'snippet' => 'Official Vatican website', 'category_id' => 15],
            
            // 16. Government
            ['url' => 'https://gov.tr', 'title' => 'Türkiye Cumhuriyeti', 'snippet' => 'Resmi devlet portalı', 'category_id' => 16],
            ['url' => 'https://whitehouse.gov', 'title' => 'The White House', 'snippet' => 'Official US government website', 'category_id' => 16],
            ['url' => 'https://tbmm.gov.tr', 'title' => 'TBMM', 'snippet' => 'Türkiye Büyük Millet Meclisi', 'category_id' => 16],
            
            // 17. Press & Media
            ['url' => 'https://cnn.com', 'title' => 'CNN', 'snippet' => 'Breaking news and latest headlines', 'category_id' => 17],
            ['url' => 'https://bbc.com', 'title' => 'BBC', 'snippet' => 'World news and current affairs', 'category_id' => 17],
            ['url' => 'https://hurriyet.com.tr', 'title' => 'Hürriyet', 'snippet' => 'Güncel haberler ve son dakika', 'category_id' => 17],
            ['url' => 'https://sabah.com.tr', 'title' => 'Sabah', 'snippet' => 'Haber portalı ve güncel gelişmeler', 'category_id' => 17],
            
            // 18. Shopping
            ['url' => 'https://amazon.com', 'title' => 'Amazon', 'snippet' => 'Online marketplace and web services', 'category_id' => 18],
            ['url' => 'https://ebay.com', 'title' => 'eBay', 'snippet' => 'Online auction and marketplace', 'category_id' => 18],
            ['url' => 'https://trendyol.com', 'title' => 'Trendyol', 'snippet' => 'Online alışveriş ve moda', 'category_id' => 18],
            ['url' => 'https://hepsiburada.com', 'title' => 'Hepsiburada', 'snippet' => 'Teknoloji ve genel alışveriş', 'category_id' => 18],
            ['url' => 'https://n11.com', 'title' => 'N11', 'snippet' => 'Online alışveriş platformu', 'category_id' => 18],
            
            // 19. Other Websites
            ['url' => 'https://reddit.com', 'title' => 'Reddit', 'snippet' => 'Social news aggregation and discussion', 'category_id' => 19],
            ['url' => 'https://medium.com', 'title' => 'Medium', 'snippet' => 'Platform for writers and readers', 'category_id' => 19],
            ['url' => 'https://eksi.net', 'title' => 'Ekşi Sözlük', 'snippet' => 'Collaborative hypertext dictionary', 'category_id' => 19]
        ];
        
        // Daha önce eklenmemiş siteleri filtrele
        $availableSites = [];
        foreach ($fallbackSites as $site) {
            if ($this->isValidSite($site['url'])) {
                $availableSites[] = $site;
            }
        }
        
        // Eğer filtreleme sonrası hiç site kalmadıysa orjinal listeyi kullan
        if (empty($availableSites)) {
            $availableSites = $fallbackSites;
        }
        
        shuffle($availableSites);
        return array_slice($availableSites, 0, $count);
    }
    
    /**
     * Siteyi analiz edip veritabanına ekler
     */
    public function processSite($siteData) {
        try {
            error_log("GoogleSearchBot: Processing site: " . json_encode($siteData));
            
            // Site bilgilerini analiz et
            $siteInfo = $this->analyzer->analyzeSite($siteData['url']);
            
            if (!$siteInfo['success']) {
                error_log("GoogleSearchBot: Site analysis failed: " . $siteInfo['error']);
                return [
                    'success' => false,
                    'error' => 'Site analizi başarısız: ' . $siteInfo['error']
                ];
            }
            
            error_log("GoogleSearchBot: Site analysis successful, generating content...");
            
            // AI ile içerik üret
            $content = $this->analyzer->generateContent($siteInfo);
            
            if (!$content || !isset($content['short_description'])) {
                error_log("GoogleSearchBot: Content generation failed");
                return [
                    'success' => false,
                    'error' => 'İçerik üretimi başarısız'
                ];
            }
            
            error_log("GoogleSearchBot: Content generated, selecting category...");
            
            // Akıllı kategori seçimi
            $categoryId = $this->analyzer->getSmartCategory($siteInfo);
            
            error_log("GoogleSearchBot: Category selected: " . $categoryId . ", checking for duplicate URL...");
            
            // Aynı URL zaten var mı kontrol et
            $checkStmt = $this->pdo->prepare("SELECT id FROM site_submissions WHERE site_url = ?");
            $checkStmt->execute([$siteInfo['url']]);
            if ($checkStmt->fetch()) {
                error_log("GoogleSearchBot: URL already exists, skipping: " . $siteInfo['url']);
                return [
                    'success' => false,
                    'error' => 'Bu URL zaten mevcut'
                ];
            }
            
            error_log("GoogleSearchBot: URL is unique, extracting logo...");
            
            // Logo çek
            $logoUrl = $this->analyzer->extractLogo($siteInfo['url']);
            error_log("GoogleSearchBot: Logo extracted: " . ($logoUrl ?: 'No logo found'));
            
            error_log("GoogleSearchBot: Saving to database...");
            
            // Veritabanına ekle
            $stmt = $this->pdo->prepare("
                INSERT INTO site_submissions (
                    site_name, site_url, contact_mail, short_description, 
                    long_description, category_id, logo_url, status, submitted_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, 'approved', NOW())
            ");
            
            $success = $stmt->execute([
                $siteInfo['title'],
                $siteInfo['url'],
                $siteInfo['email'],
                $content['short_description'],
                $content['long_description'],
                $categoryId,
                $logoUrl ?? ''
            ]);
            
            if ($success) {
                $siteId = $this->pdo->lastInsertId();
                
                error_log("GoogleSearchBot: Site saved to database with ID: " . $siteId);
                
                // Logo varsa ekle
                if (!empty($siteInfo['logo'])) {
                    $this->saveSiteLogo($siteId, $siteInfo['logo']);
                }
                
                return [
                    'success' => true,
                    'site_id' => $siteId,
                    'site_name' => $siteInfo['title'],
                    'url' => $siteInfo['url']
                ];
            } else {
                error_log("GoogleSearchBot: Database insert failed");
                return [
                    'success' => false,
                    'error' => 'Veritabanına kaydetme hatası'
                ];
            }
            
        } catch (Exception $e) {
            error_log("GoogleSearchBot: Exception in processSite: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Site logosunu kaydeder
     */
    private function saveSiteLogo($siteId, $logoUrl) {
        try {
            // Logo URL'sini site ile ilişkilendir
            $stmt = $this->pdo->prepare("UPDATE site_submissions SET favicon_url = ? WHERE id = ?");
            $stmt->execute([$logoUrl, $siteId]);
        } catch (Exception $e) {
            // Logo kaydetme hatası önemli değil
        }
    }
    
    /**
     * Bot durumunu günceller
     */
    public function updateBotStatus($status, $lastRun = null, $totalSites = null) {
        try {
            $stmt = $this->pdo->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?");
            
            $stmt->execute(['bot_status', $status, $status]);
            
            if ($lastRun !== null) {
                $stmt->execute(['bot_last_run', $lastRun, $lastRun]);
            }
            
            if ($totalSites !== null) {
                $stmt->execute(['bot_total_sites', $totalSites, $totalSites]);
            }
            
        } catch (Exception $e) {
            // Status update hatası
        }
    }
    
    /**
     * Log kaydı
     */
    public function log($message, $type = 'info') {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO bot_logs (log_type, message, created_at) 
                VALUES (?, ?, NOW())
            ");
            $stmt->execute([$type, $message]);
        } catch (Exception $e) {
            // Log hatası önemli değil
        }
    }
}
?>