<?php
require_once '../includes/config.php';

// Log tablosunu oluştur (yoksa)
try {
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS bot_logs (
            id INT AUTO_INCREMENT PRIMARY KEY,
            log_type VARCHAR(20) DEFAULT 'info',
            message TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");
} catch (Exception $e) {
    // Tablo oluşturma hatası
}

// Logları al
$logs = [];
try {
    $stmt = $pdo->query("SELECT * FROM bot_logs ORDER BY created_at DESC LIMIT 100");
    $logs = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $error = "Loglar alınamadı: " . $e->getMessage();
}

// Son eklenen siteleri al
$recentSites = [];
try {
    $stmt = $pdo->query("SELECT site_name, site_url, category_id, olusturulma_tarihi, status FROM site_submissions ORDER BY olusturulma_tarihi DESC LIMIT 20");
    $recentSites = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    // Hata durumunda boş array
}

// Log dosyasını da kontrol et
$logFile = 'bot_logs.txt';
$fileLogs = [];
if (file_exists($logFile)) {
    $fileLogs = file($logFile, FILE_IGNORE_NEW_LINES);
    $fileLogs = array_reverse(array_slice($fileLogs, -50)); // Son 50 satır
}
?>
<!DOCTYPE html>
<html lang="tr" class="dark">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Bot Logları | AI Bot Kontrol Paneli</title>
    <link href="https://fonts.googleapis.com" rel="preconnect"/>
    <link crossorigin="" href="https://fonts.gstatic.com" rel="preconnect"/>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    fontFamily: {
                        'sans': ['Inter', 'ui-sans-serif', 'system-ui'],
                    },
                    colors: {
                        primary: '#6366f1',
                        secondary: '#8b5cf6',
                        success: '#10b981',
                        danger: '#ef4444',
                        warning: '#f59e0b',
                    }
                }
            }
        }
    </script>
    <style>
        .gradient-bg {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
        
        .glass-effect {
            background: rgba(255, 255, 255, 0.1);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        
        .card-hover {
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .card-hover:hover {
            transform: translateY(-8px) scale(1.02);
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
        }
        
        .bot-card {
            background: linear-gradient(145deg, #ffffff 0%, #f8fafc 100%);
            border: 1px solid rgba(99, 102, 241, 0.1);
        }
        
        .dark .bot-card {
            background: linear-gradient(145deg, #1f2937 0%, #111827 100%);
            border: 1px solid rgba(99, 102, 241, 0.2);
        }
        
        .text-gradient {
            background: linear-gradient(135deg, #6366f1, #8b5cf6);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
        
        .log-entry {
            font-family: 'Courier New', monospace;
            font-size: 0.875rem;
            line-height: 1.5;
        }
        
        .log-info { color: #3b82f6; }
        .log-success { color: #10b981; }
        .log-warning { color: #f59e0b; }
        .log-error { color: #ef4444; }
    </style>
</head>
<body class="bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100 dark:from-slate-900 dark:via-slate-800 dark:to-slate-900 min-h-screen">
    <div class="max-w-6xl mx-auto p-6">
        <!-- Header -->
        <div class="text-center mb-8">
            <div class="inline-flex items-center justify-center w-16 h-16 rounded-full bg-gradient-to-r from-indigo-500 to-purple-600 mb-4">
                <i class="fas fa-file-alt text-3xl text-white"></i>
            </div>
            <h1 class="text-4xl font-black text-gradient mb-2">Bot Logları</h1>
            <p class="text-lg text-gray-600 dark:text-gray-300">Bot aktivitelerini görüntüleyin</p>
        </div>

        <!-- Error Message -->
        <?php if (isset($error)): ?>
        <div class="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
            <div class="flex items-center">
                <i class="fas fa-exclamation-circle text-red-600 mr-3"></i>
                <div>
                    <p class="text-red-800 font-medium">Hata!</p>
                    <p class="text-red-700 text-sm"><?php echo htmlspecialchars($error); ?></p>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Database Logs -->
        <div class="bot-card rounded-2xl shadow-lg p-8 card-hover mb-6">
            <h3 class="text-xl font-bold text-gray-900 dark:text-white mb-4 flex items-center">
                <i class="fas fa-database text-primary mr-2"></i>
                Veritabanı Logları
            </h3>
            
            <?php if (empty($logs)): ?>
            <div class="text-center py-8">
                <i class="fas fa-inbox text-4xl text-gray-400 mb-4"></i>
                <p class="text-gray-500 dark:text-gray-400">Henüz veritabanı logu bulunmuyor</p>
            </div>
            <?php else: ?>
            <div class="bg-gray-900 dark:bg-gray-800 rounded-lg p-4 max-h-96 overflow-y-auto">
                <?php foreach ($logs as $log): ?>
                <div class="log-entry mb-2">
                    <span class="text-gray-400">[<?php echo htmlspecialchars($log['created_at']); ?>]</span>
                    <span class="log-<?php echo htmlspecialchars($log['log_type']); ?>">[<?php echo strtoupper(htmlspecialchars($log['log_type'])); ?>]</span>
                    <span class="text-white"><?php echo htmlspecialchars($log['message']); ?></span>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>
        </div>

        <!-- Recent Sites -->
        <div class="bot-card rounded-2xl shadow-lg p-8 card-hover mb-6">
            <h3 class="text-xl font-bold text-gray-900 dark:text-white mb-4 flex items-center">
                <i class="fas fa-globe text-primary mr-2"></i>
                Son Eklenen Siteler (Son 20 Site)
            </h3>
            
            <?php if (empty($recentSites)): ?>
            <div class="text-center py-8">
                <i class="fas fa-inbox text-4xl text-gray-400 mb-4"></i>
                <p class="text-gray-500 dark:text-gray-400">Henüz eklenen site bulunmuyor</p>
            </div>
            <?php else: ?>
            <div class="space-y-3 max-h-96 overflow-y-auto">
                <?php foreach ($recentSites as $site): ?>
                <div class="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600">
                    <div class="flex-1">
                        <div class="flex items-center gap-3">
                            <div class="w-8 h-8 bg-gradient-to-r from-blue-500 to-blue-600 rounded-lg flex items-center justify-center">
                                <i class="fas fa-globe text-white text-sm"></i>
                            </div>
                            <div>
                                <h4 class="font-medium text-gray-900 dark:text-white"><?php echo htmlspecialchars($site['site_name']); ?></h4>
                                <p class="text-sm text-gray-600 dark:text-gray-300"><?php echo htmlspecialchars($site['site_url']); ?></p>
                            </div>
                        </div>
                    </div>
                    <div class="text-right">
                        <div class="text-sm text-gray-500 dark:text-gray-400">
                            <?php echo date('d.m.Y H:i', strtotime($site['olusturulma_tarihi'])); ?>
                        </div>
                        <div class="text-xs">
                            <?php if ($site['status'] === 'approved'): ?>
                                <span class="px-2 py-1 bg-green-100 text-green-800 rounded-full">Onaylandı</span>
                            <?php elseif ($site['status'] === 'pending'): ?>
                                <span class="px-2 py-1 bg-yellow-100 text-yellow-800 rounded-full">Beklemede</span>
                            <?php else: ?>
                                <span class="px-2 py-1 bg-gray-100 text-gray-800 rounded-full"><?php echo htmlspecialchars($site['status']); ?></span>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>
        </div>

        <!-- File Logs -->
        <div class="bot-card rounded-2xl shadow-lg p-8 card-hover mb-6">
            <h3 class="text-xl font-bold text-gray-900 dark:text-white mb-4 flex items-center">
                <i class="fas fa-file text-primary mr-2"></i>
                Dosya Logları (Son 50 Satır)
            </h3>
            
            <?php if (empty($fileLogs)): ?>
            <div class="text-center py-8">
                <i class="fas fa-inbox text-4xl text-gray-400 mb-4"></i>
                <p class="text-gray-500 dark:text-gray-400">Henüz dosya logu bulunmuyor</p>
            </div>
            <?php else: ?>
            <div class="bg-gray-900 dark:bg-gray-800 rounded-lg p-4 max-h-96 overflow-y-auto">
                <?php foreach ($fileLogs as $log): ?>
                <div class="log-entry mb-2">
                    <?php
                    // Log tipini belirle
                    $logType = 'info';
                    if (strpos($log, '[error]') !== false) $logType = 'error';
                    elseif (strpos($log, '[success]') !== false) $logType = 'success';
                    elseif (strpos($log, '[warning]') !== false) $logType = 'warning';
                    ?>
                    <span class="log-<?php echo $logType; ?>"><?php echo htmlspecialchars($log); ?></span>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>
        </div>

        <!-- Actions -->
        <div class="flex justify-center gap-4">
            <button onclick="location.reload()" 
                    class="inline-flex items-center gap-2 bg-primary hover:bg-primary-dark text-white font-semibold px-6 py-3 rounded-xl transition-all duration-200">
                <i class="fas fa-sync-alt"></i>
                Yenile
            </button>
            
            <a href="index.php" 
               class="inline-flex items-center gap-2 bg-gray-500 hover:bg-gray-600 text-white font-semibold px-6 py-3 rounded-xl transition-all duration-200">
                <i class="fas fa-arrow-left"></i>
                Bot Ana Sayfasına Dön
            </a>
        </div>
    </div>
</body>
</html>