<?php
require_once '../includes/config.php';
require_once '../includes/turkish_fix.php';
require_once 'website_analyzer.php';
require_once 'google_search_bot.php';

// Bot ayarlarını al
function getBotSetting($pdo, $key, $default = '') {
    try {
        $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
        $stmt->execute(['bot_' . $key]);
        $result = $stmt->fetchColumn();
        return $result !== false ? $result : $default;
    } catch (Exception $e) {
        return $default;
    }
}

// Log fonksiyonu
function writeLog($message, $type = 'info') {
    global $pdo;
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] [$type] $message\n";
    echo $logMessage;
    
    // Dosyaya yaz
    file_put_contents('bot_logs.txt', $logMessage, FILE_APPEND | LOCK_EX);
    
    // Veritabanına yaz
    try {
        $stmt = $pdo->prepare("INSERT INTO bot_logs (log_type, message, created_at) VALUES (?, ?, NOW())");
        $stmt->execute([$type, $message]);
    } catch (Exception $e) {
        // Veritabanı yazma hatası - sessizce geç
    }
}

// Bot durdurma kontrolü
function checkBotStop($pdo) {
    try {
        // Stop lock dosyası kontrolü
        if (file_exists('bot_stop.lock')) {
            $stop_data = json_decode(file_get_contents('bot_stop.lock'), true);
            writeLog("🛑 Bot durdurma lock dosyası bulundu - İşlem durduruluyor", 'warning');
            writeLog("Durdurma zamanı: " . ($stop_data['stop_time'] ?? 'Bilinmiyor'), 'warning');
            return true;
        }
        
        $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
        $stmt->execute(['bot_force_stop']);
        $force_stop = $stmt->fetchColumn();
        
        if ($force_stop == '1') {
            writeLog("Bot durdurma sinyali alındı - İşlem durduruluyor", 'warning');
            return true;
        }
        
        $stmt->execute(['bot_status']);
        $status = $stmt->fetchColumn();
        
        if ($status !== 'running') {
            writeLog("Bot durumu 'running' değil - İşlem durduruluyor", 'warning');
            return true;
        }
        
        return false;
    } catch (Exception $e) {
        return false;
    }
}

try {
    writeLog("=== BOT BAŞLATILIYOR ===", 'info');
    
    // İlk durdurma kontrolü
    if (checkBotStop($pdo)) {
        writeLog("Bot başlatılamadı - Durdurma sinyali aktif", 'error');
        exit;
    }
    
    // Ayarları al
    writeLog("Bot ayarları alınıyor...", 'info');
    $openaiApiKey = getBotSetting($pdo, 'openai_api_key');
    $googleApiKey = getBotSetting($pdo, 'google_search_api_key');
    $searchEngineId = getBotSetting($pdo, 'google_search_engine_id');
    $sitesPerRun = intval(getBotSetting($pdo, 'sites_per_run', '5'));
    $contentLanguage = getBotSetting($pdo, 'content_language', 'tr');
    $searchKeywords = getBotSetting($pdo, 'search_keywords', 'Foreign Language: language learning, translation, english course, spanish lessons, french tutor, german class, italian language, portuguese course, russian lessons, chinese mandarin, japanese learning, korean language, arabic course, turkish lessons, dutch language, Transport: airline booking, flight tickets, bus schedule, train tickets, shipping company, logistics service, cargo transport, delivery service, taxi booking, car rental, motorcycle rental, truck rental, boat rental, yacht charter, cruise booking, Tourism & Travel: hotel booking, vacation rental, travel agency, destination guide, adventure tourism, beach resort, mountain lodge, city tour, cultural tour, food tour, wine tasting, hiking trail, camping site, backpacking, luxury travel, Sports & Teams: football team, basketball club, soccer league, tennis academy, swimming pool, gym membership, fitness center, sports equipment, athletic wear, sports news, olympic games, world cup, championship, tournament, sports betting, Industry: manufacturing company, industrial equipment, construction company, engineering firm, factory automation, production line, quality control, safety equipment, industrial design, machinery repair, metalworking, plastic molding, textile production, chemical industry, automotive parts, Health & Life: healthcare provider, medical clinic, hospital services, wellness center, fitness program, nutrition advice, mental health, dental care, eye care, pharmacy, medical equipment, health insurance, life insurance, personal trainer, yoga instructor, Culture & Art: museum exhibition, art gallery, cultural center, music venue, theater show, dance performance, art workshop, photography studio, design agency, creative writing, poetry reading, book club, film festival, art auction, cultural heritage, Business: corporate website, professional service, consulting firm, startup company, business directory, marketing agency, accounting service, legal advice, financial planning, investment firm, insurance company, real estate agency, property management, business coaching, entrepreneurship, Internet & Tech: technology company, software development, IT service, web design, mobile app, cloud computing, cybersecurity, data analytics, artificial intelligence, machine learning, blockchain, cryptocurrency, fintech, e-commerce, digital marketing, Law & Consulting: law firm, legal service, attorney, lawyer, court case, legal advice, business law, criminal law, family law, immigration law, tax law, contract law, intellectual property, legal consultation, mediation service, Food & Agri: restaurant booking, food delivery, agricultural company, farming equipment, organic food, local produce, food truck, catering service, cooking class, wine production, coffee shop, bakery, butcher shop, fish market, grocery store, Real Estate: property listing, real estate agent, home buying, apartment rental, house sale, property investment, mortgage broker, home insurance, property management, construction company, interior design, home renovation, landscaping, property valuation, rental property, Entertainment: movie theater, gaming platform, music streaming, concert venue, comedy club, nightclub, casino, theme park, arcade, board game, video game, streaming service, podcast, radio station, live show, Education: online course, university, school, learning platform, educational resource, tutoring service, language school, driving school, music lesson, art class, cooking school, computer training, business course, certification program, skill development, Religion & Ethics: religious organization, spiritual community, church service, mosque, temple, synagogue, prayer group, meditation center, ethical foundation, charity organization, volunteer work, community service, religious education, spiritual guidance, faith-based counseling, Government: government agency, public service, municipal website, official institution, tax office, passport office, driver license, social security, health department, education department, police station, fire department, emergency service, public transportation, city planning, Press & Media: news outlet, journalist, media company, broadcasting, radio station, television channel, newspaper, magazine, online news, press release, media kit, advertising agency, public relations, social media, content creation, Shopping: e-commerce store, online shopping, retail website, marketplace, fashion store, electronics shop, home decor, beauty products, sports gear, book store, music store, toy store, jewelry shop, gift shop, discount store, Other Websites: miscellaneous website, useful tool, productivity app, utility service, reference site, directory listing, comparison tool, review site, forum, community, social network, dating site, job board, classified ads, auction site');
    $minDescLength = intval(getBotSetting($pdo, 'min_description_length', '200'));
    $maxDescLength = intval(getBotSetting($pdo, 'max_description_length', '500'));
    
    writeLog("Ayarlar alındı:", 'info');
    writeLog("- OpenAI API Key: " . (!empty($openaiApiKey) ? 'VAR' : 'YOK'), 'info');
    writeLog("- Google API Key: " . (!empty($googleApiKey) ? 'VAR' : 'YOK'), 'info');
    writeLog("- Search Engine ID: " . (!empty($searchEngineId) ? 'VAR' : 'YOK'), 'info');
    writeLog("- Sites per run: $sitesPerRun", 'info');
    writeLog("- Content Language: " . ($contentLanguage === 'en' ? 'English' : 'Türkçe'), 'info');
    writeLog("- Search keywords: " . substr($searchKeywords, 0, 100) . "...", 'info');
    
    // API anahtarları kontrolü
    if (empty($openaiApiKey)) {
        writeLog("HATA: OpenAI API anahtarı bulunamadı!", 'error');
        die("OpenAI API anahtarı gerekli. Bot ayarlarından ekleyin.");
    }
    
    if (empty($googleApiKey)) {
        writeLog("HATA: Google Search API anahtarı bulunamadı!", 'error');
        die("Google Search API anahtarı gerekli. Bot ayarlarından ekleyin.");
    }
    
    if (empty($searchEngineId)) {
        writeLog("HATA: Google Search Engine ID bulunamadı!", 'error');
        die("Google Search Engine ID gerekli. Bot ayarlarından ekleyin.");
    }
    
    // Sınıfları başlat
    writeLog("Sınıflar başlatılıyor...", 'info');
    $analyzer = new WebsiteAnalyzer($pdo, $openaiApiKey);
    $bot = new GoogleSearchBot($pdo, $googleApiKey, $searchEngineId, $analyzer);
    writeLog("Sınıflar başarıyla başlatıldı", 'success');
    
    // Log tablosunu oluştur (yoksa)
    writeLog("Log tablosu kontrol ediliyor...", 'info');
    try {
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS bot_logs (
                id INT AUTO_INCREMENT PRIMARY KEY,
                log_type VARCHAR(20) DEFAULT 'info',
                message TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        ");
        writeLog("Log tablosu hazır", 'success');
    } catch (Exception $e) {
        writeLog("Log tablosu oluşturma hatası: " . $e->getMessage(), 'error');
    }
    
    writeLog("Bot çalıştırılıyor - $sitesPerRun site işlenecek", 'info');
    
    // Rastgele siteler ara
    writeLog("Google Search API ile site arama başlatılıyor...", 'info');
    $sites = $bot->searchRandomSites($searchKeywords, $sitesPerRun);
    
    writeLog("Site arama sonucu: " . count($sites) . " site bulundu", 'info');
    
    if (empty($sites)) {
        writeLog("HATA: Hiç site bulunamadı!", 'error');
        writeLog("Sebep: Google Search API'den sonuç dönmedi veya API anahtarı geçersiz", 'error');
        writeLog("Bot durumu değiştirilmedi, tekrar deneyebilirsiniz", 'info');
        die('Hiç site bulunamadı. API anahtarlarını kontrol edin.');
    }
    
    writeLog(count($sites) . " site bulundu, işleme başlanıyor", 'success');
    
    $successCount = 0;
    $errorCount = 0;
    
    // Her siteyi işle
    foreach ($sites as $index => $siteData) {
        // Her site işleme öncesi durdurma kontrolü
        if (checkBotStop($pdo)) {
            writeLog("Bot durdurma sinyali alındı - Site işleme durduruluyor", 'warning');
            writeLog("İşlenen site sayısı: $successCount", 'info');
            break;
        }
        
        $siteNumber = $index + 1;
        writeLog("=== SITE $siteNumber/" . count($sites) . " İŞLENİYOR ===", 'info');
        writeLog("Site URL: " . ($siteData['url'] ?? 'URL bulunamadı'), 'info');
        
        try {
        $result = $bot->processSite($siteData);
        
        if ($result['success']) {
            $successCount++;
                writeLog("✅ BAŞARILI: " . $result['site_name'] . " (" . $result['url'] . ")", 'success');
                writeLog("Kategori: " . ($result['category'] ?? 'Belirlenemedi'), 'info');
                writeLog("Açıklama uzunluğu: " . strlen($result['description'] ?? ''), 'info');
        } else {
                $errorCount++;
                writeLog("❌ HATA: " . ($result['error'] ?? 'Bilinmeyen hata'), 'error');
                writeLog("Site: " . ($siteData['url'] ?? 'URL bulunamadı'), 'error');
            }
        } catch (Exception $e) {
            $errorCount++;
            writeLog("❌ EXCEPTION: " . $e->getMessage(), 'error');
            writeLog("Site: " . ($siteData['url'] ?? 'URL bulunamadı'), 'error');
        }
        
        // Site işleme sonrası durdurma kontrolü
        if (checkBotStop($pdo)) {
            writeLog("Bot durdurma sinyali alındı - Site işleme durduruluyor", 'warning');
            writeLog("İşlenen site sayısı: $successCount", 'info');
            break;
        }
        
        // Kısa bekleme
        usleep(500000); // 0.5 saniye
    }
    
    // Sonuçları logla
    writeLog("=== BOT TAMAMLANDI ===", 'info');
    writeLog("Toplam işlenen site: " . count($sites), 'info');
    writeLog("Başarılı: $successCount", 'success');
    writeLog("Hatalı: $errorCount", 'error');
    
    // Bot durumu değiştirilmiyor - sürekli çalışır durumda kalacak
    writeLog("Bot işlemi tamamlandı, durumu değiştirilmedi", 'info');
    
    // Toplam site sayısını güncelle
    try {
        $stmt = $pdo->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?");
        $stmt->execute(['bot_total_sites', $successCount, $successCount]);
        writeLog("Toplam site sayısı güncellendi: $successCount", 'success');
                        } catch (Exception $e) {
        writeLog("Toplam site sayısı güncellenemedi: " . $e->getMessage(), 'error');
    }
    
    // Son çalıştırma zamanını güncelle
    try {
        $stmt = $pdo->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?");
        $stmt->execute(['bot_last_run', date('Y-m-d H:i:s'), date('Y-m-d H:i:s')]);
        writeLog("Son çalıştırma zamanı güncellendi", 'success');
    } catch (Exception $e) {
        writeLog("Son çalıştırma zamanı güncellenemedi: " . $e->getMessage(), 'error');
    }
    
    writeLog("=== BOT BAŞARIYLA TAMAMLANDI ===", 'success');
    
} catch (Exception $e) {
    writeLog("KRITIK HATA: " . $e->getMessage(), 'error');
    writeLog("Hata dosyası: " . $e->getFile() . " Satır: " . $e->getLine(), 'error');
    
    // Bot durumu değiştirilmiyor - hata durumunda da çalışır durumda kalacak
    writeLog("Bot hata durumunda, durumu değiştirilmedi", 'info');
    
    die("Bot çalıştırılırken hata oluştu: " . $e->getMessage());
}
?>