<?php
require_once '../includes/config.php';
require_once '../includes/turkish_fix.php';
require_once 'website_analyzer.php';
require_once 'google_search_bot.php';

// Bot ayarlarını al
function getBotSetting($pdo, $key, $default = '') {
    try {
        $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
        $stmt->execute(['bot_' . $key]);
        $result = $stmt->fetchColumn();
        return $result !== false ? $result : $default;
    } catch (Exception $e) {
        return $default;
    }
}

// Log fonksiyonu
function writeLog($message, $type = 'info') {
    global $pdo;
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] [$type] $message\n";
    echo $logMessage;
    
    // Dosyaya yaz
    file_put_contents('bot_logs.txt', $logMessage, FILE_APPEND | LOCK_EX);
    
    // Veritabanına yaz
    try {
        $stmt = $pdo->prepare("INSERT INTO bot_logs (log_type, message, created_at) VALUES (?, ?, NOW())");
        $stmt->execute([$type, $message]);
    } catch (Exception $e) {
        // Veritabanı yazma hatası - sessizce geç
    }
}

// Bot durdurma kontrolü
function checkBotStop($pdo) {
    try {
        // Stop lock dosyası kontrolü
        if (file_exists('bot_stop.lock')) {
            $stop_data = json_decode(file_get_contents('bot_stop.lock'), true);
            writeLog("🛑 Bot durdurma lock dosyası bulundu - İşlem durduruluyor", 'warning');
            writeLog("Durdurma zamanı: " . ($stop_data['stop_time'] ?? 'Bilinmiyor'), 'warning');
            return true;
        }
        
        $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
        $stmt->execute(['bot_force_stop']);
        $force_stop = $stmt->fetchColumn();
        
        if ($force_stop == '1') {
            writeLog("Bot durdurma sinyali alındı - İşlem durduruluyor", 'warning');
            return true;
        }
        
        $stmt->execute(['bot_status']);
        $status = $stmt->fetchColumn();
        
        if ($status !== 'running') {
            writeLog("Bot durumu 'running' değil - İşlem durduruluyor", 'warning');
            return true;
        }
        
        return false;
    } catch (Exception $e) {
        return false;
    }
}

try {
    writeLog("=== KATEGORİ BOT BAŞLATILIYOR ===", 'info');
    
    // İlk durdurma kontrolü
    if (checkBotStop($pdo)) {
        writeLog("Kategori bot başlatılamadı - Durdurma sinyali aktif", 'error');
        exit;
    }
    
    // Ayarları al
    writeLog("Bot ayarları alınıyor...", 'info');
    $openaiApiKey = getBotSetting($pdo, 'openai_api_key');
    $googleApiKey = getBotSetting($pdo, 'google_search_api_key');
    $searchEngineId = getBotSetting($pdo, 'google_search_engine_id');
    $sitesPerRun = intval(getBotSetting($pdo, 'sites_per_run', '3'));
    $targetCategories = getBotSetting($pdo, 'target_categories');
    
    writeLog("Ayarlar alındı:", 'info');
    writeLog("- OpenAI API Key: " . (!empty($openaiApiKey) ? 'VAR' : 'YOK'), 'info');
    writeLog("- Google API Key: " . (!empty($googleApiKey) ? 'VAR' : 'YOK'), 'info');
    writeLog("- Search Engine ID: " . (!empty($searchEngineId) ? 'VAR' : 'YOK'), 'info');
    writeLog("- Sites per run: $sitesPerRun", 'info');
    writeLog("- Target categories: $targetCategories", 'info');
    
    // API anahtarları kontrolü
    if (empty($openaiApiKey)) {
        writeLog("HATA: OpenAI API anahtarı bulunamadı!", 'error');
        die("OpenAI API anahtarı gerekli. Bot ayarlarından ekleyin.");
    }
    
    if (empty($googleApiKey)) {
        writeLog("HATA: Google Search API anahtarı bulunamadı!", 'error');
        die("Google Search API anahtarı gerekli. Bot ayarlarından ekleyin.");
    }
    
    if (empty($searchEngineId)) {
        writeLog("HATA: Google Search Engine ID bulunamadı!", 'error');
        die("Google Search Engine ID gerekli. Bot ayarlarından ekleyin.");
    }
    
    // Sınıfları başlat
    writeLog("Sınıflar başlatılıyor...", 'info');
    $analyzer = new WebsiteAnalyzer($pdo, $openaiApiKey);
    $bot = new GoogleSearchBot($pdo, $googleApiKey, $searchEngineId, $analyzer);
    writeLog("Sınıflar başarıyla başlatıldı", 'success');
    
    // Log tablosunu oluştur (yoksa)
    writeLog("Log tablosu kontrol ediliyor...", 'info');
    try {
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS bot_logs (
                id INT AUTO_INCREMENT PRIMARY KEY,
                log_type VARCHAR(20) DEFAULT 'info',
                message TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        ");
        writeLog("Log tablosu hazır", 'success');
    } catch (Exception $e) {
        writeLog("Log tablosu oluşturma hatası: " . $e->getMessage(), 'error');
    }
    
    // Kategorileri al
    writeLog("Kategoriler alınıyor...", 'info');
    try {
        $stmt = $pdo->query("SELECT id, name FROM categories ORDER BY name");
        $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
        writeLog("Toplam " . count($categories) . " kategori bulundu", 'info');
        
        if (empty($categories)) {
            writeLog("HATA: Hiç kategori bulunamadı!", 'error');
            die("Veritabanında kategori bulunamadı. Önce kategoriler ekleyin.");
        }
    } catch (Exception $e) {
        writeLog("Kategori alma hatası: " . $e->getMessage(), 'error');
        die("Kategoriler alınamadı: " . $e->getMessage());
    }
    
    // Hedef kategorileri belirle
    $targetCategoryIds = [];
    if (!empty($targetCategories)) {
        $targetCategoryIds = explode(',', $targetCategories);
        $targetCategoryIds = array_map('intval', $targetCategoryIds);
        writeLog("Hedef kategoriler: " . implode(', ', $targetCategoryIds), 'info');
    } else {
        // Rastgele kategoriler seç
        $randomCategories = array_rand($categories, min(3, count($categories)));
        if (!is_array($randomCategories)) {
            $randomCategories = [$randomCategories];
        }
        $targetCategoryIds = array_map(function($index) use ($categories) {
            return $categories[$index]['id'];
        }, $randomCategories);
        writeLog("Rastgele kategoriler seçildi: " . implode(', ', $targetCategoryIds), 'info');
    }
    
    $totalSuccessCount = 0;
    $totalErrorCount = 0;
    
    // Her kategori için site ara
    foreach ($targetCategoryIds as $categoryId) {
        $category = array_filter($categories, function($cat) use ($categoryId) {
            return $cat['id'] == $categoryId;
        });
        $category = reset($category);
        
        if (!$category) {
            writeLog("Kategori ID $categoryId bulunamadı, atlanıyor", 'warning');
            continue;
        }
        
        writeLog("=== KATEGORİ: " . $category['name'] . " ===", 'info');
        
        try {
            // Kategori için site ara
            writeLog("Kategori için site arama başlatılıyor...", 'info');
            $sites = $bot->searchSitesByCategory($categoryId, $sitesPerRun);
            
            writeLog("Kategori için " . count($sites) . " site bulundu", 'info');
            
            if (empty($sites)) {
                writeLog("Bu kategori için site bulunamadı, sonraki kategoriye geçiliyor", 'warning');
                continue;
            }
            
            $categorySuccessCount = 0;
            $categoryErrorCount = 0;
            
            // Her siteyi işle
            foreach ($sites as $index => $siteData) {
                $siteNumber = $index + 1;
                writeLog("--- Site $siteNumber/" . count($sites) . " işleniyor ---", 'info');
                writeLog("Site URL: " . ($siteData['url'] ?? 'URL bulunamadı'), 'info');
                
                try {
                    $result = $bot->processSite($siteData);
                    
                    if ($result['success']) {
                        $categorySuccessCount++;
                        $totalSuccessCount++;
                        writeLog("✅ BAŞARILI: " . $result['site_name'] . " (" . $result['url'] . ")", 'success');
                        writeLog("Kategori: " . ($result['category'] ?? 'Belirlenemedi'), 'info');
                        writeLog("Açıklama uzunluğu: " . strlen($result['description'] ?? ''), 'info');
                    } else {
                        $categoryErrorCount++;
                        $totalErrorCount++;
                        writeLog("❌ HATA: " . ($result['error'] ?? 'Bilinmeyen hata'), 'error');
                        writeLog("Site: " . ($siteData['url'] ?? 'URL bulunamadı'), 'error');
                    }
                } catch (Exception $e) {
                    $categoryErrorCount++;
                    $totalErrorCount++;
                    writeLog("❌ EXCEPTION: " . $e->getMessage(), 'error');
                    writeLog("Site: " . ($siteData['url'] ?? 'URL bulunamadı'), 'error');
                }
                
                // Kısa bekleme
                usleep(500000); // 0.5 saniye
            }
            
            writeLog("Kategori '$categoryName' tamamlandı - Başarılı: $categorySuccessCount, Hatalı: $categoryErrorCount", 'info');
            
        } catch (Exception $e) {
            writeLog("Kategori '$categoryName' işlenirken hata: " . $e->getMessage(), 'error');
            $totalErrorCount++;
        }
        
        // Kategoriler arası bekleme
        usleep(1000000); // 1 saniye
    }
    
    // Sonuçları logla
    writeLog("=== KATEGORİ BOT TAMAMLANDI ===", 'info');
    writeLog("Toplam işlenen kategori: " . count($targetCategoryIds), 'info');
    writeLog("Toplam başarılı site: $totalSuccessCount", 'success');
    writeLog("Toplam hatalı site: $totalErrorCount", 'error');
    
    // Toplam site sayısını güncelle
    try {
        $stmt = $pdo->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?");
        $stmt->execute(['bot_total_sites', $totalSuccessCount, $totalSuccessCount]);
        writeLog("Toplam site sayısı güncellendi: $totalSuccessCount", 'success');
    } catch (Exception $e) {
        writeLog("Toplam site sayısı güncellenemedi: " . $e->getMessage(), 'error');
    }
    
    // Son çalıştırma zamanını güncelle
    try {
        $stmt = $pdo->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?");
        $stmt->execute(['bot_last_run', date('Y-m-d H:i:s'), date('Y-m-d H:i:s')]);
        writeLog("Son çalıştırma zamanı güncellendi", 'success');
    } catch (Exception $e) {
        writeLog("Son çalıştırma zamanı güncellenemedi: " . $e->getMessage(), 'error');
    }
    
    writeLog("=== KATEGORİ BOT BAŞARIYLA TAMAMLANDI ===", 'success');
    
} catch (Exception $e) {
    writeLog("KRITIK HATA: " . $e->getMessage(), 'error');
    writeLog("Hata dosyası: " . $e->getFile() . " Satır: " . $e->getLine(), 'error');
    
    die("Kategori bot çalıştırılırken hata oluştu: " . $e->getMessage());
}
?>