<?php
require_once '../includes/config.php';
require_once '../includes/turkish_fix.php';
require_once 'website_analyzer.php';
require_once 'google_search_bot.php';

// Bot ayarlarını al
function getBotSetting($pdo, $key, $default = '') {
    try {
        $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
        $stmt->execute(['bot_' . $key]);
        $result = $stmt->fetchColumn();
        return $result !== false ? $result : $default;
    } catch (Exception $e) {
        return $default;
    }
}

try {
    // Kategori botu için bot durumu kontrolü gerekli değil
    
    // Ayarları al
    $openaiApiKey = getBotSetting($pdo, 'openai_api_key');
    $googleApiKey = getBotSetting($pdo, 'google_search_api_key');
    $searchEngineId = getBotSetting($pdo, 'google_search_engine_id');
    $sitesPerCategory = intval(getBotSetting($pdo, 'sites_per_category', '3'));
    $targetCategories = getBotSetting($pdo, 'target_categories');
    
    // Sınıfları başlat
    $analyzer = new WebsiteAnalyzer($pdo, $openaiApiKey);
    $bot = new GoogleSearchBot($pdo, $googleApiKey, $searchEngineId, $analyzer);
    
    // Log tablosunu oluştur (yoksa)
    try {
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS bot_logs (
                id INT AUTO_INCREMENT PRIMARY KEY,
                log_type VARCHAR(20) DEFAULT 'info',
                message TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        ");
    } catch (Exception $e) {
        // Tablo oluşturma hatası
    }
    
    $bot->log("🎯 Gelişmiş kategori bazlı bot başlatılıyor...");
    
    // Hedef kategorileri belirle
    $categories = [];
    if (!empty($targetCategories)) {
        $categories = array_map('intval', explode(',', $targetCategories));
        $categories = array_filter($categories); // Boş değerleri temizle
    }
    
    // Eğer kategori seçilmemişse hata ver
    if (empty($categories)) {
        throw new Exception('Hiç kategori seçilmemiş! Lütfen kategori seçici sayfasından kategorileri seçin.');
    }
    
    // Kategori isimlerini al
    $categoryNames = [];
    foreach ($categories as $catId) {
        try {
            $stmt = $pdo->prepare("SELECT name FROM categories WHERE id = ?");
            $stmt->execute([$catId]);
            $name = $stmt->fetchColumn();
            if ($name) {
                $categoryNames[$catId] = $name;
            } else {
                $categoryNames[$catId] = "Kategori $catId";
            }
        } catch (Exception $e) {
            $categoryNames[$catId] = "Kategori $catId";
        }
    }
    
    $bot->log("🎯 Hedef kategoriler (" . count($categories) . "): " . implode(', ', array_values($categoryNames)));
    $bot->log("📊 Her kategori için " . $sitesPerCategory . " site hedefleniyor");
    $bot->log("🔢 Toplam hedef: " . (count($categories) * $sitesPerCategory) . " site");
    
    // Batch processing için progress tracking
    $totalSuccess = 0;
    $totalErrors = 0;
    $processedCategories = [];
    $startTime = time();
    
    // Paralel işlem yerine sıralı ama optimize edilmiş işlem
    foreach ($categories as $index => $categoryId) {
        $categoryName = $categoryNames[$categoryId];
        $bot->log("🔄 [$index+1/" . count($categories) . "] $categoryName kategorisi işleniyor...");
        
        try {
            // Kategori için site ara
            $sites = $bot->searchSitesByCategory($categoryId, $sitesPerCategory);
            
            if (empty($sites)) {
                $bot->log("⚠️ $categoryName için hiç site bulunamadı", 'warning');
                $processedCategories[$categoryId] = [
                    'name' => $categoryName,
                    'success' => 0,
                    'errors' => 0,
                    'status' => 'no_sites_found'
                ];
                continue;
            }
            
            $bot->log("✅ $categoryName için " . count($sites) . " site bulundu, işleniyor...");
            
            $categorySuccess = 0;
            $categoryErrors = 0;
            
            // Kategori sitelerini işle
            foreach ($sites as $siteIndex => $siteData) {
                try {
                    $result = $bot->processSiteWithCategory($siteData, $categoryId);
                    
                    if ($result['success']) {
                        $categorySuccess++;
                        $totalSuccess++;
                        $bot->log("  ✅ [" . ($siteIndex + 1) . "/" . count($sites) . "] " . $result['site_name'], 'success');
                    } else {
                        $categoryErrors++;
                        $totalErrors++;
                        $bot->log("  ❌ [" . ($siteIndex + 1) . "/" . count($sites) . "] " . ($siteData['url'] ?? 'Unknown URL') . " - " . $result['error'], 'error');
                    }
                    
                    // Rate limiting - her site arasında kısa bekleme
                    usleep(500000); // 0.5 saniye
                    
                } catch (Exception $e) {
                    $categoryErrors++;
                    $totalErrors++;
                    $bot->log("  💥 [" . ($siteIndex + 1) . "/" . count($sites) . "] Beklenmeyen hata: " . $e->getMessage(), 'error');
                }
            }
            
            $processedCategories[$categoryId] = [
                'name' => $categoryName,
                'success' => $categorySuccess,
                'errors' => $categoryErrors,
                'status' => 'completed'
            ];
            
            $bot->log("✅ $categoryName tamamlandı: $categorySuccess başarılı, $categoryErrors hata");
            
            // Kategoriler arası bekleme (API limits için)
            if ($index < count($categories) - 1) {
                $bot->log("⏱️ Kategoriler arası bekleme...");
                sleep(2);
            }
            
        } catch (Exception $e) {
            $totalErrors++;
            $processedCategories[$categoryId] = [
                'name' => $categoryName,
                'success' => 0,
                'errors' => 1,
                'status' => 'failed',
                'error' => $e->getMessage()
            ];
            $bot->log("💥 $categoryName kategorisi işlenirken hata: " . $e->getMessage(), 'error');
        }
    }
    
    // Sonuçları özetle
    $endTime = time();
    $duration = $endTime - $startTime;
    $minutes = floor($duration / 60);
    $seconds = $duration % 60;
    
    $bot->log("🏁 Toplu kategori botu tamamlandı!");
    $bot->log("⏱️ Toplam süre: {$minutes}m {$seconds}s");
    $bot->log("📊 Sonuç: $totalSuccess başarılı, $totalErrors hata");
    
    // İstatistikleri güncelle
    $totalSitesInDb = intval(getBotSetting($pdo, 'total_sites', '0')) + $totalSuccess;
    $lastRun = date('Y-m-d H:i:s') . " (Toplu kategori: $totalSuccess başarılı, $totalErrors hata)";
    
    $bot->updateBotStatus('stopped', $lastRun, $totalSitesInDb);
    
    // Web sayfası çıktısı
    ?>
    <!DOCTYPE html>
    <html lang="tr">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Toplu Kategori Bot Tamamlandı</title>
        <link href="https://fonts.googleapis.com/css2?family=Public+Sans:wght@400;500;600;700&display=swap" rel="stylesheet">
        <script src="https://cdn.tailwindcss.com"></script>
        <script>
            tailwind.config = {
                theme: {
                    extend: {
                        fontFamily: {
                            'sans': ['Public Sans', 'ui-sans-serif', 'system-ui'],
                        }
                    }
                }
            }
        </script>
    </head>
    <body class="bg-gray-50 font-sans">
        <div class="max-w-4xl mx-auto p-6">
            <div class="bg-white rounded-lg shadow-sm p-6 text-center mb-6">
                <div class="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                    <svg class="w-8 h-8 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                    </svg>
                </div>
                
                <h1 class="text-2xl font-bold text-gray-900 mb-2">🎯 Toplu Kategori Bot Tamamlandı!</h1>
                
                <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
                    <div class="bg-green-50 p-4 rounded-lg">
                        <div class="text-2xl font-bold text-green-600"><?php echo $totalSuccess; ?></div>
                        <div class="text-sm text-green-700">Başarılı Site</div>
                    </div>
                    <div class="bg-blue-50 p-4 rounded-lg">
                        <div class="text-2xl font-bold text-blue-600"><?php echo count($processedCategories); ?></div>
                        <div class="text-sm text-blue-700">İşlenen Kategori</div>
                    </div>
                    <div class="bg-purple-50 p-4 rounded-lg">
                        <div class="text-2xl font-bold text-purple-600"><?php echo "{$minutes}m {$seconds}s"; ?></div>
                        <div class="text-sm text-purple-700">Toplam Süre</div>
                    </div>
                    <div class="bg-red-50 p-4 rounded-lg">
                        <div class="text-2xl font-bold text-red-600"><?php echo $totalErrors; ?></div>
                        <div class="text-sm text-red-700">Hata Sayısı</div>
                    </div>
                </div>
                
                <div class="flex justify-center gap-4">
                    <a href="category_selector.php" 
                       class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg font-medium transition-colors">
                        🎯 Kategori Seçici
                    </a>
                    <a href="index.php" 
                       class="bg-gray-600 hover:bg-gray-700 text-white px-6 py-2 rounded-lg font-medium transition-colors">
                        🏠 Bot Paneli
                    </a>
                    <a href="../" 
                       class="bg-green-600 hover:bg-green-700 text-white px-6 py-2 rounded-lg font-medium transition-colors">
                        🌐 Ana Site
                    </a>
                </div>
            </div>
            
            <!-- Kategori Detayları -->
            <div class="bg-white rounded-lg shadow-sm p-6">
                <h2 class="text-lg font-semibold text-gray-900 mb-4">📊 Kategori Detayları</h2>
                <div class="space-y-3">
                    <?php foreach ($processedCategories as $catId => $details): ?>
                    <div class="flex items-center justify-between p-3 border border-gray-200 rounded-lg">
                        <div class="flex items-center gap-3">
                            <?php if ($details['status'] === 'completed' && $details['success'] > 0): ?>
                                <span class="w-3 h-3 bg-green-500 rounded-full"></span>
                            <?php elseif ($details['status'] === 'no_sites_found'): ?>
                                <span class="w-3 h-3 bg-yellow-500 rounded-full"></span>
                            <?php else: ?>
                                <span class="w-3 h-3 bg-red-500 rounded-full"></span>
                            <?php endif; ?>
                            
                            <div>
                                <span class="font-medium text-gray-900"><?php safeEcho($details['name']); ?></span>
                                <?php if (isset($details['error'])): ?>
                                    <p class="text-xs text-red-600"><?php safeEcho($details['error']); ?></p>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="text-right">
                            <span class="text-green-600 font-medium"><?php echo $details['success']; ?> başarılı</span>
                            <?php if ($details['errors'] > 0): ?>
                                <span class="text-red-600 font-medium ml-2"><?php echo $details['errors']; ?> hata</span>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        
        <script>
            // 15 saniye sonra kategori seçici sayfasına yönlendir
            setTimeout(function() {
                window.location.href = 'category_selector.php';
            }, 15000);
        </script>
    </body>
    </html>
    <?php
    
} catch (Exception $e) {
    // Hata durumunda bot'u durdur ve log yaz
    try {
        $stmt = $pdo->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?");
        $stmt->execute(['bot_status', 'stopped', 'stopped']);
        
        // Log da ekle
        $stmt = $pdo->prepare("INSERT INTO bot_logs (log_type, message) VALUES (?, ?)");
        $stmt->execute(['error', "Toplu kategori bot hatası: " . $e->getMessage()]);
    } catch (Exception $e2) {
        // Status update hatası
    }
    
    ?>
    <!DOCTYPE html>
    <html lang="tr">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Bot Hatası</title>
        <link href="https://fonts.googleapis.com/css2?family=Public+Sans:wght@400;500;600;700&display=swap" rel="stylesheet">
        <script src="https://cdn.tailwindcss.com"></script>
    </head>
    <body class="bg-gray-50 font-sans">
        <div class="max-w-2xl mx-auto p-6">
            <div class="bg-white rounded-lg shadow-sm p-6 text-center">
                <div class="w-16 h-16 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-4">
                    <svg class="w-8 h-8 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </div>
                
                <h1 class="text-2xl font-bold text-gray-900 mb-2">💥 Toplu Kategori Bot Hatası!</h1>
                <p class="text-gray-600 mb-6"><?php echo htmlspecialchars($e->getMessage()); ?></p>
                
                <div class="flex justify-center gap-4">
                    <a href="category_selector.php" 
                       class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg font-medium transition-colors">
                        🎯 Kategori Seçici
                    </a>
                    <a href="index.php" 
                       class="bg-gray-600 hover:bg-gray-700 text-white px-6 py-2 rounded-lg font-medium transition-colors">
                        🏠 Bot Paneline Dön
                    </a>
                </div>
            </div>
        </div>
    </body>
    </html>
    <?php
}
?>