<?php
require_once '../includes/turkish_fix.php';

class WebsiteAnalyzer {
    private $pdo;
    private $openaiApiKey;
    private $contentLanguage;
    
    public function __construct($pdo, $openaiApiKey) {
        $this->pdo = $pdo;
        $this->openaiApiKey = $openaiApiKey;
        $this->contentLanguage = $this->getContentLanguage();
    }
    
    /**
     * İçerik dilini al
     */
    private function getContentLanguage() {
        try {
            $stmt = $this->pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = 'bot_content_language'");
            $stmt->execute();
            $language = $stmt->fetchColumn();
            return $language ?: 'tr'; // Varsayılan Türkçe
        } catch (Exception $e) {
            return 'tr'; // Hata durumunda Türkçe
        }
    }
    
    /**
     * URL'den site bilgilerini çıkarır
     */
    public function analyzeSite($url) {
        $result = [
            'url' => $url,
            'title' => '',
            'description' => '',
            'logo' => '',
            'meta_description' => '',
            'domain' => '',
            'email' => '',
            'success' => false,
            'error' => ''
        ];
        
        error_log("WebsiteAnalyzer: Analyzing site: " . $url);
        
        try {
            // URL'yi düzenle
            if (!preg_match('/^https?:\/\//', $url)) {
                $url = 'https://' . $url;
            }
            
            $result['url'] = $url;
            $result['domain'] = parse_url($url, PHP_URL_HOST);
            
            error_log("WebsiteAnalyzer: Fetching content for: " . $url);
            
            // Site içeriğini al
            $html = $this->fetchWebsiteContent($url);
            if (!$html) {
                $result['error'] = 'Site içeriği alınamadı';
                error_log("WebsiteAnalyzer: Failed to fetch content for: " . $url);
                return $result;
            }
            
            error_log("WebsiteAnalyzer: Content fetched successfully, parsing HTML...");
            
            // HTML'i parse et
            $dom = new DOMDocument();
            libxml_use_internal_errors(true);
            $dom->loadHTML($html);
            libxml_clear_errors();
            
            // Site başlığını al
            $titleNodes = $dom->getElementsByTagName('title');
            if ($titleNodes->length > 0) {
                $result['title'] = fixTurkishEncoding(trim($titleNodes->item(0)->textContent));
            }
            
            // Meta description al
            $metaTags = $dom->getElementsByTagName('meta');
            foreach ($metaTags as $meta) {
                if (strtolower($meta->getAttribute('name')) === 'description') {
                    $result['meta_description'] = fixTurkishEncoding(trim($meta->getAttribute('content')));
                    break;
                }
            }
            
            // Logo bul
            $result['logo'] = $this->findSiteLogo($dom, $url);
            
            // E-posta adresi oluştur
            $result['email'] = $this->generateEmail($result['domain']);
            
            // Başlık yoksa domain adını kullan
            if (empty($result['title'])) {
                $result['title'] = ucfirst(str_replace(['www.', '.com', '.net', '.org'], '', $result['domain']));
            }
            
            $result['success'] = true;
            error_log("WebsiteAnalyzer: Analysis completed successfully for: " . $url . " - Title: " . $result['title']);
            
        } catch (Exception $e) {
            $result['error'] = $e->getMessage();
            error_log("WebsiteAnalyzer: Exception during analysis: " . $e->getMessage());
        }
        
        return $result;
    }
    
    /**
     * Site içeriğini çeker
     */
    private function fetchWebsiteContent($url) {
        $options = [
            'http' => [
                'method' => 'GET',
                'header' => [
                    'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
                    'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
                    'Accept-Language: en-US,en;q=0.5',
                    'Accept-Encoding: gzip, deflate',
                    'Connection: keep-alive'
                ],
                'timeout' => 15
            ],
            'ssl' => [
                'verify_peer' => false,
                'verify_peer_name' => false
            ]
        ];
        
        $context = stream_context_create($options);
        
        try {
            $content = @file_get_contents($url, false, $context);
            
            // GZIP decode if needed
            if ($content && isset($http_response_header)) {
                foreach ($http_response_header as $header) {
                    if (stripos($header, 'content-encoding: gzip') !== false) {
                        $content = gzdecode($content);
                        break;
                    }
                }
            }
            
            return $content;
        } catch (Exception $e) {
            return false;
        }
    }
    
    /**
     * Site logosunu bulur (hem URL hem de yerel dosya desteği)
     */
    private function findSiteLogo($dom, $baseUrl) {
        $logoUrl = '';
        
        // 1. Favicon ve site logo linklerini ara
        $linkTags = $dom->getElementsByTagName('link');
        foreach ($linkTags as $link) {
            $rel = strtolower($link->getAttribute('rel'));
            $href = $link->getAttribute('href');
            
            // Öncelik sırası: apple-touch-icon > icon > shortcut icon
            if (in_array($rel, ['apple-touch-icon', 'apple-touch-icon-precomposed'])) {
                if ($href) {
                    $logoUrl = $this->resolveUrl($href, $baseUrl);
                    break; // Apple touch icon öncelikli
                }
            } elseif (in_array($rel, ['icon', 'shortcut icon']) && empty($logoUrl)) {
                if ($href) {
                    $logoUrl = $this->resolveUrl($href, $baseUrl);
                }
            }
        }
        
        // 2. Meta tag'lerde logo ara
        if (empty($logoUrl)) {
            $metaTags = $dom->getElementsByTagName('meta');
            foreach ($metaTags as $meta) {
                $property = strtolower($meta->getAttribute('property'));
                $name = strtolower($meta->getAttribute('name'));
                $content = $meta->getAttribute('content');
                
                // Open Graph image, Twitter image vb.
                if (in_array($property, ['og:image', 'og:logo']) || 
                    in_array($name, ['twitter:image', 'msapplication-TileImage'])) {
                    if ($content) {
                        $logoUrl = $this->resolveUrl($content, $baseUrl);
                        break;
                    }
                }
            }
        }
        
        // 3. Logo class/id ile img tagları ara
        if (empty($logoUrl)) {
            $xpath = new DOMXPath($dom);
            
            // Logo arayan XPath sorguları
            $logoQueries = [
                "//img[contains(@class, 'logo')]",
                "//img[contains(@id, 'logo')]",
                "//img[contains(@alt, 'logo')]",
                "//img[contains(@src, 'logo')]",
                "//img[contains(@class, 'brand')]",
                "//img[contains(@id, 'brand')]",
                "//header//img[1]", // Header'daki ilk resim
                "//nav//img[1]"     // Nav'daki ilk resim
            ];
            
            foreach ($logoQueries as $query) {
                $nodes = $xpath->query($query);
                if ($nodes->length > 0) {
                    $src = $nodes->item(0)->getAttribute('src');
                    if ($src) {
                        $logoUrl = $this->resolveUrl($src, $baseUrl);
                        break;
                    }
                }
            }
        }
        
        // 4. Yaygın logo yollarını dene
        if (empty($logoUrl)) {
            $commonLogoPaths = [
                '/favicon.ico',
                '/logo.png',
                '/logo.jpg',
                '/logo.svg',
                '/assets/logo.png',
                '/images/logo.png',
                '/img/logo.png',
                '/static/logo.png',
                '/wp-content/uploads/logo.png'
            ];
            
            foreach ($commonLogoPaths as $path) {
                $testUrl = $this->resolveUrl($path, $baseUrl);
                if ($this->urlExists($testUrl)) {
                    $logoUrl = $testUrl;
                    break;
                }
            }
        }
        
        return $logoUrl;
    }
    
    /**
     * URL'nin mevcut olup olmadığını kontrol eder
     */
    private function urlExists($url) {
        $headers = @get_headers($url, 1);
        return $headers && strpos($headers[0], '200') !== false;
    }
    
    /**
     * Relative URL'yi absolute'a çevirir
     */
    private function resolveUrl($url, $baseUrl) {
        if (filter_var($url, FILTER_VALIDATE_URL)) {
            return $url;
        }
        
        $base = parse_url($baseUrl);
        
        if (strpos($url, '//') === 0) {
            return $base['scheme'] . ':' . $url;
        }
        
        if (strpos($url, '/') === 0) {
            return $base['scheme'] . '://' . $base['host'] . $url;
        }
        
        return $base['scheme'] . '://' . $base['host'] . '/' . ltrim($url, '/');
    }
    
    /**
     * Domain'den e-posta adresi oluşturur
     */
    private function generateEmail($domain) {
        $domain = str_replace('www.', '', $domain);
        
        $prefixes = ['info', 'contact', 'hello', 'support', 'admin'];
        $prefix = $prefixes[array_rand($prefixes)];
        
        return $prefix . '@' . $domain;
    }
    
    /**
     * OpenAI ile içerik üretir
     */
    public function generateContent($siteInfo, $minWords = 200, $maxWords = 500) {
        if (empty($this->openaiApiKey)) {
            return [
                'short_description' => $this->generateFallbackDescription($siteInfo, true),
                'long_description' => $this->generateFallbackDescription($siteInfo, false),
                'success' => false,
                'error' => 'OpenAI API key tanımlanmamış'
            ];
        }
        
        $siteName = $siteInfo['title'] ?: $siteInfo['domain'];
        $siteUrl = $siteInfo['url'];
        $metaDesc = $siteInfo['meta_description'] ?: '';
        
        // Bot ayarlarından dil tercihini al
        $language = $this->contentLanguage;
        
        // Dile göre prompt hazırla
        if ($language === 'tr') {
            $prompt = "Web Sitesi Analizi:
Site Adı: {$siteName}
URL: {$siteUrl}
Meta Açıklama: {$metaDesc}

Bu web sitesi için TÜRKÇE içerik üret. Sadece açıklamaları yaz, hiçbir başlık, etiket veya sayı kullanma.

Birinci paragraf: 1-2 cümlelik kısa özet (maksimum 40 kelime)
İkinci paragraf: Detaylı açıklama ({$minWords}-{$maxWords} kelime)

Detaylı açıklama şunları içermeli:
- Web sitesinin amacı ve misyonu  
- Sunulan hizmetler/özellikler
- Ziyaretçiler için faydalar
- İçerik çeşitliliği
- Kullanıcıların neden bu siteyi seçmesi gerektiği

Doğal, akıcı ve kullanıcı dostu Türkçe kullan. Teknik jargon yerine herkesin anlayabileceği açık terimler tercih et.

ÖNEMLİ: Hiçbir başlık, etiket, numara veya iki nokta üst üste kullanma. Sadece düz metin paragraflar yaz.";
        } else {
            $prompt = "Website Analysis:
Site Name: {$siteName}
URL: {$siteUrl}
Meta Description: {$metaDesc}

Generate content for this website in ENGLISH ONLY. Write only descriptions without any titles, labels, or numbers.

First paragraph: Brief 1-2 sentence summary (maximum 40 words)
Second paragraph: Detailed description ({$minWords}-{$maxWords} words)

Detailed description should include:
- Website's purpose and mission
- Services/features offered
- Benefits for visitors
- Content variety
- Why users should choose this site

Use natural, fluent, and user-friendly English language. Prefer clear terms that everyone can understand over technical jargon.

IMPORTANT: Do not use any titles, labels, numbers, or colons. Write only plain text paragraphs.";
        }

        try {
            $response = $this->callOpenAI($prompt);
            
            if ($response['success']) {
                $content = $response['content'];
                
                // İçeriği parse et - yeni format: iki paragraf halinde
                $shortDesc = '';
                $longDesc = '';
                
                // İçeriği satırlara böl ve boş satırları temizle
                $lines = array_filter(array_map('trim', explode("\n", $content)));
                
                if (count($lines) >= 2) {
                    // İlk paragraf kısa açıklama, ikinci paragraf uzun açıklama
                    $shortDesc = $lines[0];
                    $longDesc = implode(' ', array_slice($lines, 1));
                } elseif (count($lines) == 1) {
                    // Tek satır varsa onu iki parçaya böl
                    $sentences = explode('.', $lines[0]);
                    if (count($sentences) >= 2) {
                        $shortDesc = trim($sentences[0]) . '.';
                        $longDesc = trim(implode('.', array_slice($sentences, 1)));
                    } else {
                        $shortDesc = $lines[0];
                        $longDesc = $lines[0];
                    }
                }
                
                // Fallback: Eğer format uymazsa tüm içeriği uzun açıklama yap
                if (empty($longDesc) && !empty($content)) {
                    // Eski format için de kontrol et
                    if (preg_match('/KISA:\s*(.+?)(?=UZUN:|$)/s', $content, $matches)) {
                        $shortDesc = trim($matches[1]);
                    }
                    if (preg_match('/UZUN:\s*(.+?)$/s', $content, $matches)) {
                        $longDesc = trim($matches[1]);
                    }
                    if (preg_match('/SHORT:\s*(.+?)(?=LONG:|$)/s', $content, $matches)) {
                        $shortDesc = trim($matches[1]);
                    }
                    if (preg_match('/LONG:\s*(.+?)$/s', $content, $matches)) {
                        $longDesc = trim($matches[1]);
                    }
                    
                    // Hala boşsa
                    if (empty($longDesc)) {
                        $sentences = explode('.', $content);
                        $shortDesc = trim($sentences[0]) . '.';
                        $longDesc = trim($content);
                    }
                }
                
                // Etiketleri temizle
                $shortDesc = $this->cleanDescriptionLabels($shortDesc);
                $longDesc = $this->cleanDescriptionLabels($longDesc);
                
                // Türkçe karakterleri düzelt
                $shortDesc = fixTurkishEncoding($shortDesc);
                $longDesc = fixTurkishEncoding($longDesc);
                
                return [
                    'short_description' => $shortDesc,
                    'long_description' => $longDesc,
                    'success' => true
                ];
            } else {
                return [
                    'short_description' => $this->generateFallbackDescription($siteInfo, true),
                    'long_description' => $this->generateFallbackDescription($siteInfo, false),
                    'success' => false,
                    'error' => $response['error']
                ];
            }
            
        } catch (Exception $e) {
            return [
                'short_description' => $this->generateFallbackDescription($siteInfo, true),
                'long_description' => $this->generateFallbackDescription($siteInfo, false),
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Açıklamalardan etiketleri temizle
     */
    private function cleanDescriptionLabels($text) {
        if (empty($text)) return $text;
        
        // Türkçe etiketleri temizle - daha kapsamlı
        $text = preg_replace('/^(Kısa açıklama|Uzun açıklama|KISA AÇIKLAMA|UZUN AÇIKLAMA):\s*/i', '', $text);
        $text = preg_replace('/^(Kısa|Uzun|KISA|UZUN|Açıklama|AÇIKLAMA):\s*/i', '', $text);
        
        // İngilizce etiketleri temizle - daha kapsamlı  
        $text = preg_replace('/^(Short description|Long description|SHORT DESCRIPTION|LONG DESCRIPTION):\s*/i', '', $text);
        $text = preg_replace('/^(Short|Long|SHORT|LONG|Description|DESCRIPTION):\s*/i', '', $text);
        
        // Genel etiketler
        $text = preg_replace('/^(Summary|SUMMARY|Overview|OVERVIEW|About|ABOUT):\s*/i', '', $text);
        
        // Satır başındaki gereksiz karakterleri temizle
        $text = preg_replace('/^\s*[\-\*\•]\s*/', '', $text); // Tire, yıldız, bullet temizle
        $text = preg_replace('/^\d+\.\s*/', '', $text); // "1. " gibi numaraları temizle
        $text = preg_replace('/^\s*[:;]\s*/', '', $text); // Başlangıçtaki noktalı virgül vs temizle
        
        // Çok kez tekrar eden temizlik (iç içe etiketler için)
        for ($i = 0; $i < 3; $i++) {
            $text = preg_replace('/^(UZUN AÇIKLAMA|KISA AÇIKLAMA|LONG DESCRIPTION|SHORT DESCRIPTION):\s*/i', '', $text);
            $text = preg_replace('/^(Açıklama|Description|Summary|Overview):\s*/i', '', $text);
        }
        
        $text = trim($text);
        
        return $text;
    }
    
    /**
     * OpenAI API çağrısı
     */
    private function callOpenAI($prompt) {
        $data = [
            'model' => 'gpt-3.5-turbo',
            'messages' => [
                [
                    'role' => 'user',
                    'content' => $prompt
                ]
            ],
            'max_tokens' => 1000,
            'temperature' => 0.7
        ];
        
        $options = [
            'http' => [
                'header' => [
                    "Content-Type: application/json",
                    "Authorization: Bearer " . $this->openaiApiKey
                ],
                'method' => 'POST',
                'content' => json_encode($data),
                'timeout' => 30
            ]
        ];
        
        $context = stream_context_create($options);
        $response = @file_get_contents('https://api.openai.com/v1/chat/completions', false, $context);
        
        if ($response === false) {
            return ['success' => false, 'error' => 'API bağlantı hatası'];
        }
        
        $result = json_decode($response, true);
        
        if (isset($result['choices'][0]['message']['content'])) {
            return [
                'success' => true,
                'content' => trim($result['choices'][0]['message']['content'])
            ];
        } else {
            $error = isset($result['error']) ? $result['error']['message'] : 'Bilinmeyen API hatası';
            return ['success' => false, 'error' => $error];
        }
    }
    
    /**
     * Fallback açıklama üretir (AI çalışmazsa)
     */
    private function generateFallbackDescription($siteInfo, $isShort = false) {
        $siteName = $siteInfo['title'] ?: ucfirst(str_replace(['www.', '.com', '.net', '.org'], '', $siteInfo['domain']));
        $domain = $siteInfo['domain'];
        
        // Bot ayarlarından dil tercihini al
        $language = $this->contentLanguage;
        
        if ($language === 'tr') {
            if ($isShort) {
                $templates = [
                    "{$siteName}, kullanıcılarına kaliteli hizmet sunan güvenilir bir web sitesidir.",
                    "{$siteName} platformu, modern çözümler ve kullanıcı dostu arayüzü ile öne çıkar.",
                    "{$siteName}, alanında uzman ekibi ile hizmet veren lider bir web sitesidir."
                ];
            } else {
                $templates = [
                    "{$siteName}, {$domain} adresinde yer alan ve kullanıcılarına çeşitli dijital hizmetler sunan kapsamlı bir online platformdur. Bu web sitesi, modern teknolojiler kullanılarak geliştirilmiş olup, ziyaretçiler için kaliteli bir deneyim sunmak amacıyla tasarlanmış kullanıcı dostu bir arayüz ve işlevsel özellikler barındırmaktadır. Platform üzerinden sunulan hizmetler, güncel ihtiyaçları karşılayacak şekilde düzenlenmiş olup sürekli olarak geliştirilmekte ve güncellenmektedir. {$siteName}, dijital alanda güvenilir ve etkili çözümler arayan kullanıcılar için ideal bir tercih olarak öne çıkmaktadır.",
                    
                    "{$domain} adresinde faaliyet gösteren {$siteName}, kullanıcılarına geniş bir hizmet yelpazesi sunan dinamik bir web platformudur. Site, en güncel web teknolojileri kullanılarak geliştirilmiş olup, kullanıcı deneyimini ön planda tutan bir tasarıma sahiptir. Platform üzerinden erişilebilen hizmetler, farklı ihtiyaç ve gereksinimlere hitap edecek şekilde organize edilmiştir. {$siteName}, kaliteli içerik ve güvenilir hizmet standartları ile kullanıcılarına değer katma hedefindedir. Web sitesi, düzenli olarak güncellenen içeriği ve yenilikçi özellikleri ile sektörde kendini farklılaştırmaktadır."
                ];
            }
        } else {
            if ($isShort) {
                $templates = [
                    "{$siteName} - Online platform providing digital services and web solutions.",
                    "{$siteName} is a comprehensive web platform offering various online services.",
                    "Discover {$siteName} at {$domain} for quality digital services and solutions."
                ];
            } else {
                $templates = [
                    "{$siteName} is a comprehensive online platform located at {$domain}, providing various digital services to users worldwide. This website has been developed using modern technologies and offers a user-friendly interface with functional features designed to deliver a quality experience for visitors. The services provided through the platform are tailored to meet current needs and are continuously being improved and updated. {$siteName} stands out as an ideal choice for users seeking reliable and effective solutions in the digital space.",
                    
                    "Located at {$domain}, {$siteName} is a dynamic web platform that offers a wide variety of online services to its users. The site has been developed using cutting-edge web technologies and features a design that prioritizes user experience. The services accessible through the platform are organized to cater to different needs and requirements. {$siteName} aims to add value to its users through its commitment to quality content and reliable service standards. The website distinguishes itself in the industry with regularly updated content and innovative features."
                ];
            }
        }
        
        return $templates[array_rand($templates)];
    }
    
    /**
     * Site içeriğinin dilini algılar
     */
    public function detectLanguage($siteInfo) {
        $text = '';
        
        // Analiz edilecek metin topla
        if (!empty($siteInfo['title'])) {
            $text .= $siteInfo['title'] . ' ';
        }
        if (!empty($siteInfo['meta_description'])) {
            $text .= $siteInfo['meta_description'] . ' ';
        }
        
        $text = strtolower(trim($text));
        
        // Türkçe karakterler ve kelimeler
        $turkishChars = ['ç', 'ğ', 'ı', 'ö', 'ş', 'ü'];
        $turkishWords = [
            'hakkında', 'hakkımızda', 'anasayfa', 'iletişim', 'ürünler', 'hizmetler', 'blog', 
            'haberler', 'galeri', 'referanslar', 'ekip', 'takım', 'proje', 'çözümler',
            'kalite', 'güvenlik', 'başarı', 'deneyim', 'uzman', 'profesyonel', 'türkiye',
            'istanbul', 'ankara', 'izmir', 'bursa', 'antalya', 'adana', 'konya', 'gaziantep',
            'şirket', 'firma', 'kurum', 'kuruluş', 'organizasyon', 'dernek', 'vakıf',
            'eğitim', 'öğretim', 'kurs', 'seminar', 'sertifika', 'diploma',
            'sağlık', 'hastane', 'doktor', 'hekim', 'tedavi', 'ilaç', 'hasta',
            've', 'ile', 'için', 'olan', 'bir', 'bu', 'şu', 'o', 'en', 'çok', 'daha',
            'nasıl', 'neden', 'nerede', 'ne', 'kim', 'hangi', 'kaç', 'kadar'
        ];
        
        // İngilizce kelimeler
        $englishWords = [
            'about', 'home', 'contact', 'products', 'services', 'blog', 'news', 'gallery',
            'team', 'project', 'solutions', 'quality', 'security', 'success', 'experience',
            'expert', 'professional', 'company', 'business', 'organization', 'education',
            'health', 'hospital', 'doctor', 'treatment', 'medicine', 'patient',
            'and', 'with', 'for', 'that', 'this', 'the', 'a', 'an', 'is', 'are', 'was', 'were',
            'how', 'why', 'where', 'what', 'who', 'which', 'when', 'much', 'many', 'more'
        ];
        
        $turkishScore = 0;
        $englishScore = 0;
        
        // Türkçe karakter kontrolü
        foreach ($turkishChars as $char) {
            $turkishScore += substr_count($text, $char) * 3; // Türkçe karakterler yüksek puan
        }
        
        // Kelime kontrolü
        foreach ($turkishWords as $word) {
            if (strpos($text, $word) !== false) {
                $turkishScore += 2;
            }
        }
        
        foreach ($englishWords as $word) {
            if (strpos($text, $word) !== false) {
                $englishScore += 1;
            }
        }
        
        // Domain kontrolü (.tr, .com.tr gibi)
        $domain = $siteInfo['domain'] ?? '';
        if (strpos($domain, '.tr') !== false || strpos($domain, '.com.tr') !== false) {
            $turkishScore += 5;
        }
        
        // Sonuç döndür
        if ($turkishScore > $englishScore) {
            return 'turkish';
        } elseif ($englishScore > $turkishScore) {
            return 'english';
        } else {
            // Belirsiz durumlarda domain'e bak
            if (strpos($domain, '.tr') !== false) {
                return 'turkish';
            } else {
                return 'english'; // Varsayılan
            }
        }
    }
    
    /**
     * Site içeriğine göre akıllı kategori seçer - Geliştirilmiş versiyon
     */
    public function getSmartCategory($siteInfo) {
        $siteName = $siteInfo['title'] ?: '';
        $siteUrl = $siteInfo['url'];
        $metaDesc = $siteInfo['meta_description'] ?: '';
        $domain = $siteInfo['domain'];
        
        // Analiz edilecek text - daha kapsamlı
        $analyzeText = strtolower($siteName . ' ' . $metaDesc . ' ' . $domain . ' ' . $siteUrl);
        
        error_log("WebsiteAnalyzer: Analyzing text for category: " . substr($analyzeText, 0, 200));
        
        // Gelişmiş kategori anahtar kelimeleri (Türkçe + İngilizce)
        $categoryKeywords = [
            1 => [ // Foreign Language
                'turkish' => ['dil', 'yabancı', 'çeviri', 'tercüme', 'İngilizce', 'Almanca', 'Fransızca', 'İspanyolca', 'kurs', 'öğren'],
                'english' => ['language', 'foreign', 'translate', 'translation', 'english', 'spanish', 'french', 'german', 'learn', 'study', 'course', 'tutor', 'speak', 'fluent', 'grammar', 'vocabulary']
            ],
            2 => [ // Transport
                'turkish' => ['ulaşım', 'havayolu', 'otobüs', 'tren', 'kargo', 'nakliye', 'bilet', 'rezervasyon', 'seyahat'],
                'english' => ['transport', 'airline', 'flight', 'bus', 'train', 'shipping', 'logistics', 'cargo', 'ticket', 'booking', 'travel', 'journey', 'route', 'schedule']
            ],
            3 => [ // Tourism & Travel
                'turkish' => ['turizm', 'seyahat', 'otel', 'tatil', 'gezi', 'rezervasyon', 'rehber', 'tur', 'plaj', 'konaklama'],
                'english' => ['tourism', 'travel', 'hotel', 'vacation', 'trip', 'destination', 'adventure', 'resort', 'booking', 'holiday', 'tourist', 'guide', 'accommodation', 'beach', 'cruise']
            ],
            4 => [ // Sports & Teams
                'turkish' => ['spor', 'takım', 'fitness', 'jimnastik', 'futbol', 'basketbol', 'atletizm', 'antrenör', 'maç', 'espn', 'sports', 'athletic'],
                'english' => ['sport', 'team', 'fitness', 'gym', 'football', 'basketball', 'soccer', 'athletic', 'exercise', 'training', 'coach', 'match', 'tournament', 'league', 'player', 'workout', 'espn', 'sports', 'athletic', 'nfl', 'nba', 'mlb', 'nhl']
            ],
            5 => [ // Industry
                'turkish' => ['sanayi', 'imalat', 'inşaat', 'mühendislik', 'fabrika', 'üretim', 'endüstri', 'makine', 'teknoloji'],
                'english' => ['industry', 'manufacturing', 'construction', 'engineering', 'factory', 'industrial', 'production', 'machinery', 'automation', 'supply', 'chain']
            ],
            6 => [ // Health & Life
                'turkish' => ['sağlık', 'tıp', 'hastane', 'doktor', 'klinik', 'fitness', 'yaşam', 'diyet', 'beslenme', 'webmd', 'health', 'medical'],
                'english' => ['health', 'medical', 'wellness', 'fitness', 'healthcare', 'doctor', 'hospital', 'clinic', 'life', 'diet', 'nutrition', 'medicine', 'therapy', 'mental', 'physical', 'webmd', 'health', 'medical', 'wellness', 'symptoms', 'treatment', 'diagnosis']
            ],
            7 => [ // Culture & Art
                'turkish' => ['kültür', 'sanat', 'müze', 'galeri', 'müzik', 'sanatçı', 'sergi', 'resim', 'heykel'],
                'english' => ['culture', 'art', 'museum', 'gallery', 'music', 'artist', 'creative', 'cultural', 'design', 'painting', 'sculpture', 'exhibition', 'performance', 'theater']
            ],
            8 => [ // Business
                'turkish' => ['işletme', 'şirket', 'danışmanlık', 'girişim', 'ticaret', 'kurumsal', 'iş', 'pazarlama', 'satış'],
                'english' => ['business', 'corporate', 'company', 'consulting', 'professional', 'startup', 'enterprise', 'marketing', 'sales', 'finance', 'commerce', 'management']
            ],
            9 => [ // Internet & Tech
                'turkish' => ['teknoloji', 'yazılım', 'internet', 'bilgisayar', 'uygulama', 'web', 'tasarım', 'geliştirici', 'kod', 'programlama', 'api', 'veritabanı', 'sunucu', 'hosting'],
                'english' => ['tech', 'technology', 'software', 'developer', 'programming', 'code', 'app', 'digital', 'it', 'web', 'internet', 'computer', 'development', 'ai', 'automation', 'cloud', 'data', 'api', 'database', 'server', 'hosting', 'platform', 'tool', 'service']
            ],
            10 => [ // Law & Consulting
                'turkish' => ['hukuk', 'avukat', 'danışmanlık', 'mali', 'müşavir', 'hukuki', 'adalet', 'mahkeme'],
                'english' => ['law', 'legal', 'lawyer', 'attorney', 'consulting', 'advisor', 'financial', 'consultant', 'justice', 'court', 'litigation', 'contract']
            ],
            11 => [ // Food & Agri
                'turkish' => ['yemek', 'restoran', 'tarım', 'çiftlik', 'aşçı', 'mutfak', 'gıda', 'yemek', 'tarif'],
                'english' => ['food', 'restaurant', 'recipe', 'cooking', 'agriculture', 'farm', 'culinary', 'agri', 'chef', 'kitchen', 'cuisine', 'organic', 'farming', 'harvest']
            ],
            12 => [ // Real Estate
                'turkish' => ['emlak', 'ev', 'daire', 'kiralık', 'satılık', 'inşaat', 'gayrimenkul', 'konut'],
                'english' => ['real', 'estate', 'property', 'house', 'home', 'rent', 'buy', 'apartment', 'building', 'residential', 'commercial', 'investment', 'mortgage']
            ],
            13 => [ // Entertainment
                'turkish' => ['eğlence', 'sinema', 'film', 'oyun', 'müzik', 'konser', 'show', 'tv', 'dizi', 'imdb', 'movie', 'film'],
                'english' => ['entertainment', 'movie', 'film', 'game', 'gaming', 'music', 'show', 'theater', 'cinema', 'tv', 'series', 'concert', 'comedy', 'fun', 'imdb', 'movie', 'film', 'cinema', 'actor', 'actress', 'director', 'rating', 'review']
            ],
            14 => [ // Education
                'turkish' => ['eğitim', 'okul', 'üniversite', 'kurs', 'öğretim', 'akademik', 'ders', 'öğrenci'],
                'english' => ['education', 'school', 'university', 'course', 'learning', 'student', 'teacher', 'academic', 'study', 'lesson', 'training', 'certificate', 'degree']
            ],
            15 => [ // Religion & Ethics
                'turkish' => ['din', 'camii', 'kilise', 'inanç', 'dini', 'manevi', 'ahlak', 'ibadethane'],
                'english' => ['religion', 'church', 'faith', 'spiritual', 'religious', 'belief', 'ethics', 'moral', 'worship', 'prayer', 'sacred']
            ],
            16 => [ // Government
                'turkish' => ['devlet', 'belediye', 'kamu', 'resmi', 'hükümet', 'bakanlık', 'kurum', 'bürokrasi'],
                'english' => ['government', 'official', 'public', 'municipal', 'state', 'federal', 'agency', 'gov', 'administration', 'bureau', 'service']
            ],
            17 => [ // Press & Media
                'turkish' => ['haber', 'gazete', 'medya', 'basın', 'televizyon', 'radyo', 'haberci', 'yayın'],
                'english' => ['news', 'media', 'press', 'journalist', 'newspaper', 'magazine', 'broadcast', 'information', 'radio', 'television', 'reporting', 'journalism']
            ],
            18 => [ // Shopping
                'turkish' => ['alışveriş', 'mağaza', 'market', 'satış', 'e-ticaret', 'online', 'mağaza', 'ürün'],
                'english' => ['shop', 'shopping', 'store', 'ecommerce', 'retail', 'buy', 'sell', 'marketplace', 'fashion', 'product', 'commerce', 'online', 'purchase']
            ]
        ];
        
        $categoryScores = [];
        
        // Her kategori için skor hesapla (ağırlıklı)
        foreach ($categoryKeywords as $categoryId => $languages) {
            $score = 0;
            
            // Türkçe kelimeler için skor
            foreach ($languages['turkish'] as $keyword) {
                if (strpos($analyzeText, $keyword) !== false) {
                    $score += 2; // Türkçe kelimelere daha yüksek ağırlık
                }
            }
            
            // İngilizce kelimeler için skor
            foreach ($languages['english'] as $keyword) {
                if (strpos($analyzeText, $keyword) !== false) {
                    $score += 1;
                }
            }
            
            // Domain kontrolü - ekstra puan
            if (strpos($domain, $keyword) !== false) {
                $score += 3; // Domain'de geçen kelimeler için bonus
            }
            
            $categoryScores[$categoryId] = $score;
        }
        
        // En yüksek skoru bulan kategoriyi seç
        $maxScore = max($categoryScores);
        if ($maxScore > 0) {
            $selectedCategory = array_search($maxScore, $categoryScores);
            error_log("WebsiteAnalyzer: Category selected by keywords: " . $selectedCategory . " (score: " . $maxScore . ")");
        } else {
            // Hiç match yoksa domain analizi yap
            $selectedCategory = $this->analyzeDomainForCategory($domain);
            if (!$selectedCategory) {
                // Son çare olarak rastgele kategori seç (Other hariç)
                $availableCategories = range(1, 18);
                $selectedCategory = $availableCategories[array_rand($availableCategories)];
                error_log("WebsiteAnalyzer: Random category selected: " . $selectedCategory);
            } else {
                error_log("WebsiteAnalyzer: Category selected by domain analysis: " . $selectedCategory);
            }
        }
        
        // Kategori ID'sinin geçerliliğini kontrol et
        try {
            $stmt = $this->pdo->prepare("SELECT id FROM categories WHERE id = ?");
            $stmt->execute([$selectedCategory]);
            if ($stmt->fetchColumn()) {
                return $selectedCategory;
            } else {
                return $this->getRandomCategory();
            }
        } catch (Exception $e) {
            return $this->getRandomCategory();
        }
    }
    
    /**
     * Domain adına göre kategori analizi - Geliştirilmiş versiyon
     */
    private function analyzeDomainForCategory($domain) {
        $domain = strtolower($domain);
        error_log("WebsiteAnalyzer: Analyzing domain for category: " . $domain);
        
        $domainPatterns = [
            9 => ['tech', 'app', 'dev', 'code', 'soft', 'digital', 'cloud', 'data', 'ai', 'api', 'programming', 'software', 'platform', 'tool', 'service'],
            8 => ['business', 'corp', 'company', 'consulting', 'pro', 'enterprise', 'startup', 'finance', 'marketing'],
            14 => ['edu', 'academy', 'school', 'university', 'learn', 'education', 'student', 'teacher', 'course'],
            6 => ['health', 'medical', 'clinic', 'wellness', 'doctor', 'hospital', 'fitness', 'therapy', 'webmd', 'health', 'medical', 'symptoms', 'treatment'],
            11 => ['food', 'restaurant', 'recipe', 'cooking', 'chef', 'kitchen', 'cuisine', 'agriculture', 'farm'],
            18 => ['shop', 'store', 'buy', 'sell', 'market', 'commerce', 'retail', 'ecommerce', 'shopping'],
            17 => ['news', 'media', 'press', 'tv', 'radio', 'journalism', 'broadcast', 'magazine'],
            3 => ['travel', 'tour', 'hotel', 'vacation', 'tourism', 'booking', 'resort', 'destination'],
            16 => ['gov', 'government', 'official', 'municipal', 'state', 'federal', 'agency'],
            12 => ['realty', 'estate', 'property', 'home', 'house', 'apartment', 'rent', 'mortgage'],
            4 => ['sport', 'fitness', 'gym', 'athletic', 'team', 'football', 'basketball', 'soccer', 'espn', 'sports', 'nfl', 'nba', 'mlb', 'nhl'],
            7 => ['art', 'culture', 'museum', 'gallery', 'music', 'creative', 'design', 'theater'],
            10 => ['law', 'legal', 'attorney', 'lawyer', 'court', 'justice', 'litigation'],
            13 => ['entertainment', 'movie', 'film', 'game', 'gaming', 'show', 'cinema', 'comedy', 'imdb', 'movie', 'film', 'cinema', 'actor', 'rating'],
            15 => ['religion', 'church', 'faith', 'spiritual', 'worship', 'prayer', 'sacred'],
            1 => ['language', 'translate', 'translation', 'english', 'spanish', 'french', 'german'],
            2 => ['transport', 'airline', 'flight', 'bus', 'train', 'shipping', 'logistics', 'cargo'],
            5 => ['industry', 'manufacturing', 'construction', 'engineering', 'factory', 'industrial']
        ];
        
        foreach ($domainPatterns as $categoryId => $patterns) {
            foreach ($patterns as $pattern) {
                if (strpos($domain, $pattern) !== false) {
                    error_log("WebsiteAnalyzer: Domain matched pattern '$pattern' for category $categoryId");
                    return $categoryId;
                }
            }
        }
        
        error_log("WebsiteAnalyzer: No domain pattern matched");
        return null;
    }
    
    /**
     * Logo URL'sini çıkarır
     */
    public function extractLogo($url) {
        try {
            // URL'yi düzenle
            if (!preg_match('/^https?:\/\//', $url)) {
                $url = 'https://' . $url;
            }
            
            $domain = parse_url($url, PHP_URL_HOST);
            
            // Yaygın logo yolları
            $commonLogoPaths = [
                '/logo.png',
                '/logo.jpg',
                '/logo.svg',
                '/logo.gif',
                '/favicon.ico',
                '/images/logo.png',
                '/images/logo.jpg',
                '/images/logo.svg',
                '/img/logo.png',
                '/img/logo.jpg',
                '/img/logo.svg',
                '/assets/logo.png',
                '/assets/logo.jpg',
                '/assets/logo.svg',
                '/static/logo.png',
                '/static/logo.jpg',
                '/static/logo.svg'
            ];
            
            // Yaygın yolları dene
            foreach ($commonLogoPaths as $path) {
                $logoUrl = 'https://' . $domain . $path;
                if ($this->checkUrlExists($logoUrl)) {
                    return $logoUrl;
                }
            }
            
            // Site içeriğini al ve logo ara
            $html = $this->fetchWebsiteContent($url);
            if ($html) {
                $dom = new DOMDocument();
                @$dom->loadHTML($html);
                $xpath = new DOMXPath($dom);
                
                // Logo arayan XPath sorguları
                $logoQueries = [
                    "//img[contains(@class, 'logo')]",
                    "//img[contains(@id, 'logo')]",
                    "//img[contains(@alt, 'logo')]",
                    "//img[contains(@src, 'logo')]",
                    "//img[contains(@class, 'brand')]",
                    "//img[contains(@id, 'brand')]",
                    "//header//img[1]",
                    "//nav//img[1]"
                ];
                
                foreach ($logoQueries as $query) {
                    $nodes = $xpath->query($query);
                    if ($nodes->length > 0) {
                        $img = $nodes->item(0);
                        $src = $img->getAttribute('src');
                        if (!empty($src)) {
                            // Relative URL'yi absolute yap
                            if (strpos($src, 'http') !== 0) {
                                if (strpos($src, '/') === 0) {
                                    $src = 'https://' . $domain . $src;
                                } else {
                                    $src = 'https://' . $domain . '/' . $src;
                                }
                            }
                            return $src;
                        }
                    }
                }
            }
            
            return '';
        } catch (Exception $e) {
            error_log("WebsiteAnalyzer: Logo extraction error: " . $e->getMessage());
            return '';
        }
    }
    
    /**
     * URL'nin var olup olmadığını kontrol eder
     */
    private function checkUrlExists($url) {
        $headers = @get_headers($url, 1);
        return $headers && strpos($headers[0], '200') !== false;
    }
    
    /**
     * Rastgele kategori seçer
     */
    public function getRandomCategory() {
        try {
            $stmt = $this->pdo->query("SELECT id FROM categories ORDER BY RAND() LIMIT 1");
            $result = $stmt->fetchColumn();
            return $result ?: 1; // Varsayılan kategori ID
        } catch (Exception $e) {
            return 1;
        }
    }
}
?>