<?php
/**
 * Ban Kontrol Sistemi
 * Kullanıcı ve IP ban kontrollerini yapar
 */

/**
 * IP adresinin banlı olup olmadığını kontrol eder
 * @param PDO $pdo Veritabanı bağlantısı
 * @param string $ip_address Kontrol edilecek IP adresi
 * @return bool Ban durumu (true = banlı, false = banlı değil)
 */
function isIPBanned($pdo, $ip_address) {
    if (empty($ip_address)) {
        return false;
    }
    
    try {
        $stmt = $pdo->prepare('SELECT COUNT(*) FROM banned_ips WHERE ip_address = ? AND is_active = 1');
        $stmt->execute([$ip_address]);
        return $stmt->fetchColumn() > 0;
    } catch (Exception $e) {
        error_log('IP ban kontrolü hatası: ' . $e->getMessage());
        return false;
    }
}

/**
 * Kullanıcının banlı olup olmadığını kontrol eder
 * @param PDO $pdo Veritabanı bağlantısı
 * @param int $user_id Kullanıcı ID'si
 * @return bool Ban durumu (true = banlı, false = banlı değil)
 */
function isUserBanned($pdo, $user_id) {
    if (empty($user_id)) {
        return false;
    }
    
    try {
        $stmt = $pdo->prepare('SELECT is_banned FROM user WHERE id = ?');
        $stmt->execute([$user_id]);
        $result = $stmt->fetchColumn();
        return $result == 1;
    } catch (Exception $e) {
        error_log('Kullanıcı ban kontrolü hatası: ' . $e->getMessage());
        return false;
    }
}

/**
 * Kullanıcının IP'si veya hesabı banlı mı kontrol eder
 * @param PDO $pdo Veritabanı bağlantısı
 * @param int|null $user_id Kullanıcı ID'si (giriş yapmış kullanıcılar için)
 * @param string|null $ip_address IP adresi
 * @return array Ban durumu bilgileri
 */
function checkBanStatus($pdo, $user_id = null, $ip_address = null) {
    $result = [
        'is_banned' => false,
        'ban_type' => '',
        'message' => ''
    ];
    
    // IP ban kontrolü
    if (!empty($ip_address) && isIPBanned($pdo, $ip_address)) {
        $result['is_banned'] = true;
        $result['ban_type'] = 'ip';
        $result['message'] = 'IP adresiniz yasaklı listesinde. Bu nedenle yeni kayıt oluşturamaz ve site ekleyemezsiniz.';
        return $result;
    }
    
    // Kullanıcı ban kontrolü
    if (!empty($user_id) && isUserBanned($pdo, $user_id)) {
        $result['is_banned'] = true;
        $result['ban_type'] = 'user';
        $result['message'] = 'Hesabınız yasaklanmış durumda. Bu nedenle site ekleyemezsiniz. Giriş yapmaya devam edebilirsiniz.';
        return $result;
    }
    
    return $result;
}

/**
 * Kullanıcının IP adresini güvenli şekilde alır
 * @return string IP adresi
 */
function getUserIP() {
    // Proxy'ler için gerçek IP'yi almaya çalış
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
        $ip = $_SERVER['HTTP_CLIENT_IP'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED'])) {
        $ip = $_SERVER['HTTP_X_FORWARDED'];
    } elseif (!empty($_SERVER['HTTP_FORWARDED_FOR'])) {
        $ip = $_SERVER['HTTP_FORWARDED_FOR'];
    } elseif (!empty($_SERVER['HTTP_FORWARDED'])) {
        $ip = $_SERVER['HTTP_FORWARDED'];
    } else {
        $ip = $_SERVER['REMOTE_ADDR'] ?? '';
    }
    
    // IP adresini temizle ve doğrula
    $ip = trim($ip);
    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
        return $ip;
    }
    
    // Geçerli değilse REMOTE_ADDR'ı kullan
    return $_SERVER['REMOTE_ADDR'] ?? '';
}

/**
 * Ban mesajını kullanıcıya göster ve işlemi durdur
 * @param string $message Ban mesajı
 * @param string $redirect_url Yönlendirilecek sayfa (opsiyonel)
 */
function showBanMessage($message, $redirect_url = null) {
    if ($redirect_url) {
        header('Location: ' . $redirect_url . '?error=banned&message=' . urlencode($message));
        exit;
    }
    
    // JSON response için
    if (isset($_SERVER['CONTENT_TYPE']) && strpos($_SERVER['CONTENT_TYPE'], 'application/json') !== false) {
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'error' => 'banned',
            'message' => $message
        ]);
        exit;
    }
    
    // Normal sayfa response için
    echo '<!DOCTYPE html>
    <html lang="tr">
    <head>
        <meta charset="utf-8">
        <title>Erişim Engellendi</title>
        <style>
            body { font-family: Arial, sans-serif; margin: 50px; text-align: center; }
            .ban-message { background: #fee; border: 1px solid #fcc; padding: 20px; border-radius: 5px; color: #c00; }
        </style>
    </head>
    <body>
        <div class="ban-message">
            <h2>Erişim Engellendi</h2>
            <p>' . htmlspecialchars($message) . '</p>
            <p><a href="/">Ana Sayfaya Dön</a></p>
        </div>
    </body>
    </html>';
    exit;
}
?>