<?php
require_once 'includes/config.php';
require_once 'utils/ads_helper.php';

// Sayfa parametreleri
$page = max(1, intval($_GET['page'] ?? 1));
$per_page = 12;
$offset = ($page - 1) * $per_page;

// Arama sorgusu
$search_query = trim($_GET['q'] ?? '');

// Settings'i yükle
$settings = [];
try {
    $stmt = $pdo->query('SELECT setting_key, setting_value FROM settings');
    foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
} catch (Exception $e) {
    $settings = [];
}

// Tema ayarını al
$theme = $settings['theme'] ?? 'light';
$is_dark = $theme === 'dark';

// Sayfa tipini tespit et
$current_page_type = detectPageType();

// Veritabanı bağlantısını kontrol et
$database_available = false;
$sites_table_exists = false;
$error_message = '';

try {
    $pdo->query("SELECT 1");
    $database_available = true;
    
    // site_submissions tablosunun varlığını kontrol et
    $stmt = $pdo->query("SHOW TABLES LIKE 'site_submissions'");
    if ($stmt->rowCount() > 0) {
        $sites_table_exists = true;
    }
} catch (Exception $e) {
    $error_message = "Veritabanı bağlantı hatası: " . $e->getMessage();
}

// Arama sonuçları
$results = [];
$total_results = 0;
$total_pages = 0;

if ($database_available && $sites_table_exists && !empty($search_query)) {
    $search_term = '%' . $search_query . '%';
    
    // Toplam sonuç sayısını al
    $count_stmt = $pdo->prepare("
        SELECT COUNT(*) as total 
        FROM site_submissions s 
        LEFT JOIN categories c ON s.category_id = c.id 
        WHERE s.status = 'approved' 
        AND (s.site_name LIKE ? OR s.short_description LIKE ? OR s.long_description LIKE ? OR s.site_url LIKE ? OR c.name LIKE ?)
    ");
    $count_stmt->execute([$search_term, $search_term, $search_term, $search_term, $search_term]);
    $total_results = $count_stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Sayfalandırılmış sonuçları al
    $stmt = $pdo->prepare("
        SELECT s.id, s.site_name as title, s.site_url as url, 
               COALESCE(s.long_description, s.short_description) as description,
               s.oy_durumu as votes, s.view_count as views, s.logo_url,
               c.name as category_name, c.color as category_color, c.icon as category_icon
        FROM site_submissions s 
        LEFT JOIN categories c ON s.category_id = c.id 
        WHERE s.status = 'approved' 
        AND (s.site_name LIKE ? OR s.short_description LIKE ? OR s.long_description LIKE ? OR s.site_url LIKE ? OR c.name LIKE ?)
        ORDER BY s.oy_durumu DESC, s.olusturulma_tarihi DESC
        LIMIT " . intval($per_page) . " OFFSET " . intval($offset) . "
    ");
    $stmt->execute([$search_term, $search_term, $search_term, $search_term, $search_term]);
    $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $total_pages = ceil($total_results / $per_page);
} else if (!empty($search_query)) {
    // Fallback for when database is not available or sites table is missing
    $error_message = $error_message ?: "Veritabanı bağlantısı veya eksik 'site_submissions' tablosu nedeniyle arama sonuçları görüntülenemiyor.";
}
?>
<!DOCTYPE html>
<html lang="tr" class="<?php echo $is_dark ? 'dark' : ''; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>"<?php echo htmlspecialchars($search_query); ?>" için Arama Sonuçları - <?php echo htmlspecialchars($settings['site_title'] ?? 'Web Dizini'); ?></title>
    
    <!-- Preconnect for performance -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link rel="preconnect" href="https://cdn.jsdelivr.net">
    
    <!-- Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    
    <!-- Tailwind CSS -->
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            darkMode: 'class',
            theme: {
                extend: {
                    fontFamily: {
                        'sans': ['Inter', 'system-ui', 'sans-serif'],
                    },
                    animation: {
                        'float': 'float 6s ease-in-out infinite',
                        'gradient': 'gradient 15s ease infinite',
                    },
                    keyframes: {
                        float: {
                            '0%, 100%': { transform: 'translateY(0px)' },
                            '50%': { transform: 'translateY(-20px)' },
                        },
                        gradient: {
                            '0%, 100%': { backgroundPosition: '0% 50%' },
                            '50%': { backgroundPosition: '100% 50%' },
                        }
                    }
                }
            }
        }
    </script>
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        :root {
            --primary-color: <?php echo htmlspecialchars($settings['primary_color'] ?? '#3b82f6'); ?>;
            --gradient-start-color: <?php echo htmlspecialchars($settings['gradient_start_color'] ?? '#3b82f6'); ?>;
            --gradient-end-color: <?php echo htmlspecialchars($settings['gradient_end_color'] ?? '#8b5cf6'); ?>;
        }
        
        .gradient-text {
            background: linear-gradient(135deg, var(--gradient-start-color), var(--gradient-end-color));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
        
        .gradient-bg {
            background: linear-gradient(135deg, var(--gradient-start-color), var(--gradient-end-color));
        }
        
        .glass-effect {
            background: rgba(255, 255, 255, 0.1);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        
        .dark .glass-effect {
            background: rgba(0, 0, 0, 0.2);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .text-primary { color: var(--primary-color) !important; }
        .bg-primary { background-color: var(--primary-color) !important; }
        .border-primary { border-color: var(--primary-color) !important; }
        .hover\:bg-primary:hover { background-color: var(--primary-color) !important; }
        .hover\:text-primary:hover { color: var(--primary-color) !important; }
        
        .line-clamp-2 {
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }
        
        .line-clamp-3 {
            display: -webkit-box;
            -webkit-line-clamp: 3;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }
        
        .card-hover {
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .card-hover:hover {
            transform: translateY(-8px);
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
        }
        
        .dark .card-hover:hover {
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.5);
        }
        
        .search-highlight {
            background: linear-gradient(120deg, rgba(59, 130, 246, 0.3) 0%, rgba(59, 130, 246, 0.1) 100%);
            padding: 2px 6px;
            border-radius: 6px;
            font-weight: 500;
        }
        
        .dark .search-highlight {
            background: linear-gradient(120deg, rgba(59, 130, 246, 0.4) 0%, rgba(59, 130, 246, 0.2) 100%);
        }
        
        .floating {
            animation: float 6s ease-in-out infinite;
        }
        
        .fa {
            font-family: "Font Awesome 6 Free" !important;
            font-weight: 900;
        }
        
        .fa-solid {
            font-family: "Font Awesome 6 Free" !important;
            font-weight: 900;
        }
        
        .fa-regular {
            font-family: "Font Awesome 6 Free" !important;
            font-weight: 400;
        }
        
        .fa-brands {
            font-family: "Font Awesome 6 Brands" !important;
            font-weight: 400;
        }
    </style>
    <script>
        // Theme ayarını document.documentElement'e uygula
        document.documentElement.classList.toggle('dark', <?php echo $is_dark ? 'true' : 'false'; ?>);
    </script>
</head>
<body class="bg-gradient-to-br from-gray-50 via-white to-gray-100 dark:from-gray-900 dark:via-gray-800 dark:to-gray-900 text-gray-900 dark:text-white font-sans min-h-screen <?php echo $is_dark ? 'dark' : ''; ?>">
    <?php include 'includes/header.php'; ?>

    <main class="container mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <?php
        // Content top reklamları
        if (($settings['ads_enabled'] ?? '1') === '1') {
            echo renderAdContainer($current_page_type, 'content_top', 'ad-content-top');
        }
        ?>
        
        <!-- Search Header -->
        <div class="text-center mb-12">
            <div class="floating mb-6">
                <div class="inline-flex items-center justify-center w-20 h-20 rounded-full gradient-bg shadow-2xl">
                    <i class="fas fa-search text-3xl text-white"></i>
                </div>
            </div>
            <h1 class="text-5xl font-bold gradient-text mb-6">
                <?php if (!empty($search_query)): ?>
                    Arama Sonuçları
                <?php else: ?>
                    Arama
                <?php endif; ?>
            </h1>
            <?php if (!empty($search_query)): ?>
                <p class="text-xl text-gray-600 dark:text-gray-300 mb-2">
                    <span class="font-bold text-primary"><?php echo number_format($total_results); ?></span> sonuç bulundu 
                    <span class="font-semibold text-primary">"<?php echo htmlspecialchars($search_query); ?>"</span>
                </p>
            <?php else: ?>
                <p class="text-xl text-gray-600 dark:text-gray-300">
                    Web sitelerini bulmak için arama terimi girin
                </p>
            <?php endif; ?>
        </div>

        <!-- Search Form -->
        <div class="max-w-3xl mx-auto mb-12">
            <form action="/search" method="GET" class="relative">
                <div class="relative">
                    <i class="fas fa-search absolute left-6 top-1/2 -translate-y-1/2 text-gray-400 dark:text-gray-500 text-xl"></i>
                    <input name="q" 
                           class="w-full pl-16 pr-6 py-6 rounded-2xl bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm border-2 border-gray-200/50 dark:border-gray-700/50 focus:ring-4 focus:ring-primary/20 focus:border-primary transition-all duration-300 placeholder:text-gray-400 dark:placeholder:text-gray-500 text-xl text-gray-900 dark:text-white shadow-xl hover:shadow-2xl" 
                           placeholder="Web siteleri, kategoriler ara..." 
                           type="search" 
                           value="<?php echo htmlspecialchars($search_query); ?>"
                           autofocus>
                    <?php if (!empty($search_query)): ?>
                        <button type="button" onclick="clearSearch()" class="absolute right-6 top-1/2 -translate-y-1/2 text-gray-400 hover:text-red-500 dark:hover:text-red-400 transition-colors z-10">
                            <i class="fas fa-times text-xl"></i>
                        </button>
                    <?php endif; ?>
                </div>
            </form>
        </div>

        <!-- Database Unavailable Message -->
        <?php if (!$database_available || !$sites_table_exists): ?>
            <div class="text-center py-16 bg-red-50/80 dark:bg-red-900/20 rounded-2xl border border-red-200/50 dark:border-red-700/50 glass-effect">
                <i class="fas fa-exclamation-triangle text-6xl text-red-500 mb-6"></i>
                <h3 class="text-2xl font-semibold text-red-800 dark:text-red-300 mb-4">
                    <?php echo htmlspecialchars($error_message ?: "Arama işlevi şu anda kullanılamıyor."); ?>
                </h3>
                <p class="text-red-600 dark:text-red-400 text-lg">
                    Lütfen daha sonra tekrar deneyin veya yönetici ile iletişime geçin.
                </p>
            </div>
        <?php elseif (!empty($search_query)): ?>
            <?php if (empty($results)): ?>
                <!-- No Results -->
                <div class="text-center py-20">
                    <div class="inline-flex items-center justify-center w-24 h-24 rounded-full bg-gray-100/80 dark:bg-gray-800/80 backdrop-blur-sm mb-8">
                        <i class="fas fa-search-minus text-4xl text-gray-400 dark:text-gray-500"></i>
                    </div>
                    <h3 class="text-3xl font-semibold text-gray-900 dark:text-white mb-4">Sonuç bulunamadı</h3>
                    <p class="text-gray-600 dark:text-gray-300 text-xl mb-8">
                        "<span class="font-semibold text-primary"><?php echo htmlspecialchars($search_query); ?></span>" ile eşleşen web sitesi bulamadık
                    </p>
                    <div class="text-lg text-gray-500 dark:text-gray-400 max-w-md mx-auto">
                        <p class="mb-4">Deneyin:</p>
                        <ul class="space-y-2 text-left">
                            <li class="flex items-center gap-3">
                                <i class="fas fa-check text-green-500"></i>
                                Farklı anahtar kelimeler kullanın
                            </li>
                            <li class="flex items-center gap-3">
                                <i class="fas fa-check text-green-500"></i>
                                Yazımınızı kontrol edin
                            </li>
                            <li class="flex items-center gap-3">
                                <i class="fas fa-check text-green-500"></i>
                                Daha genel terimler kullanın
                            </li>
                        </ul>
                    </div>
                </div>
            <?php else: ?>
                <!-- Search Results -->
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mb-12">
                    <?php foreach ($results as $site): ?>
                        <div class="card-hover bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm rounded-3xl shadow-xl border border-gray-200/50 dark:border-gray-700/50 p-8 hover:shadow-2xl transition-all duration-500">
                            <!-- Category -->
                            <?php if (!empty($site['category_name'])): ?>
                                <div class="flex items-center gap-3 mb-4">
                                    <i class="<?php echo htmlspecialchars($site['category_icon'] ?? 'fas fa-folder'); ?> text-lg" style="color: <?php echo htmlspecialchars($site['category_color'] ?? $settings['primary_color'] ?? '#3b82f6'); ?>"></i>
                                    <span class="text-sm font-semibold text-white px-4 py-2 rounded-full shadow-lg" style="background: linear-gradient(135deg, <?php echo htmlspecialchars($site['category_color'] ?? $settings['primary_color'] ?? '#3b82f6'); ?>, <?php echo htmlspecialchars($site['category_color'] ?? $settings['primary_color'] ?? '#3b82f6'); ?>88);">
                                        <?php echo htmlspecialchars($site['category_name']); ?>
                                    </span>
                                </div>
                            <?php else: ?>
                                <div class="flex items-center gap-3 mb-4">
                                    <i class="fas fa-folder text-lg text-gray-400"></i>
                                    <span class="text-sm font-semibold text-gray-500 bg-gray-100/80 dark:bg-gray-700/80 px-4 py-2 rounded-full">
                                        Kategorisiz
                                    </span>
                                </div>
                            <?php endif; ?>

                            <!-- Site Logo/Icon -->
                            <div class="flex items-center gap-4 mb-4">
                                <?php if (!empty($site['logo_url'])): ?>
                                    <img src="<?php echo htmlspecialchars($site['logo_url']); ?>" 
                                         alt="<?php echo htmlspecialchars($site['title']); ?>" 
                                         class="w-12 h-12 rounded-xl object-cover shadow-lg"
                                         onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                <?php endif; ?>
                                <div class="flex items-center justify-center w-12 h-12 rounded-xl bg-gradient-to-br from-blue-500 to-purple-600 text-white font-bold text-lg shadow-lg <?php echo !empty($site['logo_url']) ? 'hidden' : ''; ?>">
                                    <?php echo strtoupper(substr($site['title'], 0, 2)); ?>
                                </div>
                                <div class="flex-1 min-w-0">
                                    <h3 class="text-xl font-bold text-gray-900 dark:text-white line-clamp-2 mb-1">
                                        <?php echo htmlspecialchars($site['title']); ?>
                                    </h3>
                                    <p class="text-sm text-gray-500 dark:text-gray-400 truncate">
                                        <?php echo htmlspecialchars(parse_url($site['url'], PHP_URL_HOST) ?: $site['url']); ?>
                                    </p>
                                </div>
                            </div>

                            <!-- Description -->
                            <p class="text-gray-600 dark:text-gray-300 line-clamp-3 mb-6 leading-relaxed">
                                <?php 
                                $description = strip_tags($site['description']);
                                if (strlen($description) > 150) {
                                    $description = substr($description, 0, 150) . '...';
                                }
                                echo htmlspecialchars($description);
                                ?>
                            </p>

                            <!-- Stats -->
                            <div class="flex items-center justify-between mb-6">
                                <div class="flex items-center gap-4 text-sm text-gray-500 dark:text-gray-400">
                                    <div class="flex items-center gap-2">
                                        <i class="fas fa-thumbs-up"></i>
                                        <span><?php echo number_format($site['votes'] ?? 0); ?></span>
                                    </div>
                                    <div class="flex items-center gap-2">
                                        <i class="fas fa-eye"></i>
                                        <span><?php echo number_format($site['views'] ?? 0); ?></span>
                                    </div>
                                </div>
                            </div>

                            <!-- Visit Button -->
                            <a href="/site/<?php echo strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $site['title']), '-')); ?>" 
                               class="block w-full text-center py-3 px-6 rounded-xl gradient-bg text-white font-semibold hover:shadow-lg transition-all duration-300 hover:scale-105">
                                <i class="fas fa-info-circle mr-2"></i>
                                Detayları Gör
                            </a>
                        </div>
                    <?php endforeach; ?>
                </div>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                    <div class="flex justify-center items-center gap-4 mt-12">
                        <?php if ($page > 1): ?>
                            <a href="?q=<?php echo urlencode($search_query); ?>&page=<?php echo $page - 1; ?>" 
                               class="flex items-center gap-2 px-6 py-3 rounded-xl bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm border border-gray-200/50 dark:border-gray-700/50 text-gray-700 dark:text-gray-300 hover:bg-primary hover:text-white transition-all duration-300 shadow-lg hover:shadow-xl">
                                <i class="fas fa-chevron-left"></i>
                                Önceki
                            </a>
                        <?php endif; ?>

                        <div class="flex items-center gap-2">
                            <?php
                            $start_page = max(1, $page - 2);
                            $end_page = min($total_pages, $page + 2);
                            
                            for ($i = $start_page; $i <= $end_page; $i++):
                            ?>
                                <a href="?q=<?php echo urlencode($search_query); ?>&page=<?php echo $i; ?>" 
                                   class="flex items-center justify-center w-12 h-12 rounded-xl font-semibold transition-all duration-300 shadow-lg hover:shadow-xl <?php echo $i === $page ? 'gradient-bg text-white' : 'bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm border border-gray-200/50 dark:border-gray-700/50 text-gray-700 dark:text-gray-300 hover:bg-primary hover:text-white'; ?>">
                                    <?php echo $i; ?>
                                </a>
                            <?php endfor; ?>
                        </div>

                        <?php if ($page < $total_pages): ?>
                            <a href="?q=<?php echo urlencode($search_query); ?>&page=<?php echo $page + 1; ?>" 
                               class="flex items-center gap-2 px-6 py-3 rounded-xl bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm border border-gray-200/50 dark:border-gray-700/50 text-gray-700 dark:text-gray-300 hover:bg-primary hover:text-white transition-all duration-300 shadow-lg hover:shadow-xl">
                                Sonraki
                                <i class="fas fa-chevron-right"></i>
                            </a>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        <?php endif; ?>
    </main>

    <?php
    // Content bottom reklamları
    if (($settings['ads_enabled'] ?? '1') === '1') {
        echo renderAdContainer($current_page_type, 'content_bottom', 'ad-content-bottom');
    }
    ?>

    <?php include 'includes/footer.php'; ?>

    <script>
        function clearSearch() {
            document.querySelector('input[name="q"]').value = '';
            document.querySelector('input[name="q"]').focus();
        }
        
        // Search form enhancement
        document.addEventListener('DOMContentLoaded', function() {
            const searchInput = document.querySelector('input[name="q"]');
            if (searchInput) {
                searchInput.addEventListener('keydown', function(e) {
                    if (e.key === 'Enter') {
                        this.closest('form').submit();
                    }
                });
            }
        });
    </script>
</body>
</html>