<?php
/**
 * Reklam Yönetim Sistemi Helper Fonksiyonları
 * Tüm reklam firmalarını destekler (AdSense, Adsterra, Media.net, Custom)
 */

require_once __DIR__ . '/../includes/config.php';

/**
 * Reklam kodunu render et
 */
function renderAdCode($ad_code, $ad_type) {
    // Boş reklam kodlarını kontrol et
    if (empty(trim($ad_code)) || 
        trim($ad_code) === '<!-- Header reklam kodu buraya gelecek -->' ||
        trim($ad_code) === '<!-- Footer reklam kodu buraya gelecek -->' ||
        trim($ad_code) === '<!-- Sidebar reklam kodu buraya gelecek -->' ||
        trim($ad_code) === '<!-- Content top reklam kodu buraya gelecek -->' ||
        trim($ad_code) === '<!-- Content bottom reklam kodu buraya gelecek -->') {
        return '';
    }
    
    // Reklam tipine göre doğrulama (sadece uyarı için)
    if ($ad_type === 'adsense') {
        if (strpos($ad_code, 'googlesyndication.com') === false && 
            strpos($ad_code, 'adsbygoogle') === false) {
            // Uyarı ver ama reklamı göster
        }
    }
    
    if ($ad_type === 'adsterra') {
        if (strpos($ad_code, 'adsterra') === false) {
            // Uyarı ver ama reklamı göster
        }
    }
    
    if ($ad_type === 'media_net') {
        if (strpos($ad_code, 'media.net') === false) {
            // Uyarı ver ama reklamı göster
        }
    }
    
    return $ad_code;
}

/**
 * Reklam container'ını render et
 */
function renderAdContainer($page_type, $position, $additional_classes = '') {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT * FROM ads 
            WHERE page_type = ? AND position = ? AND is_active = 1 
            ORDER BY display_order ASC
        ");
        $stmt->execute([$page_type, $position]);
        $ads = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return '';
    }
    
    if (empty($ads)) {
        return '';
    }
    
    $html = '';
    foreach ($ads as $ad) {
        $ad_code = renderAdCode($ad['ad_code'], $ad['ad_type']);
        if (!empty($ad_code)) {
            $html .= '<div class="ad-container ad-' . $position . ' ' . $additional_classes . '" data-ad-id="' . $ad['id'] . '">';
            $html .= $ad_code;
            $html .= '</div>';
        }
    }
    
    return $html;
}

/**
 * Reklam stilleri ve JavaScript'i döndür
 */
function getAdStyles() {
    return '
    <style>
    /* Responsive reklam stilleri */
    .ad-container {
        width: 100%;
        margin: 20px auto;
        text-align: center;
        position: relative;
    }
    
    .ad-container.ad-header {
        margin: 0 auto 10px auto;
    }
    
    .ad-container.ad-footer {
        margin: 10px auto 0 auto;
    }
    
    .ad-container.ad-sidebar {
        margin: 0 0 10px 0;
        max-width: 300px;
    }
    
    .ad-container.ad-content-top,
    .ad-container.ad-content-bottom {
        margin: 10px auto;
        max-width: 728px;
    }
    
    /* Responsive tasarım */
    @media (max-width: 768px) {
        .ad-container {
            margin: 10px auto;
        }
        
        .ad-container.ad-sidebar {
            display: none;
        }
        
        .ad-container.ad-content-top,
        .ad-container.ad-content-bottom {
            max-width: 100%;
            margin: 10px 0;
        }
    }
    
    @media (max-width: 480px) {
        .ad-container {
            margin: 5px auto;
        }
    }
    
    
    </style>
    
    <script>
    document.addEventListener("DOMContentLoaded", function() {
        const adContainers = document.querySelectorAll(".ad-container");
        
        adContainers.forEach(function(container) {
            const adContent = container.innerHTML.trim();
            if (adContent && 
                adContent !== "<!-- Header reklam kodu buraya gelecek -->" &&
                adContent !== "<!-- Footer reklam kodu buraya gelecek -->" &&
                adContent !== "<!-- Sidebar reklam kodu buraya gelecek -->" &&
                adContent !== "<!-- Content top reklam kodu buraya gelecek -->" &&
                adContent !== "<!-- Content bottom reklam kodu buraya gelecek -->") {
                
                // AdSense refresh
                if (typeof adsbygoogle !== "undefined") {
                    try {
                        (adsbygoogle = window.adsbygoogle || []).push({});
                    } catch (e) {
                        console.log("AdSense error:", e);
                    }
                }
            } else {
                container.style.display = "none";
            }
        });
        
        trackAdPerformance();
    });
    
    function trackAdPerformance() {
        const adContainers = document.querySelectorAll(".ad-container");
        
        adContainers.forEach(function(container) {
            const adId = container.getAttribute("data-ad-id");
            if (adId) {
                incrementAdView(adId);
                
                container.addEventListener("click", function() {
                    incrementAdClick(adId);
                });
            }
        });
    }
    
    function incrementAdView(adId) {
        fetch("ajax_vote.php", {
            method: "POST",
            headers: {
                "Content-Type": "application/x-www-form-urlencoded",
            },
            body: "action=ad_view&ad_id=" + adId
        }).catch(function(error) {
            console.log("Ad view tracking error:", error);
        });
    }
    
    function incrementAdClick(adId) {
        fetch("ajax_vote.php", {
            method: "POST",
            headers: {
                "Content-Type": "application/x-www-form-urlencoded",
            },
            body: "action=ad_click&ad_id=" + adId
        }).catch(function(error) {
            console.log("Ad click tracking error:", error);
        });
    }
    
    function hideAd(adId) {
        const container = document.querySelector("[data-ad-id=\"" + adId + "\"]");
        if (container) {
            container.style.display = "none";
            
            localStorage.setItem("hidden_ad_" + adId, "true");
        }
    }
    
    function checkHiddenAds() {
        const adContainers = document.querySelectorAll(".ad-container");
        adContainers.forEach(function(container) {
            const adId = container.getAttribute("data-ad-id");
            if (adId && localStorage.getItem("hidden_ad_" + adId) === "true") {
                container.style.display = "none";
            }
        });
    }
    </script>';
}

/**
 * Sayfa tipini tespit et
 */
function detectPageType() {
    $current_page = $_SERVER['REQUEST_URI'] ?? '';
    $current_page = parse_url($current_page, PHP_URL_PATH);
    
    // Ana sayfa
    if ($current_page === '/' || $current_page === '/index.php') {
        return 'home';
    }
    
    // Kategori sayfaları
    if (strpos($current_page, '/category/') === 0 || strpos($current_page, '/categorylisting.php') !== false) {
        return 'category';
    }
    
    // Site detay sayfası
    if (strpos($current_page, '/site/') === 0 || strpos($current_page, '/sitedetails.php') !== false) {
        return 'site_detail';
    }
    
    // Blog sayfaları
    if (strpos($current_page, '/blog/') === 0 || strpos($current_page, '/blog.php') !== false || strpos($current_page, '/blog_detail.php') !== false) {
        return 'blog';
    }
    
    // Yeni eklenen sayfası
    if (strpos($current_page, '/newly-added') === 0 || strpos($current_page, '/newly-added.php') !== false) {
        return 'newly_added';
    }
    
    // Popüler sayfası
    if (strpos($current_page, '/topvoted') === 0 || strpos($current_page, '/topvoted.php') !== false) {
        return 'popular';
    }
    
    // Site ekleme sayfası
    if (strpos($current_page, '/submit') === 0 || strpos($current_page, '/submitsite.php') !== false) {
        return 'add_site';
    }
    
    // Kayıt sayfası
    if (strpos($current_page, '/register') === 0 || strpos($current_page, '/register.php') !== false) {
        return 'register';
    }
    
    // Giriş sayfası
    if (strpos($current_page, '/login') === 0 || strpos($current_page, '/login.php') !== false) {
        return 'login';
    }
    
    // Arama sayfası
    if (strpos($current_page, '/search') === 0 || strpos($current_page, '/search.php') !== false) {
        return 'search';
    }
    
    // Varsayılan olarak home döndür
    return 'home';
}

/**
 * Reklam kodunu doğrula
 */
function validateAdCode($ad_code, $ad_type) {
    $result = [
        'valid' => false,
        'message' => '',
        'warnings' => []
    ];
    
    if (empty(trim($ad_code))) {
        $result['message'] = 'Reklam kodu boş olamaz';
        return $result;
    }
    
    // Reklam tipine göre doğrulama
    if ($ad_type === 'adsense') {
        if (strpos($ad_code, 'googlesyndication.com') === false) {
            $result['message'] = 'Geçersiz AdSense kodu';
            return $result;
        }
        if (strpos($ad_code, 'adsbygoogle') === false) {
            $result['warnings'][] = 'AdSense kodu eksik görünüyor';
        }
    }
    
    if ($ad_type === 'adsterra') {
        if (strpos($ad_code, 'adsterra') === false) {
            $result['message'] = 'Geçersiz Adsterra kodu';
            return $result;
        }
    }
    
    if ($ad_type === 'media_net') {
        if (strpos($ad_code, 'media.net') === false) {
            $result['message'] = 'Geçersiz Media.net kodu';
            return $result;
        }
    }
    
    // Genel güvenlik kontrolleri
    if (strpos($ad_code, '<script') !== false && strpos($ad_code, '</script>') === false) {
        $result['warnings'][] = 'Script tag kapatılmamış olabilir';
    }
    
    $result['valid'] = true;
    $result['message'] = 'Reklam kodu geçerli';
    
    return $result;
}

?>