<?php
// email_helper.php - E-posta gönderim yardımcı fonksiyonları

require_once __DIR__ . '/../includes/config.php';

// PHPMailer dosyalarını yükle
require_once __DIR__ . '/../phpmailer/src/Exception.php';
require_once __DIR__ . '/../phpmailer/src/PHPMailer.php';
require_once __DIR__ . '/../phpmailer/src/SMTP.php';

// PHPMailer namespace'ini kullan
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\SMTP;
use PHPMailer\PHPMailer\Exception;

/**
 * E-posta ayarlarını veritabanından çeker
 */
function getEmailSettings() {
    global $pdo;
    
    $settings = [];
    $stmt = $pdo->query('SELECT setting_key, setting_value FROM settings WHERE setting_key LIKE "smtp_%" OR setting_key LIKE "mail_%" OR setting_key LIKE "password_reset_%"');
    
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
    
    return $settings;
}

/**
 * E-posta gönderir (SMTP veya PHP mail)
 */
function sendEmail($to, $subject, $message, $isHtml = true) {
    $settings = getEmailSettings();
    
    // Debug logging
    error_log('Email Send Attempt - To: ' . $to . ', Subject: ' . $subject);
    error_log('SMTP Settings: ' . print_r($settings, true));
    
    // E-posta ayarları kontrolü
    if (empty($settings['mail_from_email'])) {
        error_log('E-posta ayarları eksik: mail_from_email');
        return false;
    }
    
    $from_email = $settings['mail_from_email'];
    $from_name = $settings['mail_from_name'] ?? '';
    
    // SMTP kullanılacaksa PHPMailer ile gönder
    if (!empty($settings['smtp_enabled']) && $settings['smtp_enabled'] == '1') {
        error_log('SMTP mode enabled, attempting SMTP send');
        return sendEmailSMTP($to, $subject, $message, $isHtml, $settings);
    } else {
        error_log('SMTP disabled, using PHP mail()');
        // Basit PHP mail() fonksiyonu
        return sendEmailPHP($to, $subject, $message, $isHtml, $from_email, $from_name);
    }
}

/**
 * SMTP ile e-posta gönderir (PHPMailer gerekir)
 */
function sendEmailSMTP($to, $subject, $message, $isHtml, $settings) {
    try {
        // PHPMailer'ı başlat
        $mail = new PHPMailer(true);
        
        // Debug modunu aktifleştir
        $mail->SMTPDebug = 0; // 0=kapalı, 1=client, 2=client+server, 3=verbose
        $mail->Debugoutput = function($str, $level) {
            error_log("PHPMailer [$level]: $str");
        };
        
        // SMTP ayarları
        $mail->isSMTP();
        $mail->Host = $settings['smtp_host'] ?? '';
        $mail->SMTPAuth = true;
        $mail->Username = $settings['smtp_username'] ?? '';
        $mail->Password = $settings['smtp_password'] ?? '';
        $mail->Port = intval($settings['smtp_port'] ?? 587);
        $mail->Timeout = 30; // 30 saniye timeout
        
        // Sunucu maili için SSL sertifika ayarları
        $mail->SMTPOptions = array(
            'ssl' => array(
                'verify_peer' => false,
                'verify_peer_name' => false,
                'allow_self_signed' => true,
                'cafile' => false,
                'ciphers' => 'ALL:!EXPORT:!LOW:!aNULL:!eNULL:!SSLv2:!SSLv3'
            ),
            'tls' => array(
                'verify_peer' => false,
                'verify_peer_name' => false,
                'allow_self_signed' => true
            )
        );
        
        error_log('SMTP Config - Host: ' . $mail->Host . ', Port: ' . $mail->Port . ', User: ' . $mail->Username);
        
        // Güvenlik ayarları - Sunucu maili için esnek yaklaşım
        $encryption = strtolower($settings['smtp_encryption'] ?? 'tls');
        if ($encryption === 'ssl') {
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
            if ($mail->Port == 587) {
                $mail->Port = 465; // SSL için varsayılan port
            }
        } elseif ($encryption === 'tls') {
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
            if ($mail->Port == 465) {
                $mail->Port = 587; // TLS için varsayılan port
            }
        } else {
            // Güvenlik yok
            $mail->SMTPSecure = false;
            $mail->SMTPAutoTLS = false;
            if ($mail->Port == 465 || $mail->Port == 587) {
                $mail->Port = 25; // Güvensiz bağlantı için
            }
        }
        
        error_log('SMTP Encryption: ' . $encryption);
        
        // Gönderen bilgileri
        $mail->setFrom($settings['mail_from_email'], $settings['mail_from_name'] ?? '');
        
        // Alıcı
        $mail->addAddress($to);
        
        // İçerik
        $mail->isHTML($isHtml);
        $mail->Subject = $subject;
        $mail->Body = $message;
        $mail->CharSet = 'UTF-8';
        
        error_log('Attempting to send email via SMTP...');
        $result = $mail->send();
        error_log('SMTP send result: ' . ($result ? 'SUCCESS' : 'FAILED'));
        
        return $result;
        
    } catch (Exception $e) {
        error_log('SMTP E-posta gönderim hatası: ' . $e->getMessage());
        error_log('SMTP Error Stack: ' . $e->getTraceAsString());
        return false;
    }
}

/**
 * PHP mail() fonksiyonu ile e-posta gönderir
 */
function sendEmailPHP($to, $subject, $message, $isHtml, $from_email, $from_name) {
    error_log('Using PHP mail() function');
    
    $headers = [];
    $headers[] = 'MIME-Version: 1.0';
    $headers[] = 'Content-Type: ' . ($isHtml ? 'text/html' : 'text/plain') . '; charset=UTF-8';
    $headers[] = 'From: ' . (!empty($from_name) ? "$from_name <$from_email>" : $from_email);
    $headers[] = 'Reply-To: ' . $from_email;
    $headers[] = 'X-Mailer: PHP/' . phpversion();
    
    error_log('PHP Mail Headers: ' . implode('; ', $headers));
    
    $result = mail($to, $subject, $message, implode("\r\n", $headers));
    error_log('PHP mail() result: ' . ($result ? 'SUCCESS' : 'FAILED'));
    
    return $result;
}

/**
 * Şifre sıfırlama e-postası gönderir
 */
function sendPasswordResetEmail($userEmail, $resetToken, $userName = '') {
    $settings = getEmailSettings();
    
    // Şifre sıfırlama aktif mi kontrol et
    if (empty($settings['password_reset_enabled']) || $settings['password_reset_enabled'] != '1') {
        return false;
    }
    
    $resetLink = getBaseUrl() . '/reset-password?token=' . $resetToken;
    $expireMinutes = $settings['password_reset_token_expire'] ?? 60;
    
    $subject = 'Şifre Sıfırlama Talebi';
    
    $message = '
    <html>
    <head>
        <meta charset="UTF-8">
        <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: #3b82f6; color: white; padding: 20px; text-align: center; border-radius: 5px 5px 0 0; }
            .content { background: #f9f9f9; padding: 30px; border-radius: 0 0 5px 5px; }
            .button { display: inline-block; background: #3b82f6; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; margin: 20px 0; }
            .footer { text-align: center; margin-top: 30px; color: #666; font-size: 14px; }
        </style>
    </head>
    <body>
        <div class="container">
            <div class="header">
                <h1>Şifre Sıfırlama</h1>
            </div>
            <div class="content">
                <p>Merhaba' . (!empty($userName) ? ' ' . htmlspecialchars($userName) : '') . ',</p>
                
                <p>Hesabınız için şifre sıfırlama talebinde bulundunuz. Şifrenizi sıfırlamak için aşağıdaki butona tıklayın:</p>
                
                <p style="text-align: center;">
                    <a href="' . htmlspecialchars($resetLink) . '" class="button">Şifreyi Sıfırla</a>
                </p>
                
                <p>Yukarıdaki buton çalışmazsa, aşağıdaki bağlantıyı kopyalayıp tarayıcınıza yapıştırabilirsiniz:</p>
                <p style="word-break: break-all; color: #3b82f6;">' . htmlspecialchars($resetLink) . '</p>
                
                <p><strong>Bu bağlantı ' . $expireMinutes . ' dakika geçerlidir.</strong></p>
                
                <p>Bu şifre sıfırlama talebinde bulunmadıysanız, bu e-postayı güvenle görmezden gelebilirsiniz.</p>
                
                <div class="footer">
                    <p>Bu otomatik bir e-postadır, lütfen yanıtlamayın.</p>
                </div>
            </div>
        </div>
    </body>
    </html>';
    
    return sendEmail($userEmail, $subject, $message, true);
}

/**
 * Site base URL'ini döndürür
 */
function getBaseUrl() {
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' || $_SERVER['SERVER_PORT'] == 443) ? "https://" : "http://";
    $domainName = $_SERVER['HTTP_HOST'];
    return $protocol . $domainName;
}
?>