<?php
/**
 * E-posta Doğrulama Helper
 */

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\SMTP;
use PHPMailer\PHPMailer\Exception;

if (!function_exists('sendVerificationEmail')) {
    /**
     * E-posta doğrulama maili gönderir
     */
    function sendVerificationEmail($userId, $email, $username, $settings = []) {
        try {
            // Doğrulama token'ı oluştur
            $token = bin2hex(random_bytes(32));
            $expiresAt = date('Y-m-d H:i:s', strtotime('+24 hours'));
            
            // Veritabanına kaydet
            global $pdo;
            $stmt = $pdo->prepare("INSERT INTO email_verifications (user_id, email, verification_token, expires_at) VALUES (?, ?, ?, ?)");
            $stmt->execute([$userId, $email, $token, $expiresAt]);
            
            // E-posta içeriği
            $siteName = $settings['site_title'] ?? 'Web Directory';
            $verificationUrl = getBaseUrl() . "/verify-email?token=" . $token;
            
            $subject = "E-posta Adresinizi Doğrulayın - " . $siteName;
            $message = "
            <html>
            <body>
                <h2>Merhaba " . htmlspecialchars($username) . ",</h2>
                <p>" . $siteName . " sitesine hoş geldiniz!</p>
                <p>Hesabınızı aktifleştirmek için aşağıdaki bağlantıya tıklayın:</p>
                <p><a href='" . $verificationUrl . "' style='background: #3b82f6; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>E-posta Adresimi Doğrula</a></p>
                <p>Bu bağlantı 24 saat geçerlidir.</p>
                <p>Eğer bu işlemi siz yapmadıysanız, bu e-postayı görmezden gelebilirsiniz.</p>
                <br>
                <p>İyi günler,<br>" . $siteName . " Ekibi</p>
            </body>
            </html>
            ";
            
            // E-posta gönder
            return sendEmail($email, $subject, $message, $settings);
            
        } catch (Exception $e) {
            error_log("E-posta doğrulama hatası: " . $e->getMessage());
            return false;
        }
    }
}

if (!function_exists('verifyEmailToken')) {
    /**
     * E-posta doğrulama token'ını kontrol eder
     */
    function verifyEmailToken($token) {
        try {
            global $pdo;
            
            // Token'ı kontrol et
            $stmt = $pdo->prepare("SELECT * FROM email_verifications WHERE verification_token = ? AND expires_at > NOW() AND verified = 0");
            $stmt->execute([$token]);
            $verification = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$verification) {
                return ['success' => false, 'message' => 'Invalid or expired verification link.'];
            }
            
            // Kullanıcının e-postasını doğrula
            $stmt = $pdo->prepare("UPDATE user SET email_verified = 1 WHERE id = ?");
            $stmt->execute([$verification['user_id']]);
            
            // Doğrulama kaydını güncelle
            $stmt = $pdo->prepare("UPDATE email_verifications SET verified = 1 WHERE id = ?");
            $stmt->execute([$verification['id']]);
            
            return ['success' => true, 'message' => 'Your email address has been successfully verified!'];
            
        } catch (Exception $e) {
            error_log("E-posta doğrulama hatası: " . $e->getMessage());
            return ['success' => false, 'message' => 'An error occurred during verification.'];
        }
    }
}

if (!function_exists('sendEmail')) {
    /**
     * E-posta gönderir
     */
    function sendEmail($to, $subject, $message, $settings = []) {
        try {
            // SMTP ayarları
            $smtpEnabled = $settings['smtp_enabled'] ?? '0';
            
            if ($smtpEnabled === '1') {
                // SMTP ile gönder
                return sendEmailSMTP($to, $subject, $message, $settings);
            } else {
                // PHP mail() ile gönder
                $headers = [
                    'MIME-Version: 1.0',
                    'Content-type: text/html; charset=UTF-8',
                    'From: ' . ($settings['mail_from_name'] ?? 'Site') . ' <' . ($settings['mail_from_email'] ?? 'noreply@site.com') . '>',
                    'Reply-To: ' . ($settings['mail_from_email'] ?? 'noreply@site.com'),
                    'X-Mailer: PHP/' . phpversion()
                ];
                
                return mail($to, $subject, $message, implode("\r\n", $headers));
            }
        } catch (Exception $e) {
            error_log("E-posta gönderme hatası: " . $e->getMessage());
            return false;
        }
    }
}

if (!function_exists('sendEmailSMTP')) {
    /**
     * SMTP ile e-posta gönderir
     */
    function sendEmailSMTP($to, $subject, $message, $settings = []) {
        try {
            require_once __DIR__ . '/../phpmailer/src/Exception.php';
            require_once __DIR__ . '/../phpmailer/src/PHPMailer.php';
            require_once __DIR__ . '/../phpmailer/src/SMTP.php';
            
            $mail = new PHPMailer(true);
            
            // SMTP ayarları
            $mail->isSMTP();
            $mail->Host = $settings['smtp_host'] ?? 'smtp.gmail.com';
            $mail->SMTPAuth = true;
            $mail->Username = $settings['smtp_username'] ?? '';
            $mail->Password = $settings['smtp_password'] ?? '';
            $mail->SMTPSecure = $settings['smtp_encryption'] ?? 'tls';
            $mail->Port = $settings['smtp_port'] ?? 587;
            
            // E-posta ayarları
            $mail->setFrom($settings['mail_from_email'] ?? 'noreply@site.com', $settings['mail_from_name'] ?? 'Site');
            $mail->addAddress($to);
            $mail->isHTML(true);
            $mail->Subject = $subject;
            $mail->Body = $message;
            
            return $mail->send();
            
        } catch (Exception $e) {
            error_log("SMTP e-posta hatası: " . $e->getMessage());
            return false;
        }
    }
}

if (!function_exists('getBaseUrl')) {
    /**
     * Base URL'i döndürür
     */
    function getBaseUrl() {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        $script = $_SERVER['SCRIPT_NAME'] ?? '';
        $path = dirname($script);
        
        if ($path === '/' || $path === '\\') {
            return $protocol . '://' . $host;
        }
        
        return $protocol . '://' . $host . $path;
    }
}
?>
