<?php
/**
 * reCAPTCHA v2 Helper Functions
 * Google reCAPTCHA entegrasyonu için yardımcı fonksiyonlar
 */

/**
 * reCAPTCHA ayarlarını veritabanından al
 */
function getRecaptchaSettings() {
    global $pdo;
    
    $settings = [
        'enabled' => false,
        'site_key' => '',
        'secret_key' => ''
    ];
    
    try {
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM settings WHERE setting_key IN ('recaptcha_enabled', 'recaptcha_site_key', 'recaptcha_secret_key')");
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($rows as $row) {
            switch ($row['setting_key']) {
                case 'recaptcha_enabled':
                    $settings['enabled'] = (bool)$row['setting_value'];
                    break;
                case 'recaptcha_site_key':
                    $settings['site_key'] = $row['setting_value'];
                    break;
                case 'recaptcha_secret_key':
                    $settings['secret_key'] = $row['setting_value'];
                    break;
            }
        }
    } catch (Exception $e) {
        error_log("reCAPTCHA settings error: " . $e->getMessage());
    }
    
    return $settings;
}

/**
 * reCAPTCHA HTML kodunu oluştur
 */
function renderRecaptcha() {
    $settings = getRecaptchaSettings();
    
    if (!$settings['enabled'] || empty($settings['site_key'])) {
        return '';
    }
    
    $html = '
    <!-- Google reCAPTCHA v2 -->
    <script src="https://www.google.com/recaptcha/api.js" async defer></script>
    <div class="g-recaptcha" data-sitekey="' . htmlspecialchars($settings['site_key']) . '"></div>
    ';
    
    return $html;
}

/**
 * reCAPTCHA form validation - JavaScript kodu
 */
function getRecaptchaValidationJS() {
    $settings = getRecaptchaSettings();
    
    if (!$settings['enabled']) {
        return '';
    }
    
    return '
    <script>
    // reCAPTCHA form validation
    function validateRecaptcha(formElement) {
        var recaptchaResponse = grecaptcha.getResponse();
        if (recaptchaResponse.length === 0) {
            alert("Lütfen robot olmadığınızı doğrulayın.");
            return false;
        }
        return true;
    }
    
    // Form submit olaylarını dinle
    document.addEventListener("DOMContentLoaded", function() {
        var forms = document.querySelectorAll("form[data-recaptcha=\'true\']");
        forms.forEach(function(form) {
            form.addEventListener("submit", function(e) {
                if (!validateRecaptcha(form)) {
                    e.preventDefault();
                    return false;
                }
            });
        });
    });
    </script>
    ';
}

/**
 * reCAPTCHA server-side doğrulama
 */
function verifyRecaptcha($recaptchaResponse = null) {
    $settings = getRecaptchaSettings();
    
    // reCAPTCHA devre dışıysa her zaman true dön
    if (!$settings['enabled']) {
        return true;
    }
    
    // Response belirtilmemişse POST'tan al
    if ($recaptchaResponse === null) {
        $recaptchaResponse = isset($_POST['g-recaptcha-response']) ? $_POST['g-recaptcha-response'] : '';
    }
    
    // Boş response
    if (empty($recaptchaResponse)) {
        return false;
    }
    
    // Secret key yoksa
    if (empty($settings['secret_key'])) {
        error_log("reCAPTCHA secret key is empty");
        return false;
    }
    
    // Google API\'ye doğrulama isteği gönder
    $url = 'https://www.google.com/recaptcha/api/siteverify';
    $data = [
        'secret' => $settings['secret_key'],
        'response' => $recaptchaResponse,
        'remoteip' => $_SERVER['REMOTE_ADDR'] ?? ''
    ];
    
    $options = [
        'http' => [
            'header' => "Content-type: application/x-www-form-urlencoded\r\n",
            'method' => 'POST',
            'content' => http_build_query($data)
        ]
    ];
    
    $context = stream_context_create($options);
    $result = file_get_contents($url, false, $context);
    
    if ($result === false) {
        error_log("reCAPTCHA verification failed - API request error");
        return false;
    }
    
    $resultJson = json_decode($result, true);
    
    if (!$resultJson) {
        error_log("reCAPTCHA verification failed - Invalid JSON response");
        return false;
    }
    
    return isset($resultJson['success']) && $resultJson['success'] === true;
}

/**
 * reCAPTCHA hata mesajı
 */
function getRecaptchaError() {
    $settings = getRecaptchaSettings();
    
    if (!$settings['enabled']) {
        return '';
    }
    
    return 'reCAPTCHA doğrulaması başarısız. Lütfen robot olmadığınızı doğrulayın.';
}

/**
 * Form için reCAPTCHA attribute ekle
 */
function addRecaptchaAttribute() {
    $settings = getRecaptchaSettings();
    
    if ($settings['enabled']) {
        return 'data-recaptcha="true"';
    }
    
    return '';
}

/**
 * reCAPTCHA CSS styles
 */
function getRecaptchaStyles() {
    return '
    <style>
    .g-recaptcha {
        margin: 15px 0;
        display: inline-block;
    }
    
    /* reCAPTCHA responsive */
    @media (max-width: 768px) {
        .g-recaptcha {
            transform: scale(0.9);
            transform-origin: 0 0;
        }
    }
    
    @media (max-width: 480px) {
        .g-recaptcha {
            transform: scale(0.8);
            transform-origin: 0 0;
        }
    }
    </style>
    ';
}
?>