<?php
/**
 * SEO Helper Functions
 * Dynamic meta description and title generation
 */

/**
 * Generate dynamic page title based on page type and content
 */
function generatePageTitle($pageType, $data = [], $settings = []) {
    $siteTitle = $settings['site_title'] ?? 'Site';
    $separator = $settings['seo_title_separator'] ?? ' | ';
    
    switch ($pageType) {
        case 'home':
            return $settings['meta_home_title'] ?: $siteTitle;
            
        case 'category':
            $categoryName = $data['category_name'] ?? 'Kategori';
            return $categoryName . $separator . $siteTitle;
            
        case 'site_detail':
            $siteName = $data['site_name'] ?? 'Site';
            return $siteName . $separator . $siteTitle;
            
        case 'login':
            return 'Giriş Yap' . $separator . $siteTitle;
            
        case 'register':
            return 'Kayıt Ol' . $separator . $siteTitle;
            
        case 'submit_site':
            return 'Site Ekle' . $separator . $siteTitle;
            
        case 'profile':
            $username = $data['username'] ?? 'Kullanıcı';
            return $username . ' Profili' . $separator . $siteTitle;
            
        default:
            return $siteTitle;
    }
}

/**
 * Generate dynamic meta description based on page type and content
 */
function generateMetaDescription($pageType, $data = [], $settings = []) {
    $autoMeta = ($settings['seo_auto_meta'] ?? '1') === '1';
    
    switch ($pageType) {
        case 'home':
            return $settings['site_description'] ?? '';
            
        case 'category':
            if ($autoMeta && !empty($data['category_description'])) {
                return truncateText($data['category_description'], 155);
            }
            
            $template = $settings['meta_categorylisting'] ?? '{category_name} kategorisindeki en iyi siteler | {site_title}';
            return processMetaTemplate($template, $data, $settings);
            
        case 'site_detail':
            if ($autoMeta && !empty($data['site_description'])) {
                return truncateText($data['site_description'], 155);
            }
            
            $template = $settings['meta_sitedetails'] ?? '{site_name} hakkında detaylar, yorumlar ve değerlendirmeler | {site_title}';
            return processMetaTemplate($template, $data, $settings);
            
        case 'login':
            return $settings['meta_login'] ?? 'Hesabınıza giriş yapın';
            
        case 'register':
            return $settings['meta_register'] ?? 'Ücretsiz hesap oluşturun';
            
        case 'submit_site':
            return $settings['meta_submitsite'] ?? 'Sitenizi ekleyin';
            
        case 'profile':
            return $settings['meta_profile'] ?? 'Kullanıcı profili';
            
        default:
            return $settings['site_description'] ?? '';
    }
}

/**
 * Process meta template with dynamic variables
 */
function processMetaTemplate($template, $data, $settings) {
    $replacements = [
        '{site_title}' => $settings['site_title'] ?? '',
        '{site_description}' => $settings['site_description'] ?? '',
        '{site_slogan}' => $settings['site_slogan'] ?? '',
        '{category_name}' => $data['category_name'] ?? '',
        '{category_description}' => $data['category_description'] ?? '',
        '{site_name}' => $data['site_name'] ?? '',
        '{site_description}' => $data['site_description'] ?? '',
        '{username}' => $data['username'] ?? '',
    ];
    
    return str_replace(array_keys($replacements), array_values($replacements), $template);
}

/**
 * Truncate text to specified length while preserving words
 */
function truncateText($text, $maxLength = 155) {
    $text = strip_tags($text);
    $text = html_entity_decode($text, ENT_QUOTES, 'UTF-8');
    
    if (strlen($text) <= $maxLength) {
        return $text;
    }
    
    $truncated = substr($text, 0, $maxLength);
    $lastSpace = strrpos($truncated, ' ');
    
    if ($lastSpace !== false) {
        $truncated = substr($truncated, 0, $lastSpace);
    }
    
    return $truncated . '...';
}

/**
 * Generate structured data for SEO
 */
function generateStructuredData($pageType, $data = [], $settings = []) {
    $baseUrl = 'http' . (isset($_SERVER['HTTPS']) ? 's' : '') . '://' . $_SERVER['HTTP_HOST'];
    
    $structuredData = [
        '@context' => 'https://schema.org',
        '@type' => 'WebSite',
        'name' => $settings['site_title'] ?? '',
        'description' => $settings['site_description'] ?? '',
        'url' => $baseUrl,
    ];
    
    if ($pageType === 'site_detail' && !empty($data)) {
        $structuredData = [
            '@context' => 'https://schema.org',
            '@type' => 'WebSite',
            'name' => $data['site_name'] ?? '',
            'description' => $data['site_description'] ?? '',
            'url' => $data['site_url'] ?? '',
            'category' => $data['category_name'] ?? '',
        ];
    }
    
    return json_encode($structuredData, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
}

/**
 * Get canonical URL for current page
 */
function getCanonicalUrl() {
    $protocol = 'http' . (isset($_SERVER['HTTPS']) ? 's' : '') . '://';
    $host = $_SERVER['HTTP_HOST'];
    $uri = $_SERVER['REQUEST_URI'];
    
    // Remove query parameters for clean canonical URL
    $cleanUri = explode('?', $uri)[0];
    
    return $protocol . $host . $cleanUri;
}

/**
 * Generate meta robots directive
 */
function getMetaRobots($settings = [], $pageType = '') {
    $defaultRobots = $settings['seo_robots'] ?? 'index, follow';
    
    // Special cases where you might want different robots directives
    if (in_array($pageType, ['login', 'register', 'admin'])) {
        return 'noindex, nofollow';
    }
    
    return $defaultRobots;
}
?>