<?php
/**
 * Session Güvenlik Helper
 * Güvenli session yönetimi
 */

/**
 * Güvenli session başlat
 */
function startSecureSession() {
    if (session_status() === PHP_SESSION_NONE) {
        // Session güvenlik ayarları
        ini_set('session.cookie_httponly', 1);
        ini_set('session.cookie_secure', isset($_SERVER['HTTPS']));
        ini_set('session.use_strict_mode', 1);
        ini_set('session.cookie_samesite', 'Strict');
        
        // Session timeout (2 saat)
        ini_set('session.gc_maxlifetime', 7200);
        ini_set('session.cookie_lifetime', 7200);
        
        session_start();
        
        // Session hijacking koruması
        if (!isset($_SESSION['user_agent'])) {
            $_SESSION['user_agent'] = $_SERVER['HTTP_USER_AGENT'] ?? '';
        } elseif ($_SESSION['user_agent'] !== ($_SERVER['HTTP_USER_AGENT'] ?? '')) {
            // User agent değişmiş, session'ı temizle
            session_destroy();
            session_start();
            $_SESSION['user_agent'] = $_SERVER['HTTP_USER_AGENT'] ?? '';
        }
        
        // Session fixation koruması
        if (!isset($_SESSION['session_id'])) {
            $_SESSION['session_id'] = session_id();
        } elseif ($_SESSION['session_id'] !== session_id()) {
            // Session ID değişmiş, yeni session oluştur
            session_regenerate_id(true);
            $_SESSION['session_id'] = session_id();
        }
        
        // Session timeout kontrolü
        if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity'] > 7200)) {
            // 2 saatten fazla inaktif, session'ı temizle
            session_destroy();
            session_start();
        }
        $_SESSION['last_activity'] = time();
    }
}

/**
 * Session'ı güvenli şekilde sonlandır
 */
function destroySecureSession() {
    if (session_status() === PHP_SESSION_ACTIVE) {
        // Session verilerini temizle
        $_SESSION = array();
        
        // Session cookie'sini sil
        if (ini_get("session.use_cookies")) {
            $params = session_get_cookie_params();
            setcookie(session_name(), '', time() - 42000,
                $params["path"], $params["domain"],
                $params["secure"], $params["httponly"]
            );
        }
        
        // Session'ı yok et
        session_destroy();
    }
}

/**
 * Kullanıcı giriş yapmış mı kontrol et
 */
function isLoggedIn() {
    startSecureSession();
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

/**
 * Admin kullanıcı mı kontrol et
 */
function isAdmin() {
    startSecureSession();
    return isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'admin';
}

/**
 * Kullanıcı bilgilerini güvenli şekilde al
 */
function getCurrentUser($pdo) {
    if (!isLoggedIn()) {
        return null;
    }
    
    try {
        $stmt = $pdo->prepare('SELECT id, username, email, role FROM user WHERE id = ?');
        $stmt->execute([$_SESSION['user_id']]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log('Kullanıcı bilgisi alınamadı: ' . $e->getMessage());
        return null;
    }
}

/**
 * Session'ı yenile (timeout'u sıfırla)
 */
function refreshSession() {
    if (session_status() === PHP_SESSION_ACTIVE) {
        $_SESSION['last_activity'] = time();
    }
}
?>
