<?php
/**
 * URL Helper Functions
 * SEO-friendly URL işlemleri için gerekli fonksiyonlar
 */

class URLHelper {
    
    /**
     * Türkçe karakterleri İngilizce karşılıklarına çevirir
     */
    public static function turkishToEnglish($text) {
        $turkish = ['ç', 'Ç', 'ğ', 'Ğ', 'ı', 'I', 'İ', 'ö', 'Ö', 'ş', 'Ş', 'ü', 'Ü'];
        $english = ['c', 'C', 'g', 'G', 'i', 'I', 'I', 'o', 'O', 's', 'S', 'u', 'U'];
        
        return str_replace($turkish, $english, $text);
    }
    
    /**
     * String'i URL-friendly slug'a çevirir
     */
    public static function createSlug($text) {
        // Türkçe karakterleri çevir
        $text = self::turkishToEnglish($text);
        
        // Küçük harfe çevir
        $text = strtolower($text);
        
        // Özel karakterleri ve boşlukları tire ile değiştir
        $text = preg_replace('/[^a-z0-9\-]/', '-', $text);
        
        // Birden fazla tireyi tek tire yap
        $text = preg_replace('/-+/', '-', $text);
        
        // Başta ve sonda tire varsa sil
        $text = trim($text, '-');
        
        return $text;
    }
    
    /**
     * Kategori URL'i oluşturur
     */
    public static function getCategoryURL($categoryName, $categoryId = null) {
        $slug = self::createSlug($categoryName);
        return '/category/' . $slug;
    }
    
    /**
     * Sayfa URL'i oluşturur
     */
    public static function getPageURL($pageSlug) {
        return '/' . $pageSlug;
    }
    
    /**
     * Site detay URL'i oluşturur
     */
    public static function getSiteURL($siteName, $siteId = null) {
        $slug = self::createSlug($siteName);
        return '/site/' . $slug;
    }
    
    /**
     * Kategori slug'ından kategori ID'sini bulur
     */
    public static function getCategoryBySlug($slug, $pdo) {
        // Önce direkt tüm kategorileri çek ve slug karşılaştır
        $stmt = $pdo->prepare("SELECT * FROM categories");
        $stmt->execute();
        $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($categories as $cat) {
            $catSlug = self::createSlug($cat['name']);
            if ($catSlug === $slug) {
                return $cat;
            }
        }
        
        // Bulunamadı
        return false;
    }
    
    /**
     * Site slug'ından site bilgisini bulur
     */
    public static function getSiteBySlug($slug, $pdo) {
        // Direkt tüm approved siteleri çek ve slug karşılaştır
        $stmt = $pdo->prepare("SELECT * FROM site_submissions WHERE status = 'approved'");
        $stmt->execute();
        $sites = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($sites as $site) {
            $siteSlug = self::createSlug($site['site_name']);
            if ($siteSlug === $slug) {
                return $site;
            }
        }
        
        // Bulunamadı
        return false;
    }
    
    /**
     * Logo URL'ini standardize eder
     */
    public static function getLogoURL($site_data) {
        $logo_src = '/uploads/no-logo.png'; // Varsayılan fallback (absolute path)
        
        if (!empty($site_data['logo_url'])) {
            $logo_src = $site_data['logo_url'];
        } elseif (!empty($site_data['site_logo'])) {
            $logo_path = $site_data['site_logo'];
            
            // URL ise doğrudan kullan
            if (filter_var($logo_path, FILTER_VALIDATE_URL)) {
                $logo_src = $logo_path;
            } else {
                // Yerel dosya yolu işleme
                // Clean up any existing full URLs
                $logo_path = str_replace('https://usasportstvs.com/uploads/', '', $logo_path);
                $logo_path = str_replace('https://usasportstvs.com/', '', $logo_path);
                
                // Remove admin/uploads/ prefix if exists
                if (str_starts_with($logo_path, 'admin/uploads/')) {
                    $logo_path = str_replace('admin/uploads/', '', $logo_path);
                }
                
                // Remove /admin/uploads/ prefix if exists
                if (str_starts_with($logo_path, '/admin/uploads/')) {
                    $logo_path = str_replace('/admin/uploads/', '', $logo_path);
                }
                
                // Remove uploads/ prefix if exists
                if (str_starts_with($logo_path, 'uploads/')) {
                    $logo_path = str_replace('uploads/', '', $logo_path);
                }
                
                // Remove /uploads/ prefix if exists
                if (str_starts_with($logo_path, '/uploads/')) {
                    $logo_path = str_replace('/uploads/', '', $logo_path);
                }
                
                // Create correct path with /uploads/ prefix (absolute path)
                $logo_src = '/uploads/' . $logo_path;
            }
        }
        
        return $logo_src;
    }
    
    /**
     * Mevcut URL'i yeni formata çevirir
     */
    public static function convertOldURL($url) {
        // categorylisting.php?cat_id=1 -> /category/category-name
        if (preg_match('/categorylisting\.php\?cat_id=(\d+)/', $url, $matches)) {
            // Burada veritabanından kategori adını çekmek gerekir
            return '/category/category-' . $matches[1]; // Geçici
        }
        
        // sitedetails.php?site_id=1 -> /site/site-name
        if (preg_match('/sitedetails\.php\?site_id=(\d+)/', $url, $matches)) {
            // Burada veritabanından site adını çekmek gerekir
            return '/site/site-' . $matches[1]; // Geçici
        }
        
        // .php uzantısını kaldır
        $url = preg_replace('/\.php$/', '', $url);
        
        return $url;
    }
    
    /**
     * Base URL'i döndürür
     */
    public static function getBaseURL() {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'];
        $path = dirname($_SERVER['SCRIPT_NAME']);
        
        // Localhost kontrolü
        if (strpos($host, 'localhost') !== false) {
            return $protocol . '://' . $host . rtrim($path, '/');
        }
        
        return $protocol . '://' . $host;
    }
    
    /**
     * Tam URL oluşturur
     */
    public static function makeURL($path) {
        return self::getBaseURL() . '/' . ltrim($path, '/');
    }
    
    /**
     * Canonical URL oluşturur
     */
    public static function getCanonicalURL() {
        return self::getBaseURL() . $_SERVER['REQUEST_URI'];
    }
    
    /**
     * Meta title için slug'dan okunabilir metin oluşturur
     */
    public static function slugToTitle($slug) {
        // Tireleri boşluğa çevir
        $title = str_replace('-', ' ', $slug);
        
        // Her kelimenin ilk harfini büyük yap
        $title = ucwords($title);
        
        return $title;
    }
}

/**
 * Global helper fonksiyonlar
 */

/**
 * Kategori URL'i oluştur
 */
function category_url($categoryName, $categoryId = null) {
    return URLHelper::getCategoryURL($categoryName, $categoryId);
}

/**
 * Site URL'i oluştur
 */
function site_url($siteName, $siteId = null) {
    return URLHelper::getSiteURL($siteName, $siteId);
}

/**
 * Slug oluştur
 */
function create_slug($text) {
    return URLHelper::createSlug($text);
}

/**
 * Base URL al
 */
function base_url($path = '') {
    return URLHelper::makeURL($path);
}

/**
 * Logo URL al
 */
function get_logo_url($site_data) {
    return URLHelper::getLogoURL($site_data);
}
?>