<?php
// View count helper functions
class ViewCountHelper {
    
    /**
     * Bir site görüntülendiğinde view count'u artır
     * @param PDO $pdo Database connection
     * @param int $site_id Site ID
     * @param int $user_id User ID (null if not logged in)
     * @param string $ip_address User IP address
     * @param string $user_agent User agent string
     * @param string $referrer Referrer URL
     * @return bool Success status
     */
    public static function trackView($pdo, $site_id, $user_id = null, $ip_address = '', $user_agent = '', $referrer = '') {
        try {
            // Ensure we have necessary data
            if (!$site_id || empty($ip_address)) {
                return false;
            }
            
            // Check if site_views table exists, if not create it
            $tableCheck = $pdo->query("SHOW TABLES LIKE 'site_views'");
            if ($tableCheck->rowCount() == 0) {
                // Create table if it doesn't exist
                $pdo->exec("
                    CREATE TABLE IF NOT EXISTS site_views (
                      id INT AUTO_INCREMENT PRIMARY KEY,
                      site_id INT NOT NULL,
                      user_id INT NULL,
                      ip_address VARCHAR(45),
                      user_agent TEXT,
                      referrer VARCHAR(255),
                      viewed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                      INDEX idx_site_id (site_id),
                      INDEX idx_viewed_at (viewed_at),
                      INDEX idx_ip_site (ip_address, site_id)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
                ");
            }
            
            // Check if view_count column exists in site_submissions
            $columnCheck = $pdo->query("SHOW COLUMNS FROM site_submissions LIKE 'view_count'");
            if ($columnCheck->rowCount() == 0) {
                $pdo->exec("ALTER TABLE site_submissions ADD COLUMN view_count INT DEFAULT 0");
            }
            
            // Aynı IP'den son 24 saat içinde görüntüleme var mı kontrol et (spam önleme)
            $stmt = $pdo->prepare("
                SELECT COUNT(*) FROM site_views 
                WHERE site_id = ? AND ip_address = ? 
                AND viewed_at > DATE_SUB(NOW(), INTERVAL 24 HOUR)
            ");
            $stmt->execute([$site_id, $ip_address]);
            
            if ($stmt->fetchColumn() > 0) {
                return false; // Son 24 saatte zaten görüntülemiş
            }
            
            // Yeni görüntüleme kaydı ekle
            $stmt = $pdo->prepare("
                INSERT INTO site_views (site_id, user_id, ip_address, user_agent, referrer) 
                VALUES (?, ?, ?, ?, ?)
            ");
            
            $result = $stmt->execute([
                $site_id,
                $user_id,
                $ip_address,
                substr($user_agent, 0, 500), // Limit user agent length
                substr($referrer, 0, 255)    // Limit referrer length
            ]);
            
            // Update view count in site_submissions
            if ($result) {
                $updateStmt = $pdo->prepare("
                    UPDATE site_submissions 
                    SET view_count = (
                        SELECT COUNT(*) FROM site_views WHERE site_id = ?
                    ) 
                    WHERE id = ?
                ");
                $updateStmt->execute([$site_id, $site_id]);
            }
            
            return $result;
            
        } catch (PDOException $e) {
            error_log("ViewCount Error: " . $e->getMessage());
            // Try to continue without failing completely
            try {
                // Fallback: Just increment view_count directly
                $stmt = $pdo->prepare("
                    UPDATE site_submissions 
                    SET view_count = COALESCE(view_count, 0) + 1 
                    WHERE id = ?
                ");
                return $stmt->execute([$site_id]);
            } catch (PDOException $e2) {
                error_log("ViewCount Fallback Error: " . $e2->getMessage());
                return false;
            }
        }
    }
    
    /**
     * Site'ın güncel view count'unu al
     * @param PDO $pdo Database connection
     * @param int $site_id Site ID
     * @return int View count
     */
    public static function getViewCount($pdo, $site_id) {
        try {
            // First try to get from site_submissions table
            $stmt = $pdo->prepare("SELECT view_count FROM site_submissions WHERE id = ?");
            $stmt->execute([$site_id]);
            $count = $stmt->fetchColumn();
            
            // If view_count is null or 0, try to count from site_views table
            if ($count === false || $count === null) {
                // Check if site_views table exists
                $tableCheck = $pdo->query("SHOW TABLES LIKE 'site_views'");
                if ($tableCheck->rowCount() > 0) {
                    $stmt = $pdo->prepare("SELECT COUNT(*) FROM site_views WHERE site_id = ?");
                    $stmt->execute([$site_id]);
                    $count = $stmt->fetchColumn();
                    
                    // Update the main table with the correct count
                    if ($count > 0) {
                        $updateStmt = $pdo->prepare("UPDATE site_submissions SET view_count = ? WHERE id = ?");
                        $updateStmt->execute([$count, $site_id]);
                    }
                } else {
                    $count = 0;
                }
            }
            
            return (int) $count;
        } catch (PDOException $e) {
            error_log("GetViewCount Error: " . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * En çok görüntülenen siteleri al
     * @param PDO $pdo Database connection
     * @param int $limit Limit
     * @return array Sites list
     */
    public static function getMostViewedSites($pdo, $limit = 10) {
        try {
            $stmt = $pdo->prepare("
                SELECT s.*, c.name as category_name 
                FROM site_submissions s 
                LEFT JOIN categories c ON s.category_id = c.id 
                WHERE s.status = 'approved' 
                ORDER BY s.view_count DESC, s.olusturulma_tarihi DESC 
                LIMIT ?
            ");
            $stmt->execute([$limit]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            return [];
        }
    }
    
    /**
     * Günlük view istatistiklerini al
     * @param PDO $pdo Database connection
     * @param int $days Son kaç günün verisi
     * @return array Daily stats
     */
    public static function getDailyViewStats($pdo, $days = 7) {
        try {
            $stmt = $pdo->prepare("
                SELECT 
                    DATE(viewed_at) as date,
                    COUNT(*) as views,
                    COUNT(DISTINCT site_id) as unique_sites,
                    COUNT(DISTINCT ip_address) as unique_visitors
                FROM site_views 
                WHERE viewed_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                GROUP BY DATE(viewed_at)
                ORDER BY date DESC
            ");
            $stmt->execute([$days]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            return [];
        }
    }
}
?>