<?php
require_once '../config.php';
requireAdmin();

if (!function_exists('shortenText')) {
    function shortenText(?string $text, int $limit = 80): string
    {
        $text = trim((string)$text);
        if ($text === '') {
            return '';
        }
        if (mb_strlen($text) <= $limit) {
            return $text;
        }
        return rtrim(mb_substr($text, 0, $limit - 3)) . '...';
    }
}

$adminId = $_SESSION['user_id'] ?? null;

// Yasak ekleme işlemleri
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ban_action'])) {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('activity-logs.php');
    }

    $banAction = $_POST['ban_action'] ?? '';
    
    if ($banAction === 'add_ip_ban') {
        $ipAddress = trim($_POST['ip_address'] ?? '');
        $reason = trim($_POST['reason'] ?? '');
        $notes = trim($_POST['notes'] ?? '');
        
        if (!filter_var($ipAddress, FILTER_VALIDATE_IP)) {
            flashMessage('danger', 'Geçerli bir IP adresi giriniz.');
            redirect('activity-logs.php');
        }

        try {
            $stmt = $pdo->prepare('INSERT INTO banned_ips (ip_address, reason, notes, created_by)
                VALUES (:ip, :reason, :notes, :created_by)
                ON DUPLICATE KEY UPDATE reason = VALUES(reason), notes = VALUES(notes), updated_at = NOW()');
            $stmt->execute([
                ':ip' => $ipAddress,
                ':reason' => $reason !== '' ? $reason : null,
                ':notes' => $notes !== '' ? $notes : null,
                ':created_by' => $adminId ?: null,
            ]);

            flashMessage('success', sanitize($ipAddress) . ' adresi başarıyla yasaklandı.');
        } catch (Throwable $th) {
            flashMessage('danger', 'IP yasağı eklenirken bir hata oluştu: ' . sanitize($th->getMessage()));
        }

        redirect('activity-logs.php');
    }
    
    if ($banAction === 'add_device_ban') {
        $deviceId = trim($_POST['device_id'] ?? '');
        $reason = trim($_POST['reason'] ?? '');
        $notes = trim($_POST['notes'] ?? '');
        
        if ($deviceId === '' || !preg_match('/^[a-f0-9]{32}$/', $deviceId)) {
            flashMessage('danger', 'Geçerli bir cihaz kimliği giriniz.');
            redirect('activity-logs.php');
        }

        try {
            $stmt = $pdo->prepare('INSERT INTO banned_devices (device_id, reason, notes, created_by)
                VALUES (:device, :reason, :notes, :created_by)
                ON DUPLICATE KEY UPDATE reason = VALUES(reason), notes = VALUES(notes), updated_at = NOW()');
            $stmt->execute([
                ':device' => $deviceId,
                ':reason' => $reason !== '' ? $reason : null,
                ':notes' => $notes !== '' ? $notes : null,
                ':created_by' => $adminId ?: null,
            ]);

            flashMessage('success', 'Cihaz ID ' . sanitize($deviceId) . ' başarıyla yasaklandı.');
        } catch (Throwable $th) {
            flashMessage('danger', 'Cihaz yasağı eklenirken bir hata oluştu: ' . sanitize($th->getMessage()));
        }

        redirect('activity-logs.php');
    }
}


$filters = [
    'ip' => trim($_GET['ip'] ?? ''),
    'device' => trim($_GET['device'] ?? ''),
    'user' => trim($_GET['user'] ?? ''),
    'event' => trim($_GET['event'] ?? ''),
    'q' => trim($_GET['q'] ?? ''),
    'from' => trim($_GET['from'] ?? ''),
    'to' => trim($_GET['to'] ?? ''),
];

$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$perPage = 50;

// Tarih filtrelerini belirle
$fromDate = $filters['from'] !== '' ? $filters['from'] : date('Y-m-d', strtotime('-30 days'));
$toDate = $filters['to'] !== '' ? $filters['to'] : date('Y-m-d');

// Log dosyalarından oku
$allLogs = readActivityLogsFromFiles($fromDate, $toDate);

// Filtreleme işlemleri
$filteredLogs = [];
foreach ($allLogs as $log) {
    $match = true;
    
    // IP filtresi
    if ($filters['ip'] !== '' && stripos($log['ip_address'], $filters['ip']) === false) {
        $match = false;
    }
    
    // Cihaz ID filtresi
    if ($match && $filters['device'] !== '' && stripos($log['device_id'] ?? '', $filters['device']) === false) {
        $match = false;
    }
    
    // Kullanıcı filtresi
    if ($match && $filters['user'] !== '') {
        $userMatch = false;
        if (is_numeric($filters['user']) && (int)$log['user_id'] === (int)$filters['user']) {
            $userMatch = true;
        } elseif (stripos($log['username'] ?? '', $filters['user']) !== false) {
            $userMatch = true;
        } elseif (stripos($log['email'] ?? '', $filters['user']) !== false) {
            $userMatch = true;
        }
        if (!$userMatch) {
            $match = false;
        }
    }
    
    // Etkinlik tipi filtresi
    if ($match && $filters['event'] !== '' && $log['event_type'] !== $filters['event']) {
        $match = false;
    }
    
    // Arama filtresi
    if ($match && $filters['q'] !== '') {
        $searchTerm = strtolower($filters['q']);
        $searchIn = strtolower(
            ($log['request_uri'] ?? '') . ' ' .
            ($log['referer'] ?? '') . ' ' .
            ($log['context'] ?? '')
        );
        if (strpos($searchIn, $searchTerm) === false) {
            $match = false;
        }
    }
    
    if ($match) {
        $filteredLogs[] = $log;
    }
}

// Event tiplerini topla
$eventTypes = [];
foreach ($allLogs as $log) {
    $eventType = $log['event_type'] ?? 'page_view';
    if (!in_array($eventType, $eventTypes, true)) {
        $eventTypes[] = $eventType;
    }
}
sort($eventTypes);

// Sayfalama
$totalCount = count($filteredLogs);
$totalPages = max(1, (int)ceil($totalCount / $perPage));
if ($page > $totalPages) {
    $page = $totalPages;
}
$offset = ($page - 1) * $perPage;
$logs = array_slice($filteredLogs, $offset, $perPage);

$pageTitle = 'Aktivite Log Kayıtları';
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0"><i class="fas fa-filter me-2"></i>Filtrele</h5>
        </div>
        <div class="card-body">
            <form class="row g-3" method="get">
                <div class="col-md-3">
                    <label class="form-label">IP Adresi</label>
                    <input type="text" name="ip" class="form-control" value="<?= sanitize($filters['ip']) ?>" placeholder="Örn: 185.23.45.10">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Cihaz ID</label>
                    <input type="text" name="device" class="form-control" value="<?= sanitize($filters['device']) ?>" placeholder="32 karakter">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Kullanıcı (ID veya ad)</label>
                    <input type="text" name="user" class="form-control" value="<?= sanitize($filters['user']) ?>" placeholder="Örn: 42 ya da admin">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Etkinlik Tipi</label>
                    <select name="event" class="form-select">
                        <option value="">Tümü</option>
                        <?php foreach ($eventTypes as $eventType): ?>
                            <option value="<?= sanitize($eventType) ?>" <?= $filters['event'] === $eventType ? 'selected' : '' ?>><?= sanitize($eventType) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Başlangıç Tarihi</label>
                    <input type="date" name="from" class="form-control" value="<?= sanitize($filters['from']) ?>">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Bitiş Tarihi</label>
                    <input type="date" name="to" class="form-control" value="<?= sanitize($filters['to']) ?>">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Arama</label>
                    <input type="text" name="q" class="form-control" value="<?= sanitize($filters['q']) ?>" placeholder="URL, referer veya içerik içinde arayın">
                </div>
                <div class="col-12 d-flex justify-content-end gap-2">
                    <a class="btn btn-outline-secondary" href="activity-logs.php"><i class="fas fa-rotate-right me-1"></i>Sıfırla</a>
                    <button class="btn btn-primary" type="submit"><i class="fas fa-filter me-1"></i>Filtrele</button>
                </div>
            </form>
        </div>
    </div>

    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0"><i class="fas fa-file-alt me-2"></i>Log Kayıtları <small class="text-muted">(/logs klasöründen okunuyor)</small></h5>
            <div class="d-flex align-items-center gap-2">
                <span class="badge bg-secondary">Toplam <?= $totalCount ?> kayıt</span>
            </div>
        </div>
        <div class="card-body p-0">
            <div class="table-responsive">
                <table class="table table-hover mb-0">
                    <thead class="table-light">
                        <tr>
                            <th>Zaman</th>
                            <th>Kullanıcı</th>
                            <th>IP / Konum</th>
                            <th>Cihaz</th>
                            <th>Etkinlik</th>
                            <th>İstek</th>
                            <th class="text-end">İşlemler</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($logs)): ?>
                        <tr>
                            <td colspan="7" class="text-center text-muted py-4">Kayıt bulunamadı.</td>
                        </tr>
                        <?php else: ?>
                            <?php foreach ($logs as $index => $log): ?>
                            <?php
                                $rowId = 'logDetail' . md5($log['id'] . $index);
                                $context = [];
                                if (!empty($log['context'])) {
                                    $decoded = json_decode($log['context'], true);
                                    if (is_array($decoded)) {
                                        $context = $decoded;
                                    }
                                }
                            ?>
                            <tr>
                                <td>
                                    <div><?= date('d.m.Y H:i', strtotime($log['created_at'])) ?></div>
                                    <small class="text-muted"><?= sanitize($log['request_method'] ?? '-') ?></small>
                                </td>
                                <td>
                                    <?php if (!empty($log['username'])): ?>
                                        <strong><?= sanitize($log['username']) ?></strong>
                                        <div class="text-muted small">#<?= (int)$log['user_id'] ?></div>
                                    <?php else: ?>
                                        <span class="badge bg-secondary">Misafir</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <div><strong><?= sanitize($log['ip_address']) ?></strong></div>
                                    <div class="text-muted small">
                                        <?= sanitize(trim(($log['city'] ?? '') . ' ' . ($log['country'] ?? ''))) ?: '—' ?>
                                    </div>
                                </td>
                                <td>
                                    <code><?= sanitize($log['device_id'] ?? '-') ?></code>
                                </td>
                                <td>
                                    <span class="badge bg-info text-dark"><?= sanitize($log['event_type']) ?></span>
                                    <?php if (!empty($log['referer'])): ?>
                                        <div class="text-muted small">Referer: <?= sanitize(shortenText($log['referer'], 60)) ?></div>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <div class="text-break" style="max-width:220px;">
                                        <code><?= sanitize($log['request_uri']) ?></code>
                                    </div>
                                </td>
                                <td class="text-end">
                                    <button class="btn btn-sm btn-outline-secondary" type="button" data-bs-toggle="collapse" data-bs-target="#<?= $rowId ?>" aria-expanded="false" aria-controls="<?= $rowId ?>">
                                        <i class="fas fa-circle-info"></i>
                                    </button>
                                    <button class="btn btn-sm btn-outline-danger" type="button" data-bs-toggle="modal" data-bs-target="#banIpModal" 
                                            data-ip="<?= sanitize($log['ip_address']) ?>" 
                                            data-user="<?= !empty($log['username']) ? sanitize($log['username']) : 'Misafir' ?>" 
                                            title="IP yasağı oluştur">
                                        <i class="fas fa-ban"></i>
                                    </button>
                                    <?php if (!empty($log['device_id'])): ?>
                                    <button class="btn btn-sm btn-outline-warning" type="button" data-bs-toggle="modal" data-bs-target="#banDeviceModal" 
                                            data-device="<?= sanitize($log['device_id']) ?>" 
                                            data-user="<?= !empty($log['username']) ? sanitize($log['username']) : 'Misafir' ?>" 
                                            title="Cihaz yasağı oluştur">
                                        <i class="fas fa-user-lock"></i>
                                    </button>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <tr class="collapse" id="<?= $rowId ?>">
                                <td colspan="7" class="bg-light">
                                    <div class="p-3">
                                        <h6 class="d-flex align-items-center gap-2"><i class="fas fa-code"></i>İstek Detayları</h6>
                                        <div class="row g-3">
                                            <div class="col-md-6">
                                                <div class="log-detail-block">
                                                    <strong>GET Parametreleri</strong>
                                                    <pre class="log-pre"><?= sanitize(json_encode($context['get'] ?? [], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)) ?: '[]' ?></pre>
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <div class="log-detail-block">
                                                    <strong>POST Parametreleri</strong>
                                                    <pre class="log-pre"><?= sanitize(json_encode($context['post'] ?? [], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)) ?: '[]' ?></pre>
                                                </div>
                                            </div>
                                            <div class="col-md-12">
                                                <div class="log-detail-block">
                                                    <strong>Tam İçerik</strong>
                                                    <pre class="log-pre"><?= sanitize(json_encode($context, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)) ?: '[]' ?></pre>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php if ($totalPages > 1): ?>
        <div class="card-footer bg-light border-top">
            <div class="d-flex justify-content-between align-items-center flex-wrap gap-3">
                <div class="text-muted small d-flex align-items-center gap-1">
                    <i class="fas fa-info-circle"></i>
                    <span>Toplam <strong><?= $totalCount ?></strong> kayıt, <strong><?= $totalPages ?></strong> sayfa</span>
                </div>
                
                <div class="d-flex align-items-center gap-2">
                    <?php $query = $_GET; ?>
                    
                    <a class="btn btn-sm btn-outline-primary<?= $page <= 1 ? ' disabled' : '' ?>" 
                       href="<?= $page <= 1 ? '#' : 'activity-logs.php?' . http_build_query(array_merge($query, ['page' => $page - 1])) ?>"
                       <?= $page <= 1 ? 'onclick="return false;" tabindex="-1"' : '' ?>
                       title="Önceki sayfa">
                        <i class="fas fa-chevron-left"></i>
                    </a>
                    
                    <form method="get" class="d-inline-flex align-items-center gap-2" 
                          onsubmit="var pageNum = parseInt(this.querySelector('input[name=page]').value); if(isNaN(pageNum) || pageNum < 1 || pageNum > <?= $totalPages ?>) { alert('Lütfen 1 ile <?= $totalPages ?> arasında bir sayı girin.'); return false; } return true;">
                        <?php foreach ($query as $key => $value): ?>
                            <?php if ($key !== 'page'): ?>
                                <input type="hidden" name="<?= htmlspecialchars($key, ENT_QUOTES, 'UTF-8') ?>" value="<?= htmlspecialchars($value, ENT_QUOTES, 'UTF-8') ?>">
                            <?php endif; ?>
                        <?php endforeach; ?>
                        
                        <div class="input-group input-group-sm" style="width: auto;">
                            <span class="input-group-text bg-white border-end-0 text-muted" style="font-size: 0.875rem;">Sayfa</span>
                            <input type="number" name="page" class="form-control text-center border-start-0 border-end-0" 
                                   value="<?= $page ?>" min="1" max="<?= $totalPages ?>" 
                                   style="width: 70px; font-weight: 500;" 
                                   onchange="if(this.value < 1) this.value = 1; if(this.value > <?= $totalPages ?>) this.value = <?= $totalPages ?>;">
                            <span class="input-group-text bg-white border-start-0 text-muted" style="font-size: 0.875rem;">/ <?= $totalPages ?></span>
                        </div>
                        
                        <button type="submit" class="btn btn-sm btn-primary" title="Sayfaya git">
                            <i class="fas fa-arrow-right"></i>
                        </button>
                    </form>
                    
                    <a class="btn btn-sm btn-outline-primary<?= $page >= $totalPages ? ' disabled' : '' ?>" 
                       href="<?= $page >= $totalPages ? '#' : 'activity-logs.php?' . http_build_query(array_merge($query, ['page' => $page + 1])) ?>"
                       <?= $page >= $totalPages ? 'onclick="return false;" tabindex="-1"' : '' ?>
                       title="Sonraki sayfa">
                        <i class="fas fa-chevron-right"></i>
                    </a>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- IP Yasağı Modal -->
<div class="modal fade" id="banIpModal" tabindex="-1" aria-labelledby="banIpModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="banIpModalLabel"><i class="fas fa-ban me-2 text-danger"></i>IP Adresi Yasakla</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Kapat"></button>
            </div>
            <form method="post" action="activity-logs.php">
                <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                <input type="hidden" name="ban_action" value="add_ip_ban">
                <div class="modal-body">
                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <strong>Uyarı:</strong> Bu IP adresinden gelen tüm istekler engellenecektir.
                    </div>
                    <div class="mb-3">
                        <label class="form-label">IP Adresi <span class="text-danger">*</span></label>
                        <input type="text" name="ip_address" id="banIpAddress" class="form-control" required readonly>
                        <small class="text-muted">Kullanıcı: <span id="banIpUser">-</span></small>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Sebep</label>
                        <select name="reason" class="form-select">
                            <option value="">Seçiniz...</option>
                            <option value="Spam">Spam</option>
                            <option value="Dolandırıcılık">Dolandırıcılık</option>
                            <option value="Şüpheli Aktivite">Şüpheli Aktivite</option>
                            <option value="Kötüye Kullanım">Kötüye Kullanım</option>
                            <option value="Diğer">Diğer</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Notlar</label>
                        <textarea name="notes" class="form-control" rows="3" placeholder="Ek açıklamalar..."></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <button type="submit" class="btn btn-danger"><i class="fas fa-ban me-2"></i>IP'yi Yasakla</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Cihaz Yasağı Modal -->
<div class="modal fade" id="banDeviceModal" tabindex="-1" aria-labelledby="banDeviceModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="banDeviceModalLabel"><i class="fas fa-user-lock me-2 text-warning"></i>Cihaz Yasakla</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Kapat"></button>
            </div>
            <form method="post" action="activity-logs.php">
                <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                <input type="hidden" name="ban_action" value="add_device_ban">
                <div class="modal-body">
                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <strong>Uyarı:</strong> Bu cihazdan gelen tüm istekler engellenecektir.
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Cihaz ID <span class="text-danger">*</span></label>
                        <input type="text" name="device_id" id="banDeviceId" class="form-control" required readonly>
                        <small class="text-muted">Kullanıcı: <span id="banDeviceUser">-</span></small>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Sebep</label>
                        <select name="reason" class="form-select">
                            <option value="">Seçiniz...</option>
                            <option value="Spam">Spam</option>
                            <option value="Dolandırıcılık">Dolandırıcılık</option>
                            <option value="Şüpheli Aktivite">Şüpheli Aktivite</option>
                            <option value="Kötüye Kullanım">Kötüye Kullanım</option>
                            <option value="Diğer">Diğer</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Notlar</label>
                        <textarea name="notes" class="form-control" rows="3" placeholder="Ek açıklamalar..."></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <button type="submit" class="btn btn-warning"><i class="fas fa-user-lock me-2"></i>Cihazı Yasakla</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// IP Yasağı modalı için veri doldurma
document.getElementById('banIpModal').addEventListener('show.bs.modal', function (event) {
    var button = event.relatedTarget;
    var ip = button.getAttribute('data-ip');
    var user = button.getAttribute('data-user');
    
    document.getElementById('banIpAddress').value = ip;
    document.getElementById('banIpUser').textContent = user;
});

// Cihaz Yasağı modalı için veri doldurma
document.getElementById('banDeviceModal').addEventListener('show.bs.modal', function (event) {
    var button = event.relatedTarget;
    var device = button.getAttribute('data-device');
    var user = button.getAttribute('data-user');
    
    document.getElementById('banDeviceId').value = device;
    document.getElementById('banDeviceUser').textContent = user;
});
</script>

<?php include 'includes/admin-footer.php'; ?>
