<?php
require_once '../config.php';
requireAdmin();

// Admin ekleme
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_admin'])) {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik hatası.');
    } else {
        $userId = (int)($_POST['user_id'] ?? 0);
        
        if ($userId <= 0) {
            flashMessage('danger', 'Geçersiz kullanıcı.');
        } else {
            try {
                $stmt = $pdo->prepare("UPDATE users SET is_admin = 1 WHERE id = ?");
                $stmt->execute([$userId]);
                
                flashMessage('success', 'Kullanıcı admin yetkisi aldı.');
                redirect('admins.php');
            } catch (PDOException $e) {
                flashMessage('danger', 'Admin yetkisi verilemedi: ' . $e->getMessage());
            }
        }
    }
}

// Admin yetkisi kaldırma
if (isset($_GET['remove_admin'])) {
    $removeId = (int)$_GET['remove_admin'];
    
    if ($removeId <= 0) {
        flashMessage('danger', 'Geçersiz kullanıcı.');
    } elseif ($removeId === $_SESSION['user_id']) {
        flashMessage('danger', 'Kendi admin yetkinizi kaldıramazsınız.');
    } else {
        try {
            $stmt = $pdo->prepare("UPDATE users SET is_admin = 0 WHERE id = ? AND id != ?");
            $stmt->execute([$removeId, $_SESSION['user_id']]);
            
            flashMessage('success', 'Admin yetkisi kaldırıldı.');
        } catch (PDOException $e) {
            flashMessage('danger', 'İşlem başarısız: ' . $e->getMessage());
        }
    }
    redirect('admins.php');
}

// Pagination
$totalAdminsStmt = $pdo->query("SELECT COUNT(*) FROM users WHERE is_admin = 1");
$totalAdmins = (int)$totalAdminsStmt->fetchColumn();
$pagination = getPaginationData($totalAdmins, 10);

// Admin kullanıcıları getir
$admins = $pdo->query("
    SELECT u.*, 
           COUNT(DISTINCT o.id) as order_count,
           COALESCE(SUM(o.total_amount), 0) as total_spent
    FROM users u
    LEFT JOIN orders o ON u.id = o.user_id AND o.status = 'completed'
    WHERE u.is_admin = 1
    GROUP BY u.id
    ORDER BY u.created_at DESC
    LIMIT {$pagination['limit']} OFFSET {$pagination['offset']}
")->fetchAll();

// Admin olmayan kullanıcıları getir (admin yapmak için)
$nonAdminUsers = $pdo->query("
    SELECT id, username, email, full_name
    FROM users
    WHERE is_admin = 0
    ORDER BY username ASC
    LIMIT 100
")->fetchAll();

$pageTitle = 'Admin Yönetimi';
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h3>
            <i class="fas fa-crown me-2"></i>Admin Yönetimi
        </h3>
    </div>
    
    <div class="alert alert-warning d-flex align-items-center mb-4">
        <i class="fas fa-info-circle me-2"></i>
        <div>
            <strong>Önemli Bilgilendirme:</strong> Admin yaptığınız kullanıcı aktif ise, üyeliğe giriş çıkış yapınca admin yetkisi aktif olacaktır.
        </div>
    </div>
    
    <div class="row g-4 mb-4">
        <div class="col-lg-5">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">
                        <i class="fas fa-user-plus me-2"></i>Yeni Admin Ekle
                    </h5>
                </div>
                <div class="card-body">
                    <form method="POST">
                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                        <input type="hidden" name="add_admin" value="1">
                        
                        <div class="mb-3">
                            <label class="form-label">Kullanıcı Seçin</label>
                            <select class="form-select" name="user_id" required>
                                <option value="">Kullanıcı Seçin...</option>
                                <?php foreach ($nonAdminUsers as $user): ?>
                                <option value="<?= $user['id'] ?>">
                                    <?= sanitize($user['username']) ?> - <?= sanitize($user['email']) ?> <?= $user['full_name'] ? '(' . sanitize($user['full_name']) . ')' : '' ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">Seçilen kullanıcıya admin yetkisi verilecektir.</small>
                        </div>
                        
                        <button type="submit" class="btn btn-primary w-100">
                            <i class="fas fa-crown me-2"></i>Admin Yap
                        </button>
                    </form>
                </div>
            </div>
        </div>
    </div>
    
    <div class="card">
        <div class="card-header d-flex flex-wrap justify-content-between align-items-center gap-3">
            <h5 class="mb-0"><i class="fas fa-users-gear me-2"></i>Admin Kullanıcılar</h5>
            <span class="badge bg-primary">Toplam <?= $totalAdmins ?> Admin</span>
        </div>
        <div class="card-body p-0">
            <div class="table-responsive">
                <table class="table table-hover mb-0">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Kullanıcı Adı</th>
                            <th>E-posta</th>
                            <th>Ad Soyad</th>
                            <th>Bakiye</th>
                            <th>Sipariş</th>
                            <th>Harcama</th>
                            <th>Durum</th>
                            <th>Kayıt Tarihi</th>
                            <th>İşlemler</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($admins)): ?>
                        <tr>
                            <td colspan="10" class="text-center text-muted py-4">Admin kullanıcı bulunamadı.</td>
                        </tr>
                        <?php else: ?>
                            <?php foreach ($admins as $admin): ?>
                            <tr>
                                <td><strong>#<?= $admin['id'] ?></strong></td>
                                <td>
                                    <strong><?= sanitize($admin['username']) ?></strong>
                                    <span class="badge bg-warning text-dark ms-1"><i class="fas fa-crown"></i></span>
                                </td>
                                <td><?= sanitize($admin['email']) ?></td>
                                <td><?= sanitize($admin['full_name']) ?></td>
                                <td><strong><?= formatPrice($admin['balance']) ?></strong></td>
                                <td>
                                    <span class="badge bg-info"><?= $admin['order_count'] ?></span>
                                </td>
                                <td><?= formatPrice($admin['total_spent']) ?></td>
                                <td>
                                    <?php if ($admin['is_active']): ?>
                                    <span class="badge bg-success">Aktif</span>
                                    <?php else: ?>
                                    <span class="badge bg-danger">Pasif</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <small><?= date('d.m.Y', strtotime($admin['created_at'])) ?></small>
                                </td>
                                <td>
                                    <a href="user-detail.php?id=<?= $admin['id'] ?>" class="btn btn-sm btn-outline-primary" title="Detaylar">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    <?php if ($admin['id'] !== $_SESSION['user_id']): ?>
                                    <a href="?remove_admin=<?= $admin['id'] ?>" class="btn btn-sm btn-outline-danger" onclick="return confirm('Bu kullanıcının admin yetkisini kaldırmak istediğinize emin misiniz?')" title="Admin Yetkisini Kaldır">
                                        <i class="fas fa-user-slash"></i>
                                    </a>
                                    <?php else: ?>
                                    <span class="text-muted small">(Siz)</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            <?php if ($totalAdmins > 10): ?>
            <div class="card-body border-top">
                <?= renderPagination($pagination) ?>
            </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>

