let marketplaceAdminAudioContext;

document.addEventListener('DOMContentLoaded', function () {
    const root = document.querySelector('body.admin-body');
    const sidebar = document.querySelector('[data-sidebar]');
    const overlay = document.querySelector('[data-sidebar-overlay]');
    const toggleButtons = document.querySelectorAll('[data-sidebar-toggle]');
    const closeButtons = document.querySelectorAll('[data-sidebar-close]');
    const sidebarLinks = document.querySelectorAll('.sidebar-link, .sidebar-sub__link');

    const openSidebar = () => {
        if (!root) return;
        root.classList.add('sidebar-open');
    };

    const closeSidebar = () => {
        if (!root) return;
        root.classList.remove('sidebar-open');
    };

    const toggleSidebar = () => {
        if (!root) return;
        if (root.classList.contains('sidebar-open')) {
            closeSidebar();
        } else {
            openSidebar();
        }
    };

    toggleButtons.forEach((btn) => {
        btn.addEventListener('click', toggleSidebar);
    });

    closeButtons.forEach((btn) => {
        btn.addEventListener('click', closeSidebar);
    });

    if (overlay) {
        overlay.addEventListener('click', closeSidebar);
    }

    sidebarLinks.forEach((link) => {
        link.addEventListener('click', () => {
            if (window.innerWidth < 992) {
                closeSidebar();
            }
        });
    });

    window.addEventListener('resize', () => {
        if (window.innerWidth >= 992) {
            closeSidebar();
        }
    });

    document.addEventListener('keydown', (event) => {
        if (event.key === 'Escape') {
            closeSidebar();
        }
    });

    document.querySelectorAll('[data-filter-group]').forEach((group) => {
        group.querySelectorAll('[data-filter]').forEach((chip) => {
            chip.addEventListener('click', () => {
                group.querySelectorAll('[data-filter]').forEach((el) => el.classList.remove('active'));
                chip.classList.add('active');
            });
        });
    });

    document.querySelectorAll('[data-sub-toggle]').forEach((toggle) => {
        toggle.addEventListener('click', () => {
            const target = toggle.getAttribute('data-sub-toggle');
            const parent = toggle.closest('.sidebar-group');
            const submenu = target
                ? document.querySelector(`[data-submenu="${target}"]`)
                : parent?.querySelector('.sidebar-sub');

            if (parent) {
                const isOpen = parent.classList.contains('is-open');
                parent.classList.toggle('is-open', !isOpen);
            }

            if (submenu) {
                submenu.classList.toggle('show');
            }
        });
    });

    initMarketplaceNotifications();
    initAdminThemeToggle();
    highlightSettingsLink();
    window.addEventListener('hashchange', highlightSettingsLink);

    document.querySelectorAll('[data-bulk-form]').forEach((form) => {
        const formId = form.getAttribute('data-bulk-form');
        const checkboxes = Array.from(document.querySelectorAll(`[data-bulk-target="${formId}"]`));
        const master = document.querySelector(`[data-bulk-master="${formId}"]`);
        const actionButtons = form.querySelectorAll('[data-bulk-action]');

        const updateBulkState = () => {
            const hasSelection = checkboxes.some((input) => input.checked);
            actionButtons.forEach((btn) => {
                btn.disabled = !hasSelection;
            });
        };

        checkboxes.forEach((checkbox) => {
            checkbox.addEventListener('change', updateBulkState);
        });

        if (master) {
            master.addEventListener('change', () => {
                checkboxes.forEach((checkbox) => {
                    checkbox.checked = master.checked;
                });
                updateBulkState();
            });
        }

        actionButtons.forEach((btn) => {
            const confirmMessage = btn.dataset.confirm;
            if (confirmMessage) {
                btn.addEventListener('click', (event) => {
                    if (!window.confirm(confirmMessage)) {
                        event.preventDefault();
                    }
                });
            }
        });

        updateBulkState();
    });
});

function initMarketplaceNotifications() {
    ensureAdminAudioUnlock();

    let lastCheck = new Date().toISOString();

    const poll = () => {
        fetch(`../marketplace/api/poll-admin.php?since=${encodeURIComponent(lastCheck)}`, {
            credentials: 'include'
        })
            .then((res) => res.json())
            .then((data) => {
                if (!data || !data.success) {
                    return;
                }
                lastCheck = data.timestamp || new Date().toISOString();
                if (data.count && data.count > 0) {
                    playAdminNotificationTone();
                }
            })
            .catch(() => {});
    };

    poll();
    setInterval(poll, 5000);
}

function initAdminThemeToggle() {
    const toggles = document.querySelectorAll('[data-admin-theme-toggle]');
    const root = document.querySelector('body.admin-body');
    if (!root || !toggles.length) {
        return;
    }

    const stored = localStorage.getItem('admin-theme');
    const initial = stored === 'light' || stored === 'dark' ? stored : (root.dataset.adminTheme || 'dark');

    const emitThemeChange = (theme) => {
        document.dispatchEvent(new CustomEvent('admin-theme-change', { detail: theme }));
    };

    const apply = (theme) => {
        const nextTheme = theme === 'light' ? 'light' : 'dark';
        root.dataset.adminTheme = nextTheme;
        localStorage.setItem('admin-theme', nextTheme);
        emitThemeChange(nextTheme);
        toggles.forEach((btn) => {
            const icon = btn.querySelector('i');
            if (icon) {
                icon.classList.remove('fa-moon', 'fa-sun');
                icon.classList.add(nextTheme === 'light' ? 'fa-sun' : 'fa-moon');
            }
        });
    };

    apply(initial);

    toggles.forEach((btn) => {
        btn.addEventListener('click', () => {
            const next = root.dataset.adminTheme === 'light' ? 'dark' : 'light';
            apply(next);
        });
    });
}

function highlightSettingsLink() {
    const settingsLinks = document.querySelectorAll('[data-settings-link]');
    if (!settingsLinks.length) {
        return;
    }

    const currentPath = window.location.pathname.split('/').pop();
    settingsLinks.forEach((link) => link.classList.remove('active', 'is-active'));

    let hash = window.location.hash;
    if (!hash || hash.length === 0) {
        if (currentPath === 'settings.php') {
            const defaultLink = settingsLinks[0];
            if (defaultLink) {
                defaultLink.classList.add('active', 'is-active');
            }
        }
        return;
    }

    const hashLink = document.querySelector(`[data-settings-link="${hash}"]`);
    if (hashLink) {
        hashLink.classList.add('active', 'is-active');
    }
}

function getAdminAudioContext() {
    if (typeof window.AudioContext === 'undefined' && typeof window.webkitAudioContext === 'undefined') {
        return null;
    }
    if (!marketplaceAdminAudioContext) {
        const Ctx = window.AudioContext || window.webkitAudioContext;
        marketplaceAdminAudioContext = new Ctx();
    }
    return marketplaceAdminAudioContext;
}

function ensureAdminAudioUnlock() {
    const ctx = getAdminAudioContext();
    if (!ctx) {
        return;
    }
    const unlock = () => {
        if (ctx.state === 'suspended') {
            ctx.resume().catch(() => {});
        }
    };
    document.addEventListener('pointerdown', unlock, { once: true });
    document.addEventListener('keydown', unlock, { once: true });
}

function playAdminNotificationTone() {
    const ctx = getAdminAudioContext();
    if (!ctx) {
        return;
    }

    if (ctx.state === 'suspended') {
        ctx.resume().catch(() => {});
    }

    const duration = 4.5;
    const oscillator = ctx.createOscillator();
    const gain = ctx.createGain();

    oscillator.type = 'triangle';
    oscillator.connect(gain);
    gain.connect(ctx.destination);

    const now = ctx.currentTime;
    oscillator.frequency.setValueAtTime(540, now);
    oscillator.frequency.setValueAtTime(680, now + 1.5);
    oscillator.frequency.setValueAtTime(480, now + 3);

    gain.gain.setValueAtTime(0.0001, now);
    gain.gain.exponentialRampToValueAtTime(0.25, now + 0.15);
    gain.gain.exponentialRampToValueAtTime(0.0001, now + duration);

    oscillator.start(now);
    oscillator.stop(now + duration);
}


