<?php
require_once '../config.php';
requireAdmin();

$adminExtraStyles = [
    'https://cdn.quilljs.com/1.3.7/quill.snow.css'
];
$adminExtraScriptsFoot = [
    'https://cdn.quilljs.com/1.3.7/quill.js'
];

if (!function_exists('shortenText')) {
    function shortenText(?string $text, int $limit = 120): string
    {
        $text = trim((string)$text);
        if ($text === '') {
            return '';
        }
        if (mb_strlen($text) <= $limit) {
            return $text;
        }
        return rtrim(mb_substr($text, 0, $limit - 3)) . '...';
    }
}

$adminId = $_SESSION['user_id'] ?? null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('blog.php');
    }

    $action = $_POST['action'] ?? '';

    if ($action === 'delete_post') {
        $postId = (int)($_POST['post_id'] ?? 0);
        if ($postId <= 0) {
            flashMessage('danger', 'Geçersiz blog kaydı.');
            redirect('blog.php');
        }

        try {
            $stmt = $pdo->prepare('SELECT image_path FROM blog_posts WHERE id = ?');
            $stmt->execute([$postId]);
            $post = $stmt->fetch();
            if ($post) {
                if (!empty($post['image_path'])) {
                    deleteUploadedFile($post['image_path']);
                }
                $pdo->prepare('DELETE FROM blog_posts WHERE id = ?')->execute([$postId]);
                logUserActivity($pdo, 'admin_blog_deleted', ['post_id' => $postId]);
                flashMessage('success', 'Blog yazısı silindi.');
            } else {
                flashMessage('warning', 'Silinecek blog yazısı bulunamadı.');
            }
        } catch (Throwable $th) {
            flashMessage('danger', 'Blog yazısı silinirken hata oluştu: ' . sanitize($th->getMessage()));
        }

        redirect('blog.php');
    }

    if ($action === 'save_post') {
        $postId = isset($_POST['post_id']) && $_POST['post_id'] !== '' ? (int)$_POST['post_id'] : null;
        $title = trim($_POST['title'] ?? '');
        $slugInput = trim($_POST['slug'] ?? '');
        $excerpt = trim($_POST['excerpt'] ?? '');
        $contentHtml = trim($_POST['content_html'] ?? '');
        $metaTitle = trim($_POST['meta_title'] ?? '');
        $metaDescription = trim($_POST['meta_description'] ?? '');
        $metaKeywords = trim($_POST['meta_keywords'] ?? '');
        $isPublished = isset($_POST['is_published']) ? 1 : 0;
        $removeImage = isset($_POST['remove_image']);

        if ($title === '') {
            flashMessage('danger', 'Blog başlığı zorunludur.');
            redirect('blog.php' . ($postId ? '?edit=' . $postId : ''));
        }

        if ($contentHtml === '') {
            flashMessage('danger', 'Blog içeriği boş olamaz.');
            redirect('blog.php' . ($postId ? '?edit=' . $postId : ''));
        }

        $currentPost = null;
        if ($postId) {
            $stmt = $pdo->prepare('SELECT * FROM blog_posts WHERE id = ?');
            $stmt->execute([$postId]);
            $currentPost = $stmt->fetch();
            if (!$currentPost) {
                flashMessage('danger', 'Güncellenecek blog yazısı bulunamadı.');
                redirect('blog.php');
            }
        }

        $slugBase = $slugInput !== '' ? generateSlug($slugInput) : generateSlug($title);
        $slug = $slugBase;
        $slugIndex = 2;
        $slugStmt = $pdo->prepare('SELECT id FROM blog_posts WHERE slug = ? AND id != ? LIMIT 1');
        while (true) {
            $params = [$slug, $postId ?? 0];
            $slugStmt->execute($params);
            if (!$slugStmt->fetch()) {
                break;
            }
            $slug = $slugBase . '-' . $slugIndex;
            $slugIndex++;
        }

        $metaDefaults = buildMetaDefaults($title, strip_tags($contentHtml));
        if ($metaTitle === '') {
            $metaTitle = $metaDefaults['meta_title'];
        }
        if ($metaDescription === '') {
            $metaDescription = $metaDefaults['meta_description'];
        }
        if ($metaKeywords === '') {
            $metaKeywords = $metaDefaults['meta_keywords'];
        }

        $imagePath = $currentPost['image_path'] ?? null;
        if ($removeImage && $imagePath) {
            deleteUploadedFile($imagePath);
            $imagePath = null;
        }

        try {
            $imagePath = uploadImageFile($_FILES['image'] ?? [], 'blog', $imagePath);
        } catch (Throwable $th) {
            flashMessage('danger', 'Görsel yüklenemedi: ' . sanitize($th->getMessage()));
            redirect('blog.php' . ($postId ? '?edit=' . $postId : ''));
        }

        $publishedAt = $currentPost['published_at'] ?? null;
        if ($isPublished) {
            if (!$publishedAt) {
                $publishedAt = date('Y-m-d H:i:s');
            }
        } else {
            $publishedAt = null;
        }

        try {
            if ($postId) {
                $stmt = $pdo->prepare('UPDATE blog_posts SET title = :title, slug = :slug, excerpt = :excerpt, content = :content, image_path = :image, meta_title = :meta_title, meta_description = :meta_description, meta_keywords = :meta_keywords, is_published = :is_published, published_at = :published_at, updated_at = NOW() WHERE id = :id');
                $stmt->execute([
                    ':title' => $title,
                    ':slug' => $slug,
                    ':excerpt' => $excerpt !== '' ? $excerpt : null,
                    ':content' => $contentHtml,
                    ':image' => $imagePath,
                    ':meta_title' => $metaTitle,
                    ':meta_description' => $metaDescription,
                    ':meta_keywords' => $metaKeywords,
                    ':is_published' => $isPublished,
                    ':published_at' => $publishedAt,
                    ':id' => $postId,
                ]);
                logUserActivity($pdo, 'admin_blog_updated', ['post_id' => $postId, 'slug' => $slug]);
                flashMessage('success', 'Blog yazısı güncellendi.');
            } else {
                $stmt = $pdo->prepare('INSERT INTO blog_posts (title, slug, excerpt, content, image_path, meta_title, meta_description, meta_keywords, author_id, is_published, published_at)
                    VALUES (:title, :slug, :excerpt, :content, :image, :meta_title, :meta_description, :meta_keywords, :author_id, :is_published, :published_at)');
                $stmt->execute([
                    ':title' => $title,
                    ':slug' => $slug,
                    ':excerpt' => $excerpt !== '' ? $excerpt : null,
                    ':content' => $contentHtml,
                    ':image' => $imagePath,
                    ':meta_title' => $metaTitle,
                    ':meta_description' => $metaDescription,
                    ':meta_keywords' => $metaKeywords,
                    ':author_id' => $adminId,
                    ':is_published' => $isPublished,
                    ':published_at' => $isPublished ? $publishedAt : null,
                ]);
                $postId = (int)$pdo->lastInsertId();
                logUserActivity($pdo, 'admin_blog_created', ['post_id' => $postId, 'slug' => $slug]);
                flashMessage('success', 'Yeni blog yazısı oluşturuldu.');
            }
        } catch (Throwable $th) {
            flashMessage('danger', 'Blog yazısı kaydedilemedi: ' . sanitize($th->getMessage()));
            redirect('blog.php' . ($postId ? '?edit=' . $postId : ''));
        }

        redirect('blog.php' . ($postId ? '?edit=' . $postId : ''));
    }
}

$postsStmt = $pdo->query('SELECT b.*, u.username AS author_username FROM blog_posts b LEFT JOIN users u ON b.author_id = u.id ORDER BY b.created_at DESC');
$posts = $postsStmt->fetchAll() ?: [];

$editPost = null;
if (isset($_GET['edit'])) {
    $stmt = $pdo->prepare('SELECT * FROM blog_posts WHERE id = ?');
    $stmt->execute([(int)$_GET['edit']]);
    $editPost = $stmt->fetch();
    if (!$editPost) {
        flashMessage('danger', 'Blog yazısı bulunamadı.');
        redirect('blog.php');
    }
}

$initialContent = $editPost['content'] ?? '';
$adminExtraScriptsFoot[] = [
    'content' => 'const blogEditorContainer=document.getElementById("blogEditor");if(blogEditorContainer){const quill=new Quill("#blogEditor",{theme:"snow",placeholder:"Blog içeriğini buraya yazın"});const initialContent=' . json_encode($initialContent, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) . ';if(initialContent){quill.clipboard.dangerouslyPasteHTML(initialContent);}const form=document.getElementById("blogForm");if(form){form.addEventListener("submit",function(){document.getElementById("blog_content_html").value=quill.root.innerHTML.trim();});}}',
    'type' => 'text/javascript'
];

$pageTitle = 'Blog Yönetimi';
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="row g-4">
        <div class="col-lg-7">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0"><i class="fas fa-blog me-2"></i><?= $editPost ? 'Blog Yazısını Düzenle' : 'Yeni Blog Yazısı' ?></h5>
                    <?php if ($editPost): ?>
                        <a href="blog.php" class="btn btn-sm btn-outline-secondary"><i class="fas fa-plus me-1"></i>Yeni Yazı</a>
                    <?php endif; ?>
                </div>
                <div class="card-body">
                    <form method="post" enctype="multipart/form-data" id="blogForm" class="row g-3">
                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                        <input type="hidden" name="action" value="save_post">
                        <input type="hidden" name="post_id" value="<?= $editPost ? (int)$editPost['id'] : '' ?>">
                        <input type="hidden" name="content_html" id="blog_content_html">
                        <div class="col-12">
                            <label class="form-label">Başlık *</label>
                            <input type="text" name="title" class="form-control" value="<?= sanitize($editPost['title'] ?? ($_POST['title'] ?? '')) ?>" required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Slug</label>
                            <input type="text" name="slug" class="form-control" value="<?= sanitize($editPost['slug'] ?? ($_POST['slug'] ?? '')) ?>" placeholder="Otomatik oluşturulur">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Durum</label>
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" role="switch" name="is_published" id="blog_is_published" <?= (($editPost['is_published'] ?? 0) == 1 || (!isset($editPost) && !empty($_POST['is_published']))) ? 'checked' : '' ?>>
                                <label class="form-check-label" for="blog_is_published">Yayınla</label>
                            </div>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Kısa Açıklama</label>
                            <textarea name="excerpt" class="form-control" rows="2" placeholder="Ön izleme metni (opsiyonel)"><?= sanitize($editPost['excerpt'] ?? ($_POST['excerpt'] ?? '')) ?></textarea>
                        </div>
                        <div class="col-12">
                            <label class="form-label">İçerik *</label>
                            <div id="blogEditor" style="height: 320px; background: #fff; border-radius: 8px;"></div>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Kapak Görseli</label>
                            <input type="file" name="image" class="form-control" accept="image/*">
                            <small class="text-muted">Opsiyonel. Maks 4MB.</small>
                        </div>
                        <?php if (!empty($editPost['image_path'])): ?>
                        <div class="col-md-6 d-flex align-items-center">
                            <img src="../<?= sanitize($editPost['image_path']) ?>" alt="Kapak görseli" class="img-fluid rounded" style="max-height:120px;">
                        </div>
                        <div class="col-12">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="remove_image" id="blog_remove_image">
                                <label class="form-check-label" for="blog_remove_image">Mevcut görseli kaldır</label>
                            </div>
                        </div>
                        <?php endif; ?>
                        <div class="col-md-4">
                            <label class="form-label">Meta Başlık</label>
                            <input type="text" name="meta_title" class="form-control" value="<?= sanitize($editPost['meta_title'] ?? ($_POST['meta_title'] ?? '')) ?>" placeholder="Otomatik oluşturulur">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Meta Açıklama</label>
                            <input type="text" name="meta_description" class="form-control" value="<?= sanitize($editPost['meta_description'] ?? ($_POST['meta_description'] ?? '')) ?>" placeholder="Otomatik oluşturulur">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Meta Anahtar Kelimeler</label>
                            <input type="text" name="meta_keywords" class="form-control" value="<?= sanitize($editPost['meta_keywords'] ?? ($_POST['meta_keywords'] ?? '')) ?>" placeholder="Otomatik oluşturulur">
                        </div>
                        <div class="col-12 text-end">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save me-2"></i><?= $editPost ? 'Yazıyı Güncelle' : 'Yazıyı Oluştur' ?>
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        <div class="col-lg-5">
            <div class="card h-100">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0"><i class="fas fa-list me-2"></i>Blog Yazıları</h5>
                    <span class="badge bg-secondary">Toplam <?= count($posts) ?> kayıt</span>
                </div>
                <div class="card-body p-0">
                    <div class="table-responsive">
                        <table class="table table-hover mb-0">
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>Başlık</th>
                                    <th>Durum</th>
                                    <th>Yazar</th>
                                    <th>Tarih</th>
                                    <th class="text-end">İşlemler</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($posts)): ?>
                                <tr>
                                    <td colspan="6" class="text-center text-muted py-4">Henüz blog yazısı oluşturulmadı.</td>
                                </tr>
                                <?php else: ?>
                                    <?php foreach ($posts as $post): ?>
                                    <tr>
                                        <td>#<?= (int)$post['id'] ?></td>
                                        <td>
                                            <strong><?= sanitize($post['title']) ?></strong>
                                            <?php if (!empty($post['excerpt'])): ?>
                                                <div class="text-muted small"><?= sanitize(shortenText($post['excerpt'], 60)) ?></div>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ((int)$post['is_published'] === 1): ?>
                                                <span class="badge bg-success">Yayında</span>
                                            <?php else: ?>
                                                <span class="badge bg-warning text-dark">Taslak</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?= $post['author_username'] ? sanitize($post['author_username']) : '<span class="text-muted">Belirsiz</span>' ?></td>
                                        <td>
                                            <div class="small text-muted">Oluştu: <?= date('d.m.Y', strtotime($post['created_at'])) ?></div>
                                            <div class="small text-muted">Güncel: <?= date('d.m.Y', strtotime($post['updated_at'])) ?></div>
                                        </td>
                                        <td class="text-end">
                                            <a href="blog.php?edit=<?= (int)$post['id'] ?>" class="btn btn-sm btn-outline-primary"><i class="fas fa-pen"></i></a>
                                            <form method="post" class="d-inline" onsubmit="return confirm('Bu blog yazısını silmek istediğinize emin misiniz?');">
                                                <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                                <input type="hidden" name="action" value="delete_post">
                                                <input type="hidden" name="post_id" value="<?= (int)$post['id'] ?>">
                                                <button type="submit" class="btn btn-sm btn-outline-danger"><i class="fas fa-trash"></i></button>
                                            </form>
                                            <?php if ((int)$post['is_published'] === 1): ?>
                                                <a href="<?= htmlspecialchars(routeUrl('blog_post', ['slug' => $post['slug']]), ENT_QUOTES, 'UTF-8') ?>" class="btn btn-sm btn-link" target="_blank"><i class="fas fa-up-right-from-square"></i></a>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>

