<?php
require_once '../config.php';
requireAdmin();

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_action'])) {
    $selectedIds = array_map('intval', $_POST['selected_ids'] ?? []);
    $selectedIds = array_values(array_filter($selectedIds));

    if (empty($selectedIds)) {
        flashMessage('warning', 'Toplu işlem için kategori seçmediniz.');
        redirect('categories.php');
    }

    $placeholders = implode(',', array_fill(0, count($selectedIds), '?'));

    try {
        if ($_POST['bulk_action'] === 'activate') {
            $stmt = $pdo->prepare("UPDATE categories SET is_active = 1 WHERE id IN ($placeholders)");
            $stmt->execute($selectedIds);
            flashMessage('success', 'Seçili kategoriler aktifleştirildi.');
        } elseif ($_POST['bulk_action'] === 'delete') {
            $pdo->beginTransaction();
            $imageStmt = $pdo->prepare("SELECT image, banner_image FROM categories WHERE id IN ($placeholders)");
            $imageStmt->execute($selectedIds);
            $images = $imageStmt->fetchAll();

            $pdo->prepare("DELETE FROM categories WHERE id IN ($placeholders)")->execute($selectedIds);
            $pdo->commit();

            foreach ($images as $row) {
                if (!empty($row['image'])) {
                    deleteUploadedFile($row['image']);
                }
                if (!empty($row['banner_image'])) {
                    deleteUploadedFile($row['banner_image']);
                }
            }

            flashMessage('success', 'Seçili kategoriler silindi.');
        }
    } catch (Throwable $th) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        flashMessage('danger', 'Toplu kategori işlemi sırasında hata oluştu: ' . $th->getMessage());
    }

    redirect('categories.php');
}

try {
    $pdo->query("ALTER TABLE categories ADD COLUMN image VARCHAR(255) NULL AFTER icon");
} catch (PDOException $e) {
    // sütun zaten varsa hata yoksayılır
}

try {
    $pdo->query("ALTER TABLE categories ADD COLUMN banner_image VARCHAR(255) NULL AFTER image");
} catch (PDOException $e) {
    // sütun zaten varsa hata yoksayılır
}

try {
    $pdo->query("ALTER TABLE categories ADD COLUMN parent_id INT NULL DEFAULT NULL AFTER id");
} catch (PDOException $e) {
    // sütun zaten varsa hata yoksayılır
}

try {
    $pdo->query("ALTER TABLE categories ADD COLUMN display_order INT NOT NULL DEFAULT 0 AFTER parent_id");
} catch (PDOException $e) {
    // sütun zaten varsa hata yoksayılır
}

try {
    $pdo->query("ALTER TABLE categories ADD COLUMN meta_title VARCHAR(150) NULL AFTER banner_image");
} catch (PDOException $e) {
    // sütun zaten varsa hata yoksayılır
}

try {
    $pdo->query("ALTER TABLE categories ADD COLUMN meta_description VARCHAR(255) NULL AFTER meta_title");
} catch (PDOException $e) {
    // sütun zaten varsa hata yoksayılır
}

try {
    $pdo->query("ALTER TABLE categories ADD INDEX idx_categories_parent (parent_id)");
} catch (PDOException $e) {
    // indeks zaten varsa hata yoksayılır
}

if (!function_exists('renderCategoryOptionTree')) {
    function renderCategoryOptionTree(array $tree, ?int $selected = null, ?int $exclude = null, int $depth = 0): void
    {
        foreach ($tree as $node) {
            if ($exclude !== null && (int)$node['id'] === $exclude) {
                continue;
            }
            $label = ($depth > 0 ? str_repeat('— ', $depth) : '') . sanitize($node['name']);
            $value = (int)$node['id'];
            $isSelected = $selected !== null && $selected === $value ? ' selected' : '';
            echo '<option value="' . $value . '"' . $isSelected . '>' . $label . '</option>';
            if (!empty($node['children'])) {
                renderCategoryOptionTree($node['children'], $selected, $exclude, $depth + 1);
            }
        }
    }
}

// Kategori ekleme/güncelleme
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik hatası.');
    } else {
        $id = isset($_POST['id']) ? (int)$_POST['id'] : null;
        $name = safe_trim($_POST['name'] ?? '');
        $providedSlug = safe_trim($_POST['slug'] ?? '');
        $slug = slugify($providedSlug !== '' ? $providedSlug : $name);
        $description = safe_trim($_POST['description'] ?? '');
        $icon = safe_trim($_POST['icon'] ?? 'fa-tags');
        $isActive = isset($_POST['is_active']) ? 1 : 0;
        $parentId = isset($_POST['parent_id']) && $_POST['parent_id'] !== '' ? (int)$_POST['parent_id'] : null;
        $displayOrder = isset($_POST['display_order']) ? (int)$_POST['display_order'] : 0;
        $metaTitle = safe_trim($_POST['meta_title'] ?? '');
        $metaDescription = safe_trim($_POST['meta_description'] ?? '');
        $deliveryType = isset($_POST['delivery_type']) && in_array($_POST['delivery_type'], ['auto', 'manual']) ? $_POST['delivery_type'] : 'auto';

        $currentImage = null;
        $currentBannerImage = null;

        $allCategories = getCategories($pdo, false);
        $categoryIndex = buildCategoryIndex($allCategories);

        if ($parentId && !isset($categoryIndex['byId'][$parentId])) {
            $parentId = null;
        }

        if ($id) {
            $stmt = $pdo->prepare("SELECT * FROM categories WHERE id = ?");
            $stmt->execute([$id]);
            $existing = $stmt->fetch();
            if (!$existing) {
                flashMessage('danger', 'Kategori bulunamadı.');
                redirect('categories.php');
            }
            $currentImage = $existing['image'] ?? null;
            $currentBannerImage = $existing['banner_image'] ?? null;

            if ($parentId === $id) {
                $parentId = null;
            }

            if ($parentId) {
                $descendants = getCategoryDescendantIds($categoryIndex, $id);
                if (in_array($parentId, $descendants, true)) {
                    flashMessage('danger', 'Bir kategori kendi alt kategorisine bağlanamaz.');
                    redirect('categories.php?edit=' . (int)$id);
                }
            }
        }

        // Görsel kaldırma kontrolü
        if (isset($_POST['remove_image']) && $_POST['remove_image'] === '1') {
            if ($currentImage) {
                deleteUploadedFile($currentImage);
            }
            $currentImage = null;
        }
        
        if (isset($_POST['remove_banner_image']) && $_POST['remove_banner_image'] === '1') {
            if ($currentBannerImage) {
                deleteUploadedFile($currentBannerImage);
            }
            $currentBannerImage = null;
        }

        try {
            // Sadece görsel kaldırılmadıysa yeni görsel yükle
            if (!isset($_POST['remove_image'])) {
                $currentImage = uploadImageFile($_FILES['image'] ?? [], 'categories', $currentImage);
            }
            if (!isset($_POST['remove_banner_image'])) {
                $currentBannerImage = uploadImageFile($_FILES['banner_image'] ?? [], 'categories/banners', $currentBannerImage);
            }
        } catch (Exception $e) {
            flashMessage('danger', 'Kategori görselleri yüklenemedi: ' . $e->getMessage());
            redirect('categories.php' . ($id ? '?edit=' . (int)$id : ''));
        }

        $metaTitle = $metaTitle !== '' ? $metaTitle : null;
        $metaDescription = $metaDescription !== '' ? $metaDescription : null;

        try {
            if ($id) {
                $stmt = $pdo->prepare("UPDATE categories SET name = ?, slug = ?, description = ?, icon = ?, image = ?, banner_image = ?, is_active = ?, parent_id = ?, display_order = ?, meta_title = ?, meta_description = ?, delivery_type = ? WHERE id = ?");
                $stmt->execute([
                    $name,
                    $slug,
                    $description,
                    $icon,
                    $currentImage,
                    $currentBannerImage,
                    $isActive,
                    $parentId,
                    $displayOrder,
                    $metaTitle,
                    $metaDescription,
                    $deliveryType,
                    $id
                ]);
                flashMessage('success', 'Kategori güncellendi.');
            } else {
                $stmt = $pdo->prepare("INSERT INTO categories (name, slug, description, icon, image, banner_image, is_active, parent_id, display_order, meta_title, meta_description, delivery_type) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([
                    $name,
                    $slug,
                    $description,
                    $icon,
                    $currentImage,
                    $currentBannerImage,
                    $isActive,
                    $parentId,
                    $displayOrder,
                    $metaTitle,
                    $metaDescription,
                    $deliveryType
                ]);
                flashMessage('success', 'Kategori eklendi.');
            }
            redirect('categories.php');
        } catch (PDOException $e) {
            flashMessage('danger', 'İşlem başarısız: ' . $e->getMessage());
        }
    }
}

// Kategori silme
if (isset($_GET['delete'])) {
    try {
        $stmt = $pdo->prepare("SELECT image, banner_image FROM categories WHERE id = ?");
        $stmt->execute([$_GET['delete']]);
        $cat = $stmt->fetch();
        if ($cat && !empty($cat['image'])) {
            deleteUploadedFile($cat['image']);
        }
        if ($cat && !empty($cat['banner_image'])) {
            deleteUploadedFile($cat['banner_image']);
        }

        $stmt = $pdo->prepare("DELETE FROM categories WHERE id = ?");
        $stmt->execute([$_GET['delete']]);
        flashMessage('success', 'Kategori silindi.');
        redirect('categories.php');
    } catch (PDOException $e) {
        flashMessage('danger', 'Silme başarısız. Bu kategoriye ait ürünler olabilir.');
    }
}

// Kategorileri getir
$allCategories = getCategories($pdo, false);
$categoryTree = buildCategoryTree($allCategories);
$categoryIndex = buildCategoryIndex($allCategories);

// Hiyerarşik kategori listesi oluştur
$hierarchicalCategories = [];
function flattenCategoryTree($categories, $level = 0) {
    $result = [];
    foreach ($categories as $category) {
        $category['level'] = $level;
        $result[] = $category;
        if (!empty($category['children'])) {
            $result = array_merge($result, flattenCategoryTree($category['children'], $level + 1));
        }
    }
    return $result;
}
$hierarchicalCategories = flattenCategoryTree($categoryTree);

// Düzenlenecek kategori
$editCategory = null;
if (isset($_GET['edit'])) {
    $stmt = $pdo->prepare("SELECT * FROM categories WHERE id = ?");
    $stmt->execute([$_GET['edit']]);
    $editCategory = $stmt->fetch();
}

$pageTitle = 'Kategori Yönetimi';
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="row">
        <div class="col-lg-4">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">
                        <i class="fas fa-<?= $editCategory ? 'edit' : 'plus' ?> me-2"></i>
                        <?= $editCategory ? 'Kategori Düzenle' : 'Yeni Kategori Ekle' ?>
                    </h5>
                </div>
                <div class="card-body">
                    <form method="POST" action="" enctype="multipart/form-data">
                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                        <?php if ($editCategory): ?>
                        <input type="hidden" name="id" value="<?= $editCategory['id'] ?>">
                        <?php endif; ?>
                        
                        <div class="mb-3">
                            <label class="form-label">Kategori Adı</label>
                            <input type="text" class="form-control" name="name" 
                                   value="<?= $editCategory ? sanitize($editCategory['name']) : '' ?>" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Slug (URL)</label>
                            <input type="text" class="form-control" name="slug" 
                                   value="<?= $editCategory ? sanitize($editCategory['slug']) : '' ?>" required>
                            <small class="text-muted">Örnek: oyun-pinleri</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Üst Kategori</label>
                            <select class="form-select" name="parent_id">
                                <option value="">Üst kategori yok</option>
                                <?php
                                $selectedParent = $editCategory['parent_id'] ?? null;
                                $currentId = $editCategory['id'] ?? null;
                                renderCategoryOptionTree($categoryTree, $selectedParent ? (int)$selectedParent : null, $currentId ? (int)$currentId : null);
                                ?>
                            </select>
                            <small class="text-muted">Üst kategori seçerseniz bu kategori vitrinde alt kategori olarak listelenir.</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Görüntülenme Sırası</label>
                            <input type="number" class="form-control" name="display_order" value="<?= $editCategory ? (int)$editCategory['display_order'] : 0 ?>">
                            <small class="text-muted">Küçük sayılar üst tarafta görünür.</small>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Açıklama</label>
                            <textarea class="form-control" name="description" rows="3"><?= $editCategory ? sanitize($editCategory['description']) : '' ?></textarea>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Meta Başlığı</label>
                            <input type="text" class="form-control" name="meta_title" value="<?= $editCategory ? sanitize($editCategory['meta_title']) : '' ?>" maxlength="150">
                            <small class="text-muted">Kategori sayfası için isteğe bağlı meta başlığı.</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Meta Açıklaması</label>
                            <textarea class="form-control" name="meta_description" rows="2" maxlength="255"><?= $editCategory ? sanitize($editCategory['meta_description']) : '' ?></textarea>
                            <small class="text-muted">Arama motorlarında gösterilecek kısa açıklama (en fazla 255 karakter).</small>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">İkon (Font Awesome)</label>
                            <input type="text" class="form-control" name="icon" 
                                   value="<?= $editCategory ? sanitize($editCategory['icon']) : 'fa-tag' ?>" required>
                            <small class="text-muted">Örnek: fa-gamepad, fa-gift</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Kategori Kart Görseli</label>
                            <?php if ($editCategory && !empty($editCategory['image'])): ?>
                            <div class="mb-2">
                                <img src="../<?= sanitize($editCategory['image']) ?>" alt="Kategori görseli" class="img-fluid rounded shadow-sm" style="max-height: 140px;">
                                <div class="form-check mt-2">
                                    <input class="form-check-input" type="checkbox" name="remove_image" value="1" id="remove_image">
                                    <label class="form-check-label text-danger" for="remove_image">
                                        <i class="fa-solid fa-trash"></i> Bu görseli kaldır
                                    </label>
                                </div>
                            </div>
                            <?php endif; ?>
                            <input type="file" class="form-control" name="image" accept="image/*">
                            <small class="text-muted">Kart görünümü için PNG, JPG, WEBP veya GIF (maks. 4MB). Görselsiz de kayıt yapılabilir.</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Üst Banner Görseli</label>
                            <?php if ($editCategory && !empty($editCategory['banner_image'])): ?>
                            <div class="mb-2">
                                <img src="../<?= sanitize($editCategory['banner_image']) ?>" alt="Kategori banner" class="img-fluid rounded shadow-sm" style="max-height: 160px;">
                                <div class="form-check mt-2">
                                    <input class="form-check-input" type="checkbox" name="remove_banner_image" value="1" id="remove_banner_image">
                                    <label class="form-check-label text-danger" for="remove_banner_image">
                                        <i class="fa-solid fa-trash"></i> Bu görseli kaldır
                                    </label>
                                </div>
                            </div>
                            <?php endif; ?>
                            <input type="file" class="form-control" name="banner_image" accept="image/*">
                            <small class="text-muted">Kategori sayfasının üst bölümünde kullanılacak geniş görsel (önerilen 1440x360). Görselsiz de kayıt yapılabilir.</small>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Ürün Kodları Teslimat Tipi</label>
                            <select class="form-select" name="delivery_type">
                                <option value="auto" <?= (!$editCategory || ($editCategory['delivery_type'] ?? 'auto') === 'auto') ? 'selected' : '' ?>>
                                    Otomatik Teslimat (Kod eklenince stoktan düşer ve sipariş otomatik tamamlanır)
                                </option>
                                <option value="manual" <?= ($editCategory && ($editCategory['delivery_type'] ?? 'auto') === 'manual') ? 'selected' : '' ?>>
                                    Manuel Teslimat (Admin manuel onaylar)
                                </option>
                            </select>
                            <small class="text-muted">Oto teslimat: Kod eklenince stoktan otomatik düşer. Manuel: Admin codes.php'den kendisi gönderir.</small>
                        </div>

                        <div class="mb-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="is_active" id="is_active"
                                       <?= $editCategory && $editCategory['is_active'] ? 'checked' : (!$editCategory ? 'checked' : '') ?>>
                                <label class="form-check-label" for="is_active">
                                    Aktif
                                </label>
                            </div>
                        </div>
                        
                        <div class="d-flex gap-2">
                            <button type="submit" class="btn btn-primary flex-fill">
                                <i class="fas fa-save me-2"></i>Kaydet
                            </button>
                            <?php if ($editCategory): ?>
                            <a href="categories.php" class="btn btn-secondary">
                                <i class="fas fa-times"></i>
                            </a>
                            <?php endif; ?>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        
        <div class="col-lg-8">
            <div class="card">
                <div class="card-header d-flex flex-wrap justify-content-between align-items-center gap-3">
                    <h5 class="mb-0">
                        <i class="fas fa-list me-2"></i>Kategoriler
                    </h5>
                    <form method="post" id="categoryBulkForm" class="bulk-action-toolbar" data-bulk-form="categoryBulkForm">
                        <div class="d-flex align-items-center gap-2">
                            <span class="text-muted small">Seçili kategoriler için:</span>
                            <button type="submit" name="bulk_action" value="activate" class="btn btn-sm btn-success" disabled data-bulk-action>
                                <i class="fas fa-check me-1"></i>Toplu Onayla
                            </button>
                            <button type="submit" name="bulk_action" value="delete" class="btn btn-sm btn-outline-danger" disabled data-bulk-action data-confirm="Seçili kategorileri silmek istediğinize emin misiniz?">
                                <i class="fas fa-trash-alt me-1"></i>Toplu Sil
                            </button>
                        </div>
                    </form>
                </div>
                <div class="card-body p-0">
                    <div class="table-responsive">
                        <table class="table table-hover table-bulk mb-0">
                            <thead>
                                <tr>
                                    <th class="text-center" style="width:48px;">
                                        <input type="checkbox" class="form-check-input" data-bulk-master="categoryBulkForm">
                                    </th>
                                    <th>ID</th>
                                    <th>Görsel</th>
                                    <th>Kategori</th>
                                    <th>Üst Kategori</th>
                                    <th>Slug</th>
                                    <th>Teslimat Tipi</th>
                                    <th>Sıra</th>
                                    <th>Durum</th>
                                    <th>İşlemler</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($hierarchicalCategories as $cat): ?>
                                <?php 
                                    $level = $cat['level'] ?? 0;
                                    $indent = str_repeat('&nbsp;&nbsp;&nbsp;&nbsp;', $level);
                                    $isChild = $level > 0;
                                    $rowClass = $isChild ? 'category-child-row' : 'category-parent-row';
                                ?>
                                <tr class="<?= $rowClass ?>" data-level="<?= $level ?>">
                                    <td class="text-center">
                                        <input type="checkbox" class="form-check-input" name="selected_ids[]" value="<?= $cat['id'] ?>" form="categoryBulkForm" data-bulk-target="categoryBulkForm">
                                    </td>
                                    <td><strong>#<?= $cat['id'] ?></strong></td>
                                    <td>
                                        <?php if (!empty($cat['image'])): ?>
                                            <img src="../<?= sanitize($cat['image']) ?>" alt="<?= sanitize($cat['name']) ?>" style="height: 48px; width: 48px; object-fit: cover; border-radius: 8px;">
                                        <?php else: ?>
                                            <span class="text-muted">-</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($isChild): ?>
                                            <span class="category-indent-marker">
                                                <i class="fas fa-level-up-alt fa-rotate-90 text-muted me-2"></i>
                                            </span>
                                        <?php endif; ?>
                                        <strong><?= sanitize($cat['name']) ?></strong>
                                        <?php if (!empty($cat['description'])): ?>
                                            <br>
                                            <small class="text-muted"><?= sanitize($cat['description']) ?></small>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if (!empty($cat['parent_id'])): ?>
                                            <?php 
                                                $parentCat = $categoryIndex['byId'][$cat['parent_id']] ?? null;
                                                if ($parentCat):
                                            ?>
                                            <span class="badge bg-secondary"><?= sanitize($parentCat['name']) ?></span>
                                            <?php endif; ?>
                                        <?php else: ?>
                                            <span class="text-muted">—</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><code><?= sanitize($cat['slug']) ?></code></td>
                                    <td>
                                        <?php 
                                            $deliveryType = $cat['delivery_type'] ?? 'auto';
                                            if ($deliveryType === 'auto'):
                                        ?>
                                            <span class="badge bg-success"><i class="fas fa-bolt me-1"></i>Otomatik</span>
                                        <?php else: ?>
                                            <span class="badge bg-warning text-dark"><i class="fas fa-hand me-1"></i>Manuel</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?= (int)$cat['display_order'] ?></td>
                                    <td>
                                        <?php if ($cat['is_active']): ?>
                                        <span class="badge bg-success">Aktif</span>
                                        <?php else: ?>
                                        <span class="badge bg-secondary">Pasif</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <div class="btn-group">
                                            <a href="?edit=<?= $cat['id'] ?>" class="btn btn-sm btn-outline-primary">
                                                <i class="fas fa-edit"></i>
                                            </a>
                                            <a href="?delete=<?= $cat['id'] ?>" class="btn btn-sm btn-outline-danger confirm-delete">
                                                <i class="fas fa-trash"></i>
                                            </a>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>
