<?php
require_once '../config.php';
requireAdmin();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $currentStatus = $_POST['status'] ?? ($_GET['status'] ?? 'pending');
    
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız. Lütfen tekrar deneyin.');
        redirect('deposits.php?status=' . urlencode($currentStatus));
    }
}

// Bekleyen onaylar için işlem
if (isset($_POST['approve_deposit'])) {
    $orderId = (int)$_POST['order_id'];
    $statusFilter = $_POST['status'] ?? 'pending';
    
    try {
        $pdo->beginTransaction();
        
        // Önce order bilgilerini alalım
        $orderStmt = $pdo->prepare('SELECT user_id, total_amount FROM orders WHERE id = ? AND order_type = "deposit" AND status = "pending"');
        $orderStmt->execute([$orderId]);
        $order = $orderStmt->fetch();
        
        if (!$order) {
            $pdo->rollBack();
            flashMessage('warning', 'İşlem bulunamadı veya zaten işlenmiş.');
            redirect('deposits.php?status=' . urlencode($statusFilter));
        }
        
        $userId = $order['user_id'];
        $amount = $order['total_amount'];
        
        // Order'ı completed yap
        $updateOrderStmt = $pdo->prepare('UPDATE orders SET status = "completed", updated_at = NOW() WHERE id = ?');
        $updateOrderStmt->execute([$orderId]);
        
        // Kullanıcının bakiyesine ekle
        $updateBalanceStmt = $pdo->prepare('UPDATE users SET balance = balance + ? WHERE id = ?');
        $updateBalanceStmt->execute([$amount, $userId]);
        
        // Güncellenen bakiyeyi kontrol et
        $checkBalanceStmt = $pdo->prepare('SELECT balance FROM users WHERE id = ?');
        $checkBalanceStmt->execute([$userId]);
        $newBalance = $checkBalanceStmt->fetchColumn();
        
        $pdo->commit();
        flashMessage('success', 'Para yatırma işlemi onaylandı. ' . formatPrice($amount) . ' bakiye eklendi. Yeni bakiye: ' . formatPrice($newBalance));
        
    } catch (Throwable $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        flashMessage('danger', 'Hata: ' . $e->getMessage());
    }
    
    redirect('deposits.php?status=' . urlencode($statusFilter));
}

if (isset($_POST['reject_deposit'])) {
    $orderId = (int)$_POST['order_id'];
    $statusFilter = $_POST['status'] ?? 'pending';
    
    try {
        $stmt = $pdo->prepare('UPDATE orders SET status = "cancelled", updated_at = NOW() WHERE id = ? AND order_type = "deposit" AND status = "pending"');
        $stmt->execute([$orderId]);
        
        if ($stmt->rowCount() > 0) {
            flashMessage('success', 'Para yatırma işlemi reddedildi.');
        } else {
            flashMessage('warning', 'İşlem bulunamadı.');
        }
    } catch (Throwable $e) {
        flashMessage('danger', 'Hata: ' . $e->getMessage());
    }
    
    redirect('deposits.php?status=' . urlencode($statusFilter));
}

// Durum filtresi
$allowedStatuses = ['pending', 'completed', 'cancelled', 'all'];
$statusFilter = $_GET['status'] ?? 'pending';
if (!in_array($statusFilter, $allowedStatuses, true)) {
    $statusFilter = 'pending';
}

// Özet istatistikler
$summaryStmt = $pdo->query('
    SELECT 
        status,
        COUNT(*) AS total_count,
        SUM(total_amount) AS total_amount,
        SUM(commission_amount) AS total_commission
    FROM orders
    WHERE order_type = "deposit"
    GROUP BY status
');
$summaryRows = $summaryStmt ? $summaryStmt->fetchAll(PDO::FETCH_ASSOC) : [];
$summary = [
    'pending' => ['count' => 0, 'amount' => 0.0, 'commission' => 0.0],
    'completed' => ['count' => 0, 'amount' => 0.0, 'commission' => 0.0],
    'cancelled' => ['count' => 0, 'amount' => 0.0, 'commission' => 0.0],
];
foreach ($summaryRows as $row) {
    $statusKey = strtolower((string)$row['status']);
    if (isset($summary[$statusKey])) {
        $summary[$statusKey]['count'] = (int)($row['total_count'] ?? 0);
        $summary[$statusKey]['amount'] = (float)($row['total_amount'] ?? 0);
        $summary[$statusKey]['commission'] = (float)($row['total_commission'] ?? 0);
    }
}

// Para yatırma listesi
$listSql = '
    SELECT o.*, u.username, u.email, u.full_name,
           t.id AS transaction_id, t.amount AS transaction_amount, t.description AS transaction_description
    FROM orders o
    JOIN users u ON o.user_id = u.id
    LEFT JOIN transactions t ON t.order_id = o.id AND t.type = "deposit"
    WHERE o.order_type = "deposit"
';
$listParams = [];
if ($statusFilter !== 'all') {
    $listSql .= ' AND o.status = ?';
    $listParams[] = $statusFilter;
}
$listSql .= ' ORDER BY o.created_at DESC LIMIT 100';

$listStmt = $pdo->prepare($listSql);
$listStmt->execute($listParams);
$deposits = $listStmt->fetchAll(PDO::FETCH_ASSOC);

$pageTitle = 'Para Yatırmaları';
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h4 class="mb-1"><i class="fas fa-wallet me-2"></i>Para Yatırmaları</h4>
            <p class="text-muted mb-0">Kullanıcıların bakiye yükleme işlemlerini görüntüleyin ve yönetin.</p>
        </div>
    </div>

    <div class="row g-3 mb-4">
        <div class="col-md-4">
            <div class="card h-100">
                <div class="card-body">
                    <h6 class="text-muted">Bekleyen İşlemler</h6>
                    <div class="metric-value text-warning"><?= number_format($summary['pending']['count'], 0, ',', '.') ?></div>
                    <div class="small text-muted">Toplam Tutar: <?= formatPrice($summary['pending']['amount']) ?></div>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card h-100">
                <div class="card-body">
                    <h6 class="text-muted">Tamamlanan İşlemler</h6>
                    <div class="metric-value text-success"><?= number_format($summary['completed']['count'], 0, ',', '.') ?></div>
                    <div class="small text-muted">Toplam Ödeme: <?= formatPrice($summary['completed']['amount']) ?></div>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card h-100">
                <div class="card-body">
                    <h6 class="text-muted">Toplam Komisyon</h6>
                    <div class="metric-value text-primary"><?= formatPrice($summary['completed']['commission']) ?></div>
                    <div class="small text-muted">Tamamlanan işlemlerden</div>
                </div>
            </div>
        </div>
    </div>

    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <div>
                <h5 class="mb-0">Para Yatırma İşlemleri</h5>
                <small class="text-muted">Filtreleyerek işlemleri görüntüleyin.</small>
            </div>
            <div class="btn-group">
                <?php 
                $labelMap = ['all' => 'Tümü', 'pending' => 'Bekleyen', 'completed' => 'Tamamlanan', 'cancelled' => 'İptal']; 
                foreach ($allowedStatuses as $statusOption): 
                ?>
                    <a href="deposits.php?status=<?= urlencode($statusOption) ?>" class="btn btn-sm btn<?= $statusFilter === $statusOption ? '' : '-outline' ?>-primary">
                        <?= sanitize($labelMap[$statusOption] ?? ucfirst($statusOption)) ?>
                    </a>
                <?php endforeach; ?>
            </div>
        </div>
        <div class="card-body p-0">
            <?php if (empty($deposits)): ?>
                <div class="p-4 text-center text-muted">Seçili kritere göre görüntülenecek para yatırma işlemi bulunamadı.</div>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead class="table-light">
                            <tr>
                                <th>#</th>
                                <th>Kullanıcı</th>
                                <th>Ödeme Yöntemi</th>
                                <th>İşlem Tarihi</th>
                                <th>Net Tutar</th>
                                <th>Komisyon</th>
                                <th>Brüt Tutar</th>
                                <th>Durum</th>
                                <th>İşlem</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($deposits as $deposit): ?>
                                <?php
                                    $status = strtolower((string)$deposit['status']);
                                    $badgeClass = 'bg-warning text-dark';
                                    $statusLabel = 'Beklemede';
                                    if ($status === 'completed') {
                                        $badgeClass = 'bg-success';
                                        $statusLabel = 'Tamamlandı';
                                    } elseif ($status === 'cancelled') {
                                        $badgeClass = 'bg-danger';
                                        $statusLabel = 'İptal';
                                    }
                                    
                                    // Transaction description'dan bilgileri çıkar
                                    $desc = $deposit['transaction_description'] ?? '';
                                    $senderName = '-';
                                    $note = '';
                                    if (preg_match('/Gönderen:\s*([^-]+)/', $desc, $matches)) {
                                        $senderName = trim($matches[1]);
                                    }
                                    if (preg_match('/Not:\s*(.+)/', $desc, $matches)) {
                                        $note = trim($matches[1]);
                                    }
                                ?>
                                <tr>
                                    <td><?= (int)$deposit['id'] ?></td>
                                    <td>
                                        <strong><?= sanitize($deposit['username']) ?></strong>
                                        <div class="small text-muted"><?= sanitize($deposit['email']) ?></div>
                                        <?php if (!empty($deposit['full_name'])): ?>
                                            <div class="small text-muted"><?= sanitize($deposit['full_name']) ?></div>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="badge bg-primary text-uppercase"><?= sanitize($deposit['payment_method'] ?? '-') ?></span>
                                        <?php if ($deposit['payment_method'] === 'bank_transfer' && $senderName !== '-'): ?>
                                            <div class="small text-muted">Gönderen: <?= sanitize($senderName) ?></div>
                                        <?php endif; ?>
                                    </td>
                                    <td><?= date('d.m.Y H:i', strtotime($deposit['created_at'])) ?></td>
                                    <td><strong><?= formatPrice($deposit['total_amount']) ?></strong></td>
                                    <td class="text-success">+<?= formatPrice($deposit['commission_amount']) ?></td>
                                    <td><strong><?= formatPrice($deposit['payable_amount']) ?></strong></td>
                                    <td>
                                        <span class="badge <?= $badgeClass ?>"><?= sanitize($statusLabel) ?></span>
                                        <?php if (!empty($deposit['updated_at']) && $deposit['updated_at'] !== $deposit['created_at']): ?>
                                            <div class="small text-muted"><?= date('d.m.Y H:i', strtotime($deposit['updated_at'])) ?></div>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($status === 'pending'): ?>
                                            <div class="d-flex gap-2">
                                                <form method="post" class="d-inline">
                                                    <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                                    <input type="hidden" name="order_id" value="<?= (int)$deposit['id'] ?>">
                                                    <input type="hidden" name="status" value="<?= sanitize($statusFilter) ?>">
                                                    <button type="submit" name="approve_deposit" class="btn btn-sm btn-success" title="Onayla"
                                                            onclick="return confirm('Bu işlemi onaylamak istediğinize emin misiniz? Kullanıcının bakiyesine <?= formatPrice($deposit['total_amount']) ?> eklenecektir.')">
                                                        <i class="fas fa-check"></i>
                                                    </button>
                                                </form>
                                                <form method="post" class="d-inline">
                                                    <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                                    <input type="hidden" name="order_id" value="<?= (int)$deposit['id'] ?>">
                                                    <input type="hidden" name="status" value="<?= sanitize($statusFilter) ?>">
                                                    <button type="submit" name="reject_deposit" class="btn btn-sm btn-danger" title="Reddet"
                                                            onclick="return confirm('Bu işlemi reddetmek istediğinize emin misiniz?')">
                                                        <i class="fas fa-times"></i>
                                                    </button>
                                                </form>
                                            </div>
                                        <?php else: ?>
                                            <a href="order-detail.php?id=<?= $deposit['id'] ?>" class="btn btn-sm btn-outline-primary">
                                                <i class="fas fa-eye"></i> Detay
                                            </a>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>
