<?php
require_once '../config.php';
requireAdmin();

$action = $_POST['action'] ?? $_GET['action'] ?? null;
$editId = isset($_GET['edit']) ? (int)$_GET['edit'] : 0;
$editItem = null;

if ($editId > 0) {
    $stmt = $pdo->prepare('SELECT * FROM discount_codes WHERE id = ?');
    $stmt->execute([$editId]);
    $editItem = $stmt->fetch();
}

// Kod oluştur
if ($action === 'save' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('discount-codes.php');
    }

    $code = strtoupper(trim($_POST['code'] ?? ''));
    $description = trim($_POST['description'] ?? '');
    $discountType = $_POST['discount_type'] ?? 'percentage';
    $discountValue = (float)($_POST['discount_value'] ?? 0);
    $minPurchase = (float)($_POST['min_purchase_amount'] ?? 0);
    $maxDiscount = !empty($_POST['max_discount_amount']) ? (float)$_POST['max_discount_amount'] : null;
    $usageLimit = !empty($_POST['usage_limit']) ? (int)$_POST['usage_limit'] : null;
    $startDate = !empty($_POST['start_date']) ? $_POST['start_date'] : null;
    $endDate = !empty($_POST['end_date']) ? $_POST['end_date'] : null;
    $isActive = isset($_POST['is_active']) ? 1 : 0;

    if ($code === '') {
        flashMessage('danger', 'Kod alanı zorunludur.');
        redirect('discount-codes.php');
    }

    if (!in_array($discountType, ['percentage', 'fixed'])) {
        $discountType = 'percentage';
    }

    if ($discountValue <= 0) {
        flashMessage('danger', 'İndirim değeri 0\'dan büyük olmalıdır.');
        redirect('discount-codes.php');
    }

    if ($discountType === 'percentage' && $discountValue > 100) {
        flashMessage('danger', 'Yüzde indirimi 100\'den büyük olamaz.');
        redirect('discount-codes.php');
    }

    try {
        if ($editId > 0 && $editItem) {
            // Güncelle
            $stmt = $pdo->prepare('
                UPDATE discount_codes 
                SET code = ?, description = ?, discount_type = ?, discount_value = ?, 
                    min_purchase_amount = ?, max_discount_amount = ?, usage_limit = ?,
                    start_date = ?, end_date = ?, is_active = ?, updated_at = NOW()
                WHERE id = ?
            ');
            $stmt->execute([
                $code, $description, $discountType, $discountValue,
                $minPurchase, $maxDiscount, $usageLimit,
                $startDate, $endDate, $isActive, $editId
            ]);
            flashMessage('success', 'İndirim kodu güncellendi.');
        } else {
            // Yeni ekle
            $stmt = $pdo->prepare('
                INSERT INTO discount_codes 
                (code, description, discount_type, discount_value, min_purchase_amount, 
                 max_discount_amount, usage_limit, start_date, end_date, is_active, created_by)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ');
            $stmt->execute([
                $code, $description, $discountType, $discountValue, $minPurchase,
                $maxDiscount, $usageLimit, $startDate, $endDate, $isActive, $_SESSION['user_id']
            ]);
            flashMessage('success', 'Yeni indirim kodu oluşturuldu.');
        }
    } catch (Exception $e) {
        flashMessage('danger', 'Hata: ' . $e->getMessage());
    }

    redirect('discount-codes.php');
}

// Sil
if ($action === 'delete' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('discount-codes.php');
    }

    $deleteId = (int)($_POST['id'] ?? 0);
    if ($deleteId > 0) {
        try {
            $pdo->prepare('DELETE FROM discount_codes WHERE id = ?')->execute([$deleteId]);
            flashMessage('success', 'İndirim kodu silindi.');
        } catch (Exception $e) {
            flashMessage('danger', 'Silme hatası: ' . $e->getMessage());
        }
    }
    redirect('discount-codes.php');
}

// Toggle
if ($action === 'toggle' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('discount-codes.php');
    }

    $toggleId = (int)($_POST['id'] ?? 0);
    if ($toggleId > 0) {
        try {
            $pdo->prepare('UPDATE discount_codes SET is_active = 1 - is_active WHERE id = ?')->execute([$toggleId]);
            flashMessage('success', 'Durum güncellendi.');
        } catch (Exception $e) {
            flashMessage('danger', 'Hata: ' . $e->getMessage());
        }
    }
    redirect('discount-codes.php');
}

// Liste
$stmt = $pdo->query('
    SELECT dc.*, COUNT(dcu.id) as total_uses 
    FROM discount_codes dc
    LEFT JOIN discount_code_usage dcu ON dc.id = dcu.discount_code_id
    GROUP BY dc.id
    ORDER BY dc.created_at DESC
');
$codes = $stmt->fetchAll();

$pageTitle = 'İndirim Kodları';
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="row g-4">
        <div class="col-lg-4">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">
                        <i class="fas fa-tag me-2"></i>
                        <?= $editItem ? 'İndirim Kodunu Düzenle' : 'Yeni İndirim Kodu' ?>
                    </h5>
                </div>
                <div class="card-body">
                    <form method="post">
                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                        <input type="hidden" name="action" value="save">

                        <div class="mb-3">
                            <label class="form-label">Kod *</label>
                            <input type="text" name="code" class="form-control text-uppercase" 
                                   value="<?= sanitize($editItem['code'] ?? '') ?>" 
                                   placeholder="Örn: INDIRIM50" required>
                            <small class="text-muted">Büyük harflerle yazılacak</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Açıklama</label>
                            <textarea name="description" class="form-control" rows="2"><?= sanitize($editItem['description'] ?? '') ?></textarea>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">İndirim Tipi *</label>
                            <select name="discount_type" class="form-select" required>
                                <option value="percentage" <?= (($editItem['discount_type'] ?? 'percentage') === 'percentage') ? 'selected' : '' ?>>Yüzde (%)</option>
                                <option value="fixed" <?= (($editItem['discount_type'] ?? '') === 'fixed') ? 'selected' : '' ?>>Sabit Tutar (TL)</option>
                            </select>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">İndirim Değeri *</label>
                            <input type="number" name="discount_value" class="form-control" step="0.01" min="0"
                                   value="<?= sanitize($editItem['discount_value'] ?? '') ?>" required>
                            <small class="text-muted">% için: 1-100 arası, TL için tutar</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Min. Sepet Tutarı (TL)</label>
                            <input type="number" name="min_purchase_amount" class="form-control" step="0.01" min="0"
                                   value="<?= sanitize($editItem['min_purchase_amount'] ?? '0') ?>">
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Maks. İndirim Tutarı (TL)</label>
                            <input type="number" name="max_discount_amount" class="form-control" step="0.01" min="0"
                                   value="<?= sanitize($editItem['max_discount_amount'] ?? '') ?>">
                            <small class="text-muted">% indirimler için üst limit</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Kullanım Limiti</label>
                            <input type="number" name="usage_limit" class="form-control" min="1"
                                   value="<?= sanitize($editItem['usage_limit'] ?? '') ?>">
                            <small class="text-muted">Boş = Sınırsız</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Başlangıç Tarihi</label>
                            <input type="datetime-local" name="start_date" class="form-control"
                                   value="<?= $editItem && $editItem['start_date'] ? date('Y-m-d\TH:i', strtotime($editItem['start_date'])) : '' ?>">
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Bitiş Tarihi</label>
                            <input type="datetime-local" name="end_date" class="form-control"
                                   value="<?= $editItem && $editItem['end_date'] ? date('Y-m-d\TH:i', strtotime($editItem['end_date'])) : '' ?>">
                        </div>

                        <div class="mb-3">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" name="is_active" id="is_active"
                                       <?= ($editItem ? (int)$editItem['is_active'] : 1) ? 'checked' : '' ?>>
                                <label class="form-check-label" for="is_active">Aktif</label>
                            </div>
                        </div>

                        <div class="d-grid gap-2">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save me-2"></i><?= $editItem ? 'Güncelle' : 'Oluştur' ?>
                            </button>
                            <?php if ($editItem): ?>
                                <a href="discount-codes.php" class="btn btn-secondary">
                                    <i class="fas fa-times me-2"></i>İptal
                                </a>
                            <?php endif; ?>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-lg-8">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0"><i class="fas fa-list me-2"></i>İndirim Kodları</h5>
                    <span class="badge bg-primary"><?= count($codes) ?> kod</span>
                </div>
                <div class="card-body p-0">
                    <div class="table-responsive">
                        <table class="table table-hover mb-0">
                            <thead>
                                <tr>
                                    <th>Kod</th>
                                    <th>Tip</th>
                                    <th>Değer</th>
                                    <th>Kullanım</th>
                                    <th>Geçerlilik</th>
                                    <th>Durum</th>
                                    <th class="text-end">İşlemler</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($codes)): ?>
                                    <tr>
                                        <td colspan="7" class="text-center text-muted py-4">Henüz indirim kodu eklenmemiş.</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($codes as $code): ?>
                                        <?php
                                        $now = new DateTime();
                                        $isExpired = false;
                                        $isLimitReached = false;
                                        $isNotStarted = false;

                                        if ($code['end_date'] && new DateTime($code['end_date']) < $now) {
                                            $isExpired = true;
                                        }
                                        if ($code['start_date'] && new DateTime($code['start_date']) > $now) {
                                            $isNotStarted = true;
                                        }
                                        if ($code['usage_limit'] && $code['used_count'] >= $code['usage_limit']) {
                                            $isLimitReached = true;
                                        }
                                        ?>
                                        <tr>
                                            <td>
                                                <strong class="text-primary"><?= sanitize($code['code']) ?></strong>
                                                <?php if ($code['description']): ?>
                                                    <br><small class="text-muted"><?= sanitize($code['description']) ?></small>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php if ($code['discount_type'] === 'percentage'): ?>
                                                    <span class="badge bg-info">% İndirim</span>
                                                <?php else: ?>
                                                    <span class="badge bg-success">Sabit TL</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php if ($code['discount_type'] === 'percentage'): ?>
                                                    <strong>%<?= number_format($code['discount_value'], 0) ?></strong>
                                                <?php else: ?>
                                                    <strong><?= formatPrice($code['discount_value']) ?></strong>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <span class="badge <?= $isLimitReached ? 'bg-danger' : 'bg-secondary' ?>">
                                                    <?= (int)$code['used_count'] ?> / <?= $code['usage_limit'] ? (int)$code['usage_limit'] : '∞' ?>
                                                </span>
                                            </td>
                                            <td>
                                                <?php if ($isExpired): ?>
                                                    <span class="badge bg-danger">Süresi Doldu</span>
                                                <?php elseif ($isNotStarted): ?>
                                                    <span class="badge bg-warning">Başlamadı</span>
                                                <?php elseif ($code['end_date']): ?>
                                                    <small><?= date('d.m.Y H:i', strtotime($code['end_date'])) ?></small>
                                                <?php else: ?>
                                                    <span class="text-muted">Süresiz</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php if ($code['is_active']): ?>
                                                    <span class="badge bg-success">Aktif</span>
                                                <?php else: ?>
                                                    <span class="badge bg-secondary">Pasif</span>
                                                <?php endif; ?>
                                            </td>
                                            <td class="text-end">
                                                <a href="discount-codes.php?edit=<?= (int)$code['id'] ?>" class="btn btn-sm btn-outline-primary">
                                                    <i class="fas fa-edit"></i>
                                                </a>
                                                <form method="post" class="d-inline" onsubmit="return confirm('Bu kodu silmek istediğinize emin misiniz?');">
                                                    <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                                    <input type="hidden" name="action" value="toggle">
                                                    <input type="hidden" name="id" value="<?= (int)$code['id'] ?>">
                                                    <button type="submit" class="btn btn-sm btn-outline-warning">
                                                        <i class="fas fa-power-off"></i>
                                                    </button>
                                                </form>
                                                <form method="post" class="d-inline" onsubmit="return confirm('Bu kodu silmek istediğinize emin misiniz?');">
                                                    <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                                    <input type="hidden" name="action" value="delete">
                                                    <input type="hidden" name="id" value="<?= (int)$code['id'] ?>">
                                                    <button type="submit" class="btn btn-sm btn-outline-danger">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </form>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>

