<?php
require_once '../config.php';
requireAdmin();

$action = $_POST['action'] ?? $_GET['action'] ?? null;
$editId = isset($_GET['edit']) ? (int)$_GET['edit'] : 0;
$editItem = null;

if ($editId > 0) {
    $stmt = $pdo->prepare('SELECT * FROM gift_codes WHERE id = ?');
    $stmt->execute([$editId]);
    $editItem = $stmt->fetch();
}

// Otomatik kod oluştur
function generateGiftCode($length = 12) {
    $characters = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
    $code = '';
    for ($i = 0; $i < $length; $i++) {
        $code .= $characters[rand(0, strlen($characters) - 1)];
    }
    return $code;
}

// Yeni kod oluştur
if ($action === 'save' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('gift-codes.php');
    }

    $code = strtoupper(trim($_POST['code'] ?? ''));
    if ($code === '') {
        $code = generateGiftCode();
    }
    
    $description = trim($_POST['description'] ?? '');
    $amount = (float)($_POST['amount'] ?? 0);
    $usageLimit = !empty($_POST['usage_limit']) ? (int)$_POST['usage_limit'] : 1;
    $startDate = !empty($_POST['start_date']) ? $_POST['start_date'] : null;
    $endDate = !empty($_POST['end_date']) ? $_POST['end_date'] : null;
    $isActive = isset($_POST['is_active']) ? 1 : 0;

    if ($amount <= 0) {
        flashMessage('danger', 'Bakiye miktarı 0\'dan büyük olmalıdır.');
        redirect('gift-codes.php');
    }

    try {
        if ($editId > 0 && $editItem) {
            // Güncelle
            $stmt = $pdo->prepare('
                UPDATE gift_codes 
                SET code = ?, description = ?, amount = ?, usage_limit = ?,
                    start_date = ?, end_date = ?, is_active = ?, updated_at = NOW()
                WHERE id = ?
            ');
            $stmt->execute([
                $code, $description, $amount, $usageLimit,
                $startDate, $endDate, $isActive, $editId
            ]);
            flashMessage('success', 'Hediye kodu güncellendi.');
        } else {
            // Yeni ekle
            $stmt = $pdo->prepare('
                INSERT INTO gift_codes 
                (code, description, amount, usage_limit, start_date, end_date, is_active, created_by)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ');
            $stmt->execute([
                $code, $description, $amount, $usageLimit,
                $startDate, $endDate, $isActive, $_SESSION['user_id']
            ]);
            flashMessage('success', 'Yeni hediye kodu oluşturuldu: ' . $code);
        }
    } catch (Exception $e) {
        flashMessage('danger', 'Hata: ' . $e->getMessage());
    }

    redirect('gift-codes.php');
}

// Sil
if ($action === 'delete' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('gift-codes.php');
    }

    $deleteId = (int)($_POST['id'] ?? 0);
    if ($deleteId > 0) {
        try {
            $pdo->prepare('DELETE FROM gift_codes WHERE id = ?')->execute([$deleteId]);
            flashMessage('success', 'Hediye kodu silindi.');
        } catch (Exception $e) {
            flashMessage('danger', 'Silme hatası: ' . $e->getMessage());
        }
    }
    redirect('gift-codes.php');
}

// Toggle
if ($action === 'toggle' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('gift-codes.php');
    }

    $toggleId = (int)($_POST['id'] ?? 0);
    if ($toggleId > 0) {
        try {
            $pdo->prepare('UPDATE gift_codes SET is_active = 1 - is_active WHERE id = ?')->execute([$toggleId]);
            flashMessage('success', 'Durum güncellendi.');
        } catch (Exception $e) {
            flashMessage('danger', 'Hata: ' . $e->getMessage());
        }
    }
    redirect('gift-codes.php');
}

// Liste
$stmt = $pdo->query('
    SELECT gc.*, u.username as used_by_username
    FROM gift_codes gc
    LEFT JOIN users u ON gc.used_by_user_id = u.id
    ORDER BY gc.created_at DESC
');
$codes = $stmt->fetchAll();

$pageTitle = 'Hediye Kodları';
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="row g-4">
        <div class="col-lg-4">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">
                        <i class="fas fa-gift me-2"></i>
                        <?= $editItem ? 'Hediye Kodunu Düzenle' : 'Yeni Hediye Kodu' ?>
                    </h5>
                </div>
                <div class="card-body">
                    <form method="post">
                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                        <input type="hidden" name="action" value="save">

                        <div class="mb-3">
                            <label class="form-label">Kod</label>
                            <input type="text" name="code" class="form-control text-uppercase" 
                                   value="<?= sanitize($editItem['code'] ?? '') ?>" 
                                   placeholder="Boş bırakın, otomatik oluşturulsun">
                            <small class="text-muted">Boş bırakırsanız otomatik oluşturulur</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Açıklama</label>
                            <textarea name="description" class="form-control" rows="2"><?= sanitize($editItem['description'] ?? '') ?></textarea>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Bakiye Miktarı (TL) *</label>
                            <input type="number" name="amount" class="form-control" step="0.01" min="0"
                                   value="<?= sanitize($editItem['amount'] ?? '') ?>" required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Kullanım Limiti</label>
                            <input type="number" name="usage_limit" class="form-control" min="1"
                                   value="<?= sanitize($editItem['usage_limit'] ?? '1') ?>">
                            <small class="text-muted">Genelde 1 (tek kullanımlık)</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Başlangıç Tarihi</label>
                            <input type="datetime-local" name="start_date" class="form-control"
                                   value="<?= $editItem && $editItem['start_date'] ? date('Y-m-d\TH:i', strtotime($editItem['start_date'])) : '' ?>">
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Bitiş Tarihi</label>
                            <input type="datetime-local" name="end_date" class="form-control"
                                   value="<?= $editItem && $editItem['end_date'] ? date('Y-m-d\TH:i', strtotime($editItem['end_date'])) : '' ?>">
                        </div>

                        <div class="mb-3">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" name="is_active" id="is_active"
                                       <?= ($editItem ? (int)$editItem['is_active'] : 1) ? 'checked' : '' ?>>
                                <label class="form-check-label" for="is_active">Aktif</label>
                            </div>
                        </div>

                        <div class="d-grid gap-2">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save me-2"></i><?= $editItem ? 'Güncelle' : 'Oluştur' ?>
                            </button>
                            <?php if ($editItem): ?>
                                <a href="gift-codes.php" class="btn btn-secondary">
                                    <i class="fas fa-times me-2"></i>İptal
                                </a>
                            <?php endif; ?>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-lg-8">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0"><i class="fas fa-list me-2"></i>Hediye Kodları</h5>
                    <span class="badge bg-primary"><?= count($codes) ?> kod</span>
                </div>
                <div class="card-body p-0">
                    <div class="table-responsive">
                        <table class="table table-hover mb-0">
                            <thead>
                                <tr>
                                    <th>Kod</th>
                                    <th>Tutar</th>
                                    <th>Kullanım</th>
                                    <th>Durum</th>
                                    <th>Kullanan</th>
                                    <th class="text-end">İşlemler</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($codes)): ?>
                                    <tr>
                                        <td colspan="6" class="text-center text-muted py-4">Henüz hediye kodu eklenmemiş.</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($codes as $code): ?>
                                        <?php
                                        $now = new DateTime();
                                        $isExpired = false;
                                        $isNotStarted = false;

                                        if ($code['end_date'] && new DateTime($code['end_date']) < $now) {
                                            $isExpired = true;
                                        }
                                        if ($code['start_date'] && new DateTime($code['start_date']) > $now) {
                                            $isNotStarted = true;
                                        }
                                        ?>
                                        <tr>
                                            <td>
                                                <strong class="text-primary"><?= sanitize($code['code']) ?></strong>
                                                <?php if ($code['description']): ?>
                                                    <br><small class="text-muted"><?= sanitize($code['description']) ?></small>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <strong class="text-success"><?= formatPrice($code['amount']) ?></strong>
                                            </td>
                                            <td>
                                                <span class="badge <?= $code['is_used'] ? 'bg-danger' : 'bg-secondary' ?>">
                                                    <?= (int)$code['used_count'] ?> / <?= (int)$code['usage_limit'] ?>
                                                </span>
                                            </td>
                                            <td>
                                                <?php if ($code['is_used']): ?>
                                                    <span class="badge bg-danger">Kullanıldı</span>
                                                <?php elseif (!$code['is_active']): ?>
                                                    <span class="badge bg-secondary">Pasif</span>
                                                <?php elseif ($isExpired): ?>
                                                    <span class="badge bg-warning">Süresi Doldu</span>
                                                <?php elseif ($isNotStarted): ?>
                                                    <span class="badge bg-info">Başlamadı</span>
                                                <?php else: ?>
                                                    <span class="badge bg-success">Aktif</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php if ($code['used_by_username']): ?>
                                                    <span class="text-primary"><?= sanitize($code['used_by_username']) ?></span>
                                                    <br><small class="text-muted"><?= $code['used_at'] ? date('d.m.Y H:i', strtotime($code['used_at'])) : '' ?></small>
                                                <?php else: ?>
                                                    <span class="text-muted">-</span>
                                                <?php endif; ?>
                                            </td>
                                            <td class="text-end">
                                                <?php if (!$code['is_used']): ?>
                                                    <a href="gift-codes.php?edit=<?= (int)$code['id'] ?>" class="btn btn-sm btn-outline-primary">
                                                        <i class="fas fa-edit"></i>
                                                    </a>
                                                    <form method="post" class="d-inline">
                                                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                                        <input type="hidden" name="action" value="toggle">
                                                        <input type="hidden" name="id" value="<?= (int)$code['id'] ?>">
                                                        <button type="submit" class="btn btn-sm btn-outline-warning">
                                                            <i class="fas fa-power-off"></i>
                                                        </button>
                                                    </form>
                                                <?php endif; ?>
                                                <form method="post" class="d-inline" onsubmit="return confirm('Bu kodu silmek istediğinize emin misiniz?');">
                                                    <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                                    <input type="hidden" name="action" value="delete">
                                                    <input type="hidden" name="id" value="<?= (int)$code['id'] ?>">
                                                    <button type="submit" class="btn btn-sm btn-outline-danger">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </form>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>

